package org.terracotta.ehcachedx.monitor.probe;

import static org.terracotta.ehcachedx.monitor.probe.CacheManagerService.createMetaDataDateFormat;

import java.util.Collections;
import java.util.Date;
import java.util.SortedSet;
import java.util.TreeSet;

import org.terracotta.ehcachedx.monitor.probe.counter.sampled.SampledCounter;
import org.terracotta.ehcachedx.monitor.probe.counter.sampled.TimeStampedCounterValue;
import org.terracotta.ehcachedx.monitor.util.StringUtils;

/**
 * A Statistic type.
 * 
 * Keep this in sync with the Statistic in Monitor
 */
public enum Statistic {
	removed, put, updated, expired, evicted, hit, hitMemory, hitOffHeap, hitDisk, hitRatio(
			true), offHeapUsage(true), miss, missMemory, missOffHeap, missDisk, totalCount(
			0) {
		@Override
		long normalize(SampledCacheStatistics stats, long value) {
			return value;
		}
	},
	diskCount(0) {
		@Override
		long normalize(SampledCacheStatistics stats, long value) {
			return value;
		}
	},
	offHeapCount(0) {
		@Override
		long normalize(SampledCacheStatistics stats, long value) {
			return value;
		}
	},
	memoryCount(0) {
		@Override
		long normalize(SampledCacheStatistics stats, long value) {
			return value;
		}
	},
	memorySize(0) {
		@Override
		long normalize(SampledCacheStatistics stats, long value) {
			return value;
		}
	},
	offHeapSize(0) {
		@Override
		long normalize(SampledCacheStatistics stats, long value) {
			return value;
		}
	},
	offHeapUsed(0) {
		@Override
		long normalize(SampledCacheStatistics stats, long value) {
			return value;
		}
	};

	private final static SortedSet<String> NAMES;

	static {
		SortedSet<String> names = new TreeSet<String>();
		for (Statistic name : Statistic.values()) {
			names.add(name.name());
		}

		NAMES = Collections.unmodifiableSortedSet(names);
	}

	public static SortedSet<String> getNames() {
		return NAMES;
	}

	private boolean average;
	private int decimals;

	Statistic() {
		this(1, false);
	}

	Statistic(int decimals) {
		this(decimals, false);
	}

	Statistic(boolean average) {
		this(1, average);
	}

	Statistic(int decimals, boolean average) {
		this.decimals = decimals;
		this.average = average;
	}

	public int getDecimals() {
		return decimals;
	}

	public boolean isAverage() {
		return average;
	}

	long normalize(SampledCacheStatistics stats, long value) {
		// TODO The constant 10 looks suspicious here.
		return (value * 10) / stats.getCounterConfig().getIntervalSecs();
	}

	private String convertSample(SampledCacheStatistics stats, long value) {
		String valueWithoutDecimals = String.valueOf(normalize(stats, value));
		return StringUtils.introduceDecimals(valueWithoutDecimals, decimals);
	}

	String getSample(SampledCacheStatistics stats) {
		SampledCounter counter = stats.getCounters().get(this);
		if (null == counter) {
			return null;
		}
		long value = counter.getMostRecentSample().getCounterValue();
		return convertSample(stats, value);
	}

	// TODO Don't expect this to scale well in terms of speed or memory.

	SampleHistoryEntry[] getSampleHistory(SampledCacheStatistics stats) {
		SampledCounter counter = stats.getCounters().get(this);
		TimeStampedCounterValue[] values = counter.getAllSampleValues();
		SampleHistoryEntry[] result = new SampleHistoryEntry[values.length];
		for (int i = 0; i < values.length; i++) {
			TimeStampedCounterValue value = values[values.length - (i + 1)];
			result[i] = new SampleHistoryEntry(convertSample(stats,
					value.getCounterValue()), createMetaDataDateFormat()
					.format(new Date(value.getTimeStamp())));
		}
		return result;
	}

}
