/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */
package org.terracotta.ehcachedx.monitor.probe;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.ehcachedx.license.LicenseResolver;
import org.terracotta.ehcachedx.monitor.common.AuthenticationService;
import org.terracotta.ehcachedx.monitor.common.DxException;
import org.terracotta.ehcachedx.monitor.common.GeneralService;
import org.terracotta.ehcachedx.monitor.common.LicenseService;
import org.terracotta.ehcachedx.monitor.common.LifeCycleService;
import org.terracotta.ehcachedx.monitor.common.Service;
import org.terracotta.ehcachedx.monitor.common.rest.RestConstants;
import org.terracotta.ehcachedx.monitor.common.rest.RestResponse;
import org.terracotta.ehcachedx.monitor.common.handler.AuthenticationHandler;
import org.terracotta.ehcachedx.monitor.common.handler.DxResourceHandler;
import org.terracotta.ehcachedx.monitor.common.handler.LicenseHandler;
import org.terracotta.ehcachedx.monitor.common.handler.RedirectHandler;
import org.terracotta.ehcachedx.monitor.common.handler.RestHandler;
import org.terracotta.ehcachedx.monitor.common.handler.RestHandlerList;
import org.terracotta.ehcachedx.monitor.probe.counter.sampled.SampledCounterConfig;
import org.terracotta.ehcachedx.monitor.probe.util.PortUtils;
import org.terracotta.ehcachedx.monitor.util.ExceptionUtils;
import org.terracotta.ehcachedx.monitor.util.HandlerUtils;
import org.terracotta.ehcachedx.org.mortbay.jetty.Server;
import org.terracotta.ehcachedx.org.mortbay.jetty.handler.HandlerList;
import org.terracotta.ehcachedx.org.mortbay.jetty.handler.ResourceHandler;
import org.terracotta.ehcachedx.org.mortbay.resource.Resource;
import org.terracotta.ehcachedx.org.mortbay.thread.QueuedThreadPool;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;
import java.util.StringTokenizer;

import static org.terracotta.ehcachedx.monitor.util.RestProxy.registerProbe;

/**
 * Cannot move this to probe without breaking one test?
 */
public class ProbeDxService implements Service {

    private static final Logger LOG = LoggerFactory.getLogger(ProbeDxService.class);

    //Not configurable in probe
    public static int PROBE_SAMPLING_INTERVAL_SECONDS = 10;
    public static final int PROBE_SAMPLING_HISTORY = 10;


    private final String address;
    private final String probeName;
    private final boolean bindStrictly;
    private final String monitorAddress;
    private final Integer monitorPort;
    private final RedirectHandler redirectHandler;
    private final LicenseHandler licenseHandler;
    private final LicenseService licenseService;
    private final LifeCycleService lifeCycleService;

    private final GeneralService generalService;
    private final Thread registerWithServer;
    private RestHandler probeHandler;

    private ProbeService probeService;
    private String probeHandlerName;
    private boolean daemon = false;
    private Server httpServer;
    private Integer port;

    public ProbeDxService(String address, int port) throws UnknownHostException {
        this(new Config().address(address).port(port));
    }

    public ProbeDxService(final Config config) throws UnknownHostException {
        this.probeName = config.getName();
        this.bindStrictly = config.isBindStrictly();

        this.licenseHandler = new LicenseHandler();

        Map<String, String> redirects = new HashMap<String, String>();
        redirects.put("/", HandlerUtils.MONITOR_PATH_PREFIX);
        this.redirectHandler = new RedirectHandler(redirects);

        this.generalService = new GeneralService(PROBE_SAMPLING_HISTORY, PROBE_SAMPLING_INTERVAL_SECONDS);
        this.lifeCycleService = new LifeCycleService(this, licenseHandler);
        this.licenseService = new LicenseService();

        if (config.getAddress() != null && config.getAddress().length() != 0) {
            this.address = config.getAddress();
        } else {
            this.address = calculateHostAddress();
        }

        this.port = config.getPort();
        if (port == null || 0 == port) {
            assignFreePort(false);
        }

        this.monitorAddress = config.getServerAddress();
        this.monitorPort = config.getServerPort();
        this.registerWithServer = new Thread(new RegisterWithServer());
    }

    public synchronized boolean isDaemon() {
        return daemon;
    }

    public synchronized void setDaemon(boolean daemon) {
        this.daemon = daemon;
    }

    /**
     * Hardcoded 10 seconds and 10 history items with reset on sample and 0 starting value
     * If the monitor samples less than once per 100 seconds data will be lost!
     */
    public static SampledCounterConfig createCounterConfig() {


        return new SampledCounterConfig(PROBE_SAMPLING_INTERVAL_SECONDS, PROBE_SAMPLING_HISTORY, true, 0L);

    }

    public synchronized void init(String handlerName, RestHandler handler) {
        this.probeHandlerName = handlerName;
        this.probeHandler = handler;
        this.probeService = new ProbeService();
    }

    public String getAddress() {
        return address;
    }

    public synchronized Integer getPort() {
        return port;
    }

    public synchronized void assignFreePort(boolean forced) throws IllegalStateException {
        if (httpServer != null && httpServer.isStarting()) {
            throw new IllegalStateException("Cannot change the port of an already started DX service.");
        }

        port = PortUtils.getFreePort();
        if (forced) {
            LOG.warn("Resolving port conflict by automatically using a free TCP/IP port to listen on: " + port);
        } else {
            LOG.debug("Automatically finding a free TCP/IP port to listen on: " + port);
        }
    }

    private String calculateHostAddress() throws UnknownHostException {
        return InetAddress.getLocalHost().getHostAddress();
    }

    public synchronized void start() throws DxException {
        if (httpServer != null) {
            throw new DxException("The DX service has already been started beforehand.");
        }

        httpServer = new Server(bindStrictly ? address : null, port);

        // ensure that the threads are running as daemon threads if needed
        QueuedThreadPool threadPool = new QueuedThreadPool();
        threadPool.setMaxThreads(3);
        threadPool.setDaemon(daemon);
        httpServer.setThreadPool(threadPool);
        httpServer.setStopAtShutdown(true);

        registerHandlers();

        startBackend();

        // register probe with monitor
        registerWithServer.start();
    }

    /**
     * @throws org.terracotta.ehcachedx.monitor.common.DxException
     *
     */
    private synchronized void startBackend() throws DxException {
        try {
            httpServer.start();
            LOG.info("Started probe at http://" + address + ":" + port + HandlerUtils.MONITOR_PATH_PREFIX + RestHandlerList.LIST_METHODS_PATH);
            waitUntilRunning();
        } catch (Exception e) {
            throw new DxException("Unable to start Probe DX service for address '" + address + "' and port '" + port + "'.", e);
        }
    }

    private synchronized void registerHandlers() {
        HandlerList handlers = new HandlerList();
        handlers.addHandler(redirectHandler);

        // register the class path resource handler
        ResourceHandler resourceHandler = new DxResourceHandler();
        resourceHandler.setBaseResource(Resource.newClassPathResource("content"));

        handlers.addHandler(resourceHandler);

        RestHandlerList restHandlers = new RestHandlerList();

            restHandlers.addHandler(new RestHandler(probeService));
        restHandlers.addHandler(new RestHandler(generalService));

        restHandlers.addHandler(licenseHandler);
        restHandlers.addHandler(new RestHandler(licenseService));

        if (probeHandler != null) {
            restHandlers.addHandler(probeHandler);
        }
        restHandlers.addHandler(new RestHandler(lifeCycleService));

        handlers.addHandler(restHandlers);
        httpServer.setHandler(handlers);
    }

    private synchronized boolean usesExternalServer() {
        return monitorAddress != null && monitorPort != null &&
                (!monitorAddress.equals(address) || !monitorPort.equals(port));
    }

    public void waitUntilRunning() throws InterruptedException {
        while (!isRunning()) {
            Thread.sleep(100);
        }
    }

    public synchronized boolean isRunning() {
        if (httpServer == null) {
            return false;
        }

        return httpServer.isStarted() && !registerWithServer.isAlive();
    }

    public synchronized void stop() throws DxException {
        if (httpServer != null) {
            try {
                httpServer.stop();
                if (probeService != null) {
                    LOG.info("Stopped probe for " + address + ":" + port);
                } else {
                    LOG.info("Stopped monitor for " + address + ":" + port);
                }
            } catch (Exception e) {
                throw new DxException("Unable to stop the DX service.", e);
            }
            httpServer = null;

            lifeCycleService.stop();
        }
    }

    public LicenseHandler getLicenseHandler() {
        return licenseHandler;
    }


    public static class Config {
        private String address;
        private Integer port;
        private String name;
        private String monitorAddress;
        private Integer monitorPort;
        private boolean bindStrictly;
        private boolean memoryMeasurement;

        public Config() {
            // default constructor
        }

        public Config(String address, Integer port, String name, String monitorAddress, Integer monitorPort, boolean bindStrictly) {
            address(address);
            port(port);
            name(name);
            monitorAddress(monitorAddress);
            monitorPort(monitorPort);
            bindStrictly(bindStrictly);
            memoryMeasurement(true);
        }

        public Config(String address, Integer port, String name, String monitorAddress, Integer monitorPort,
                      boolean bindStrictly, boolean memoryMeasurement) {
            address(address);
            port(port);
            name(name);
            monitorAddress(monitorAddress);
            monitorPort(monitorPort);
            bindStrictly(bindStrictly);
            memoryMeasurement(memoryMeasurement);
        }

        Config address(String address) {
            this.address = address;
            return this;
        }

        Config port(Integer port) {
            this.port = port;
            return this;
        }

        Config name(String name) {
            this.name = name;
            return this;
        }

        Config monitorAddress(String monitorAddress) {
            this.monitorAddress = monitorAddress;
            return this;
        }

        Config monitorPort(Integer monitorPort) {
            this.monitorPort = monitorPort;
            return this;
        }

        Config bindStrictly(boolean bindStrictly) {
            this.bindStrictly = bindStrictly;
            return this;
        }

        Config memoryMeasurement(boolean memoryMeasurement) {
            this.memoryMeasurement = memoryMeasurement;
            return this;
        }

        public String getAddress() {
            return address;
        }

        public Integer getPort() {
            return port;
        }

        public String getName() {
            return name;
        }

        public String getServerAddress() {
            return monitorAddress;
        }

        public Integer getServerPort() {
            return monitorPort;
        }

        public boolean isBindStrictly() {
            return bindStrictly;
        }

        public boolean isMemoryMeasurement() {
            return memoryMeasurement;
        }
    }

    private class RegisterWithServer implements Runnable {
        public void run() {
            String response = null;

            final String localAddress;
            final Integer localPort;
            final String localProbeName;
            final String localProbeHandlerName;
            final String localServerAddress;
            final Integer localServerPort;

            synchronized (ProbeDxService.this) {
                localAddress = address;
                localPort = port;
                localProbeName = probeName;
                localProbeHandlerName = probeHandlerName;
                localServerAddress = monitorAddress;
                localServerPort = monitorPort;
            }

            if (usesExternalServer()) {
                // register with the monitor if needed
                try {
                    response = registerProbe(localAddress, localPort, localProbeName, localProbeHandlerName, localServerAddress, localServerPort);
                    LOG.info("ProbeDXService Registered with monitor at address '" + localServerAddress + "' and port '" + localServerPort + "'");
                } catch (IOException e) {
                    LOG.error("Couldn't register ProbeDXService with monitor at address '" + localServerAddress + "' and port '" + localServerPort + "'\n" + ExceptionUtils.getExceptionStackTrace(e));
                }
            }

            String license = null;
            if (response != null && !RestResponse.OK.equals(response)) {
                StringTokenizer tokenizer = new StringTokenizer(response.trim(), ":\n");
                while (tokenizer.hasMoreTokens()) {
                    String key = tokenizer.nextToken();
                    if (tokenizer.hasMoreTokens()) {
                        String value = tokenizer.nextToken();
                        synchronized (ProbeDxService.this) {
                            if (RestConstants.ELEMENT_LICENSE.equals(key.trim())) {
                                license = value.trim();
                            }
                        }
                    }
                }
            }
            if (license == null) {
                licenseHandler.registerLicense(null);
                LOG.info("Null license registered");
            } else {
                licenseHandler.registerLicense(LicenseResolver.BASE64_PREFIX + license);
                LOG.info("License obtained from Monitor and registered in probe: " + license);                
            }
        }
    }
}