/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */
package org.terracotta.ehcachedx.monitor.probe;

import net.sf.ehcache.Element;

import java.sql.Date;
import java.util.Collections;
import java.util.SortedSet;
import java.util.TreeSet;

import static org.terracotta.ehcachedx.monitor.probe.CacheManagerService.createMetaDataDateFormat;

/**
 * Used only by CacheManagerService
 */
public abstract class ElementMetaData {
    private final static SortedSet<String> NAMES;

    static {
        SortedSet<String> names = new TreeSet<String>();
        for (MetaData metaData : MetaData.values()) {
            names.add(metaData.name());
        }

        NAMES = Collections.unmodifiableSortedSet(names);
    }

    public static String getMetaData(String name, Element element) {
        String result;
        try {
            MetaData metaData = MetaData.valueOf(name);
            result = metaData.getMetaData(element);
        } catch (IllegalArgumentException e) {
            result = null;
        }
        return result;
    }

    public static SortedSet<String> getNames() {
        return NAMES;
    }

    private enum MetaData {
        valueType {
            @Override
            public String getMetaData(Element element) {
                if (element.getObjectValue() != null) {
                    return element.getObjectValue().getClass().getName();
                } else {
                    return "";
                }
            }},

        created {
            @Override
            public String getMetaData(Element element) {
                return formatTimestamp(element.getCreationTime());
            }},

        accessed {
            @Override
            public String getMetaData(Element element) {
                return formatTimestamp(element.getLastAccessTime());
            }},

        updated {
            @Override
            public String getMetaData(Element element) {
                return formatTimestamp(element.getLastUpdateTime());
            }},

        expired {
            @Override
            public String getMetaData(Element element) {
                return String.valueOf(element.isExpired());
            }},

        eternal {
            @Override
            public String getMetaData(Element element) {
                return String.valueOf(element.isEternal());
            }},

        timeToIdle {
            @Override
            public String getMetaData(Element element) {
                return String.valueOf(element.getTimeToIdle());
            }},

        timeToLive {
            @Override
            public String getMetaData(Element element) {
                return String.valueOf(element.getTimeToLive());
            }},

        expirationTime {
            @Override
            public String getMetaData(Element element) {
                if (Long.MAX_VALUE == element.getExpirationTime()) {
                    return "";
                }
                return formatTimestamp(element.getExpirationTime());
            }},

        hits {
            @Override
            public String getMetaData(Element element) {
                return String.valueOf(element.getHitCount());
            }};

        abstract public String getMetaData(Element element);

        protected String formatTimestamp(long timestamp) {
            if (0 == timestamp) {
                return "";
            }
            return createMetaDataDateFormat().format(new Date(timestamp));
        }
    }
}