/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */
package org.terracotta.ehcachedx.monitor.probe;

import java.util.Collections;
import java.util.SortedSet;
import java.util.TreeSet;

import net.sf.ehcache.Ehcache;
import net.sf.ehcache.store.MemoryStoreEvictionPolicy;

/**
 * Used only by CacheManagerService
 */
public abstract class CacheConfig {
	private final static SortedSet<String> NAMES;

	static {
		SortedSet<String> names = new TreeSet<String>();
		for (Config config : Config.values()) {
			names.add(config.name());
		}

		NAMES = Collections.unmodifiableSortedSet(names);
	}

	public static Object getConfigValue(String name, Ehcache cache) {
		Object result;
		try {
			Config config = Config.valueOf(name);
			result = config.getConfigValue(cache);
		} catch (IllegalArgumentException e) {
			result = null;
		}
		return result;
	}

	public static SortedSet<String> getNames() {
		return NAMES;
	}

	private enum Config {
		statistics {
			@Override
			public Object getConfigValue(Ehcache cache) {
				try {
					return cache.getCacheConfiguration().getStatistics();
				} catch (NoSuchMethodError e) {
					return "na";
				}
			}
		},
		evictionPolicy {
			@Override
			public Object getConfigValue(Ehcache cache) {
				MemoryStoreEvictionPolicy policy = cache
						.getCacheConfiguration().getMemoryStoreEvictionPolicy();
				return policy.toString();
			}
		},
		maxElementsInMemory {
			@Override
			public Object getConfigValue(Ehcache cache) {
				return cache.getCacheConfiguration().getMaxElementsInMemory();
			}
		},
		maxElementsOnDisk {
			@Override
			public Object getConfigValue(Ehcache cache) {
				return cache.getCacheConfiguration().getMaxElementsOnDisk();
			}
		},
		timeToLiveSeconds {
			@Override
			public Object getConfigValue(Ehcache cache) {
				return cache.getCacheConfiguration().getTimeToLiveSeconds();
			}
		},
		timeToIdleSeconds {
			@Override
			public Object getConfigValue(Ehcache cache) {
				return cache.getCacheConfiguration().getTimeToIdleSeconds();
			}
		},
		eternal {
			@Override
			public Object getConfigValue(Ehcache cache) {
				return cache.getCacheConfiguration().isEternal();
			}
		},
		diskPersistent {
			@Override
			public Object getConfigValue(Ehcache cache) {
				return cache.getCacheConfiguration().isDiskPersistent();
			}
		},
		overflowToDisk {
			@Override
			public Object getConfigValue(Ehcache cache) {
				return cache.getCacheConfiguration().isOverflowToDisk();
			}
		},
		overflowToOffHeap {
			@Override
			public Object getConfigValue(Ehcache cache) {
				try {
					return cache.getCacheConfiguration().isOverflowToOffHeap();
				} catch (NoSuchMethodError e) {
					return "na";
				}
			}
		},
		maxMemoryOffHeap {
			@Override
			public Object getConfigValue(Ehcache cache) {
				try {
					return cache.getCacheConfiguration().getMaxMemoryOffHeap();
				} catch (NoSuchMethodError e) {
					return 0;
				}
			}
		},
		diskExpiryThreadIntervalSeconds {
			@Override
			public Object getConfigValue(Ehcache cache) {
				return cache.getCacheConfiguration()
						.getDiskExpiryThreadIntervalSeconds();
			}
		},
		diskSpoolBufferSizeMB {
			@Override
			public Object getConfigValue(Ehcache cache) {
				return cache.getCacheConfiguration().getDiskSpoolBufferSizeMB();
			}
		};

		abstract public Object getConfigValue(Ehcache cache);
	}
}