/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */
package org.terracotta.ehcachedx.monitor.probe.counter.sampled;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.Callable;


public class PullSampledCounterImpl extends SampledCounterImpl {

    private static final Logger LOG = LoggerFactory.getLogger(PullSampledCounterImpl.class);

    private final boolean calculateDifference;
    private final Callable<Long> callable;

    private volatile long previous = 0;

    public PullSampledCounterImpl(PullSampledCounterConfig config) {
        super(config);
        this.calculateDifference = config.getCalculateDifference();
        this.callable = config.getCallable();
    }

    @Override
    public synchronized void setValue(long newValue) {
        previous = value.get();
        value.set(newValue);
    }

    @Override
    protected synchronized void recordSample() {
        if (callable != null) {
            try {
                Long result = callable.call();
                if (null == result) {
                    setValue(0L);
                } else {
                    setValue(result);
                }
            } catch (Exception e) {
                LOG.error(e.getMessage());
            }
        }

        final long now = System.currentTimeMillis();
        TimeStampedCounterValue timedSample;
        if (calculateDifference) {
            synchronized (this) {
                // the value that's pulled by the callable is a ever-increasing counter, so the sample is the
                // difference between the latest pulled value and the previous one
                timedSample = new TimeStampedCounterValue(now, value.get() - previous);
            }
        } else {
            timedSample = new TimeStampedCounterValue(now, value.get());
        }

        history.push(timedSample);
    }
}
