/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */
package org.terracotta.ehcachedx.monitor.probe;

import net.sf.ehcache.CacheException;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import org.terracotta.ehcachedx.monitor.probe.counter.CounterManager;
import org.terracotta.ehcachedx.monitor.probe.counter.sampled.PullSampledCounterConfig;
import org.terracotta.ehcachedx.monitor.probe.counter.sampled.SampledCounter;
import org.terracotta.ehcachedx.monitor.probe.counter.sampled.SampledCounterConfig;
import org.terracotta.ehcachedx.monitor.probe.counter.sampled.memory.CacheMemorySizeCalculator;
import org.terracotta.ehcachedx.monitor.probe.counter.sampled.memory.MemorySamplerSupport;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Callable;

/**
 * Used by CacheManagerService
 */
public class SampledCacheStatistics {


    private final SampledCounterConfig counterConfig;

    private final SampledCounter removedCounter;
    private final SampledCounter putCounter;
    private final SampledCounter updatedCounter;
    private final SampledCounter expiredCounter;
    private final SampledCounter evictedCounter;
    private final SampledCounter hitCounter;
    private final SampledCounter hitMemoryCounter;
    private final SampledCounter hitDiskCounter;
    private final SampledCounter hitRatioCounter;
    private final SampledCounter missCounter;
    private final SampledCounter totalCountCounter;
    private final SampledCounter diskCountCounter;
    private final SampledCounter memoryCountCounter;
    //should be a gauge, not counter.
    private final SampledCounter memorySizeCounter;

    private boolean memoryMeasurement;

    private final MemorySamplerSupport memorySamplerSupport;

    public Map<Statistic, SampledCounter> getCounters() {
        return counters;
    }

    private final Map<Statistic, SampledCounter> counters;

    public SampledCacheStatistics(final Ehcache cache, CounterManager counterManager, SampledCounterConfig counterConfig,
                                  boolean memoryMeasurement) {
        this.counterConfig = counterConfig;
        this.memoryMeasurement = memoryMeasurement;
        memorySamplerSupport = new MemorySamplerSupport(memoryMeasurement);

        // store dedicated fields for each counter to be able to access them directly without look-ups
        removedCounter = (SampledCounter) counterManager.createCounter(counterConfig);
        putCounter = (SampledCounter) counterManager.createCounter(counterConfig);
        updatedCounter = (SampledCounter) counterManager.createCounter(counterConfig);
        expiredCounter = (SampledCounter) counterManager.createCounter(counterConfig);
        evictedCounter = (SampledCounter) counterManager.createCounter(counterConfig);
        hitCounter = (SampledCounter) counterManager.createCounter(new PullSampledCounterConfig(counterConfig, true, new Callable<Long>() {
            public Long call() throws Exception {
                return cache.getStatistics().getCacheHits();
            }
        }));
        hitMemoryCounter = (SampledCounter) counterManager.createCounter(new PullSampledCounterConfig(counterConfig, true, new Callable<Long>() {
            public Long call() throws Exception {
                return cache.getStatistics().getInMemoryHits();
            }
        }));
        hitDiskCounter = (SampledCounter) counterManager.createCounter(new PullSampledCounterConfig(counterConfig, true, new Callable<Long>() {
            public Long call() throws Exception {
                return cache.getStatistics().getOnDiskHits();
            }
        }));
        hitRatioCounter = (SampledCounter) counterManager.createCounter(new PullSampledCounterConfig(counterConfig, false, new Callable<Long>() {
            private long previousHits = 0;
            private long previousMisses = 0;

            public Long call() throws Exception {
                long currentHits = cache.getStatistics().getCacheHits();
                long currentMisses = cache.getStatistics().getCacheMisses();
                long hitsDelta = currentHits - previousHits;
                long missesDelta = currentMisses - previousMisses;
                previousHits = currentHits;
                previousMisses = currentMisses;
                long total = hitsDelta + missesDelta;
                if (0 == total) {
                    return 0L;
                }

                // returning as per milli
                return 1000 * hitsDelta / total;
            }
        }));
        missCounter = (SampledCounter) counterManager.createCounter(new PullSampledCounterConfig(counterConfig, true, new Callable<Long>() {
            public Long call() throws Exception {
                return cache.getStatistics().getCacheMisses();
            }
        }));
        totalCountCounter = (SampledCounter) counterManager.createCounter(new PullSampledCounterConfig(counterConfig, false, new Callable<Long>() {
            public Long call() throws Exception {
                return (long) cache.getSize();
            }
        }));
        diskCountCounter = (SampledCounter) counterManager.createCounter(new PullSampledCounterConfig(counterConfig, false, new Callable<Long>() {
            public Long call() throws Exception {
                return (long) cache.getDiskStoreSize();
            }
        }));
        memoryCountCounter = (SampledCounter) counterManager.createCounter(new PullSampledCounterConfig(counterConfig, false, new Callable<Long>() {
            public Long call() throws Exception {
                return cache.getMemoryStoreSize();
            }
        }));
        memorySizeCounter = (SampledCounter) counterManager.createCounter(new PullSampledCounterConfig(counterConfig, false, new CacheMemorySizeCalculator(cache, memorySamplerSupport.getStatistics()))
        );

        // create a map of the available counters and what they're named as
        counters = new HashMap<Statistic, SampledCounter>();
        counters.put(Statistic.removed, removedCounter);
        counters.put(Statistic.put, putCounter);
        counters.put(Statistic.updated, updatedCounter);
        counters.put(Statistic.expired, expiredCounter);
        counters.put(Statistic.evicted, evictedCounter);
        counters.put(Statistic.hit, hitCounter);
        counters.put(Statistic.hitMemory, hitMemoryCounter);
        counters.put(Statistic.hitDisk, hitDiskCounter);
        counters.put(Statistic.hitRatio, hitRatioCounter);
        counters.put(Statistic.miss, missCounter);
        counters.put(Statistic.totalCount, totalCountCounter);
        counters.put(Statistic.diskCount, diskCountCounter);
        counters.put(Statistic.memoryCount, memoryCountCounter);
        counters.put(Statistic.memorySize, memorySizeCounter);

        if (Statistic.values().length != counters.size()) {
            throw new AssertionError("Implementation error: not all SampledCacheStatistics names have been registered with a counter");
        }
    }

    public void dispose() {
        removedCounter.shutdown();
        putCounter.shutdown();
        updatedCounter.shutdown();
        expiredCounter.shutdown();
        evictedCounter.shutdown();
        hitCounter.shutdown();
        hitMemoryCounter.shutdown();
        hitDiskCounter.shutdown();
        hitRatioCounter.shutdown();
        missCounter.shutdown();
        totalCountCounter.shutdown();
        diskCountCounter.shutdown();
        memoryCountCounter.shutdown();
        memorySizeCounter.shutdown();
    }


    public String getSample(String name) {
        Statistic nameInstance;
        try {
            nameInstance = Statistic.valueOf(name);
        } catch (IllegalArgumentException e) {
            return null;
        }
        return nameInstance.getSample(this);
    }

    public SampleHistoryEntry[] getAggregatedSampleHistory(String name, long fromTimeSeconds, int timeInterval, int sampleRate, int offset) {
        SampleHistoryEntry[] history = getSampleHistory(name);
        SampleRange sampleRange = new SampleRange(fromTimeSeconds, timeInterval, sampleRate, offset, history.length - 1);

        // TODO Copying memory may be costly.
        SampleHistoryEntry[] result = new SampleHistoryEntry[sampleRange.getNumSamples()];
        System.arraycopy(history, sampleRange.getFirstSample(), result, 0, sampleRange.getNumSamples());

        // TODO Perform aggregation.
        // TODO For hitRate, will need to get underlying two values, total them up and then divide.
        // TODO This approach will vary based on the stat requested.
        // TODO How do we calculate timestamp for aggregated values - the timestamp of the last sample?

        // TODO Calculate numResults and samplesPerResult for aggregation purposes.
//        int numResults;
//        if (numSamples <= 100) {
//            numResults = numSamples; // no aggregation required
//        } else {
//            numResults = xxx;
//        }

        return result;
    }

    public SampleHistoryEntry[] getSampleHistory(String name) {
        Statistic nameInstance;
        try {
            nameInstance = Statistic.valueOf(name);
        } catch (IllegalArgumentException e) {
            return null;
        }
        return nameInstance.getSampleHistory(this);
    }

    public void notifyElementRemoved(Ehcache ehcache, Element element) throws CacheException {
        removedCounter.increment();
    }

    public void notifyElementPut(Ehcache ehcache, Element element) throws CacheException {
        putCounter.increment();
        memorySamplerSupport.notifyMemorySampler(ehcache, element, putCounter.getValue());
    }

    public void notifyElementUpdated(Ehcache ehcache, Element element) throws CacheException {
        updatedCounter.increment();
        //On update we also want to notify. But the memory sampler uses the putCounter for its algorithm
        memorySamplerSupport.notifyMemorySampler(ehcache, element, putCounter.getValue());
    }

    public void notifyElementExpired(Ehcache ehcache, Element element) {
        expiredCounter.increment();
    }

    public void notifyElementEvicted(Ehcache ehcache, Element element) {
        evictedCounter.increment();
    }

    public void notifyRemoveAll(Ehcache ehcache) {
        // no-op
    }

    public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    public SampledCounterConfig getCounterConfig() {
        return counterConfig;
    }

}
