/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */
package org.terracotta.ehcachedx.monitor.probe;

import net.sf.ehcache.CacheException;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Status;
import net.sf.ehcache.distribution.CacheManagerPeerListener;
import org.terracotta.ehcachedx.monitor.common.CacheManagerServiceAPI;
import org.terracotta.ehcachedx.monitor.common.DxException;
import org.terracotta.ehcachedx.monitor.common.handler.RestHandler;

import java.net.UnknownHostException;
import java.util.Collections;
import java.util.List;

public class ProbePeerListener implements CacheManagerPeerListener {

    public static final String SCHEME_PROBE = "Probe";

    private volatile CacheManagerService cacheManagerService;
    private volatile ProbeDxService service;

    private Status status;

    public ProbePeerListener(final ProbeDxService.Config config, final CacheManager cacheManager) throws UnknownHostException {

        this.service = new ProbeDxService(config);
        this.service.setDaemon(true);
        this.cacheManagerService = new CacheManagerService(this.service.createCounterConfig(), cacheManager, config.isMemoryMeasurement());

        String cacheManagerName = cacheManager.getName();
        if (cacheManagerName.startsWith(CacheManager.class.getName())) {
            cacheManagerName = cacheManagerName.substring(CacheManager.class.getPackage().getName().length() + 1);
        }
        this.service.init(cacheManagerName, new RestHandler(cacheManagerService, CacheManagerServiceAPI.METHOD_PREFIX));

        synchronized (this) {
            this.status = Status.STATUS_UNINITIALISED;
        }
    }

    public void init() throws CacheException {
        cacheManagerService.init();
        
        try {
            service.start();
        } catch (DxException e) {
            throw new CacheException("Unexpected error while starting the DX service.", e);
        }
    }

    public void dispose() throws CacheException {
        try {
            service.stop();
        } catch (DxException e) {
            throw new CacheException("Unexpected error while stopping the DX service.", e);
        }
        service = null;

        cacheManagerService.shutdown();
        cacheManagerService = null;

        synchronized (this) {
            status = Status.STATUS_SHUTDOWN;
        }
    }

    public List getBoundCachePeers() {
        return Collections.emptyList();
    }

    public Status getStatus() {
        synchronized (this) {
            if (Status.STATUS_UNINITIALISED == status) {
                if (service != null && service.isRunning()) {
                    status = Status.STATUS_ALIVE;
                }
            }
            
            return status;
        }
    }

    public String getUniqueResourceIdentifier() {
        return ProbePeerListener.class.getName() + ":" + service.getPort();
    }

    /**
     * If a conflict is detected in unique resource use, this method signals the listener to attempt
     * automatic resolution of the resource conflict.
     *
     * @throws IllegalStateException if the status of the listener is not {@link net.sf.ehcache.Status#STATUS_UNINITIALISED}
     */
    public void attemptResolutionOfUniqueResourceConflict() throws IllegalStateException, CacheException {
        if (service != null) {
            service.assignFreePort(true);
        }
    }

    public String getScheme() {
        return SCHEME_PROBE;
    }

    public void notifyCacheAdded(final String cacheName) {
        if (cacheManagerService != null) {
            cacheManagerService.registerCache(cacheName);
        }
    }

    public void notifyCacheRemoved(final String cacheName) {
        if (cacheManagerService != null) {
            cacheManagerService.unregisterCache(cacheName);
        }
    }
}
