/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */
package org.terracotta.ehcachedx.monitor.probe;

import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.ELEMENT_CACHE;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.ELEMENT_CONFIG_VALUES;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.ELEMENT_KEY;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.ELEMENT_NAME;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.ELEMENT_SAMPLE;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.ELEMENT_SAMPLED_STATISTIC_VALUES;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.ELEMENT_TIMESTAMP;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.ELEMENT_VALUE;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.PARAM_CACHE;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.PARAM_COUNT;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.PARAM_FROM_DATE_TIME;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.PARAM_KEY;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.PARAM_NAME;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.PARAM_OFFSET;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.PARAM_QUERY;
import static org.terracotta.ehcachedx.monitor.common.rest.RestConstants.PARAM_TIME_INTERVAL;

import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import net.sf.ehcache.config.CacheConfiguration;
import net.sf.ehcache.distribution.CacheManagerPeerProvider;
import net.sf.ehcache.event.RegisteredEventListeners;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.ehcachedx.monitor.common.CacheManagerServiceAPI;
import org.terracotta.ehcachedx.monitor.common.rest.ResponseElement;
import org.terracotta.ehcachedx.monitor.common.rest.RestMethod;
import org.terracotta.ehcachedx.monitor.common.rest.RestRequest;
import org.terracotta.ehcachedx.monitor.common.rest.RestResponse;
import org.terracotta.ehcachedx.monitor.probe.counter.CounterManager;
import org.terracotta.ehcachedx.monitor.probe.counter.CounterManagerImpl;
import org.terracotta.ehcachedx.monitor.probe.counter.sampled.SampledCounterConfig;

public class CacheManagerService implements CacheManagerServiceAPI {

	private static final Logger LOG = LoggerFactory
			.getLogger(CacheManagerService.class);

	private static final String REPLICATION_MODE_RMI = "RMI";
	private static final String REPLICATION_MODE_JGROUPS = "JGroups";
	private static final String REPLICATION_MODE_JMS = "JMS";
	private static final String REPLICATION_MODE_CUSTOM = "Custom";
	private static final String REPLICATION_MODE_TERRACOTTA = "Terracotta";
	private static final String REPLICATION_MODE_NONE = "None";
	private static final String REPLICATION_MODE_MIXED = "Mixed";

	private final ConcurrentMap<String, SampledCacheStatisticsWrapper> cacheStatistics = new ConcurrentHashMap<String, SampledCacheStatisticsWrapper>();
	private final CounterManager counterManager = new CounterManagerImpl();
	private final SampledCounterConfig counterConfig;
	private final CacheManager cacheManager;
	private final boolean memoryMeasurement;

	public CacheManagerService(SampledCounterConfig counterConfig,
			CacheManager cacheManager, boolean memoryMeasurement) {
		this.counterConfig = counterConfig;
		this.cacheManager = cacheManager;
		this.memoryMeasurement = memoryMeasurement;
	}

	public void init() {
		// register all the caches that are already present in the cache manager
		synchronized (cacheStatistics) {
			for (String cacheName : cacheManager.getCacheNames()) {
				registerCache(cacheName);
			}
		}
	}

	public void shutdown() {
		counterManager.shutdown();
	}

	@RestMethod
	public void getCacheManagerName(RestResponse response) throws IOException {
		String cacheManagerName = cacheManager.getName();
		if (cacheManagerName.startsWith(CacheManager.class.getName() + "@")) {
			cacheManagerName = cacheManagerName.substring(CacheManager.class
					.getPackage().getName().length() + 1);
		}
		response.value(cacheManagerName);
	}

	@RestMethod
	public void getCacheManagerStatus(RestResponse response) throws IOException {
		response.value(cacheManager.getStatus().toString());
	}

	@RestMethod
	public void getCacheNames(RestResponse response) throws IOException {
		SortedSet<String> set = new TreeSet<String>(Arrays.asList(cacheManager
				.getCacheNames()));
		response.childType(ELEMENT_NAME).children(set);
	}

	@RestMethod
	public void getCacheStatisticsState(RestResponse response)
			throws IOException {
		Object result = Boolean.TRUE;
		for (String cacheName : cacheManager.getCacheNames()) {
			Ehcache cache = cacheManager.getEhcache(cacheName);
			if (cache != null) {
				try {
					Method method = cache.getClass().getDeclaredMethod(
							"isSampledStatisticsEnabled");
					if (method == null) {
						result = "na";
						break;
					} else {
						boolean sampledStatsEnabled = ((Boolean) method
								.invoke(cache)).booleanValue();
						if (!sampledStatsEnabled) {
							result = Boolean.FALSE;
							break;
						}
					}
				} catch (Exception e) {
					result = "na";
					break;
				}
			}
		}
		response.value(result.toString());
	}

	@RestMethod
	public void enableCacheStatistics(RestRequest request, RestResponse response)
			throws IOException {
		for (String cacheName : cacheManager.getCacheNames()) {
			Ehcache cache = cacheManager.getEhcache(cacheName);
			if (cache != null) {
				try {
					Method method = cache.getClass().getDeclaredMethod(
							"setSampledStatisticsEnabled", Boolean.TYPE);
					if (method != null) {
						method.invoke(cache, true);
						CacheConfiguration cacheConfig = cache
								.getCacheConfiguration();
						method = cacheConfig.getClass().getDeclaredMethod(
								"setStatistics", Boolean.TYPE);
						if (method != null) {
							method.invoke(cacheConfig, true);
						}
					}
				} catch (Exception e) {
					/**/
				}
			}
		}
		response.ok();
	}

	@RestMethod
	public void getCacheCount(RestResponse response) throws IOException {
		response.value(cacheManager.getCacheNames().length);
	}

	@RestMethod
	public void getReplicationMode(RestResponse response) throws IOException {
		Collection<CacheManagerPeerProvider> providers;
		try {
			try {
				Field field = cacheManager.getClass().getDeclaredField(
						"cacheManagerPeerProvider");
				field.setAccessible(true);
				CacheManagerPeerProvider provider = (CacheManagerPeerProvider) field
						.get(cacheManager);
				providers = new ArrayList<CacheManagerPeerProvider>();
				if (provider != null) {
					providers.add(provider);
				}
			} catch (NoSuchFieldException e) {
				Field field = cacheManager.getClass().getDeclaredField(
						"cacheManagerPeerProviders");
				field.setAccessible(true);
				Map<String, CacheManagerPeerProvider> providersMap = (Map<String, CacheManagerPeerProvider>) field
						.get(cacheManager);
				if (providersMap != null) {
					providers = providersMap.values();
				} else {
					providers = Collections.emptyList();
				}
			}
		} catch (Exception e) {
			LOG.error("Unsupported Ehcache version, can't detect the replication mode");
			response.value("unknown");
			return;
		}

		final Set<String> replicationModes = new HashSet<String>();

		for (CacheManagerPeerProvider provider : providers) {
			if (provider != null) {
				if (provider.getClass().getName()
						.contains("RMICacheManagerPeerProvider")) {
					replicationModes.add(REPLICATION_MODE_RMI);
				} else if (provider.getClass().getName()
						.contains("JGroupsCacheManagerPeerProvider")) {
					replicationModes.add(REPLICATION_MODE_JGROUPS);
				} else if (provider.getClass().getName()
						.contains("JMSCacheManagerPeerProvider")) {
					replicationModes.add(REPLICATION_MODE_JMS);
				} else {
					replicationModes.add(REPLICATION_MODE_CUSTOM);
				}
			}
		}

		// check if the cache is Terracotta clustered
		if (Boolean.getBoolean("tc.active")) {
			String[] names = cacheManager.getCacheNames();
			if (names.length > 0) {
				for (String name : names) {
					Ehcache cache = cacheManager.getEhcache(name);
					CacheConfiguration config = cache.getCacheConfiguration();
					try {
						Method method = CacheConfiguration.class
								.getDeclaredMethod("isTerracottaClustered");
						method.setAccessible(true);
						Object result = method.invoke(config);
						if ((Boolean) result) {
							replicationModes.add(REPLICATION_MODE_TERRACOTTA);
						} else if (0 == replicationModes.size()
								|| replicationModes
										.contains(REPLICATION_MODE_TERRACOTTA)) {
							// only add the 'None' mode if no peer provider
							// registered a replication mode beforehand
							// or if a 'Terracotta' mode was already added, the
							// latter will for sure result in a
							// 'Mixed' mode at the end of this method
							replicationModes.add(REPLICATION_MODE_NONE);
						}
					} catch (NoSuchMethodException e) {
						// ignore, this is not echache 1.7
					} catch (InvocationTargetException e) {
						// ignore, this is not echache 1.7
					} catch (IllegalAccessException e) {
						// ignore, this is not echache 1.7
					}
				}
			}
		}

		// check the consistency of the replication mode for caches in the same
		// cache manager
		// if different replication modes are used, the 'Mixed' keyword is
		// returned
		if (0 == replicationModes.size()) {
			response.value(REPLICATION_MODE_NONE);
		} else if (1 == replicationModes.size()) {
			response.value(replicationModes.iterator().next());
		} else {
			response.value(REPLICATION_MODE_MIXED);
		}
	}

	@RestMethod(required = { PARAM_CACHE, PARAM_COUNT }, optional = { PARAM_QUERY })
	public void getCacheKeys(RestRequest request, RestResponse response)
			throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);
		String countString = request.getParameter(PARAM_COUNT);
		String query = request.getParameter(PARAM_QUERY);

		int count = Integer.parseInt(countString);
		if (count <= 0) {
			throw new IllegalArgumentException("count needs to be positive");
		}

		Ehcache cache = cacheManager.getEhcache(cacheName);
		response.childType(ELEMENT_KEY);
		if (cache != null) {
			List keys = cache.getKeysWithExpiryCheck();
			Iterator it = keys.iterator();
			while (count > 0 && it.hasNext()) {
				Object key = it.next();
				if (null == query || queryKey(key, query)) {
					count--;
					response.child(key);
				}
			}
		}
	}

	@RestMethod
	public void getCacheElementMetaDataNames(RestRequest request,
			RestResponse response) throws IOException {
		response.childType(PARAM_NAME).children(ElementMetaData.getNames());
	}

	@RestMethod(required = { PARAM_CACHE, PARAM_KEY, PARAM_NAME })
	public void getCacheElementMetaDataValue(RestRequest request,
			RestResponse response) throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);
		String key = request.getParameter(PARAM_KEY);
		String metaDataName = request.getParameter(PARAM_NAME);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			Element element = cache.getQuiet(key);
			if (element != null) {
				response.value(ElementMetaData.getMetaData(metaDataName,
						element));
			}
		}
	}

	@RestMethod(required = { PARAM_CACHE, PARAM_KEY, PARAM_NAME })
	public void getCacheElementMetaDataValues(RestRequest request,
			RestResponse response) throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);
		String key = request.getParameter(PARAM_KEY);
		String[] metaDataNames = request.getParameterValues(PARAM_NAME);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			Element element = cache.getQuiet(key);
			if (element != null) {
				for (String name : metaDataNames) {
					String metaData = ElementMetaData
							.getMetaData(name, element);
					if (metaData != null) {
						response.child(name, metaData);
					}
				}
			}
		}
	}

	@RestMethod(required = { PARAM_CACHE, PARAM_KEY })
	public void getCacheElementMetaDataAllValues(RestRequest request,
			RestResponse response) throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);
		String key = request.getParameter(PARAM_KEY);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			Element element = cache.getQuiet(key);
			if (element != null) {
				for (String name : ElementMetaData.getNames()) {
					response.child(name,
							ElementMetaData.getMetaData(name, element));
				}
			}
		}
	}

	@RestMethod(required = { PARAM_CACHE })
	public void getCacheElementCount(RestRequest request, RestResponse response)
			throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			response.value(cache.getSize());
		}
	}

	@RestMethod(required = { PARAM_CACHE })
	public void getCacheElementCountMemory(RestRequest request,
			RestResponse response) throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			response.value(cache.getMemoryStoreSize());
		}
	}

	@RestMethod(required = { PARAM_CACHE })
	public void getCacheElementCountDisk(RestRequest request,
			RestResponse response) throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			response.value(cache.getDiskStoreSize());
		}
	}

	@RestMethod(required = { PARAM_CACHE })
	public void getCacheElementMemorySize(RestRequest request,
			RestResponse response) throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			response.value(cache.calculateInMemorySize());
		}
	}

	@RestMethod(required = { PARAM_CACHE })
	public void removeAllFromCache(RestRequest request, RestResponse response)
			throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			cache.removeAll();
			response.ok();
		}
	}

	@RestMethod(required = { PARAM_CACHE, PARAM_QUERY })
	public void removeQueryFromCache(RestRequest request, RestResponse response)
			throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);
		String query = request.getParameter(PARAM_QUERY);

		int count = 0;
		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			List<Object> keys = cache.getKeysWithExpiryCheck();
			for (Object key : keys) {
				if (queryKey(key, query)) {
					if (cache.removeQuiet(key)) {
						count++;
					}
				}
			}
			response.value(count);
		}
	}

	@RestMethod(required = { PARAM_CACHE, PARAM_KEY })
	public void removeFromCache(RestRequest request, RestResponse response)
			throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);
		String key = request.getParameter(PARAM_KEY);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			if (cache.removeQuiet(key)) {
				response.value("removed");
			} else {
				response.value("key not found");
			}
		}
	}

	@RestMethod(required = { PARAM_CACHE })
	public void flushCache(RestRequest request, RestResponse response)
			throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			cache.flush();
			response.ok();
		}
	}

	@RestMethod(required = { PARAM_CACHE })
	public void evictExpiredElements(RestRequest request, RestResponse response)
			throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			cache.evictExpiredElements();
			response.ok();
		}
	}

	@RestMethod(required = { PARAM_CACHE, PARAM_NAME })
	public void getCacheConfigValue(RestRequest request, RestResponse response)
			throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);
		String configName = request.getParameter(PARAM_NAME);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			response.value(CacheConfig.getConfigValue(configName, cache));
		}
	}

	@RestMethod(required = { PARAM_CACHE, PARAM_NAME })
	public void getCacheConfigValues(RestRequest request, RestResponse response)
			throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);
		String[] configNames = request.getParameterValues(PARAM_NAME);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			for (String configName : configNames) {
				Object value = CacheConfig.getConfigValue(configName, cache);
				if (value != null) {
					response.child(configName, value);
				}
			}
		}
	}

	@RestMethod(required = { PARAM_CACHE })
	public void getCacheConfigAllValues(RestRequest request,
			RestResponse response) throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);

		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			for (String name : CacheConfig.getNames()) {
				response.child(name, CacheConfig.getConfigValue(name, cache));
			}
		}
	}

	// TODO Write system tests.
	@RestMethod
	public void getAllCachesConfigAllValues(RestResponse response)
			throws IOException {
		for (String cacheName : cacheManager.getCacheNames()) {
			Ehcache cache = cacheManager.getEhcache(cacheName);
			if (cache != null) {
				ResponseElement cacheElement = response.begin(ELEMENT_CACHE);
				cacheElement.child(ELEMENT_NAME, cacheName);
				ResponseElement configValuesElement = cacheElement
						.begin(ELEMENT_CONFIG_VALUES);

				for (String name : CacheConfig.getNames()) {
					configValuesElement.child(name,
							CacheConfig.getConfigValue(name, cache));
				}

				configValuesElement.end();
				cacheElement.end();
			}
		}
	}

	@RestMethod
	public void getCacheConfigNames(RestRequest request, RestResponse response)
			throws IOException {
		response.childType(ELEMENT_NAME).children(CacheConfig.getNames());
	}

	// TODO Remove duplication between the next 3 methods.
	@RestMethod(required = { PARAM_CACHE, PARAM_NAME })
	public void getSampledStatisticValue(RestRequest request,
			RestResponse response) throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);
		String statName = request.getParameter(PARAM_NAME);

		SampledCacheStatistics stats = getStats(cacheName);
		if (stats != null) {
			String result = stats.getSample(statName);
			if (result != null) {
				response.value(result);
			}
		}
	}

	@RestMethod(required = { PARAM_CACHE, PARAM_NAME })
	public void getSampledStatisticValues(RestRequest request,
			RestResponse response) throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);
		String[] statNames = request.getParameterValues(PARAM_NAME);

		SampledCacheStatistics stats = getStats(cacheName);
		if (stats != null) {
			for (String statName : statNames) {
				String result = stats.getSample(statName);
				if (result != null) {
					response.child(statName, result);
				}
			}
		}
	}

	@RestMethod(required = { PARAM_CACHE })
	public void getSampledStatisticAllValues(RestRequest request,
			RestResponse response) throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);

		SampledCacheStatistics stats = getStats(cacheName);
		if (stats != null) {
			for (String statName : Statistic.getNames()) {
				String result = stats.getSample(statName);
				if (result != null) {
					response.child(statName, result);
				}
			}
		}
	}

	// TODO Write system tests.
	@RestMethod(required = { PARAM_NAME })
	public void getAllCachesSampledStatisticValues(RestRequest request,
			RestResponse response) throws IOException {
		String[] statNames = request.getParameterValues(PARAM_NAME);
		for (String cacheName : cacheManager.getCacheNames()) {
			Ehcache cache = cacheManager.getEhcache(cacheName);
			if (cache != null) {
				ResponseElement cacheElement = response.begin(ELEMENT_CACHE);
				cacheElement.child(ELEMENT_NAME, cacheName);
				ResponseElement sampledStatisticValuesElement = cacheElement
						.begin(ELEMENT_SAMPLED_STATISTIC_VALUES);

				SampledCacheStatistics stats = getStats(cacheName);
				if (stats != null) {
					for (String statName : statNames) {
						String result = stats.getSample(statName);
						if (result != null) {
							sampledStatisticValuesElement.child(statName,
									result);
						}
					}
				}

				sampledStatisticValuesElement.end();
				cacheElement.end();
			}
		}
	}

	@RestMethod(required = { PARAM_CACHE, PARAM_NAME }, optional = {
			PARAM_FROM_DATE_TIME, PARAM_TIME_INTERVAL, PARAM_OFFSET })
	public void getSampledStatisticHistory(RestRequest request,
			RestResponse response) throws IOException {
		String cacheName = request.getParameter(PARAM_CACHE);
		String statName = request.getParameter(PARAM_NAME);
		String fromDateTime = request.getParameter(PARAM_FROM_DATE_TIME);
		String timeIntervalParam = request.getParameter(PARAM_TIME_INTERVAL);
		String offset = request.getParameter(PARAM_OFFSET);

		int sampleRate = counterConfig.getIntervalSecs();

		long fromTimeSeconds = 0; // Zero represents live data.
		if (fromDateTime != null) {
			String pattern = "yyyy/MM/dd HH:mm:ss";
			DateFormat df = new SimpleDateFormat(pattern);
			df.setLenient(false);
			try {
				Date date = df.parse(fromDateTime);
				if (date.after(new Date())) {
					throw new IllegalArgumentException(PARAM_FROM_DATE_TIME
							+ " of '" + fromDateTime
							+ "' cannot be in the future");
				}
				fromTimeSeconds = date.getTime() / 1000;
			} catch (ParseException e) {
				throw new IllegalArgumentException("Can't parse "
						+ PARAM_FROM_DATE_TIME + "field '" + fromDateTime
						+ "' with pattern '" + pattern + "'");
			}
		}

		int timeInterval = 60; // seconds
		if (timeIntervalParam != null) {
			timeInterval = Integer.parseInt(timeIntervalParam);
		}
		if (timeInterval < sampleRate) {
			throw new IllegalArgumentException(
					PARAM_TIME_INTERVAL
							+ " needs to be greater than or equal to the sample rate of "
							+ sampleRate + " seconds");
		}
		if (timeInterval > 60 * 60 * 24) {
			throw new IllegalArgumentException(PARAM_TIME_INTERVAL
					+ " cannot be greater than " + sampleRate
					+ " seconds (1 day)");
		}

		int offsetFactor = 0;
		if (offset != null) {
			if ("forward".equals(offset)) {
				offsetFactor = 1;
			} else if ("back".equals(offset)) {
				offsetFactor = -1;
			} else {
				throw new IllegalArgumentException(PARAM_OFFSET + "'" + offset
						+ "'" + " must be either 'forward' or 'back'");
			}
		}

		SampledCacheStatistics stats = getStats(cacheName);
		if (stats != null) {
			for (SampleHistoryEntry entry : stats.getAggregatedSampleHistory(
					statName, fromTimeSeconds, timeInterval, sampleRate,
					offsetFactor)) {
				response.begin(ELEMENT_SAMPLE)
						.child(ELEMENT_VALUE, entry.getValue())
						.child(ELEMENT_TIMESTAMP, entry.getDate()).end();
			}
		}
	}

	private SampledCacheStatistics getStats(String cacheName) {
		SampledCacheStatisticsWrapper wrapper = cacheStatistics.get(cacheName);
		SampledCacheStatistics stats = null;
		if (wrapper != null) {
			stats = wrapper.getStats();
		}
		return stats;
	}

	private boolean queryKey(Object key, String query) {
		return String.valueOf(key).indexOf(query) != -1;
	}

	public void registerCache(String cacheName) {
		Ehcache cache = cacheManager.getEhcache(cacheName);
		RegisteredEventListeners listeners = cache
				.getCacheEventNotificationService();
		SampledCacheStatistics stats = new SampledCacheStatistics(cache,
				counterManager, counterConfig, memoryMeasurement);
		SampledCacheStatisticsWrapper wrapper = new SampledCacheStatisticsWrapper(
				stats);
		// ensure that the listener is registered atomically
		synchronized (cacheStatistics) {
			listeners.registerListener(wrapper);
			cacheStatistics.put(cacheName, wrapper);
		}
	}

	public void unregisterCache(String cacheName) {
		RegisteredEventListeners listeners = null;
		Ehcache cache = cacheManager.getEhcache(cacheName);
		if (cache != null) {
			listeners = cache.getCacheEventNotificationService();
		}

		synchronized (cacheStatistics) {
			SampledCacheStatisticsWrapper wrapper = cacheStatistics
					.remove(cacheName);
			// ensure that the listener is unregistered atomically
			if (wrapper != null && listeners != null) {
				listeners.unregisterListener(wrapper);
			}
		}
	}

	public static SimpleDateFormat createMetaDataDateFormat() {
		return new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
	}
}
