package org.telegram.telegrambots.meta.api.methods.groupadministration;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.ToString;
import lombok.experimental.SuperBuilder;
import lombok.experimental.Tolerate;
import lombok.extern.jackson.Jacksonized;
import org.telegram.telegrambots.meta.api.methods.botapimethods.BotApiMethod;
import org.telegram.telegrambots.meta.exceptions.TelegramApiRequestException;
import org.telegram.telegrambots.meta.exceptions.TelegramApiValidationException;

/**
 * @author Ruben Bermudez
 * @version 3.1
 * Use this method to generate a new primary invite link for a chat; any previously generated primary link is revoked.
 * The bot must be an administrator in the chat for this to work and must have the appropriate admin rights.
 * Returns the new invite link as String on success.
 *
 * @apiNote Each administrator in a chat generates their own invite links.
 * Bots can't use invite links generated by other administrators.
 * If you want your bot to work with invite links, it will need to generate its own link using exportChatInviteLink or by calling the getChat method.
 * If your bot needs to generate a new primary invite link replacing its previous one, use exportChatInviteLink again.
 */
@EqualsAndHashCode(callSuper = false)
@Getter
@Setter
@ToString
@RequiredArgsConstructor
@SuperBuilder
@Jacksonized
@JsonIgnoreProperties(ignoreUnknown = true)
public class ExportChatInviteLink extends BotApiMethod<String> {
    public static final String PATH = "exportChatInviteLink";

    private static final String CHATID_FIELD = "chat_id";

    @JsonProperty(CHATID_FIELD)
    @NonNull
    private String chatId; ///< Unique identifier for the target chat or username of the target channel (in the format @channelusername)

    @Tolerate
    public void setChatId(@NonNull Long chatId) {
        this.chatId = chatId.toString();
    }

    @Override
    public String getMethod() {
        return PATH;
    }

    @Override
    public String deserializeResponse(String answer) throws TelegramApiRequestException {
        return deserializeResponse(answer, String.class);
    }

    @Override
    public void validate() throws TelegramApiValidationException {
        if (chatId.isEmpty()) {
            throw new TelegramApiValidationException("ChatId can't be empty", this);
        }
    }

    public static abstract class ExportChatInviteLinkBuilder<C extends ExportChatInviteLink, B extends ExportChatInviteLinkBuilder<C, B>> extends BotApiMethodBuilder<String, C, B> {
        @Tolerate
        public ExportChatInviteLinkBuilder<C, B> chatId(@NonNull Long chatId) {
            this.chatId = chatId.toString();
            return this;
        }
    }
}
