/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.session.data.redis;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.function.Function;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jspecify.annotations.Nullable;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.core.NestedExceptionUtils;
import org.springframework.dao.NonTransientDataAccessException;
import org.springframework.data.redis.connection.Message;
import org.springframework.data.redis.connection.MessageListener;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.BoundSetOperations;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.serializer.JdkSerializationRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.util.ByteUtils;
import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.scheduling.support.CronExpression;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.session.DelegatingIndexResolver;
import org.springframework.session.FindByIndexNameSessionRepository;
import org.springframework.session.FlushMode;
import org.springframework.session.IndexResolver;
import org.springframework.session.MapSession;
import org.springframework.session.PrincipalNameIndexResolver;
import org.springframework.session.SaveMode;
import org.springframework.session.Session;
import org.springframework.session.SessionIdGenerator;
import org.springframework.session.UuidSessionIdGenerator;
import org.springframework.session.data.redis.RedisSessionExpirationStore;
import org.springframework.session.data.redis.RedisSessionMapper;
import org.springframework.session.events.SessionCreatedEvent;
import org.springframework.session.events.SessionDeletedEvent;
import org.springframework.session.events.SessionExpiredEvent;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

public class RedisIndexedSessionRepository
implements FindByIndexNameSessionRepository<RedisSession>,
MessageListener,
InitializingBean,
DisposableBean {
    private static final Log logger = LogFactory.getLog(RedisIndexedSessionRepository.class);
    private static final String SPRING_SECURITY_CONTEXT = "SPRING_SECURITY_CONTEXT";
    public static final String DEFAULT_CLEANUP_CRON = "0 * * * * *";
    public static final int DEFAULT_DATABASE = 0;
    public static final String DEFAULT_NAMESPACE = "spring:session";
    private int database = 0;
    private String namespace = "spring:session:";
    private String sessionCreatedChannelPrefix;
    private byte[] sessionCreatedChannelPrefixBytes;
    private String sessionDeletedChannel;
    private byte[] sessionDeletedChannelBytes;
    private String sessionExpiredChannel;
    private byte[] sessionExpiredChannelBytes;
    private String expiredKeyPrefix;
    private byte[] expiredKeyPrefixBytes;
    private final RedisOperations<String, Object> sessionRedisOperations;
    private RedisSessionExpirationStore expirationStore;
    private ApplicationEventPublisher eventPublisher = event -> {};
    private Duration defaultMaxInactiveInterval = Duration.ofSeconds(1800L);
    private IndexResolver<Session> indexResolver = new DelegatingIndexResolver(new IndexResolver[]{new PrincipalNameIndexResolver()});
    private RedisSerializer<Object> defaultSerializer = new JdkSerializationRedisSerializer();
    private FlushMode flushMode = FlushMode.ON_SAVE;
    private SaveMode saveMode = SaveMode.ON_SET_ATTRIBUTE;
    private String cleanupCron = "0 * * * * *";
    private @Nullable ThreadPoolTaskScheduler taskScheduler;
    private SessionIdGenerator sessionIdGenerator = UuidSessionIdGenerator.getInstance();
    private BiFunction<String, Map<String, Object>, MapSession> redisSessionMapper = new RedisSessionMapper();

    public RedisIndexedSessionRepository(RedisOperations<String, Object> sessionRedisOperations) {
        Assert.notNull(sessionRedisOperations, (String)"sessionRedisOperations cannot be null");
        this.sessionRedisOperations = sessionRedisOperations;
        this.expirationStore = new MinuteBasedRedisSessionExpirationStore(sessionRedisOperations, this::getExpirationsKey);
        this.configureSessionChannels();
    }

    public void afterPropertiesSet() {
        if (!"-".equals(this.cleanupCron)) {
            this.taskScheduler = RedisIndexedSessionRepository.createTaskScheduler();
            this.taskScheduler.initialize();
            this.taskScheduler.schedule(this::cleanUpExpiredSessions, (Trigger)new CronTrigger(this.cleanupCron));
        }
    }

    private static ThreadPoolTaskScheduler createTaskScheduler() {
        ThreadPoolTaskScheduler taskScheduler = new ThreadPoolTaskScheduler();
        taskScheduler.setThreadNamePrefix("spring-session-");
        return taskScheduler;
    }

    public void destroy() {
        if (this.taskScheduler != null) {
            this.taskScheduler.destroy();
        }
    }

    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        Assert.notNull((Object)applicationEventPublisher, (String)"applicationEventPublisher cannot be null");
        this.eventPublisher = applicationEventPublisher;
    }

    public void setDefaultMaxInactiveInterval(Duration defaultMaxInactiveInterval) {
        Assert.notNull((Object)defaultMaxInactiveInterval, (String)"defaultMaxInactiveInterval must not be null");
        this.defaultMaxInactiveInterval = defaultMaxInactiveInterval;
    }

    @Deprecated(since="3.0.0")
    public void setDefaultMaxInactiveInterval(int defaultMaxInactiveInterval) {
        this.setDefaultMaxInactiveInterval(Duration.ofSeconds(defaultMaxInactiveInterval));
    }

    public void setIndexResolver(IndexResolver<Session> indexResolver) {
        Assert.notNull(indexResolver, (String)"indexResolver cannot be null");
        this.indexResolver = indexResolver;
    }

    public void setDefaultSerializer(RedisSerializer<Object> defaultSerializer) {
        Assert.notNull(defaultSerializer, (String)"defaultSerializer cannot be null");
        this.defaultSerializer = defaultSerializer;
    }

    public void setFlushMode(FlushMode flushMode) {
        Assert.notNull((Object)flushMode, (String)"flushMode cannot be null");
        this.flushMode = flushMode;
    }

    public void setSaveMode(SaveMode saveMode) {
        Assert.notNull((Object)saveMode, (String)"saveMode must not be null");
        this.saveMode = saveMode;
    }

    public void setCleanupCron(String cleanupCron) {
        Assert.notNull((Object)cleanupCron, (String)"cleanupCron must not be null");
        if (!"-".equals(cleanupCron)) {
            Assert.isTrue((boolean)CronExpression.isValidExpression((String)cleanupCron), (String)"cleanupCron must be valid");
        }
        this.cleanupCron = cleanupCron;
    }

    public void setDatabase(int database) {
        this.database = database;
        this.configureSessionChannels();
    }

    private void configureSessionChannels() {
        this.sessionCreatedChannelPrefix = this.namespace + "event:" + this.database + ":created:";
        this.sessionCreatedChannelPrefixBytes = this.sessionCreatedChannelPrefix.getBytes();
        this.sessionDeletedChannel = "__keyevent@" + this.database + "__:del";
        this.sessionDeletedChannelBytes = this.sessionDeletedChannel.getBytes();
        this.sessionExpiredChannel = "__keyevent@" + this.database + "__:expired";
        this.sessionExpiredChannelBytes = this.sessionExpiredChannel.getBytes();
        this.expiredKeyPrefix = this.namespace + "sessions:expires:";
        this.expiredKeyPrefixBytes = this.expiredKeyPrefix.getBytes();
    }

    public RedisOperations<String, Object> getSessionRedisOperations() {
        return this.sessionRedisOperations;
    }

    public void save(RedisSession session) {
        session.save();
    }

    public void cleanUpExpiredSessions() {
        this.expirationStore.cleanupExpiredSessions();
    }

    public @Nullable RedisSession findById(String id) {
        return this.getSession(id, false);
    }

    public Map<String, RedisSession> findByIndexNameAndIndexValue(String indexName, String indexValue) {
        if (!PRINCIPAL_NAME_INDEX_NAME.equals(indexName)) {
            return Collections.emptyMap();
        }
        String principalKey = this.getPrincipalKey(indexValue);
        Set sessionIds = this.sessionRedisOperations.boundSetOps((Object)principalKey).members();
        if (sessionIds == null) {
            return Collections.emptyMap();
        }
        HashMap<String, RedisSession> sessions = new HashMap<String, RedisSession>(sessionIds.size());
        for (Object id : sessionIds) {
            RedisSession session = this.findById((String)id);
            if (session == null) continue;
            sessions.put(session.getId(), session);
        }
        return sessions;
    }

    private @Nullable RedisSession getSession(String id, boolean allowExpired) {
        Map entries = this.getSessionBoundHashOperations(id).entries();
        if (entries == null || entries.isEmpty()) {
            return null;
        }
        MapSession loaded = this.redisSessionMapper.apply(id, entries);
        if (loaded == null || !allowExpired && loaded.isExpired()) {
            return null;
        }
        RedisSession result = new RedisSession(loaded, false);
        result.originalLastAccessTime = loaded.getLastAccessedTime();
        return result;
    }

    public void deleteById(String sessionId) {
        RedisSession session = this.getSession(sessionId, true);
        if (session == null) {
            return;
        }
        this.cleanupPrincipalIndex(session);
        this.expirationStore.remove(sessionId);
        String expireKey = this.getExpiredKey(session.getId());
        this.sessionRedisOperations.delete((Object)expireKey);
        session.setMaxInactiveInterval(Duration.ZERO);
        this.save(session);
    }

    public RedisSession createSession() {
        MapSession cached = new MapSession(this.sessionIdGenerator);
        cached.setMaxInactiveInterval(this.defaultMaxInactiveInterval);
        RedisSession session = new RedisSession(cached, true);
        session.flushImmediateIfNecessary();
        return session;
    }

    public void onMessage(Message message, byte @Nullable [] pattern) {
        byte[] messageChannel = message.getChannel();
        if (ByteUtils.startsWith((byte[])messageChannel, (byte[])this.sessionCreatedChannelPrefixBytes)) {
            Map entries;
            String channel = new String(messageChannel);
            String sessionId = channel.substring(channel.lastIndexOf(":") + 1);
            MapSession loaded = this.redisSessionMapper.apply(sessionId, entries = (Map)this.defaultSerializer.deserialize(message.getBody()));
            if (loaded != null) {
                RedisSession session = new RedisSession(loaded, false);
                this.handleCreated(session);
            }
            return;
        }
        byte[] messageBody = message.getBody();
        if (!ByteUtils.startsWith((byte[])messageBody, (byte[])this.expiredKeyPrefixBytes)) {
            return;
        }
        boolean isDeleted = Arrays.equals(messageChannel, this.sessionDeletedChannelBytes);
        if (isDeleted || Arrays.equals(messageChannel, this.sessionExpiredChannelBytes)) {
            int endIndex;
            String body = new String(messageBody);
            int beginIndex = body.lastIndexOf(":") + 1;
            String sessionId = body.substring(beginIndex, endIndex = body.length());
            RedisSession session = this.getSession(sessionId, true);
            if (session == null) {
                logger.warn((Object)("Unable to publish SessionDestroyedEvent for session " + sessionId));
                return;
            }
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Publishing SessionDestroyedEvent for session " + sessionId));
            }
            this.cleanupPrincipalIndex(session);
            this.expirationStore.remove(session.getId());
            if (isDeleted) {
                this.handleDeleted(session);
            } else {
                this.handleExpired(session);
            }
        }
    }

    private void cleanupPrincipalIndex(RedisSession session) {
        String sessionId = session.getId();
        Map indexes = this.indexResolver.resolveIndexesFor((Session)session);
        String principal = (String)indexes.get(PRINCIPAL_NAME_INDEX_NAME);
        if (principal != null) {
            this.sessionRedisOperations.boundSetOps((Object)this.getPrincipalKey(principal)).remove(new Object[]{sessionId});
        }
    }

    private void handleCreated(RedisSession session) {
        this.publishEvent((ApplicationEvent)new SessionCreatedEvent((Object)this, (Session)session));
    }

    private void handleDeleted(RedisSession session) {
        this.publishEvent((ApplicationEvent)new SessionDeletedEvent((Object)this, (Session)session));
    }

    private void handleExpired(RedisSession session) {
        this.publishEvent((ApplicationEvent)new SessionExpiredEvent((Object)this, (Session)session));
    }

    private void publishEvent(ApplicationEvent event) {
        try {
            this.eventPublisher.publishEvent(event);
        }
        catch (Throwable ex) {
            logger.error((Object)("Error publishing " + String.valueOf(event) + "."), ex);
        }
    }

    public void setRedisKeyNamespace(String namespace) {
        Assert.hasText((String)namespace, (String)"namespace cannot be null or empty");
        this.namespace = namespace.trim() + ":";
        this.configureSessionChannels();
    }

    public void setExpirationStore(RedisSessionExpirationStore expirationStore) {
        Assert.notNull((Object)expirationStore, (String)"expirationStore cannot be null");
        this.expirationStore = expirationStore;
    }

    String getSessionKey(String sessionId) {
        return this.namespace + "sessions:" + sessionId;
    }

    String getPrincipalKey(String principalName) {
        return this.namespace + "index:" + FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME + ":" + principalName;
    }

    String getExpirationsKey(long expiration) {
        return this.namespace + "expirations:" + expiration;
    }

    private String getExpiredKey(String sessionId) {
        return this.getExpiredKeyPrefix() + sessionId;
    }

    private String getSessionCreatedChannel(String sessionId) {
        return this.getSessionCreatedChannelPrefix() + sessionId;
    }

    private String getExpiredKeyPrefix() {
        return this.expiredKeyPrefix;
    }

    public String getSessionCreatedChannelPrefix() {
        return this.sessionCreatedChannelPrefix;
    }

    public String getSessionDeletedChannel() {
        return this.sessionDeletedChannel;
    }

    public String getSessionExpiredChannel() {
        return this.sessionExpiredChannel;
    }

    private BoundHashOperations<String, String, Object> getSessionBoundHashOperations(String sessionId) {
        String key = this.getSessionKey(sessionId);
        return this.sessionRedisOperations.boundHashOps((Object)key);
    }

    static String getSessionAttrNameKey(String attributeName) {
        return "sessionAttr:" + attributeName;
    }

    public void setSessionIdGenerator(SessionIdGenerator sessionIdGenerator) {
        Assert.notNull((Object)sessionIdGenerator, (String)"sessionIdGenerator cannot be null");
        this.sessionIdGenerator = sessionIdGenerator;
    }

    public void setRedisSessionMapper(BiFunction<String, Map<String, Object>, MapSession> redisSessionMapper) {
        Assert.notNull(redisSessionMapper, (String)"redisSessionMapper cannot be null");
        this.redisSessionMapper = redisSessionMapper;
    }

    private final class MinuteBasedRedisSessionExpirationStore
    implements RedisSessionExpirationStore {
        private static final String SESSION_EXPIRES_PREFIX = "expires:";
        private final RedisOperations<String, Object> redis;
        private final Function<Long, String> lookupExpirationKey;

        MinuteBasedRedisSessionExpirationStore(RedisOperations<String, Object> redis, Function<Long, String> lookupExpirationKey) {
            this.redis = redis;
            this.lookupExpirationKey = lookupExpirationKey;
        }

        @Override
        public void save(RedisSession session) {
            long originalRoundedUp;
            Long originalExpiration = session.originalLastAccessTime != null ? Long.valueOf(session.originalLastAccessTime.plus(session.getMaxInactiveInterval()).toEpochMilli()) : null;
            String keyToExpire = SESSION_EXPIRES_PREFIX + session.getId();
            long toExpire = MinuteBasedRedisSessionExpirationStore.roundUpToNextMinute(MinuteBasedRedisSessionExpirationStore.expiresInMillis(session));
            if (originalExpiration != null && toExpire != (originalRoundedUp = MinuteBasedRedisSessionExpirationStore.roundUpToNextMinute(originalExpiration))) {
                String expireKey = this.getExpirationKey(originalRoundedUp);
                this.redis.boundSetOps((Object)expireKey).remove(new Object[]{keyToExpire});
            }
            String expirationsKey = RedisIndexedSessionRepository.this.getExpirationsKey(toExpire);
            long sessionExpireInSeconds = session.getMaxInactiveInterval().getSeconds();
            long fiveMinutesAfterExpires = sessionExpireInSeconds + TimeUnit.MINUTES.toSeconds(5L);
            this.redis.boundSetOps((Object)expirationsKey).expire(fiveMinutesAfterExpires, TimeUnit.SECONDS);
            String expireKey = this.getExpirationKey(toExpire);
            BoundSetOperations expireOperations = this.redis.boundSetOps((Object)expireKey);
            expireOperations.add(new Object[]{keyToExpire});
        }

        @Override
        public void remove(String sessionId) {
            RedisSession session = RedisIndexedSessionRepository.this.getSession(sessionId, true);
            if (session != null) {
                long toExpire = MinuteBasedRedisSessionExpirationStore.roundUpToNextMinute(MinuteBasedRedisSessionExpirationStore.expiresInMillis(session));
                String expireKey = this.getExpirationKey(toExpire);
                String entryToRemove = SESSION_EXPIRES_PREFIX + session.getId();
                this.redis.boundSetOps((Object)expireKey).remove(new Object[]{entryToRemove});
            }
        }

        @Override
        public void cleanupExpiredSessions() {
            long now = System.currentTimeMillis();
            long prevMin = MinuteBasedRedisSessionExpirationStore.roundDownMinute(now);
            String expirationKey = this.getExpirationKey(prevMin);
            Set sessionsToExpire = this.redis.boundSetOps((Object)expirationKey).members();
            this.redis.delete((Object)expirationKey);
            if (CollectionUtils.isEmpty((Collection)sessionsToExpire)) {
                return;
            }
            for (Object sessionId : sessionsToExpire) {
                this.touch(RedisIndexedSessionRepository.this.getSessionKey((String)sessionId));
            }
        }

        private void touch(String sessionKey) {
            RedisIndexedSessionRepository.this.sessionRedisOperations.hasKey((Object)sessionKey);
        }

        String getExpirationKey(long expires) {
            return this.lookupExpirationKey.apply(expires);
        }

        private static long expiresInMillis(Session session) {
            return session.getLastAccessedTime().plus(session.getMaxInactiveInterval()).toEpochMilli();
        }

        private static long roundUpToNextMinute(long timeInMs) {
            Instant instant = Instant.ofEpochMilli(timeInMs).plus(1L, ChronoUnit.MINUTES);
            Instant nextMinute = instant.truncatedTo(ChronoUnit.MINUTES);
            return nextMinute.toEpochMilli();
        }

        private static long roundDownMinute(long timeInMs) {
            Instant downMinute = Instant.ofEpochMilli(timeInMs).truncatedTo(ChronoUnit.MINUTES);
            return downMinute.toEpochMilli();
        }
    }

    public final class RedisSession
    implements Session {
        private final MapSession cached;
        private @Nullable Instant originalLastAccessTime;
        private Map<String, Object> delta = new HashMap<String, Object>();
        private boolean isNew;
        private @Nullable String originalPrincipalName;
        private String originalSessionId;

        RedisSession(MapSession cached, boolean isNew) {
            this.cached = cached;
            this.isNew = isNew;
            this.originalSessionId = cached.getId();
            Map indexes = RedisIndexedSessionRepository.this.indexResolver.resolveIndexesFor((Session)this);
            this.originalPrincipalName = (String)indexes.get(FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME);
            if (this.isNew) {
                this.delta.put("creationTime", cached.getCreationTime().toEpochMilli());
                this.delta.put("maxInactiveInterval", (int)cached.getMaxInactiveInterval().getSeconds());
                this.delta.put("lastAccessedTime", cached.getLastAccessedTime().toEpochMilli());
            }
            if (this.isNew || RedisIndexedSessionRepository.this.saveMode == SaveMode.ALWAYS) {
                this.getAttributeNames().forEach(attributeName -> this.delta.put(RedisIndexedSessionRepository.getSessionAttrNameKey(attributeName), cached.getAttribute(attributeName)));
            }
        }

        public void setLastAccessedTime(Instant lastAccessedTime) {
            this.cached.setLastAccessedTime(lastAccessedTime);
            this.delta.put("lastAccessedTime", this.getLastAccessedTime().toEpochMilli());
            this.flushImmediateIfNecessary();
        }

        public boolean isExpired() {
            return this.cached.isExpired();
        }

        public Instant getCreationTime() {
            return this.cached.getCreationTime();
        }

        public String getId() {
            return this.cached.getId();
        }

        public String changeSessionId() {
            String newSessionId = RedisIndexedSessionRepository.this.sessionIdGenerator.generate();
            this.cached.setId(newSessionId);
            return newSessionId;
        }

        public Instant getLastAccessedTime() {
            return this.cached.getLastAccessedTime();
        }

        public void setMaxInactiveInterval(Duration interval) {
            this.cached.setMaxInactiveInterval(interval);
            this.delta.put("maxInactiveInterval", (int)this.getMaxInactiveInterval().getSeconds());
            this.flushImmediateIfNecessary();
        }

        public Duration getMaxInactiveInterval() {
            return this.cached.getMaxInactiveInterval();
        }

        public <T> @Nullable T getAttribute(String attributeName) {
            Object attributeValue = this.cached.getAttribute(attributeName);
            if (attributeValue != null && RedisIndexedSessionRepository.this.saveMode.equals((Object)SaveMode.ON_GET_ATTRIBUTE)) {
                this.delta.put(RedisIndexedSessionRepository.getSessionAttrNameKey(attributeName), attributeValue);
            }
            return (T)attributeValue;
        }

        public Set<String> getAttributeNames() {
            return this.cached.getAttributeNames();
        }

        public void setAttribute(String attributeName, Object attributeValue) {
            this.cached.setAttribute(attributeName, attributeValue);
            this.delta.put(RedisIndexedSessionRepository.getSessionAttrNameKey(attributeName), attributeValue);
            this.flushImmediateIfNecessary();
        }

        public void removeAttribute(String attributeName) {
            this.cached.removeAttribute(attributeName);
            this.delta.put(RedisIndexedSessionRepository.getSessionAttrNameKey(attributeName), null);
            this.flushImmediateIfNecessary();
        }

        private void flushImmediateIfNecessary() {
            if (RedisIndexedSessionRepository.this.flushMode == FlushMode.IMMEDIATE) {
                this.save();
            }
        }

        private void save() {
            this.saveChangeSessionId();
            this.saveDelta();
        }

        private void saveDelta() {
            if (this.delta.isEmpty()) {
                return;
            }
            String sessionId = this.getId();
            RedisIndexedSessionRepository.this.getSessionBoundHashOperations(sessionId).putAll(this.delta);
            String principalSessionKey = RedisIndexedSessionRepository.getSessionAttrNameKey(FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME);
            String securityPrincipalSessionKey = RedisIndexedSessionRepository.getSessionAttrNameKey(RedisIndexedSessionRepository.SPRING_SECURITY_CONTEXT);
            if (this.delta.containsKey(principalSessionKey) || this.delta.containsKey(securityPrincipalSessionKey)) {
                String principal;
                if (this.originalPrincipalName != null) {
                    String originalPrincipalRedisKey = RedisIndexedSessionRepository.this.getPrincipalKey(this.originalPrincipalName);
                    RedisIndexedSessionRepository.this.sessionRedisOperations.boundSetOps((Object)originalPrincipalRedisKey).remove(new Object[]{sessionId});
                }
                Map indexes = RedisIndexedSessionRepository.this.indexResolver.resolveIndexesFor((Session)this);
                this.originalPrincipalName = principal = (String)indexes.get(FindByIndexNameSessionRepository.PRINCIPAL_NAME_INDEX_NAME);
                if (principal != null) {
                    String principalRedisKey = RedisIndexedSessionRepository.this.getPrincipalKey(principal);
                    RedisIndexedSessionRepository.this.sessionRedisOperations.boundSetOps((Object)principalRedisKey).add(new Object[]{sessionId});
                }
            }
            if (this.isNew) {
                String sessionCreatedKey = RedisIndexedSessionRepository.this.getSessionCreatedChannel(this.getId());
                RedisIndexedSessionRepository.this.sessionRedisOperations.convertAndSend(sessionCreatedKey, this.delta);
                this.isNew = false;
            }
            long sessionExpireInSeconds = this.getMaxInactiveInterval().getSeconds();
            this.createShadowKey(sessionExpireInSeconds);
            long fiveMinutesAfterExpires = sessionExpireInSeconds + TimeUnit.MINUTES.toSeconds(5L);
            RedisIndexedSessionRepository.this.sessionRedisOperations.boundHashOps((Object)RedisIndexedSessionRepository.this.getSessionKey(this.getId())).expire(fiveMinutesAfterExpires, TimeUnit.SECONDS);
            RedisIndexedSessionRepository.this.expirationStore.save(this);
            this.delta = new HashMap<String, Object>(this.delta.size());
        }

        private void createShadowKey(long sessionExpireInSeconds) {
            BoundValueOperations valueOps;
            String keyToExpire = "expires:" + this.getId();
            String sessionKey = RedisIndexedSessionRepository.this.getSessionKey(keyToExpire);
            if (sessionExpireInSeconds < 0L) {
                valueOps = RedisIndexedSessionRepository.this.sessionRedisOperations.boundValueOps((Object)sessionKey);
                valueOps.append("");
                valueOps.persist();
                RedisIndexedSessionRepository.this.sessionRedisOperations.boundHashOps((Object)RedisIndexedSessionRepository.this.getSessionKey(this.getId())).persist();
            }
            if (sessionExpireInSeconds == 0L) {
                RedisIndexedSessionRepository.this.sessionRedisOperations.delete((Object)sessionKey);
            } else {
                valueOps = RedisIndexedSessionRepository.this.sessionRedisOperations.boundValueOps((Object)sessionKey);
                valueOps.append("");
                valueOps.expire(sessionExpireInSeconds, TimeUnit.SECONDS);
            }
        }

        private void saveChangeSessionId() {
            String sessionId = this.getId();
            if (sessionId.equals(this.originalSessionId)) {
                return;
            }
            if (!this.isNew) {
                String originalSessionIdKey = RedisIndexedSessionRepository.this.getSessionKey(this.originalSessionId);
                String sessionIdKey = RedisIndexedSessionRepository.this.getSessionKey(sessionId);
                try {
                    RedisIndexedSessionRepository.this.sessionRedisOperations.rename((Object)originalSessionIdKey, (Object)sessionIdKey);
                }
                catch (NonTransientDataAccessException ex) {
                    this.handleErrNoSuchKeyError(ex);
                }
                String originalExpiredKey = RedisIndexedSessionRepository.this.getExpiredKey(this.originalSessionId);
                String expiredKey = RedisIndexedSessionRepository.this.getExpiredKey(sessionId);
                try {
                    RedisIndexedSessionRepository.this.sessionRedisOperations.rename((Object)originalExpiredKey, (Object)expiredKey);
                }
                catch (NonTransientDataAccessException ex) {
                    this.handleErrNoSuchKeyError(ex);
                }
                if (this.originalPrincipalName != null) {
                    String originalPrincipalRedisKey = RedisIndexedSessionRepository.this.getPrincipalKey(this.originalPrincipalName);
                    RedisIndexedSessionRepository.this.sessionRedisOperations.boundSetOps((Object)originalPrincipalRedisKey).remove(new Object[]{this.originalSessionId});
                    RedisIndexedSessionRepository.this.sessionRedisOperations.boundSetOps((Object)originalPrincipalRedisKey).add(new Object[]{sessionId});
                }
                RedisIndexedSessionRepository.this.expirationStore.remove(this.originalSessionId);
            }
            this.originalSessionId = sessionId;
        }

        private void handleErrNoSuchKeyError(NonTransientDataAccessException ex) {
            String message = NestedExceptionUtils.getMostSpecificCause((Throwable)ex).getMessage();
            if (!StringUtils.startsWithIgnoreCase((String)message, (String)"ERR no such key")) {
                throw ex;
            }
        }
    }
}

