/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.integration.aws.metadata;

import com.amazonaws.AmazonWebServiceRequest;
import com.amazonaws.handlers.AsyncHandler;
import com.amazonaws.services.dynamodbv2.AmazonDynamoDB;
import com.amazonaws.services.dynamodbv2.AmazonDynamoDBAsync;
import com.amazonaws.services.dynamodbv2.document.AttributeUpdate;
import com.amazonaws.services.dynamodbv2.document.DynamoDB;
import com.amazonaws.services.dynamodbv2.document.Expected;
import com.amazonaws.services.dynamodbv2.document.Item;
import com.amazonaws.services.dynamodbv2.document.Table;
import com.amazonaws.services.dynamodbv2.document.spec.DeleteItemSpec;
import com.amazonaws.services.dynamodbv2.document.spec.UpdateItemSpec;
import com.amazonaws.services.dynamodbv2.model.AmazonDynamoDBException;
import com.amazonaws.services.dynamodbv2.model.AttributeDefinition;
import com.amazonaws.services.dynamodbv2.model.BillingMode;
import com.amazonaws.services.dynamodbv2.model.ConditionalCheckFailedException;
import com.amazonaws.services.dynamodbv2.model.CreateTableRequest;
import com.amazonaws.services.dynamodbv2.model.CreateTableResult;
import com.amazonaws.services.dynamodbv2.model.DescribeTableRequest;
import com.amazonaws.services.dynamodbv2.model.KeySchemaElement;
import com.amazonaws.services.dynamodbv2.model.KeyType;
import com.amazonaws.services.dynamodbv2.model.ProvisionedThroughput;
import com.amazonaws.services.dynamodbv2.model.ResourceNotFoundException;
import com.amazonaws.services.dynamodbv2.model.ReturnValue;
import com.amazonaws.services.dynamodbv2.model.ScalarAttributeType;
import com.amazonaws.services.dynamodbv2.model.TimeToLiveSpecification;
import com.amazonaws.services.dynamodbv2.model.UpdateTimeToLiveRequest;
import com.amazonaws.waiters.FixedDelayStrategy;
import com.amazonaws.waiters.MaxAttemptsRetryStrategy;
import com.amazonaws.waiters.PollingStrategy;
import com.amazonaws.waiters.Waiter;
import com.amazonaws.waiters.WaiterHandler;
import com.amazonaws.waiters.WaiterParameters;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.integration.metadata.ConcurrentMetadataStore;
import org.springframework.util.Assert;

public class DynamoDbMetadataStore
implements ConcurrentMetadataStore,
InitializingBean {
    public static final String DEFAULT_TABLE_NAME = "SpringIntegrationMetadataStore";
    private static final Log logger = LogFactory.getLog(DynamoDbMetadataStore.class);
    private static final String KEY = "KEY";
    private static final String VALUE = "VALUE";
    private static final String TTL = "TTL";
    private final AmazonDynamoDBAsync dynamoDB;
    private final Table table;
    private final CountDownLatch createTableLatch = new CountDownLatch(1);
    private int createTableRetries = 25;
    private int createTableDelay = 1;
    private BillingMode billingMode = BillingMode.PAY_PER_REQUEST;
    private long readCapacity = 1L;
    private long writeCapacity = 1L;
    private Integer timeToLive;
    private volatile boolean initialized;

    public DynamoDbMetadataStore(AmazonDynamoDBAsync dynamoDB) {
        this(dynamoDB, DEFAULT_TABLE_NAME);
    }

    public DynamoDbMetadataStore(AmazonDynamoDBAsync dynamoDB, String tableName) {
        Assert.notNull((Object)dynamoDB, (String)"'dynamoDB' must not be null.");
        Assert.hasText((String)tableName, (String)"'tableName' must not be empty.");
        this.dynamoDB = dynamoDB;
        this.table = new DynamoDB((AmazonDynamoDB)this.dynamoDB).getTable(tableName);
    }

    public void setCreateTableRetries(int createTableRetries) {
        this.createTableRetries = createTableRetries;
    }

    public void setCreateTableDelay(int createTableDelay) {
        this.createTableDelay = createTableDelay;
    }

    public void setBillingMode(BillingMode billingMode) {
        Assert.notNull((Object)billingMode, (String)"'billingMode' must not be null");
        this.billingMode = billingMode;
    }

    public void setReadCapacity(long readCapacity) {
        this.readCapacity = readCapacity;
    }

    public void setWriteCapacity(long writeCapacity) {
        this.writeCapacity = writeCapacity;
    }

    public void setTimeToLive(int timeToLive) {
        this.timeToLive = timeToLive;
    }

    public void afterPropertiesSet() {
        try {
            if (this.isTableAvailable()) {
                return;
            }
            CreateTableRequest createTableRequest = new CreateTableRequest().withTableName(this.table.getTableName()).withKeySchema(new KeySchemaElement[]{new KeySchemaElement(KEY, KeyType.HASH)}).withAttributeDefinitions(new AttributeDefinition[]{new AttributeDefinition(KEY, ScalarAttributeType.S)}).withBillingMode(this.billingMode);
            if (BillingMode.PROVISIONED.equals((Object)this.billingMode)) {
                createTableRequest.withProvisionedThroughput(new ProvisionedThroughput(Long.valueOf(this.readCapacity), Long.valueOf(this.writeCapacity)));
            }
            this.dynamoDB.createTableAsync(createTableRequest, (AsyncHandler)new AsyncHandler<CreateTableRequest, CreateTableResult>(){

                public void onError(Exception e) {
                    logger.error((Object)("Cannot create DynamoDb table: " + DynamoDbMetadataStore.this.table.getTableName()), (Throwable)e);
                    DynamoDbMetadataStore.this.createTableLatch.countDown();
                }

                public void onSuccess(CreateTableRequest request, CreateTableResult createTableResult) {
                    Waiter waiter = DynamoDbMetadataStore.this.dynamoDB.waiters().tableExists();
                    WaiterParameters waiterParameters = new WaiterParameters((AmazonWebServiceRequest)new DescribeTableRequest(DynamoDbMetadataStore.this.table.getTableName())).withPollingStrategy(new PollingStrategy((PollingStrategy.RetryStrategy)new MaxAttemptsRetryStrategy(DynamoDbMetadataStore.this.createTableRetries), (PollingStrategy.DelayStrategy)new FixedDelayStrategy(DynamoDbMetadataStore.this.createTableDelay)));
                    waiter.runAsync(waiterParameters, (WaiterHandler)new WaiterHandler<DescribeTableRequest>(){

                        public void onWaitSuccess(DescribeTableRequest request) {
                            DynamoDbMetadataStore.this.updateTimeToLiveIfAny();
                            DynamoDbMetadataStore.this.createTableLatch.countDown();
                            DynamoDbMetadataStore.this.table.describe();
                        }

                        public void onWaitFailure(Exception e) {
                            logger.error((Object)("Cannot describe DynamoDb table: " + DynamoDbMetadataStore.this.table.getTableName()), (Throwable)e);
                            DynamoDbMetadataStore.this.createTableLatch.countDown();
                        }
                    });
                }
            });
        }
        finally {
            this.initialized = true;
        }
    }

    private boolean isTableAvailable() {
        try {
            this.table.describe();
            this.updateTimeToLiveIfAny();
            this.createTableLatch.countDown();
            return true;
        }
        catch (ResourceNotFoundException e) {
            if (logger.isInfoEnabled()) {
                logger.info((Object)("No table '" + this.table.getTableName() + "'. Creating one..."));
            }
            return false;
        }
    }

    private void updateTimeToLiveIfAny() {
        block3: {
            if (this.timeToLive != null) {
                UpdateTimeToLiveRequest updateTimeToLiveRequest = new UpdateTimeToLiveRequest().withTableName(this.table.getTableName()).withTimeToLiveSpecification(new TimeToLiveSpecification().withAttributeName(TTL).withEnabled(Boolean.valueOf(this.timeToLive > 0)));
                try {
                    this.dynamoDB.updateTimeToLive(updateTimeToLiveRequest);
                }
                catch (AmazonDynamoDBException e) {
                    if (!logger.isWarnEnabled()) break block3;
                    logger.warn((Object)"The error during 'updateTimeToLive' request", (Throwable)e);
                }
            }
        }
    }

    private void awaitForActive() {
        Assert.state((boolean)this.initialized, () -> "The component has not been initialized: " + this + ".\n Is it declared as a bean?");
        try {
            this.createTableLatch.await(this.createTableRetries * this.createTableDelay, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("The DynamoDb table " + this.table.getTableName() + " has not been created during " + this.createTableRetries * this.createTableDelay + " seconds");
        }
    }

    public void put(String key, String value) {
        Assert.hasText((String)key, (String)"'key' must not be empty.");
        Assert.hasText((String)value, (String)"'value' must not be empty.");
        this.awaitForActive();
        Item item = new Item().withPrimaryKey(KEY, (Object)key).withString(VALUE, value);
        if (this.timeToLive != null && this.timeToLive > 0) {
            item = item.withLong(TTL, (System.currentTimeMillis() + (long)this.timeToLive.intValue()) / 1000L);
        }
        this.table.putItem(item);
    }

    public String get(String key) {
        Assert.hasText((String)key, (String)"'key' must not be empty.");
        this.awaitForActive();
        Item item = this.table.getItem(KEY, (Object)key);
        return DynamoDbMetadataStore.getValueIfAny(item);
    }

    public String putIfAbsent(String key, String value) {
        Assert.hasText((String)key, (String)"'key' must not be empty.");
        Assert.hasText((String)value, (String)"'value' must not be empty.");
        this.awaitForActive();
        UpdateItemSpec updateItemSpec = new UpdateItemSpec().withPrimaryKey(KEY, (Object)key).withAttributeUpdate(new AttributeUpdate[]{new AttributeUpdate(VALUE).put((Object)value)}).withExpected(new Expected[]{new Expected(KEY).notExist()});
        if (this.timeToLive != null && this.timeToLive > 0) {
            updateItemSpec = updateItemSpec.addAttributeUpdate(new AttributeUpdate(TTL).put((Object)((System.currentTimeMillis() + (long)this.timeToLive.intValue()) / 1000L)));
        }
        try {
            this.table.updateItem(updateItemSpec);
            return null;
        }
        catch (ConditionalCheckFailedException e) {
            return this.get(key);
        }
    }

    public boolean replace(String key, String oldValue, String newValue) {
        Assert.hasText((String)key, (String)"'key' must not be empty.");
        Assert.hasText((String)oldValue, (String)"'value' must not be empty.");
        Assert.hasText((String)newValue, (String)"'newValue' must not be empty.");
        this.awaitForActive();
        UpdateItemSpec updateItemSpec = new UpdateItemSpec().withPrimaryKey(KEY, (Object)key).withAttributeUpdate(new AttributeUpdate[]{new AttributeUpdate(VALUE).put((Object)newValue)}).withExpected(new Expected[]{new Expected(VALUE).eq((Object)oldValue)}).withReturnValues(ReturnValue.UPDATED_NEW);
        if (this.timeToLive != null && this.timeToLive > 0) {
            updateItemSpec = updateItemSpec.addAttributeUpdate(new AttributeUpdate(TTL).put((Object)((System.currentTimeMillis() + (long)this.timeToLive.intValue()) / 1000L)));
        }
        try {
            return this.table.updateItem(updateItemSpec).getItem() != null;
        }
        catch (ConditionalCheckFailedException e) {
            return false;
        }
    }

    public String remove(String key) {
        Assert.hasText((String)key, (String)"'key' must not be empty.");
        this.awaitForActive();
        Item item = this.table.deleteItem(new DeleteItemSpec().withPrimaryKey(KEY, (Object)key).withReturnValues(ReturnValue.ALL_OLD)).getItem();
        return DynamoDbMetadataStore.getValueIfAny(item);
    }

    private static String getValueIfAny(Item item) {
        if (item != null) {
            return item.getString(VALUE);
        }
        return null;
    }

    public String toString() {
        return "DynamoDbMetadataStore{table=" + this.table + ", createTableRetries=" + this.createTableRetries + ", createTableDelay=" + this.createTableDelay + ", billingMode=" + this.billingMode + ", readCapacity=" + this.readCapacity + ", writeCapacity=" + this.writeCapacity + ", timeToLive=" + this.timeToLive + '}';
    }
}

