/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.vault.config;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import org.springframework.boot.system.SystemProperties;
import org.springframework.cloud.vault.config.GcpIamCredentialsAuthenticationFactory;
import org.springframework.cloud.vault.config.VaultProperties;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;
import org.springframework.vault.authentication.AppRoleAuthentication;
import org.springframework.vault.authentication.AppRoleAuthenticationOptions;
import org.springframework.vault.authentication.AwsEc2Authentication;
import org.springframework.vault.authentication.AwsEc2AuthenticationOptions;
import org.springframework.vault.authentication.AwsIamAuthentication;
import org.springframework.vault.authentication.AwsIamAuthenticationOptions;
import org.springframework.vault.authentication.AzureMsiAuthentication;
import org.springframework.vault.authentication.AzureMsiAuthenticationOptions;
import org.springframework.vault.authentication.ClientAuthentication;
import org.springframework.vault.authentication.ClientCertificateAuthentication;
import org.springframework.vault.authentication.ClientCertificateAuthenticationOptions;
import org.springframework.vault.authentication.CubbyholeAuthentication;
import org.springframework.vault.authentication.CubbyholeAuthenticationOptions;
import org.springframework.vault.authentication.GcpComputeAuthentication;
import org.springframework.vault.authentication.GcpComputeAuthenticationOptions;
import org.springframework.vault.authentication.GitHubAuthentication;
import org.springframework.vault.authentication.GitHubAuthenticationOptions;
import org.springframework.vault.authentication.KubernetesAuthentication;
import org.springframework.vault.authentication.KubernetesAuthenticationOptions;
import org.springframework.vault.authentication.KubernetesServiceAccountTokenFile;
import org.springframework.vault.authentication.PcfAuthentication;
import org.springframework.vault.authentication.PcfAuthenticationOptions;
import org.springframework.vault.authentication.ResourceCredentialSupplier;
import org.springframework.vault.authentication.TokenAuthentication;
import org.springframework.vault.support.VaultToken;
import org.springframework.web.client.RestOperations;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider;
import software.amazon.awssdk.regions.Region;

class ClientAuthenticationFactory {
    private static final boolean googleCredentialsPresent = ClassUtils.isPresent((String)"com.google.auth.oauth2.GoogleCredentials", (ClassLoader)ClientAuthenticationFactory.class.getClassLoader());
    private final VaultProperties vaultProperties;
    private final RestOperations restOperations;
    private final RestOperations externalRestOperations;

    ClientAuthenticationFactory(VaultProperties vaultProperties, RestOperations restOperations, RestOperations externalRestOperations) {
        this.vaultProperties = vaultProperties;
        this.restOperations = restOperations;
        this.externalRestOperations = externalRestOperations;
    }

    ClientAuthentication createClientAuthentication() {
        return switch (this.vaultProperties.getAuthentication()) {
            case VaultProperties.AuthenticationMethod.APPROLE -> this.appRoleAuthentication(this.vaultProperties);
            case VaultProperties.AuthenticationMethod.AWS_EC2 -> this.awsEc2Authentication(this.vaultProperties);
            case VaultProperties.AuthenticationMethod.AWS_IAM -> this.awsIamAuthentication(this.vaultProperties);
            case VaultProperties.AuthenticationMethod.AZURE_MSI -> this.azureMsiAuthentication(this.vaultProperties);
            case VaultProperties.AuthenticationMethod.CERT -> this.certificateAuthentication(this.vaultProperties);
            case VaultProperties.AuthenticationMethod.CUBBYHOLE -> this.cubbyholeAuthentication();
            case VaultProperties.AuthenticationMethod.GCP_GCE -> this.gcpGceAuthentication(this.vaultProperties);
            case VaultProperties.AuthenticationMethod.GCP_IAM -> this.gcpIamAuthentication(this.vaultProperties);
            case VaultProperties.AuthenticationMethod.GITHUB -> this.gitHubAuthentication(this.vaultProperties);
            case VaultProperties.AuthenticationMethod.KUBERNETES -> this.kubernetesAuthentication(this.vaultProperties);
            case VaultProperties.AuthenticationMethod.PCF -> this.pcfAuthentication(this.vaultProperties);
            case VaultProperties.AuthenticationMethod.TOKEN -> this.tokenAuthentication(this.vaultProperties);
            default -> throw new UnsupportedOperationException(String.format("Client authentication %s not supported", new Object[]{this.vaultProperties.getAuthentication()}));
        };
    }

    private ClientAuthentication appRoleAuthentication(VaultProperties vaultProperties) {
        AppRoleAuthenticationOptions options = ClientAuthenticationFactory.getAppRoleAuthenticationOptions(vaultProperties);
        return new AppRoleAuthentication(options, this.restOperations);
    }

    static AppRoleAuthenticationOptions getAppRoleAuthenticationOptions(VaultProperties vaultProperties) {
        VaultProperties.AppRoleProperties appRole = vaultProperties.getAppRole();
        AppRoleAuthenticationOptions.AppRoleAuthenticationOptionsBuilder builder = AppRoleAuthenticationOptions.builder().path(appRole.getAppRolePath());
        if (StringUtils.hasText((String)appRole.getRole())) {
            builder.appRole(appRole.getRole());
        }
        AppRoleAuthenticationOptions.RoleId roleId = ClientAuthenticationFactory.getRoleId(vaultProperties, appRole);
        AppRoleAuthenticationOptions.SecretId secretId = ClientAuthenticationFactory.getSecretId(vaultProperties, appRole);
        builder.roleId(roleId).secretId(secretId);
        return builder.build();
    }

    private static AppRoleAuthenticationOptions.RoleId getRoleId(VaultProperties vaultProperties, VaultProperties.AppRoleProperties appRole) {
        if (StringUtils.hasText((String)appRole.getRoleId())) {
            return AppRoleAuthenticationOptions.RoleId.provided((String)appRole.getRoleId());
        }
        if (StringUtils.hasText((String)vaultProperties.getToken()) && StringUtils.hasText((String)appRole.getRole())) {
            return AppRoleAuthenticationOptions.RoleId.pull((VaultToken)VaultToken.of((String)vaultProperties.getToken()));
        }
        if (StringUtils.hasText((String)vaultProperties.getToken())) {
            return AppRoleAuthenticationOptions.RoleId.wrapped((VaultToken)VaultToken.of((String)vaultProperties.getToken()));
        }
        throw new IllegalArgumentException("Cannot configure RoleId. Any of role-id, initial token, or initial token and role name must be configured.");
    }

    private static AppRoleAuthenticationOptions.SecretId getSecretId(VaultProperties vaultProperties, VaultProperties.AppRoleProperties appRole) {
        if (StringUtils.hasText((String)appRole.getSecretId())) {
            return AppRoleAuthenticationOptions.SecretId.provided((String)appRole.getSecretId());
        }
        if (StringUtils.hasText((String)vaultProperties.getToken()) && StringUtils.hasText((String)appRole.getRole())) {
            return AppRoleAuthenticationOptions.SecretId.pull((VaultToken)VaultToken.of((String)vaultProperties.getToken()));
        }
        if (StringUtils.hasText((String)vaultProperties.getToken())) {
            return AppRoleAuthenticationOptions.SecretId.wrapped((VaultToken)VaultToken.of((String)vaultProperties.getToken()));
        }
        return AppRoleAuthenticationOptions.SecretId.absent();
    }

    private ClientAuthentication awsEc2Authentication(VaultProperties vaultProperties) {
        VaultProperties.AwsEc2Properties awsEc2 = vaultProperties.getAwsEc2();
        AwsEc2AuthenticationOptions.Nonce nonce = StringUtils.hasText((String)awsEc2.getNonce()) ? AwsEc2AuthenticationOptions.Nonce.provided((char[])awsEc2.getNonce().toCharArray()) : AwsEc2AuthenticationOptions.Nonce.generated();
        AwsEc2AuthenticationOptions authenticationOptions = AwsEc2AuthenticationOptions.builder().role(awsEc2.getRole()).path(awsEc2.getAwsEc2Path()).nonce(nonce).identityDocumentUri(awsEc2.getIdentityDocument()).build();
        return new AwsEc2Authentication(authenticationOptions, this.restOperations, this.externalRestOperations);
    }

    ClientAuthentication awsIamAuthentication(VaultProperties vaultProperties) {
        VaultProperties.AwsIamProperties awsIam = vaultProperties.getAwsIam();
        AwsIamAuthenticationOptions.AwsIamAuthenticationOptionsBuilder builder = AwsIamAuthenticationOptions.builder();
        AwsCredentialsProvider credentialsProvider = AwsCredentialProvider.getAwsCredentialsProvider();
        if (StringUtils.hasText((String)awsIam.getRegion())) {
            builder.region(Region.of((String)awsIam.getRegion()));
        }
        if (StringUtils.hasText((String)awsIam.getRole())) {
            builder.role(awsIam.getRole());
        }
        if (StringUtils.hasText((String)awsIam.getServerName())) {
            builder.serverName(awsIam.getServerName());
        }
        if (awsIam.getEndpointUri() != null) {
            builder.endpointUri(awsIam.getEndpointUri());
        }
        builder.path(awsIam.getAwsPath()).credentialsProvider(credentialsProvider);
        AwsIamAuthenticationOptions options = builder.credentialsProvider(credentialsProvider).build();
        return new AwsIamAuthentication(options, this.restOperations);
    }

    private ClientAuthentication azureMsiAuthentication(VaultProperties vaultProperties) {
        VaultProperties.AzureMsiProperties azureMsi = vaultProperties.getAzureMsi();
        Assert.hasText((String)azureMsi.getRole(), (String)"Azure role (spring.cloud.vault.azure-msi.role) must not be empty");
        AzureMsiAuthenticationOptions options = AzureMsiAuthenticationOptions.builder().role(azureMsi.getRole()).path(azureMsi.getAzurePath()).instanceMetadataUri(azureMsi.getMetadataService()).identityTokenServiceUri(azureMsi.getIdentityTokenService()).build();
        return new AzureMsiAuthentication(options, this.restOperations, this.externalRestOperations);
    }

    private ClientAuthentication cubbyholeAuthentication() {
        Assert.hasText((String)this.vaultProperties.getToken(), (String)"Initial Token (spring.cloud.vault.token) for Cubbyhole authentication must not be empty");
        CubbyholeAuthenticationOptions options = CubbyholeAuthenticationOptions.builder().wrapped().initialToken(VaultToken.of((String)this.vaultProperties.getToken())).build();
        return new CubbyholeAuthentication(options, this.restOperations);
    }

    private ClientAuthentication gcpGceAuthentication(VaultProperties vaultProperties) {
        VaultProperties.GcpGceProperties gcp = vaultProperties.getGcpGce();
        Assert.hasText((String)gcp.getRole(), (String)"Role (spring.cloud.vault.gcp-gce.role) must not be empty");
        GcpComputeAuthenticationOptions.GcpComputeAuthenticationOptionsBuilder builder = GcpComputeAuthenticationOptions.builder().path(gcp.getGcpPath()).role(gcp.getRole());
        if (StringUtils.hasText((String)gcp.getServiceAccount())) {
            builder.serviceAccount(gcp.getServiceAccount());
        }
        return new GcpComputeAuthentication(builder.build(), this.restOperations, this.externalRestOperations);
    }

    private ClientAuthentication gcpIamAuthentication(VaultProperties vaultProperties) {
        if (googleCredentialsPresent) {
            return GcpIamCredentialsAuthenticationFactory.create(vaultProperties, this.restOperations);
        }
        throw new IllegalStateException("Cannot create authentication mechanism for GCP IAM. This method requires the google-auth-library-oauth2-http dependency.");
    }

    private ClientAuthentication gitHubAuthentication(VaultProperties vaultProperties) {
        GitHubAuthenticationOptions options = GitHubAuthenticationOptions.builder().path(vaultProperties.getGithub().getGithubPath()).token(this.getGitHubToken(vaultProperties)).build();
        return new GitHubAuthentication(options, this.restOperations);
    }

    private String getGitHubToken(VaultProperties vaultProperties) {
        VaultProperties.GithubProperties github = vaultProperties.getGithub();
        String token = github.getToken();
        if (StringUtils.hasText((String)token)) {
            return token;
        }
        if (github.isAllowCliToken()) {
            return ClientAuthenticationFactory.invokeGitHubCli("auth", "token");
        }
        throw new IllegalStateException("GitHub Token (spring.cloud.vault.github.token) must not be empty");
    }

    private static String invokeGitHubCli(String command, String subcommand) {
        ProcessBuilder processBuilder = new ProcessBuilder("gh", command, subcommand);
        try {
            Process process = processBuilder.start();
            int exitCode = process.waitFor();
            if (exitCode != 0) {
                String error = new String(process.getErrorStream().readAllBytes());
                throw new IllegalStateException(String.format("Cannot not retrieve GitHub token using GitHub CLI\n\tCommand: %s (Exit code %d)\n\tError:   %s", StringUtils.collectionToDelimitedString(processBuilder.command(), (String)" "), exitCode, error));
            }
            return new String(process.getInputStream().readAllBytes()).strip();
        }
        catch (IOException | InterruptedException ex) {
            throw new IllegalStateException(String.format("Cannot not retrieve GitHub token using GitHub CLI\n\tCommand: %s", StringUtils.collectionToDelimitedString(processBuilder.command(), (String)" ")), ex);
        }
    }

    private ClientAuthentication kubernetesAuthentication(VaultProperties vaultProperties) {
        VaultProperties.KubernetesProperties kubernetes = vaultProperties.getKubernetes();
        Assert.hasText((String)kubernetes.getRole(), (String)"Role (spring.cloud.vault.kubernetes.role) must not be empty");
        Assert.hasText((String)kubernetes.getServiceAccountTokenFile(), (String)"Service account token file (spring.cloud.vault.kubernetes.service-account-token-file) must not be empty");
        KubernetesAuthenticationOptions options = KubernetesAuthenticationOptions.builder().path(kubernetes.getKubernetesPath()).role(kubernetes.getRole()).jwtSupplier((Supplier)new KubernetesServiceAccountTokenFile(kubernetes.getServiceAccountTokenFile())).build();
        return new KubernetesAuthentication(options, this.restOperations);
    }

    private ClientAuthentication pcfAuthentication(VaultProperties vaultProperties) {
        VaultProperties.PcfProperties pcfProperties = vaultProperties.getPcf();
        Assert.isTrue((boolean)ClassUtils.isPresent((String)"org.bouncycastle.crypto.signers.PSSSigner", (ClassLoader)this.getClass().getClassLoader()), (String)"BouncyCastle (bcpkix-jdk15on) must be on the classpath");
        Assert.hasText((String)pcfProperties.getRole(), (String)"Role (spring.cloud.vault.pcf.role) must not be empty");
        PcfAuthenticationOptions.PcfAuthenticationOptionsBuilder builder = PcfAuthenticationOptions.builder().role(pcfProperties.getRole()).path(pcfProperties.getPcfPath());
        if (pcfProperties.getInstanceCertificate() != null) {
            builder.instanceCertificate((Supplier)new ResourceCredentialSupplier(pcfProperties.getInstanceCertificate()));
        }
        if (pcfProperties.getInstanceKey() != null) {
            builder.instanceKey((Supplier)new ResourceCredentialSupplier(pcfProperties.getInstanceKey()));
        }
        return new PcfAuthentication(builder.build(), this.restOperations);
    }

    private ClientAuthentication certificateAuthentication(VaultProperties vaultProperties) {
        ClientCertificateAuthenticationOptions options = ClientCertificateAuthenticationOptions.builder().path(vaultProperties.getSsl().getCertAuthPath()).build();
        return new ClientCertificateAuthentication(options, this.restOperations);
    }

    private ClientAuthentication tokenAuthentication(VaultProperties vaultProperties) {
        if (StringUtils.hasText((String)vaultProperties.getToken())) {
            return new TokenAuthentication(vaultProperties.getToken());
        }
        Path vaultTokenPath = Paths.get(SystemProperties.get((String[])new String[]{"user.home"}), ".vault-token");
        if (Files.exists(vaultTokenPath, new LinkOption[0])) {
            try {
                return new TokenAuthentication(new String(Files.readAllBytes(vaultTokenPath), StandardCharsets.UTF_8));
            }
            catch (IOException ex) {
                throw new IllegalStateException(String.format("Could not retrieve vault token from %s", vaultTokenPath), ex);
            }
        }
        throw new IllegalStateException("Cannot create authentication mechanism for TOKEN. This method requires either a Token (spring.cloud.vault.token) or a token file at ~/.vault-token.");
    }

    private static class AwsCredentialProvider {
        private AwsCredentialProvider() {
        }

        private static AwsCredentialsProvider getAwsCredentialsProvider() {
            final DefaultCredentialsProvider backingCredentialsProvider = DefaultCredentialsProvider.create();
            final AwsCredentials firstAccess = backingCredentialsProvider.resolveCredentials();
            final AtomicReference<AwsCredentials> once = new AtomicReference<AwsCredentials>(firstAccess);
            return new AwsCredentialsProvider(){

                public AwsCredentials resolveCredentials() {
                    if (once.compareAndSet(firstAccess, null)) {
                        return firstAccess;
                    }
                    return backingCredentialsProvider.resolveCredentials();
                }
            };
        }
    }
}

