/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.sleuth.instrument.jdbc;

import java.net.URI;
import java.sql.Connection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.sql.CommonDataSource;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.cloud.sleuth.Span;
import org.springframework.cloud.sleuth.SpanAndScope;
import org.springframework.cloud.sleuth.Tracer;
import org.springframework.cloud.sleuth.docs.AssertingSpan;
import org.springframework.cloud.sleuth.docs.AssertingSpanBuilder;
import org.springframework.cloud.sleuth.docs.DocumentedSpan;
import org.springframework.cloud.sleuth.docs.EventValue;
import org.springframework.cloud.sleuth.docs.TagKey;
import org.springframework.cloud.sleuth.instrument.jdbc.SleuthJdbcSpan;
import org.springframework.cloud.sleuth.instrument.jdbc.TraceListenerStrategySpanCustomizer;
import org.springframework.cloud.sleuth.instrument.jdbc.TraceType;
import org.springframework.lang.Nullable;
import org.springframework.util.StringUtils;

class TraceListenerStrategy<CON, STMT, RS> {
    private static final Log log = LogFactory.getLog(TraceListenerStrategy.class);
    private final Map<CON, ConnectionInfo> openConnections = new ConcurrentHashMap<CON, ConnectionInfo>();
    private static final Pattern URL_SERVICE_NAME_FINDER = Pattern.compile("sleuthServiceName=(.*?)(?:&|$)");
    private final Tracer tracer;
    private final List<TraceType> traceTypes;
    private final List<TraceListenerStrategySpanCustomizer> customizers;

    TraceListenerStrategy(Tracer tracer, List<TraceType> traceTypes, List<TraceListenerStrategySpanCustomizer> customizers) {
        this.tracer = tracer;
        this.traceTypes = traceTypes;
        this.customizers = customizers;
    }

    void beforeGetConnection(CON connectionKey, @Nullable CommonDataSource dataSource, String dataSourceName) {
        if (log.isTraceEnabled()) {
            log.trace((Object)("Before get connection key [" + connectionKey + "] - current span is [" + this.tracer.currentSpan() + "]"));
        }
        SpanAndScope spanAndScope = null;
        if (this.traceTypes.contains((Object)TraceType.CONNECTION)) {
            AssertingSpanBuilder connectionSpanBuilder = AssertingSpanBuilder.of((DocumentedSpan)SleuthJdbcSpan.JDBC_CONNECTION_SPAN, (Span.Builder)this.tracer.spanBuilder()).name(SleuthJdbcSpan.JDBC_CONNECTION_SPAN.getName());
            connectionSpanBuilder.remoteServiceName(dataSourceName);
            connectionSpanBuilder.kind(Span.Kind.CLIENT);
            this.customizers.stream().filter(customizer -> customizer.isApplicable(dataSource)).forEach(customizer -> customizer.customizeConnectionSpan(dataSource, (Span.Builder)connectionSpanBuilder));
            AssertingSpan connectionSpan = connectionSpanBuilder.start();
            spanAndScope = new SpanAndScope((Span)connectionSpan, this.tracer.withSpan((Span)connectionSpan));
            if (log.isTraceEnabled()) {
                log.trace((Object)("Started client span before connection [" + connectionSpan + "] - current span is [" + this.tracer.currentSpan() + "]"));
            }
        }
        ConnectionInfo connectionInfo = new ConnectionInfo(spanAndScope);
        this.openConnections.put(connectionKey, connectionInfo);
    }

    void afterGetConnection(CON connectionKey, Connection connection, Throwable t) {
        if (log.isTraceEnabled()) {
            log.trace((Object)("After get connection [" + connectionKey + "]. Current span is [" + this.tracer.currentSpan() + "]"));
        }
        this.openConnections.get(connectionKey).getSpan().ifPresent(spanAndScope -> this.parseServerIpAndPort(connection, spanAndScope.getSpan()));
        if (t != null) {
            ConnectionInfo connectionInfo = this.openConnections.remove(connectionKey);
            connectionInfo.getSpan().ifPresent(connectionSpan -> {
                this.parseServerIpAndPort(connection, connectionSpan.getSpan());
                if (log.isTraceEnabled()) {
                    log.trace((Object)("Closing client span due to exception [" + connectionSpan.getSpan() + "] - current span is [" + this.tracer.currentSpan() + "]"));
                }
                connectionSpan.getSpan().error(t);
                connectionSpan.close();
                if (log.isTraceEnabled()) {
                    log.trace((Object)("Current span [" + this.tracer.currentSpan() + "]"));
                }
            });
        }
    }

    void beforeQuery(CON connectionKey, Connection connection, STMT statementKey, String dataSourceName) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)("Before query - connection [" + connectionKey + "] and current span [" + this.tracer.currentSpan() + "]"));
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            if (log.isTraceEnabled()) {
                log.trace((Object)"Connection may be closed after statement preparation, but before statement execution");
            }
            return;
        }
        SpanAndScope spanAndScope = null;
        if (this.traceTypes.contains((Object)TraceType.QUERY)) {
            AssertingSpanBuilder statementSpanBuilder = AssertingSpanBuilder.of((DocumentedSpan)SleuthJdbcSpan.JDBC_QUERY_SPAN, (Span.Builder)this.tracer.spanBuilder()).name(String.format(SleuthJdbcSpan.JDBC_QUERY_SPAN.getName(), "query"));
            statementSpanBuilder.remoteServiceName(dataSourceName);
            this.parseServerIpAndPort(connection, (Span.Builder)statementSpanBuilder);
            statementSpanBuilder.kind(Span.Kind.CLIENT);
            Span statementSpan = statementSpanBuilder.start();
            spanAndScope = new SpanAndScope(statementSpan, this.tracer.withSpan(statementSpan));
            if (log.isTraceEnabled()) {
                log.trace((Object)("Started client span before query [" + statementSpan + "] - current span is [" + this.tracer.currentSpan() + "]"));
            }
        }
        StatementInfo statementInfo = new StatementInfo(spanAndScope);
        connectionInfo.getNestedStatements().put(statementKey, statementInfo);
    }

    void addQueryRowCount(CON connectionKey, STMT statementKey, int rowCount) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)("Add query row count for connection key [" + connectionKey + "]"));
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            if (log.isTraceEnabled()) {
                log.trace((Object)"Connection is already closed");
            }
            return;
        }
        StatementInfo statementInfo = connectionInfo.getNestedStatements().get(statementKey);
        statementInfo.getSpan().ifPresent(statementSpan -> AssertingSpan.of((DocumentedSpan)SleuthJdbcSpan.JDBC_QUERY_SPAN, (Span)statementSpan.getSpan()).tag((TagKey)SleuthJdbcSpan.QueryTags.ROW_COUNT, String.valueOf(rowCount)));
    }

    void afterQuery(CON connectionKey, STMT statementKey, String sql, Throwable t) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)("After query for connection key [" + connectionKey + "]"));
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            if (log.isTraceEnabled()) {
                log.trace((Object)("Connection may be closed after statement preparation, but before statement execution. Current span is [" + this.tracer.currentSpan() + "]"));
            }
            return;
        }
        StatementInfo statementInfo = connectionInfo.getNestedStatements().get(statementKey);
        statementInfo.getSpan().ifPresent(statementSpan -> {
            this.updateQuerySpan(sql, t, (SpanAndScope)statementSpan);
            statementSpan.close();
            if (log.isTraceEnabled()) {
                log.trace((Object)("Current span [" + this.tracer.currentSpan() + "]"));
            }
        });
    }

    private void updateQuerySpan(String sql, Throwable t, SpanAndScope statementSpan) {
        AssertingSpan.of((DocumentedSpan)SleuthJdbcSpan.JDBC_QUERY_SPAN, (Span)statementSpan.getSpan()).tag((TagKey)SleuthJdbcSpan.QueryTags.QUERY, sql).name(this.spanName(sql));
        if (t != null) {
            statementSpan.getSpan().error(t);
        }
        if (log.isTraceEnabled()) {
            log.trace((Object)("Closing statement span [" + statementSpan + "] - current span is [" + this.tracer.currentSpan() + "]"));
        }
    }

    void beforeResultSetNext(CON connectionKey, Connection connection, STMT statementKey, RS resultSetKey, String dataSourceName) {
        if (log.isTraceEnabled()) {
            log.trace((Object)"Before result set next");
        }
        if (!this.traceTypes.contains((Object)TraceType.FETCH)) {
            return;
        }
        ConnectionInfo connectionInfo = this.openConnections.get(connectionKey);
        if (connectionInfo == null) {
            if (log.isTraceEnabled()) {
                log.trace((Object)"No connection info, skipping");
            }
            return;
        }
        if (connectionInfo.getNestedResultSetSpans().containsKey(resultSetKey)) {
            if (log.isTraceEnabled()) {
                log.trace((Object)"ResultSet span is already created");
            }
            return;
        }
        AssertingSpanBuilder resultSetSpanBuilder = AssertingSpanBuilder.of((DocumentedSpan)SleuthJdbcSpan.JDBC_RESULT_SET_SPAN, (Span.Builder)this.tracer.spanBuilder()).name(SleuthJdbcSpan.JDBC_RESULT_SET_SPAN.getName());
        resultSetSpanBuilder.remoteServiceName(dataSourceName);
        resultSetSpanBuilder.kind(Span.Kind.CLIENT);
        this.parseServerIpAndPort(connection, (Span.Builder)resultSetSpanBuilder);
        AssertingSpan resultSetSpan = resultSetSpanBuilder.start();
        SpanAndScope SpanAndScope2 = new SpanAndScope((Span)resultSetSpan, this.tracer.withSpan((Span)resultSetSpan));
        if (log.isTraceEnabled()) {
            log.trace((Object)("Started client result set span [" + resultSetSpan + "] - current span is [" + this.tracer.currentSpan() + "]"));
        }
        connectionInfo.getNestedResultSetSpans().put(resultSetKey, SpanAndScope2);
        StatementInfo statementInfo = connectionInfo.getNestedStatements().get(statementKey);
        if (statementInfo != null) {
            statementInfo.getNestedResultSetSpans().put(resultSetKey, SpanAndScope2);
        }
    }

    void afterResultSetClose(CON connectionKey, RS resultSetKey, int rowCount, Throwable t) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)"After result set close");
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            return;
        }
        SpanAndScope resultSetSpan = connectionInfo.getNestedResultSetSpans().remove(resultSetKey);
        if (resultSetSpan == null) {
            return;
        }
        if (rowCount != -1) {
            AssertingSpan.of((DocumentedSpan)SleuthJdbcSpan.JDBC_RESULT_SET_SPAN, (Span)resultSetSpan.getSpan()).tag((TagKey)SleuthJdbcSpan.QueryTags.ROW_COUNT, String.valueOf(rowCount));
        }
        if (t != null) {
            resultSetSpan.getSpan().error(t);
        }
        if (log.isTraceEnabled()) {
            log.trace((Object)("Closing client result set span [" + resultSetSpan + "] - current span is [" + this.tracer.currentSpan() + "]"));
        }
        resultSetSpan.close();
        if (log.isTraceEnabled()) {
            log.trace((Object)("Current span [" + this.tracer.currentSpan() + "]"));
        }
    }

    void afterStatementClose(CON connectionKey, STMT statementKey) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)"After statement close");
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            return;
        }
        StatementInfo statementInfo = connectionInfo.getNestedStatements().remove(statementKey);
        if (statementInfo != null) {
            statementInfo.getNestedResultSetSpans().forEach((resultSetKey, span) -> {
                connectionInfo.getNestedResultSetSpans().remove(resultSetKey);
                if (log.isTraceEnabled()) {
                    log.trace((Object)("Closing span after statement close [" + span.getSpan() + "] - current span is [" + this.tracer.currentSpan() + "]"));
                }
                span.close();
                if (log.isTraceEnabled()) {
                    log.trace((Object)("Current span [" + this.tracer.currentSpan() + "]"));
                }
            });
            statementInfo.getNestedResultSetSpans().clear();
        }
    }

    void afterCommit(CON connectionKey, Throwable t) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)"After commit");
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            return;
        }
        connectionInfo.getSpan().ifPresent(connectionSpan -> {
            if (t != null) {
                connectionSpan.getSpan().error(t);
            }
            AssertingSpan.of((DocumentedSpan)SleuthJdbcSpan.JDBC_QUERY_SPAN, (Span)connectionSpan.getSpan()).event((EventValue)SleuthJdbcSpan.QueryEvents.COMMIT);
        });
    }

    void afterRollback(CON connectionKey, Throwable t) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)"After rollback");
        }
        if ((connectionInfo = this.openConnections.get(connectionKey)) == null) {
            return;
        }
        connectionInfo.getSpan().ifPresent(connectionSpan -> {
            if (t != null) {
                connectionSpan.getSpan().error(t);
            } else {
                connectionSpan.getSpan().error((Throwable)new JdbcException("Transaction rolled back"));
            }
            AssertingSpan.of((DocumentedSpan)SleuthJdbcSpan.JDBC_QUERY_SPAN, (Span)connectionSpan.getSpan()).event((EventValue)SleuthJdbcSpan.QueryEvents.ROLLBACK);
        });
    }

    void afterConnectionClose(CON connectionKey, Throwable t) {
        ConnectionInfo connectionInfo;
        if (log.isTraceEnabled()) {
            log.trace((Object)("After connection close with key [" + connectionKey + "]"));
        }
        if ((connectionInfo = this.openConnections.remove(connectionKey)) == null) {
            return;
        }
        connectionInfo.getNestedResultSetSpans().values().forEach(SpanAndScope::close);
        connectionInfo.getNestedStatements().values().forEach(statementInfo -> statementInfo.getSpan().ifPresent(SpanAndScope::close));
        if (log.isTraceEnabled()) {
            log.trace((Object)("Current span after closing statements [" + this.tracer.currentSpan() + "]"));
        }
        connectionInfo.getSpan().ifPresent(connectionSpan -> {
            if (t != null) {
                connectionSpan.getSpan().error(t);
            }
            if (log.isTraceEnabled()) {
                log.trace((Object)("Closing span after connection close [" + connectionSpan.getSpan() + "] - current span is [" + this.tracer.currentSpan() + "]"));
            }
            connectionSpan.close();
            if (log.isTraceEnabled()) {
                log.trace((Object)("Current span [" + this.tracer.currentSpan() + "]"));
            }
        });
    }

    private String spanName(String sql) {
        return sql.substring(0, sql.indexOf(32)).toLowerCase(Locale.ROOT);
    }

    private void parseServerIpAndPort(Connection connection, Span.Builder span) {
        if (connection == null) {
            return;
        }
        UrlAndRemoteServiceName urlAndRemoteServiceName = this.parseServerIpAndPort(connection);
        span.remoteServiceName(urlAndRemoteServiceName.remoteServiceName);
        URI url = urlAndRemoteServiceName.url;
        if (url != null) {
            span.remoteIpAndPort(url.getHost(), url.getPort());
        }
    }

    private void parseServerIpAndPort(Connection connection, Span span) {
        if (connection == null) {
            return;
        }
        UrlAndRemoteServiceName urlAndRemoteServiceName = this.parseServerIpAndPort(connection);
        span.remoteServiceName(urlAndRemoteServiceName.remoteServiceName);
        URI url = urlAndRemoteServiceName.url;
        if (url != null) {
            span.remoteIpAndPort(url.getHost(), url.getPort());
        }
    }

    private UrlAndRemoteServiceName parseServerIpAndPort(Connection connection) {
        String remoteServiceName = "";
        try {
            String databaseName;
            String parsedServiceName;
            String urlAsString = connection.getMetaData().getURL().substring(5);
            URI url = URI.create(urlAsString.replace(" ", ""));
            Matcher matcher = URL_SERVICE_NAME_FINDER.matcher(url.toString());
            if (matcher.find() && matcher.groupCount() == 1 && (parsedServiceName = matcher.group(1)) != null && !parsedServiceName.isEmpty()) {
                remoteServiceName = parsedServiceName;
            }
            if (!StringUtils.hasText((String)remoteServiceName) && (databaseName = connection.getCatalog()) != null && !databaseName.isEmpty()) {
                remoteServiceName = databaseName;
            }
            return new UrlAndRemoteServiceName(url, remoteServiceName);
        }
        catch (Exception e) {
            return new UrlAndRemoteServiceName(null, remoteServiceName);
        }
    }

    private static final class JdbcException
    extends RuntimeException {
        JdbcException(String message) {
            super(message);
        }
    }

    private final class UrlAndRemoteServiceName {
        final URI url;
        final String remoteServiceName;

        private UrlAndRemoteServiceName(URI url, String remoteServiceName) {
            this.url = url;
            this.remoteServiceName = remoteServiceName;
        }
    }

    private final class StatementInfo {
        private final SpanAndScope span;
        private final Map<RS, SpanAndScope> nestedResultSetSpans = new ConcurrentHashMap();

        private StatementInfo(SpanAndScope span) {
            this.span = span;
        }

        Optional<SpanAndScope> getSpan() {
            return Optional.ofNullable(this.span);
        }

        Map<RS, SpanAndScope> getNestedResultSetSpans() {
            return this.nestedResultSetSpans;
        }
    }

    private final class ConnectionInfo {
        private final SpanAndScope span;
        private final Map<STMT, StatementInfo> nestedStatements = new ConcurrentHashMap();
        private final Map<RS, SpanAndScope> nestedResultSetSpans = new ConcurrentHashMap();

        private ConnectionInfo(SpanAndScope span) {
            this.span = span;
        }

        Optional<SpanAndScope> getSpan() {
            return Optional.ofNullable(this.span);
        }

        Map<STMT, StatementInfo> getNestedStatements() {
            return this.nestedStatements;
        }

        Map<RS, SpanAndScope> getNestedResultSetSpans() {
            return this.nestedResultSetSpans;
        }
    }
}

