/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.ai.vectorstore.redis.cache.semantic;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import java.lang.reflect.Type;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import org.jspecify.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.ai.chat.cache.semantic.SemanticCache;
import org.springframework.ai.chat.messages.AssistantMessage;
import org.springframework.ai.chat.model.ChatResponse;
import org.springframework.ai.chat.model.Generation;
import org.springframework.ai.embedding.EmbeddingModel;
import org.springframework.ai.vectorstore.SearchRequest;
import org.springframework.ai.vectorstore.VectorStore;
import org.springframework.ai.vectorstore.redis.RedisVectorStore;
import org.springframework.util.Assert;
import redis.clients.jedis.JedisPooled;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.search.Document;
import redis.clients.jedis.search.Query;
import redis.clients.jedis.search.SearchResult;

public final class DefaultSemanticCache
implements SemanticCache {
    private static final Logger logger = LoggerFactory.getLogger(DefaultSemanticCache.class);
    private static final String DEFAULT_INDEX_NAME = "semantic-cache-index";
    private static final String DEFAULT_PREFIX = "semantic-cache:";
    private static final Integer DEFAULT_BATCH_SIZE = 100;
    private static final double DEFAULT_SIMILARITY_THRESHOLD = 0.8;
    private final VectorStore vectorStore;
    private final double similarityThreshold;
    private final boolean useDistanceThreshold;
    private final Gson gson;
    private final String prefix;
    private final String indexName;

    private DefaultSemanticCache(VectorStore vectorStore, double similarityThreshold, String indexName, String prefix, boolean useDistanceThreshold) {
        this.vectorStore = vectorStore;
        this.similarityThreshold = similarityThreshold;
        this.useDistanceThreshold = useDistanceThreshold;
        this.prefix = prefix;
        this.indexName = indexName;
        this.gson = this.createGson();
    }

    private Gson createGson() {
        return new GsonBuilder().registerTypeAdapter(Duration.class, (Object)new DurationAdapter()).registerTypeAdapter(ChatResponse.class, (Object)new ChatResponseAdapter()).create();
    }

    @Override
    public VectorStore getStore() {
        return this.vectorStore;
    }

    @Override
    public void set(String query, ChatResponse response) {
        List existing;
        String responseJson = this.gson.toJson((Object)response);
        Assert.state((response.getResult() != null ? 1 : 0) != 0, (String)"expected a non-empty response");
        String responseText = response.getResult().getOutput().getText();
        HashMap<String, String> metadata = new HashMap<String, String>();
        metadata.put("response", responseJson);
        metadata.put("response_text", Objects.requireNonNullElse(responseText, ""));
        org.springframework.ai.document.Document document = org.springframework.ai.document.Document.builder().text(query).metadata(metadata).build();
        VectorStore vectorStore = this.vectorStore;
        if (vectorStore instanceof RedisVectorStore) {
            RedisVectorStore redisVectorStore = (RedisVectorStore)vectorStore;
            existing = redisVectorStore.searchByRange(query, this.similarityThreshold);
            if (logger.isDebugEnabled()) {
                logger.debug("Using RedisVectorStore's native VECTOR_RANGE query to find similar documents for replacement");
            }
        } else {
            existing = this.vectorStore.similaritySearch(SearchRequest.builder().query(query).topK(1).similarityThreshold(this.similarityThreshold).build());
        }
        if (!existing.isEmpty()) {
            if (logger.isDebugEnabled()) {
                logger.debug("Replacing similar document with id={} and score={}", (Object)((org.springframework.ai.document.Document)existing.get(0)).getId(), (Object)((org.springframework.ai.document.Document)existing.get(0)).getScore());
            }
            this.vectorStore.delete(List.of(((org.springframework.ai.document.Document)existing.get(0)).getId()));
        }
        this.vectorStore.add(List.of(document));
    }

    @Override
    public void set(String query, ChatResponse response, Duration ttl) {
        List existing;
        String docId = UUID.randomUUID().toString();
        String responseJson = this.gson.toJson((Object)response);
        Assert.state((response.getResult() != null ? 1 : 0) != 0, (String)"expected a non-empty response");
        String responseText = response.getResult().getOutput().getText();
        HashMap<String, String> metadata = new HashMap<String, String>();
        metadata.put("response", responseJson);
        metadata.put("response_text", Objects.requireNonNullElse(responseText, ""));
        org.springframework.ai.document.Document document = org.springframework.ai.document.Document.builder().id(docId).text(query).metadata(metadata).build();
        VectorStore vectorStore = this.vectorStore;
        if (vectorStore instanceof RedisVectorStore) {
            RedisVectorStore redisVectorStore = (RedisVectorStore)vectorStore;
            existing = redisVectorStore.searchByRange(query, this.similarityThreshold);
            if (logger.isDebugEnabled()) {
                logger.debug("Using RedisVectorStore's native VECTOR_RANGE query to find similar documents for replacement (TTL version)");
            }
        } else {
            existing = this.vectorStore.similaritySearch(SearchRequest.builder().query(query).topK(1).similarityThreshold(this.similarityThreshold).build());
        }
        if (!existing.isEmpty()) {
            if (logger.isDebugEnabled()) {
                logger.debug("Replacing similar document with id={} and score={}", (Object)((org.springframework.ai.document.Document)existing.get(0)).getId(), (Object)((org.springframework.ai.document.Document)existing.get(0)).getScore());
            }
            this.vectorStore.delete(List.of(((org.springframework.ai.document.Document)existing.get(0)).getId()));
        }
        this.vectorStore.add(List.of(document));
        vectorStore = this.vectorStore;
        if (vectorStore instanceof RedisVectorStore) {
            RedisVectorStore redisStore = (RedisVectorStore)vectorStore;
            String key = this.prefix + docId;
            redisStore.getJedis().expire(key, ttl.getSeconds());
        }
    }

    @Override
    public Optional<ChatResponse> get(String query) {
        String responseJson;
        List similar;
        VectorStore vectorStore;
        double effectiveThreshold = this.similarityThreshold;
        if (this.useDistanceThreshold) {
            effectiveThreshold = 1.0 - this.similarityThreshold / 2.0;
            if (logger.isDebugEnabled()) {
                logger.debug("Converting distance threshold {} to similarity threshold {}", (Object)this.similarityThreshold, (Object)effectiveThreshold);
            }
        }
        if ((vectorStore = this.vectorStore) instanceof RedisVectorStore) {
            RedisVectorStore redisVectorStore = (RedisVectorStore)vectorStore;
            similar = redisVectorStore.searchByRange(query, effectiveThreshold);
            if (logger.isDebugEnabled()) {
                logger.debug("Using RedisVectorStore's native VECTOR_RANGE query with threshold {}", (Object)effectiveThreshold);
            }
        } else {
            if (logger.isDebugEnabled()) {
                logger.debug("Falling back to standard similarity search (vectorStore is not RedisVectorStore)");
            }
            similar = this.vectorStore.similaritySearch(SearchRequest.builder().query(query).topK(5).similarityThreshold(effectiveThreshold).build());
        }
        if (similar.isEmpty()) {
            if (logger.isDebugEnabled()) {
                logger.debug("No documents met the similarity threshold criteria");
            }
            return Optional.empty();
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Query: '{}', found {} matches with similarity >= {}", new Object[]{query, similar.size(), this.similarityThreshold});
            for (org.springframework.ai.document.Document doc : similar) {
                logger.debug("  - Document: id={}, score={}, raw_vector_score={}", new Object[]{doc.getId(), doc.getScore(), doc.getMetadata().getOrDefault("vector_score", "N/A")});
            }
        }
        org.springframework.ai.document.Document mostSimilar = (org.springframework.ai.document.Document)similar.get(0);
        if (logger.isDebugEnabled()) {
            logger.debug("Using most similar document: id={}, score={}", (Object)mostSimilar.getId(), (Object)mostSimilar.getScore());
        }
        if ((responseJson = (String)mostSimilar.getMetadata().get("response")) == null) {
            return Optional.empty();
        }
        try {
            ChatResponse response = (ChatResponse)this.gson.fromJson(responseJson, ChatResponse.class);
            return Optional.of(response);
        }
        catch (JsonParseException e) {
            return Optional.empty();
        }
    }

    @Override
    public void clear() {
        Optional nativeClient = this.vectorStore.getNativeClient();
        if (nativeClient.isPresent()) {
            JedisPooled jedis = (JedisPooled)nativeClient.get();
            boolean moreRecords = true;
            while (moreRecords) {
                Query query = new Query("*");
                query.limit(Integer.valueOf(0), DEFAULT_BATCH_SIZE);
                query.setNoContent();
                SearchResult searchResult = jedis.ftSearch(this.indexName, query);
                if (searchResult.getTotalResults() > 0L) {
                    Pipeline pipeline = jedis.pipelined();
                    try {
                        for (Document doc : searchResult.getDocuments()) {
                            pipeline.jsonDel(doc.getId());
                        }
                        pipeline.syncAndReturnAll();
                        continue;
                    }
                    finally {
                        if (pipeline != null) {
                            pipeline.close();
                        }
                        continue;
                    }
                }
                moreRecords = false;
            }
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    private static class DurationAdapter
    implements JsonSerializer<Duration>,
    JsonDeserializer<Duration> {
        private DurationAdapter() {
        }

        public JsonElement serialize(Duration duration, Type type, JsonSerializationContext context) {
            return new JsonPrimitive((Number)duration.toSeconds());
        }

        public Duration deserialize(JsonElement json, Type type, JsonDeserializationContext context) throws JsonParseException {
            return Duration.ofSeconds(json.getAsLong());
        }
    }

    private static class ChatResponseAdapter
    implements JsonSerializer<ChatResponse>,
    JsonDeserializer<ChatResponse> {
        private ChatResponseAdapter() {
        }

        public JsonElement serialize(ChatResponse response, Type type, JsonSerializationContext context) {
            AssistantMessage output;
            JsonObject jsonObject = new JsonObject();
            String responseText = "";
            if (response.getResults() != null && !response.getResults().isEmpty() && (output = ((Generation)response.getResults().get(0)).getOutput()) != null) {
                responseText = output.getText();
            }
            jsonObject.addProperty("fullText", responseText);
            JsonArray generations = new JsonArray();
            for (Generation generation : response.getResults()) {
                JsonObject generationObj = new JsonObject();
                AssistantMessage output2 = generation.getOutput();
                generationObj.addProperty("text", output2.getText());
                generations.add((JsonElement)generationObj);
            }
            jsonObject.add("generations", (JsonElement)generations);
            return jsonObject;
        }

        public ChatResponse deserialize(JsonElement json, Type type, JsonDeserializationContext context) throws JsonParseException {
            JsonObject jsonObject = json.getAsJsonObject();
            String fullText = "";
            if (jsonObject.has("fullText")) {
                fullText = jsonObject.get("fullText").getAsString();
            }
            if (!fullText.isEmpty()) {
                ArrayList<Generation> generations = new ArrayList<Generation>();
                generations.add(new Generation(new AssistantMessage(fullText)));
                return ChatResponse.builder().generations(generations).build();
            }
            ArrayList<Generation> generations = new ArrayList<Generation>();
            JsonArray generationsArray = jsonObject.getAsJsonArray("generations");
            for (JsonElement element : generationsArray) {
                JsonObject generationObj = element.getAsJsonObject();
                String text = generationObj.get("text").getAsString();
                generations.add(new Generation(new AssistantMessage(text)));
            }
            return ChatResponse.builder().generations(generations).build();
        }
    }

    public static class Builder {
        private @Nullable VectorStore vectorStore;
        private @Nullable EmbeddingModel embeddingModel;
        private double similarityThreshold = 0.8;
        private boolean useDistanceThreshold = false;
        private String indexName = "semantic-cache-index";
        private String prefix = "semantic-cache:";
        private @Nullable JedisPooled jedisClient;

        public Builder vectorStore(VectorStore vectorStore) {
            this.vectorStore = vectorStore;
            return this;
        }

        public Builder embeddingModel(EmbeddingModel embeddingModel) {
            this.embeddingModel = embeddingModel;
            return this;
        }

        public Builder similarityThreshold(double threshold) {
            this.similarityThreshold = threshold;
            return this;
        }

        public Builder distanceThreshold(double threshold) {
            this.similarityThreshold = threshold;
            this.useDistanceThreshold = true;
            return this;
        }

        public Builder indexName(String indexName) {
            this.indexName = indexName;
            return this;
        }

        public Builder prefix(String prefix) {
            this.prefix = prefix;
            return this;
        }

        public Builder jedisClient(JedisPooled jedisClient) {
            this.jedisClient = jedisClient;
            return this;
        }

        public DefaultSemanticCache build() {
            if (this.vectorStore == null) {
                if (this.jedisClient == null) {
                    throw new IllegalStateException("Either vectorStore or jedisClient must be provided");
                }
                if (this.embeddingModel == null) {
                    throw new IllegalStateException("EmbeddingModel must be provided");
                }
                this.vectorStore = RedisVectorStore.builder((JedisPooled)this.jedisClient, (EmbeddingModel)this.embeddingModel).indexName(this.indexName).prefix(this.prefix).metadataFields(new RedisVectorStore.MetadataField[]{RedisVectorStore.MetadataField.text((String)"response"), RedisVectorStore.MetadataField.text((String)"response_text"), RedisVectorStore.MetadataField.numeric((String)"ttl")}).initializeSchema(true).build();
                VectorStore vectorStore = this.vectorStore;
                if (vectorStore instanceof RedisVectorStore) {
                    RedisVectorStore redisStore = (RedisVectorStore)vectorStore;
                    redisStore.afterPropertiesSet();
                }
            }
            return new DefaultSemanticCache(this.vectorStore, this.similarityThreshold, this.indexName, this.prefix, this.useDistanceThreshold);
        }
    }
}

