/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.authentication;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.JwtBuilder;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import io.jsonwebtoken.SignatureException;
import io.jsonwebtoken.impl.crypto.MacProvider;
import java.security.Key;
import java.util.Base64;
import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.concurrent.Immutable;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.sonar.api.Startable;
import org.sonar.api.config.Configuration;
import org.sonar.api.server.ServerSide;
import org.sonar.api.utils.System2;
import org.sonar.core.util.UuidFactory;
import org.sonar.server.authentication.event.AuthenticationEvent;
import org.sonar.server.authentication.event.AuthenticationException;

@ServerSide
public class JwtSerializer
implements Startable {
    private static final SignatureAlgorithm SIGNATURE_ALGORITHM = SignatureAlgorithm.HS256;
    private final Configuration config;
    private final System2 system2;
    private final UuidFactory uuidFactory;
    private SecretKey secretKey;

    public JwtSerializer(Configuration config, System2 system2, UuidFactory uuidFactory) {
        this.config = config;
        this.system2 = system2;
        this.uuidFactory = uuidFactory;
    }

    @VisibleForTesting
    SecretKey getSecretKey() {
        return this.secretKey;
    }

    public void start() {
        Optional encodedKey = this.config.get("sonar.auth.jwtBase64Hs256Secret");
        this.secretKey = encodedKey.isPresent() ? JwtSerializer.decodeSecretKeyProperty((String)encodedKey.get()) : JwtSerializer.generateSecretKey();
    }

    String encode(JwtSession jwtSession) {
        this.checkIsStarted();
        long now = this.system2.now();
        JwtBuilder jwtBuilder = Jwts.builder().setId(this.uuidFactory.create()).setSubject(jwtSession.getUserLogin()).setIssuedAt(new Date(now)).setExpiration(new Date(now + jwtSession.getExpirationTimeInSeconds() * 1000L)).signWith(SIGNATURE_ALGORITHM, (Key)this.secretKey);
        for (Map.Entry<String, Object> entry : jwtSession.getProperties().entrySet()) {
            jwtBuilder.claim(entry.getKey(), entry.getValue());
        }
        return jwtBuilder.compact();
    }

    Optional<Claims> decode(String token) {
        this.checkIsStarted();
        Claims claims = null;
        try {
            claims = (Claims)Jwts.parser().setSigningKey((Key)this.secretKey).parseClaimsJws(token).getBody();
            Objects.requireNonNull(claims.getId(), "Token id hasn't been found");
            Objects.requireNonNull(claims.getSubject(), "Token subject hasn't been found");
            Objects.requireNonNull(claims.getExpiration(), "Token expiration date hasn't been found");
            Objects.requireNonNull(claims.getIssuedAt(), "Token creation date hasn't been found");
            return Optional.of(claims);
        }
        catch (ExpiredJwtException | SignatureException e) {
            return Optional.empty();
        }
        catch (Exception e) {
            throw AuthenticationException.newBuilder().setSource(AuthenticationEvent.Source.jwt()).setLogin(claims == null ? null : claims.getSubject()).setMessage(e.getMessage()).build();
        }
    }

    String refresh(Claims token, int expirationTimeInSeconds) {
        this.checkIsStarted();
        long now = this.system2.now();
        JwtBuilder jwtBuilder = Jwts.builder();
        for (Map.Entry entry : token.entrySet()) {
            jwtBuilder.claim((String)entry.getKey(), entry.getValue());
        }
        jwtBuilder.setExpiration(new Date(now + (long)(expirationTimeInSeconds * 1000))).signWith(SIGNATURE_ALGORITHM, (Key)this.secretKey);
        return jwtBuilder.compact();
    }

    private static SecretKey generateSecretKey() {
        return MacProvider.generateKey((SignatureAlgorithm)SIGNATURE_ALGORITHM);
    }

    private static SecretKey decodeSecretKeyProperty(String base64SecretKey) {
        byte[] decodedKey = Base64.getDecoder().decode(base64SecretKey);
        return new SecretKeySpec(decodedKey, 0, decodedKey.length, SIGNATURE_ALGORITHM.getJcaName());
    }

    private void checkIsStarted() {
        Preconditions.checkNotNull((Object)this.secretKey, (String)"%s not started", (Object[])new Object[]{this.getClass().getName()});
    }

    public void stop() {
        this.secretKey = null;
    }

    @Immutable
    static class JwtSession {
        private final String userLogin;
        private final long expirationTimeInSeconds;
        private final Map<String, Object> properties;

        JwtSession(String userLogin, long expirationTimeInSeconds) {
            this(userLogin, expirationTimeInSeconds, Collections.emptyMap());
        }

        JwtSession(String userLogin, long expirationTimeInSeconds, Map<String, Object> properties) {
            this.userLogin = Objects.requireNonNull(userLogin, "User login cannot be null");
            this.expirationTimeInSeconds = expirationTimeInSeconds;
            this.properties = properties;
        }

        String getUserLogin() {
            return this.userLogin;
        }

        long getExpirationTimeInSeconds() {
            return this.expirationTimeInSeconds;
        }

        Map<String, Object> getProperties() {
            return this.properties;
        }
    }
}

