/*
 * Copyright © 2009 Benny Bottema (benny@bennybottema.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.simplejavamail.mailer.internal;

import jakarta.mail.MessagingException;
import jakarta.mail.Session;
import lombok.val;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.simplejavamail.api.email.Email;
import org.simplejavamail.api.internal.authenticatedsockssupport.socks5server.AnonymousSocks5Server;
import org.simplejavamail.api.mailer.EmailTooBigException;
import org.simplejavamail.api.mailer.config.OperationalConfig;
import org.simplejavamail.mailer.internal.util.TransportRunner;

import java.util.concurrent.atomic.AtomicInteger;

import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static org.simplejavamail.mailer.internal.MailerException.GENERIC_ERROR;
import static org.simplejavamail.mailer.internal.MailerException.MAILER_ERROR;
import static org.simplejavamail.mailer.internal.MailerException.UNKNOWN_ERROR;

/**
 * Separate closure that can be executed directly or from a thread.
 * <p>
 * Note that this Runnable implementation is <strong>not</strong> thread related, it is just to encapsulate the code to
 * be run directly or from a <em>real</em> Runnable.
 */
class SendMailClosure extends AbstractProxyServerSyncingClosure {

	@NotNull private final OperationalConfig operationalConfig;
	@NotNull private final Session session;
	@NotNull private final Email email;
	private final boolean transportModeLoggingOnly;

	SendMailClosure(@NotNull OperationalConfig operationalConfig, @NotNull Session session, @NotNull Email email, @Nullable AnonymousSocks5Server proxyServer, boolean transportModeLoggingOnly, @NotNull AtomicInteger smtpConnectionCounter) {
		super(smtpConnectionCounter, proxyServer);
		this.operationalConfig = operationalConfig;
		this.session = session;
		this.email = email;
		this.transportModeLoggingOnly = transportModeLoggingOnly;
	}

	@Override
	public void executeClosure() {
		LOGGER.trace("sending email...");
		try {
			if (transportModeLoggingOnly) {
				SessionBasedEmailToMimeMessageConverter.convertAndLogMimeMessage(session, email);
				LOGGER.info("TRANSPORT_MODE_LOGGING_ONLY: skipping actual sending...");
			} else if (operationalConfig.getCustomMailer() != null) {
				val message = SessionBasedEmailToMimeMessageConverter.convertAndLogMimeMessage(session, email);
				operationalConfig.getCustomMailer().sendMessage(operationalConfig, session, email, message);
			} else {
				TransportRunner.sendMessage(operationalConfig.getClusterKey(), session, email);
			}
		} catch (final MessagingException e) {
			handleException(e, GENERIC_ERROR);
		} catch (final MailerException | EmailTooBigException e) {
			handleException(e, MAILER_ERROR);
		} catch (final Exception e) {
			handleException(e, UNKNOWN_ERROR);
		}
	}

	private void handleException(final Exception e, String errorMsg) {
		LOGGER.trace("Failed to send email {}\n{}\n\t{}", email.getId(), email, errorMsg);
		val emailId = ofNullable(email.getId())
				.map(id -> format("ID: '%s'", id))
				.orElse(format("Subject: '%s'", email.getSubject()));
		throw new MailerException(format(errorMsg, emailId), e);
	}
}