'use strict';
import EchartsHelper from '@/widget-resources/js/echarts/echarts-helper.js';

angular.module('app.sunburst.service', [])
    .factory('sunburstService', sunburstService);

sunburstService.$inject = ['VIZ_COLORS'];

function sunburstService(VIZ_COLORS) {
    /**
     * @name getConfig
     * @param {array} keys  semoss data keys
     * @param {object} chartData semoss chart data
     * @param {object} uiOptions semoss uioptions
     * @desc creates series config for sunburst
     * @return {object} the config object
     */
    function getConfig(keys, chartData, uiOptions) {
        var hierarchy = [],
            backgroundColorStyle = false,
            returnObj;

        // Set Background Color (check for watermark)
        if (/\S/.test(uiOptions.watermark)) {
            backgroundColorStyle = {
                type: 'pattern',
                image: paintWaterMark(uiOptions.watermark),
                repeat: 'repeat'
            };
        }

        chartData.values.forEach(function (value) {
            createPath(value[0], value.slice(1), hierarchy, uiOptions, keys);
        });

        // echarts calculates parent total value when it its not there
        // so instead of keeping track of total value ourselves we let
        // baidu handle it
        hierarchy.forEach(function (rootNode) {
            // if no children, deleting value will not let us paint chart
            if (rootNode.children.length > 0) {
                delete rootNode.value;
            }
        });

        addLevelColors(hierarchy, EchartsHelper.setColors(uiOptions, undefined, VIZ_COLORS).color);

        returnObj = {
            // TODO also pass in labels to setColors
            color: EchartsHelper.setColors(uiOptions, undefined, VIZ_COLORS).color,
            series: [{
                type: 'sunburst',
                data: hierarchy,
                sort: getSortOrder(uiOptions.sortSunburst),
                label: getLabels(uiOptions.displayValues, uiOptions.customizeSunburstLabel),
                // emphasis: getEmphasis(),
                radius: ['50%', '85%']
            }],
            tooltip: getToolTip(uiOptions.showTooltips)
        }

        if (backgroundColorStyle) {
            returnObj.backgroundColor = backgroundColorStyle;
        }

        return returnObj;
    }

    function getToolTip(show) {
        return {
            show: show,
            trigger: 'item',
            confine: true,
            formatter: function (info)  {
                var returnArray = [];

                if (info.marker) {
                    returnArray.push(cleanValue(info.marker));
                }

                if (info.name) {
                    returnArray.push(cleanValue(info.name));
                }

                if (info.value) {
                    returnArray.push('<br>Value: ' + cleanValue(info.value));
                }

                return returnArray.join('');
            }
        };
    }

    function cleanValue(item) {
        if (typeof item === 'string') {
            return item.replace(/_/g, ' ');
        } else if (typeof item === 'number') {
            return item.toLocaleString(undefined, {
                minimumFractionDigits: 0,
                maximumFractionDigits: 3
            });
        }
        return item;
    }

    /**
     * @name addLevelColors
     * @param {array} hierarchy parent nodes of current level
     * @param {array} colors  ui options colors
     * @desc goes thru color array for each child node
     * @return {void}
     */
    function addLevelColors(hierarchy, colors) {
        hierarchy.forEach(function (node, idx) {
            // dont want to overwrite a cbv or something
            if (!node.itemStyle.color) {
                node.itemStyle.color = colors[idx % colors.length];
            }

            addLevelColors(node.children, colors);
        });
    }

    /**
     * @name createPath
     * @param {string} parent - name of parent node
     * @param {string[]} path - name of descendent nodes
     * @param {object[]} currentLevel - current level in hierarchy
     * @param {obj} uiOptions semoss uiOptions
     * @param {array} keys semoss keys
     * @desc function to wrap up semoss data as hierarchical data.
     *       the first value is the root, the next value is the child, the next
     *       is the grandchild, etc... the last value is the value of that node.
     *       go thru current level, if no parent create it. add children.
     *       combine their values if duplicate.
     * @return {void}
     */
    function createPath(parent, path, currentLevel, uiOptions, keys) {
        var parentFound = false;

        if (path.length === 1) {
            // only one level, will need to just throw in the parents
            if (keys.length === 2) {
                currentLevel.push(
                    createNode(parent, path[path.length - 1], uiOptions)
                );
            }

            return;
        }

        currentLevel.forEach(function (node) {
            var child, nodeNum;

            if (parent === node.name) {
                parentFound = true;
                child = childAlreadyExists(node, path[0]);
                if (child) {
                    child.value += path[path.length - 1];
                } else {
                    nodeNum = node.children.length;
                    node.children.push(
                        createNode(path[0], path[path.length - 1], uiOptions, nodeNum)
                    );
                }

                createPath(
                    path[0],
                    path.slice(1),
                    node.children,
                    uiOptions,
                    keys
                );
            }
        });

        if (!parentFound) {
            currentLevel.push(
                createNode(parent, path[path.length - 1], uiOptions)
            );

            // need to add child to parent as well
            currentLevel[currentLevel.length - 1].children.push(
                createNode(path[0], path[path.length - 1], uiOptions)
            );

            createPath(
                path[0],
                path.slice(1),
                currentLevel[currentLevel.length - 1].children,
                uiOptions,
                keys
            );
        }
    }

    /**
     * @name childAlreadyExists
     * @param {object} node the node we are looking at
     * @param {string} childToCheck the name of the child to look for
     * @desc determines if the node already has this child
     * @return {boolean} true if it does false otherwise
     */
    function childAlreadyExists(node, childToCheck) {
        var i, child;

        for (i = 0; i < node.children.length; i++) {
            child = node.children[i];
            if (child.name === childToCheck) {
                return child;
            }
        }

        return false;
    }

    /**
     * @name createNode
     * @param {string} name name of node
     * @param {number} value node value
     * @param {obj} uiOptions semoss uiOptions
     * @desc creates a node in the sunburst
     * @return {object} sunburst node
     */
    function createNode(name, value, uiOptions) {
        var node = {
                name: name,
                value: value,
                children: [],
                itemStyle: {}
            }, i, j, rule, cleanName, valueToColor;
        if (uiOptions.colorByValue && uiOptions.colorByValue.length > 0) {
            cleanName = name;
            for (i = 0; i < uiOptions.colorByValue.length; i++) {
                rule = uiOptions.colorByValue[i];
                for (j = 0; j < rule.valuesToColor.length; j++) {
                    valueToColor = rule.valuesToColor[j];

                    if (cleanName === valueToColor || value === valueToColor) {
                        node.itemStyle.color = rule.color;
                    }
                }
            }
        }
        return node;
    }

    /**
     * @name getLabels
     * @param {bool} displayValues uiOptions.displayValues
     * @param {obj} settings uiOptions.customizeSunburstLabel
     * @desc customize sunburst labels
     * @return {object} label object
     */
    function getLabels(displayValues, settings) {
        var show = false;

        if (displayValues) {
            show = true;
        }

        if (settings.orientation === 'horizontal') {
            settings.orientation = 0;
        }

        return {
            'show': show,
            'rotate': settings.orientation,
            'fontSize': settings.fontSize
        };
    }

    /**
     * @name getSortOrder
     * @param {bool} param uiOptions.sortSunburst
     * @desc customize sunburst sort
     * @return {string} asc or desc
     */
    function getSortOrder(param) {
        if (param) {
            return 'asc';
        }
        return 'desc';
    }

    /**
    * @name paintWaterMark
    * @desc paints a custom watermark on the viz
    * @param {string} watermark - string of the watermark text
    * @returns {Object} - canvas details
    */
    function paintWaterMark(watermark) {
        var canvas = document.createElement('canvas'),
            ctx = canvas.getContext('2d');
        canvas.width = canvas.height = 100;
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        ctx.globalAlpha = 0.08;
        ctx.font = '20px Microsoft Yahei';
        ctx.translate(50, 50);
        ctx.rotate(-Math.PI / 4);
        if (watermark) {
            ctx.fillText(watermark, 0, 0);
        }
        return canvas;
    }

    // /**
    //  * @name getEmphasis
    //  * @desc customize sunburst emphasis
    //  * @return {object} emphasis object
    //  */
    // function getEmphasis() {
    //     var emphasisObj = {};

    //     emphasisObj.label = {
    //         'show': true
    //     };

    //     return emphasisObj;
    // }

    return {
        getConfig: getConfig
    };
}
