'use strict';

import echarts from 'echarts';
import EchartsHelper from '@/widget-resources/js/echarts/echarts-helper.js';
import './sunburst-echarts.service.js';

/**
 *
 * @name sunburst-echarts
 * @desc sunburst-echarts chart directive for creating and visualizing a sunburst chart
 */

export default angular.module('app.sunburst-echarts.directive', [
    'app.sunburst.service'
]).directive('sunburstEcharts', sunburstEcharts);

sunburstEcharts.$inject = ['sunburstService'];

function sunburstEcharts(sunburstService) {
    sunburstChartLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^widget'],
        priority: 300,
        link: sunburstChartLink
    };

    function sunburstChartLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];

        /** ************* Main Event Listeners ************************/
        var resizeListener,
            updateTaskListener,
            updateOrnamentsListener,
            addDataListener,
            modeListener,
            /** *************** ECharts ****************************/
            eChartsConfig = {},
            sunburstChart,
            clickTimer;

        /**
         * @name initialize
         * @desc creates the visualization on the chart div
         * @returns {void}
         */
        function initialize() {
            // bind listeners
            resizeListener = scope.widgetCtrl.on('resize-widget', resizeViz);
            updateTaskListener = scope.widgetCtrl.on('update-task', setData);
            updateOrnamentsListener = scope.widgetCtrl.on('update-ornaments', setData);
            addDataListener = scope.widgetCtrl.on('added-data', setData);
            modeListener = scope.widgetCtrl.on('update-mode', toggleMode);

            scope.$on('$destroy', destroy);

            setData();
        }

        /**
         * @name setData
         * @desc setData for the visualization and paints
         * @returns {void}
         */
        function setData() {
            var selectedLayout = scope.widgetCtrl.getWidget('view.visualization.layout'),
                individualTools = scope.widgetCtrl.getWidget('view.visualization.tools.individual.' + selectedLayout) || {},
                sharedTools = scope.widgetCtrl.getWidget('view.visualization.tools.shared'),
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.' + selectedLayout),
                layerIndex = 0,
                data = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.data'),
                uiOptions = angular.extend(sharedTools, individualTools);

            data.values = data.values;

            eChartsConfig.chartConfig = sunburstService.getConfig(keys, data, uiOptions);
            eChartsConfig.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.callbacks = scope.widgetCtrl.getEventCallbacks();
            eChartsConfig.comments = scope.widgetCtrl.getWidget('view.visualization.commentData');

            paint(true);
        }

        /**
         * @name paint
         * @param {bool} showAnimation - weather to show animation or not
         * @desc paints the visualization
         * @returns {void}
         */
        function paint(showAnimation) {
            if (sunburstChart) {
                sunburstChart.clear();
                sunburstChart.dispose();
            }
            sunburstChart = echarts.init(ele[0].firstElementChild);

            eChartsConfig.chartConfig.series[0].animation = showAnimation;

            eChartsConfig.textStyle = {
                fontFamily: 'Libre Franklin'
            };

            // use configuration item and data specified to show chart
            EchartsHelper.setOption(sunburstChart, eChartsConfig.chartConfig);

            // Add event listeners
            initializeEvents();
        }

        /**
         * @name initializeEvents
         * @desc creates the event layer
         * @returns {void}
         */
        function initializeEvents() {
            // TODO add back in click events... temporarily disabled so user can interact with sunburst levels
            // sunburstChart.on('click', eChartClicked);

            // it is necessary to initialize comment mode so the nodes are painted
            EchartsHelper.initializeCommentMode({
                comments: eChartsConfig.comments,
                currentMode: eChartsConfig.currentMode,
                saveCb: eChartsConfig.callbacks.commentMode.onSave
            });

            if (typeof eChartsConfig.callbacks.defaultMode.onKeyUp === 'function' || typeof eChartsConfig.callbacks.defaultMode.onKeyDown === 'function') {
                sunburstChart._dom.tabIndex = 1;
                if (typeof eChartsConfig.callbacks.defaultMode.onKeyUp === 'function') {
                    sunburstChart._dom.addEventListener('keyup', function (e) {
                        eChartsConfig.callbacks.defaultMode.onKeyUp({
                            eventType: 'onKeyUp',
                            key: e.key,
                            event: e,
                            keyCode: e.keyCode
                        });
                    });
                }
                if (typeof eChartsConfig.callbacks.defaultMode.onKeyDown === 'function') {
                    sunburstChart._dom.addEventListener('keydown', function (e) {
                        eChartsConfig.callbacks.defaultMode.onKeyDown({
                            eventType: 'onKeyDown',
                            key: e.key,
                            event: e,
                            keyCode: e.keyCode
                        });
                    });
                }
            }
        }

        /**
         * @name eChartClicked
         * @desc single click event from echarts
         * @param {object} event - echarts event sent back on click
         * @returns {void}
         */
        function eChartClicked(event) {
            if (clickTimer) {
                clearTimeout(clickTimer);
                clickCallback(event, 'doubleClick');
            } else {
                clickTimer = setTimeout(clickCallback.bind(null, event, 'click'), 250);
            }
        }

        /**
         * @name clickCallback
         * @desc click callback event
         * @param {object} event - echarts event sent back on click
         * @param {string} type - click or double click
         * @returns {void}
         */
        function clickCallback(event, type) {
            var returnObj = {
                data: {}
            };
            returnObj.data[eChartsConfig.legendLabels] = [event.name];
            if (type === 'click') {
                eChartsConfig.callbacks.defaultMode.onClick(returnObj);
            } else {
                eChartsConfig.callbacks.defaultMode.onDoubleClick(returnObj);
            }
            clickTimer = null;
        }

        /**
         * @name toggleMode
         * @desc switches the jv mode to the new specified mode
         * @returns {void}
         */
        function toggleMode() {
            eChartsConfig.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
        }

        /**
         * @name resizeViz
         * @desc reruns the jv paint function
         * @returns {void}
         */
        function resizeViz() {
            paint(false);
        }

        /**
         * @name destroy
         * @desc destroys listeners and dom elements outside of the scope
         * @returns {void}
         */
        function destroy() {
            resizeListener();
            updateTaskListener();
            updateOrnamentsListener();
            addDataListener();
            modeListener();
        }

        // Start Visualization Creation
        initialize();
    }
}
