'use strict';

import echarts from 'echarts';
import EchartsHelper from '@/widget-resources/js/echarts/echarts-helper.js';
import visualizationUniversal from '@/core/store/visualization/visualization.js';

import './polar-bar-echarts.service.js';

/**
 *
 * @name polar-bar-echarts
 * @desc polar-bar-echarts chart directive for creating and visualizing a column chart
 */

export default angular.module('app.polar-bar-echarts.directive', [
    'app.polar-bar.service'
]).directive('polarBarEcharts', polarBarEcharts);

polarBarEcharts.$inject = ['VIZ_COLORS', 'polarBarService'];

function polarBarEcharts(VIZ_COLORS, polarBarService) {
    polarBarChartLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^widget', '^visualization'],
        priority: 300,
        link: polarBarChartLink
    };

    function polarBarChartLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];
        scope.visualizationCtrl = ctrl[1];
        var dataTypes;

        /** ************* Main Event Listeners ************************/
        var resizeListener,
            updateTaskListener,
            updateOrnamentsListener,
            addDataListener,
            modeListener,
            /** *************** ECharts ****************************/
            eChartsConfig,
            polarBarChart,
            clickTimer,
            hoverTimer;

        /**
         * @name initialize
         * @desc creates the visualization on the chart div
         * @returns {void}
         */
        function initialize() {
            // bind listeners
            resizeListener = scope.widgetCtrl.on('resize-widget', resizeViz);
            updateTaskListener = scope.widgetCtrl.on('update-task', setData);
            updateOrnamentsListener = scope.widgetCtrl.on('update-ornaments', setData);
            addDataListener = scope.widgetCtrl.on('added-data', setData);
            modeListener = scope.widgetCtrl.on('update-mode', toggleMode);

            scope.$on('$destroy', destroy);

            setData();
        }

        /**
         * @name setData
         * @desc setData for the visualization and paints
         * @returns {void}
         */
        function setData() {
            var layerIndex = 0,
                selectedLayout = scope.widgetCtrl.getWidget('view.visualization.layout'),
                individual = scope.widgetCtrl.getWidget('view.visualization.tools.individual.' + selectedLayout) || {},
                sharedTools = scope.widgetCtrl.getWidget('view.visualization.tools.shared'),
                colorBy = scope.widgetCtrl.getWidget('view.visualization.colorByValue'),
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.' + selectedLayout),
                data = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.data'),
                groupBy = {},
                groupByInstance,
                groupByInfo = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.groupByInfo'),
                uiOptions;

            uiOptions = angular.extend(sharedTools, individual);
            uiOptions.colorByValue = colorBy;
            getDataTypes(keys, uiOptions);

            if (groupByInfo && groupByInfo.viewType) {
                groupBy = formatDataForGroupBy(data, groupByInfo);
                data = groupBy.data;
                groupByInstance = groupBy.name;
            }

            eChartsConfig = polarBarService.getConfig('barPolar', data, uiOptions, colorBy, groupByInstance, keys);
            eChartsConfig.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.callbacks = scope.widgetCtrl.getEventCallbacks();
            eChartsConfig.comments = scope.widgetCtrl.getWidget('view.visualization.commentData');
            eChartsConfig.groupByInfo = groupByInfo;
            uiOptions = EchartsHelper.setColors(uiOptions, eChartsConfig.legendHeaders, VIZ_COLORS);
            paint();
        }

        /**
         * @name getDataTypes
         * @desc gets the data formatting options for each dimension
         * @param {Object} keys - object of data keys
         * @param {object} options- uiOptions
         */
        function getDataTypes (keys, options) {
            dataTypes = {
                labelType: '',
                valueType: ''
            };
            let k, j, i,
                labelName,
                valueName,
                formatType,
                formatDimension;

            
            for (k = 0; k < keys.length; k++) {
                if (keys[k].model === 'label') {
                    dataTypes.labelType = visualizationUniversal.mapFormatOpts(keys[k]);
                    labelName = keys[k].alias;
                }
                //only use formatting rules from 1st series selected to configure Y axis
                if (keys[k].model === 'value' && dataTypes.valueType === '') {
                    dataTypes.valueType = visualizationUniversal.mapFormatOpts(keys[k]);
                    valueName = keys[k].alias;
                }
                if (keys[k].model === 'tooltip' || keys[k].model === 'value') {
                    dataTypes[keys[k].alias] = [];
                    formatType = visualizationUniversal.mapFormatOpts(keys[k]);
                    dataTypes[keys[k].alias].push(formatType);
                    if (options.formatDataValues) {
                        for (j = 0; j < options.formatDataValues.formats.length; j++){
                            formatType = options.formatDataValues.formats[j];
                            if(keys[k].alias === formatType.dimension){
                                dataTypes[formatType.dimension] = [];
                                dataTypes[formatType.dimension].push(formatType);
                            }
                        }
                    } 
                }
            }

            if(options.formatDataValues){
                for(i = 0; i < options.formatDataValues.formats.length; i++){
                    formatDimension = options.formatDataValues.formats[i].dimension;
                    if(formatDimension === labelName){
                        dataTypes.labelType = options.formatDataValues.formats[i];
                    }
                    if(formatDimension === valueName){
                        dataTypes.valueType = options.formatDataValues.formats[i];
                    }
                }
            }
        }

        /**
         * @name formatDataForGroupBy
         * @desc formats data when Group By exists
         * @param {object} data orginial data
         * @param {object} groupBy groupBy object
         * @returns {void}
         */
        function formatDataForGroupBy(data, groupBy) {
            var formattedData = data,
                groupByIndex,
                name,
                i,
                instanceIdx,
                returnObj = {};

            if (groupBy.viewType === 'Individual Instance') {
                groupByIndex = data.headers.indexOf(groupBy.selectedDim);
                if (groupByIndex === -1) {
                    // return data;
                    groupByIndex = data.headers.length;
                }

                if (typeof groupBy.instanceIndex === 'string') {
                    instanceIdx = parseInt(groupBy.instanceIndex, 10);
                }
                // Create name for title
                name = groupBy.selectedDim + ' : ' + groupBy.uniqueInstances[instanceIdx];
                // Remove Group By dimension from data headers and values
                formattedData.headers.splice(groupByIndex, 1);
                formattedData.rawHeaders.splice(groupByIndex, 1);

                // Remove any added data from brush/click
                for (i = 0; i < data.values.length; i++) {
                    if (data.values[i][groupByIndex] !== groupBy.uniqueInstances[instanceIdx]) {
                        data.values.splice(i, 1);
                        i--;
                    }
                }

                for (i = 0; i < data.values.length; i++) {
                    data.values[i].splice(groupByIndex, 1);
                }
                returnObj.name = name;
                returnObj.data = data;
            }

            return returnObj;
        }

        /**
         * @name paint
         * @desc paints the visualization
         * @returns {void}
         */
        function paint() {
            if (polarBarChart) {
                polarBarChart.clear();
                polarBarChart.dispose();
            }
            // TODO also think abou abstracting some of these options to variables for more customizabilty from uiOptions
            polarBarChart = echarts.init(ele[0].firstElementChild);

            var option = {},
                i,
                dataEmpty;

            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType) {
                dataEmpty = true;
                for (i = 0; i < eChartsConfig.data.length; i++) {
                    if (eChartsConfig.data[i].data.length !== 0) {
                        dataEmpty = false;
                        break;
                    }
                }

                if (eChartsConfig.groupByInfo.viewType === 'Individual Instance') {
                    option.graphic = [];
                    if (dataEmpty) {
                        option.graphic = option.graphic.concat({
                            id: 'textGroup',
                            type: 'group',
                            right: 'center',
                            top: '60%',
                            children: [
                                {
                                    type: 'rect',
                                    top: 'center',
                                    right: 'center',
                                    shape: {
                                        width: 200,
                                        height: 40
                                    },
                                    style: {
                                        fill: '#fff',
                                        stroke: '#999',
                                        lineWidth: 2,
                                        shadowBlur: 8,
                                        shadowOffsetX: 3,
                                        shadowOffsetY: 3,
                                        shadowColor: 'rgba(0,0,0,0.3)'
                                    }
                                },
                                {
                                    type: 'text',
                                    right: 'center',
                                    top: 'center',
                                    style: {
                                        text: 'There is no data for this instance.',
                                        textAlign: 'center'
                                    }
                                }
                            ]
                        });
                    }
                }
            }

            if (eChartsConfig.backgroundColorStyle) {
                option.backgroundColor = eChartsConfig.backgroundColorStyle;
            }
            option.color = eChartsConfig.options.echartsColor;
            option.tooltip = {
                show: eChartsConfig.options.showTooltips,
                formatter: function (info) {
                    var returnArray = [],
                        tooltipName = info[0].name,
                        j,
                        formatType,
                        tooltipType;

                    if (tooltipName) {
                        returnArray.push('<b>' + visualizationUniversal.formatValue(tooltipName, dataTypes.labelType) + '</b>' + '<br>');
                    }

                    for (j = 0; j < info.length; j++) {
                        formatType = dataTypes[info[j].seriesName][0];

                        if (!info[j].data || info[j].componentSubType === 'line') {
                            continue;
                        }
                        if (info[j].marker) {
                            returnArray.push(info[j].marker);
                        }
                        if (info[j].seriesName) {
                            returnArray.push('' + cleanValue(info[j].seriesName) + ': ' + visualizationUniversal.formatValue(info[j].value, formatType) + '<br>');
                        }
                    }

                    if (info[0].data.tooltip) {
                        for (j = 0; j < info[0].data.tooltip.length; j++) {
                            tooltipType = dataTypes[info[0].data.tooltip[j].header][0];
                            if (eChartsConfig.legendHeaders.indexOf(info[0].data.tooltip[j].header) === -1) {
                                returnArray.push('' + cleanValue(info[0].data.tooltip[j].header) + ': ' + visualizationUniversal.formatValue(info[0].data.tooltip[j].value || 0, tooltipType) + '<br>');
                            }
                        }
                    }

                    return returnArray.join('');
                },
                trigger: 'axis',
                axisPointer: {
                    type: eChartsConfig.axisPointer
                },
                confine: true
            };
            option.legend = {
                type: 'scroll',
                data: eChartsConfig.legendHeaders,
                show: eChartsConfig.showLegend,
                orient: 'horizontal',
                left: 'left',
                pageButtonPosition: 'start',
                formatter: function (value) {
                    return value.replace(/_/g, ' ');
                }
            };
            //option.angleAxis = eChartsConfig.angleAxis;
            option.angleAxis = {
                type: 'value',
                min: eChartsConfig.angleAxis.min,
                max: eChartsConfig.angleAxis.max,
                //data: eChartsConfig.angleAxis,
                axisLabel: {
                    formatter: function (obj) {
                        //var test = eChartsConfig.angleAxis;
                        if (obj) {
                            return visualizationUniversal.formatValue(obj, dataTypes.valueType);
                        }
                        //return value.replace(/_/g, ' ');
                    }
                }
            },
            option.radiusAxis = {
                type: 'category',
                data: eChartsConfig.legendData,
                z: 10,
                axisLabel: {
                    show: eChartsConfig.options.toggleXLabel,
                    formatter: function (value) {
                        return visualizationUniversal.formatValue(value, dataTypes.labelType);
                        //return value.replace(/_/g, ' ');
                    }
                }
            };
            option.polar = {
                center: ['50%', '50%'],
                radius: '75%'
            };
            option.dataZoom = getDataZoom(eChartsConfig.options.togglePolarZoom) || [];
            // option.barWidth = eChartsConfig.barWidth;
            option.series = eChartsConfig.data;

            option.textStyle = {
                fontFamily: 'Libre Franklin'
            };

            // use configuration item and data specified to show chart
            EchartsHelper.setOption(polarBarChart, option);

            // Add event listeners
            initializeEvents();
        }

        function cleanValue(item) {
            if (typeof item === 'string') {
                return item.replace(/_/g, ' ');
            } else if (typeof item === 'number') {
                return item.toLocaleString(undefined, {
                    minimumFractionDigits: 0,
                    maximumFractionDigits: 3
                });
            }
            return item;
        }

        /**
         * @name getDataZoom
         * @desc defines data zoom on polar bar chart
         * @param {string} param - echartsConfig.options.togglePolarZoom
         * @returns {array} dataZoom array of slider and inside zoom
         */
        function getDataZoom(param) {
            var dataZoom;
            if (param === 'Radius Zoom') {
                dataZoom = [
                    {
                        type: 'inside',
                        radiusAxisIndex: 0
                    },
                    {
                        type: 'slider',
                        radiusAxisIndex: 0,
                        backgroundColor: 'rgba(255,255,255,1)',
                        left: '95%',
                        top: '10%',
                        bottom: '50%',
                        orient: 'vertical',
                        borderColor: '#EFEFEF',
                        showDetail: false,
                        dataBackground: {
                            lineStyle: {
                                color: 'rgb(210,210,210)'
                            },
                            areaStyle: {
                                color: 'rgb(210,210,210)'
                            }
                        },
                        handleIcon: 'M10.7,11.9H9.3c-4.9,0.3-8.8,4.4-8.8,9.4c0,5,3.9,9.1,8.8,9.4h1.3c4.9-0.3,8.8-4.4,8.8-9.4C19.5,16.3,15.6,12.2,10.7,11.9z M13.3,24.4H6.7v-1.2h6.6z M13.3,22H6.7v-1.2h6.6z M13.3,19.6H6.7v-1.2h6.6z',
                        width: 20,
                        handleSize: 24,
                        handleStyle: {
                            color: 'rgba(79, 164, 222, 0.6)'
                        },
                        fillerColor: 'rgba(154,201,248,0.3)'
                    }
                ];
            } else if (param === 'Angle Zoom') {
                dataZoom = [
                    {
                        type: 'inside',
                        angleAxisIndex: 0
                    },
                    {
                        type: 'slider',
                        angleAxisIndex: 0,
                        backgroundColor: 'rgba(255,255,255,1)',
                        borderColor: '#EFEFEF',
                        showDetail: false,
                        dataBackground: {
                            lineStyle: {
                                color: 'rgb(210,210,210)'
                            },
                            areaStyle: {
                                color: 'rgb(210,210,210)'
                            }
                        },
                        height: 20,
                        handleIcon: 'M10.7,11.9H9.3c-4.9,0.3-8.8,4.4-8.8,9.4c0,5,3.9,9.1,8.8,9.4h1.3c4.9-0.3,8.8-4.4,8.8-9.4C19.5,16.3,15.6,12.2,10.7,11.9z M13.3,24.4H6.7v-1.2h6.6z M13.3,22H6.7v-1.2h6.6z M13.3,19.6H6.7v-1.2h6.6z',
                        handleSize: 24,
                        handleStyle: {
                            color: 'rgba(79, 164, 222, 0.6)'
                        },
                        fillerColor: 'rgba(154,201,248,0.3)'
                    }
                ];
            } else {
                dataZoom = [];
            }
            return dataZoom;
        }

        /**
         * @name initializeEvents
         * @desc creates the event layer
         * @returns {void}
         */
        function initializeEvents() {
            // Event: On single click open new tab with wikipedia page of selected label
            polarBarChart.on('click', eChartClicked);
            polarBarChart.on('mouseover', eChartMouse);
            polarBarChart.on('mouseout', eChartMouseOut);
            polarBarChart._dom.addEventListener('mouseout', mouseOut);

            // Context Menu
            polarBarChart.on('contextmenu', function (e) {
                scope.visualizationCtrl.setContextMenuDataFromClick(e, {
                    name: [eChartsConfig.legendLabels]
                });
            });
            polarBarChart._dom.addEventListener('contextmenu', scope.visualizationCtrl.openContextMenu);

            // it is necessary to initialize comment mode so the nodes are painted
            EchartsHelper.initializeCommentMode({
                comments: eChartsConfig.comments,
                currentMode: eChartsConfig.currentMode,
                saveCb: eChartsConfig.callbacks.commentMode.onSave
            });

            if (typeof eChartsConfig.callbacks.defaultMode.onKeyUp === 'function' || typeof eChartsConfig.callbacks.defaultMode.onKeyDown === 'function') {
                polarBarChart._dom.tabIndex = 1;
                if (typeof eChartsConfig.callbacks.defaultMode.onKeyUp === 'function') {
                    polarBarChart._dom.addEventListener('keyup', function (e) {
                        eChartsConfig.callbacks.defaultMode.onKeyUp({
                            eventType: 'onKeyUp',
                            key: e.key,
                            event: e,
                            keyCode: e.keyCode
                        });
                    });
                }
                if (typeof eChartsConfig.callbacks.defaultMode.onKeyDown === 'function') {
                    polarBarChart._dom.addEventListener('keydown', function (e) {
                        eChartsConfig.callbacks.defaultMode.onKeyDown({
                            eventType: 'onKeyDown',
                            key: e.key,
                            event: e,
                            keyCode: e.keyCode
                        });
                    });
                }
            }
        }

        /**
         * @name eChartClicked
         * @desc single click event from echarts
         * @param {object} event - echarts event sent back on click
         * @returns {void}
         */
        function eChartClicked(event) {
            if (clickTimer) {
                clearTimeout(clickTimer);
                eventCallback(event, 'onDoubleClick');
            } else {
                clickTimer = setTimeout(eventCallback.bind(null, event, 'onClick'), 250);
            }
        }

        /**
         * @name eChartMouse
         * @desc onHover event for echarts
         * @param {object} event - echarts event sent back on hover
         * @returns {void}
         */
        function eChartMouse(event) {
            if (hoverTimer) {
                clearTimeout(hoverTimer);
            }
            hoverTimer = setTimeout(eventCallback.bind(null, event, 'onHover'), 2000);
        }

        /**
         * @name eChartMouseOut
         * @desc offHover event for echarts
         * @param {object} event - echarts event sent back on offHover
         * @returns {void}
         */
        function eChartMouseOut(event) {
            var currentEvent = scope.widgetCtrl.getEvent('currentEvent');
            if (currentEvent.type === 'onHover') {
                eventCallback(event, 'onMouseOut');
            }
        }

        /**
         * @name mouseOut
         * @desc clears timers on mouse out of canvas
         * @returns {void}
         */
        function mouseOut() {
            clearTimeout(hoverTimer);
        }

        /**
         * @name eventCallback
         * @desc click callback event
         * @param {object} event - echarts event sent back on click
         * @param {string} type - click or double click
         * @returns {void}
         */
        function eventCallback(event, type) {
            var returnObj = {
                data: {}
            };
            returnObj.data[eChartsConfig.legendLabels] = [event.name];
            eChartsConfig.callbacks.defaultMode[type](returnObj);
            clickTimer = null;
            clearTimeout(hoverTimer);
            hoverTimer = null;
        }

        /**
         * @name executeCallback
         * @desc execute callback for a specific type with params
         * @param {string} type - type of callback to execute
         * @param {object} args - arguments to pass into the callback
         * @returns {void}
         */
        // TODO determine if hardcoding events or not... Use below for current mode checks
        // function executeCallback(type, args) {
        //     var cb;
        //     if (eChartsConfig.currentMode && eChartsConfig.callbacks.hasOwnProperty(eChartsConfig.currentMode)) {
        //         cb = eChartsConfig.callbacks[eChartsConfig.currentMode][type];
        //         if (typeof cb === 'function') {
        //             cb.apply(null, args);
        //         }
        //     }
        // }

        /**
         * @name toggleMode
         * @desc switches the jv mode to the new specified mode
         * @returns {void}
         */
        function toggleMode() {
            eChartsConfig.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
        }

        /**
         * @name resizeViz
         * @desc reruns the jv paint function
         * @returns {void}
         */
        function resizeViz() {
            polarBarChart.resize();
        }

        /**
         * @name destroy
         * @desc destroys listeners and dom elements outside of the scope
         * @returns {void}
         */
        function destroy() {
            resizeListener();
            updateTaskListener();
            updateOrnamentsListener();
            addDataListener();
            modeListener();
        }

        // Start Visualization Creation
        initialize();
    }
}
