'use strict';

import { PREVIEW_LIMIT } from '@/core/constants.js';
import './pipeline-query.scss';

export default angular.module('app.pipeline.pipeline-query',
    []
).directive('pipelineQuery', pipelineQueryDirective);

pipelineQueryDirective.$inject = ['$timeout', 'ENDPOINT', 'semossCoreService'];

function pipelineQueryDirective($timeout, ENDPOINT, semossCoreService) {
    pipelineQueryCtrl.$inject = [];
    pipelineQueryLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        template: require('./pipeline-query.directive.html'),
        scope: {},
        require: ['^widget', '^pipelineComponent'],
        controller: pipelineQueryCtrl,
        controllerAs: 'pipelineQuery',
        bindToController: {},
        link: pipelineQueryLink
    };

    function pipelineQueryCtrl() {}

    function pipelineQueryLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];
        scope.pipelineComponentCtrl = ctrl[1];

        scope.pipelineQuery.frameType = undefined;
        scope.pipelineQuery.customFrameName = {
            name: '',
            valid: true,
            message: ''
        };
        scope.pipelineQuery.selectedApp = '';

        scope.pipelineQuery.query = '';

        scope.pipelineQuery.updateFrame = updateFrame;
        scope.pipelineQuery.updateApp = updateApp;
        scope.pipelineQuery.updateQuery = updateQuery;
        scope.pipelineQuery.previewQuery = previewQuery;
        scope.pipelineQuery.importQuery = importQuery;
        scope.pipelineQuery.validateFrameName = validateFrameName;

        /**
         * @name setFrameData
         * @desc set the frame type
         * @return {void}
         */
        function setFrameData() {
            scope.pipelineQuery.frameType = scope.widgetCtrl.getOptions('initialFrameType');
        }

        /**
         * @name updateFrame
         * @param {string} type - type
         * @desc update the frame type
         * @return {void}
         */
        function updateFrame(type) {
            scope.widgetCtrl.setOptions('initialFrameType', type);
        }

        /**
         * @name getApps
         * @desc set the initial data
         * @returns {void}
         */
        function getApps() {
            let selectedApp = semossCoreService.app.get('selectedApp'),
                keepSelected = false,
                reset = true,
                qsComponent;

            qsComponent = scope.pipelineComponentCtrl.getComponent('parameters.QUERY_STRUCT.value');

            if (scope.pipelineComponentCtrl.getComponent('parameters.SELECTED_APP.value')) {
                scope.pipelineQuery.selectedApp = scope.pipelineComponentCtrl.getComponent('parameters.SELECTED_APP.value');
                if (qsComponent && qsComponent.engineName) {
                    keepSelected = true;
                    reset = false;
                }
            } else if (selectedApp && selectedApp !== 'NEWSEMOSSAPP') {
                scope.pipelineQuery.selectedApp = selectedApp;
            }

            if (reset || keepSelected) {
                updateApp(reset);
            }
        }

        /**
         * @name updateApp
         * @desc update the information based on the selected APP
         * @param {boolean} reset - should we reset the information or use the information from the QUERY_STRUCT
         * @returns {void}
         */
        function updateApp(reset) {
            var queryComponent = scope.pipelineComponentCtrl.getComponent('parameters.QUERY_STRUCT.value');
            scope.pipelineQuery.customFrameName.name = scope.pipelineComponentCtrl.createFrameName(scope.pipelineQuery.selectedApp.display);
            if (reset || !queryComponent.query) {
                scope.pipelineQuery.query = '';
            } else if (queryComponent.query) {
                scope.pipelineQuery.query = queryComponent.query;
            }

            previewQuery();
        }

        /**
         * @name updateQuery
         * @param {event} event - DOM event of the action
         * @desc update the frame type
         * @return {void}
         */
        function updateQuery(event) {
            if (event && event.ctrlKey && event.keyCode === 13) {
                previewQuery(true);
            }
        }


        /**
         * @name previewQuery
         * @param {boolean} alert - message on errors
         * @desc import the query
         * @returns {void}
         */
        function previewQuery(alert) {
            var parameters = {},
                valid = true;

            if (scope.pipelineQuery.query.length === 0) {
                if (alert) {
                    scope.widgetCtrl.alert('warn', 'Query is empty. Please enter a query.');
                }
                valid = false;
            }

            if (valid) {
                parameters = buildParameters(true);
            }

            scope.pipelineComponentCtrl.previewComponent(parameters);
        }

        /**
         * @name importQuery
         * @param {boolean} visualize if true visualize frame
         * @desc import the query
         * @returns {void}
         */
        function importQuery(visualize) {
            let callback,
                parameters,
                options = {};

            if (scope.pipelineQuery.query.length === 0) {
                scope.widgetCtrl.alert('warn', 'Query is empty. Please enter a query.');
                return;
            }

            parameters = buildParameters();

            options = {};
            options.name = scope.pipelineQuery.selectedApp.display;
            options.icon = scope.pipelineQuery.selectedApp.image;


            if (visualize) {
                callback = scope.pipelineComponentCtrl.visualizeComponent;
            }

            scope.pipelineComponentCtrl.executeComponent(parameters, options, callback);
        }

        /**
         * @name buildParameters
         * @param {boolean} preview - true if coming from preview
         * @desc build the parameters for the current module
         * @returns {object} - map of the paramters to value
         */
        function buildParameters(preview) {
            return {
                'IMPORT_FRAME': {
                    'name': scope.pipelineComponentCtrl.getComponent('parameters.IMPORT_FRAME.value.name') || scope.pipelineQuery.customFrameName.name,
                    'type': scope.pipelineComponentCtrl.getComponent('parameters.IMPORT_FRAME.value.type') || scope.widgetCtrl.getOptions('initialFrameType'),
                    'override': true
                },
                'QUERY_STRUCT': {
                    'qsType': 'RAW_ENGINE_QUERY',
                    'engineName': scope.pipelineQuery.selectedApp.value,
                    'query': scope.pipelineQuery.query,
                    'limit': preview ? PREVIEW_LIMIT : -1
                },
                'SELECTED_APP': scope.pipelineQuery.selectedApp
            };
        }

        /**
         * @name validateFrameName
         * @desc checks if the frame name entered by the user is valid
         * @returns {void}
         */
        function validateFrameName() {
            let results = scope.pipelineComponentCtrl.validateFrameName(scope.pipelineQuery.customFrameName.name);
            scope.pipelineQuery.customFrameName.valid = results.valid;
            scope.pipelineQuery.customFrameName.message = results.message;
        }

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize() {
            ele[0].querySelector('textarea').focus();
            setFrameData();
            getApps();

            scope.$on('$destroy', function () {});
            scope.pipelineQuery.customFrameName.name = scope.pipelineComponentCtrl.createFrameName(scope.pipelineQuery.selectedApp.display);
        }

        initialize();
    }
}
