'use strict';

import * as d3 from 'd3';
import d3tip from '@/widget-resources/js/d3_v4-tip/index.js';
import '@/widget-resources/js/d3-parcoords/d3.parcoords_v4.js';
import '@/widget-resources/css/d3-charts.css';
import './parallel-coordinates.scss';


export default angular.module('app.parallel-coordinates-standard.directive', [])
    .directive('parallelCoordinatesStandard', parallelCoordinatesStandard);

parallelCoordinatesStandard.$inject = ['$filter', '$timeout', '_', 'semossCoreService'];

function parallelCoordinatesStandard($filter, $timeout, _, semossCoreService) {
    parcoordsLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];
    parcoordsCtrl.$inject = [];

    return {
        restrict: 'E',
        require: ['^widget'],
        link: parcoordsLink,
        scope: {},
        bindToController: {},
        controllerAs: 'parcoordsCtrl',
        controller: parcoordsCtrl
    };

    function parcoordsCtrl() {}

    function parcoordsLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];
        /** **************Main Event Listeners*************************/
        var // declare/initialize local variables
            chart,
            tooltipId,
            parcoordsData = [],
            dimensions,
            heightToggle = false,
            widthToggle = false,
            updateStatus = false,
            numcols = 0,
            parcoordsHighlightData = [],
            uniqueRowCount = 0,
            uriMap = {},
            tip,
            parcoords,
            parcoordsDiv,
            customMargins = {
                top: 50,
                left: 130,
                right: 0,
                bottom: 10
            },
            d3 = window.d3;

        function setData() {
            var newData,
                dataTableAlignArray = [],
                key,
                datum,
                tempData = [],
                tempDatum,
                uri,
                dimCount,
                individualTools = scope.widgetCtrl.getWidget('view.visualization.tools.individual.ParallelCoordinates') || {},
                sharedTools = scope.widgetCtrl.getWidget('view.visualization.tools.shared'),
                color = semossCoreService.visualization.getColorPalette(sharedTools.colorName)[0],
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.ParallelCoordinates'),
                layerIndex = 0,
                data = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.data'),
                uniqueCount,
                i,
                len;

            newData = semossCoreService.visualization.getTableData(data.headers, data.values, data.rawHeaders);
            newData.dataTableAlign = semossCoreService.visualization.getDataTableAlign(keys);
            newData.uiOptions = angular.extend(sharedTools, individualTools);

            d3.select(chart).selectAll('*').remove();
            d3.select(chart)
                .attr('class', 'parallel-coordinates-standard');

            parcoordsDiv = d3.select(chart)
                .append('div')
                .attr('class', 'parallel-coordinates-standard__chart');

            dimensions = chart.getBoundingClientRect();

            parcoords = d3.parcoords({
                color: color
            })(parcoordsDiv.node())
                .alpha(0.8)
                .mode('queue') // progressive rendering
                .rate(300)
                .height(d3.max([dimensions.height - customMargins.top - customMargins.bottom, 500]))
                .width(d3.max([dimensions.width - customMargins.left - customMargins.right, 700]))
                .margin(customMargins);

            if (!_.isEmpty(newData) && parcoordsData[0] !== newData) {
                // sets all tools to original values
                uniqueRowCount = 0;
                heightToggle = false;
                widthToggle = false;
                parcoordsData = [];

                dataTableAlignArray = [];
                for (key in newData.dataTableAlign) {
                    if (newData.dataTableAlign.hasOwnProperty(key)) {
                        dataTableAlignArray.push(newData.dataTableAlign[key]);
                    }
                }

                // set parcoords data to the correct format
                for (datum in newData.viewData) {
                    if (newData.viewData.hasOwnProperty(datum)) {
                        tempDatum = {};
                        for (key in newData.viewData[datum]) {
                            if (dataTableAlignArray.indexOf(key) > -1) {
                                // var filteredKey = $filter('replaceUnderscores')(key);
                                uri = '' + newData.viewData[datum][key];
                                tempDatum[key] = uri; // may need to change key back to filtered
                                // if (uri.indexOf('http://') > -1) {
                                //     uriMap[uri] = {
                                //         uri: uri,
                                //         header: key
                                //     };
                                // }
                            }
                        }
                        tempData[datum] = tempDatum;
                    }
                }

                parcoordsData[0] = tempData;

                for (i = 0, len = newData.viewHeaders.length; i < len; i++) {
                    uniqueCount = _.uniq(_.map(parcoordsData[0], newData.viewHeaders[i])).length;

                    uniqueRowCount = uniqueRowCount < uniqueCount ? uniqueCount : uniqueRowCount;
                }

                if (parcoordsData[0].length === 0) { // no data
                    return;
                }
                update(parcoordsData[0]);

                if (parcoords.dimensions().length === undefined) {
                    dimCount = Object.keys(parcoords.dimensions()).length;
                } else {
                    dimCount = parcoords.dimensions().length;
                }

                if (dimCount !== _.size(parcoordsData[0][0])) {
                    scope.widgetCtrl.alert('warn', 'Some axes are not shown because they have too many distinct values to display. Please filter/reduce your data to see axes');
                }
                if (newData.uiOptions) {
                    if (newData.uiOptions.heightFitToScreen === true) {
                        heightFitToScreen();
                    } else if (newData.uiOptions.widthFitToScreen === true) {
                        widthFitToScreen();
                    }
                }
            }
        }

        function highlightSelectedItem(selectedItem) {
            var allLabels = parcoordsDiv.selectAll('text:not(.label)'),
                selectedPLabel = [],
                i,
                tempData,
                selectedObjects = [];

            for (i in selectedItem) {
                if (selectedItem.hasOwnProperty(i)) {
                    selectedPLabel.push(
                        parcoordsDiv.selectAll('text:not(.label)').filter(function (d) {
                            return d === $filter('shortenAndReplaceUnderscores')(selectedItem[i].uri);
                        })
                    );
                }
            }

            allLabels.style({
                'font-weight': 'normal'
            });

            for (i in selectedPLabel) {
                if (selectedPLabel.hasOwnProperty(i)) {
                    selectedPLabel[i].style({
                        'font-weight': 'bold'
                    });
                }
            }

            tempData = parcoords.data();

            for (i in selectedItem) {
                if (selectedItem.hasOwnProperty(i)) {
                    selectedObjects.push(tempData.filter(function (obj) {
                        if (_.includes(_.values(obj), $filter('shortenAndReplaceUnderscores')(selectedItem[i].uri))) {
                            return true;
                        }
                    }));
                }
            }

            for (i in selectedObjects) {
                if (!_.isEmpty(selectedObjects[i])) {
                    parcoords.highlight(selectedObjects[i]);
                }
            }
        }

        function resizeViz() {
            var height;

            dimensions = chart.getBoundingClientRect();

            d3.selectAll('#' + tooltipId).remove();

            parcoords
                .autoscale()
                .height(
                    heightToggle ?
                        Math.max(uniqueRowCount * 15 + 50, dimensions.height - customMargins.top - customMargins.bottom) :
                        dimensions.height - customMargins.top - customMargins.bottom
                )
                .width(
                    widthToggle ?
                        d3.max([uniqueRowCount * 10, numcols * 320]) :
                        dimensions.width - customMargins.left - customMargins.right
                )
                .render()
                .shadows()
                .reorderable()
                .brushable();

            if (parcoordsHighlightData.length > 0) {
                parcoords.highlight(parcoordsHighlightData);
            }

            tip = d3tip()
                .attr('class', 'd3-tip')
                .attr('id', tooltipId)
                .html(function (d) {
                    if (isNaN(d)) {
                        return $filter('shortenAndReplaceUnderscores')(d);
                    }
                    if (d > 1) {
                        return d3.format(',')(d);
                    }
                    return d3.format(',.2f')(d);
                });

            parcoordsDiv.selectAll('.axis text').call(tip);
            parcoordsDiv.selectAll('.axis text')
                .attr('title', function (d) {
                    if (isNaN(d)) {
                        return $filter('shortenAndReplaceUnderscores')(d);
                    }
                    if (d > 1) {
                        return d3.format(',')(d);
                    }
                    return d3.format(',.2f')(d);
                })
                .on('mouseover', tip.show)
                .on('mouseout', tip.hide);

            // reattach the dblclick event when resizing; it gets removed when you resize height/width
            parcoordsDiv.selectAll('text:not(.label)')
                .on('dblclick', function (element) {
                    var filteredEle = $filter('replaceSpaces')(element),
                        selectedItem;
                    if (uriMap[filteredEle]) {
                        selectedItem = [{
                            uri: uriMap[filteredEle].uri,
                            name: (element),
                            axisName: uriMap[filteredEle].header
                        }];
                        highlightSelectedItem(selectedItem);
                    }
                });

            height = parcoordsDiv.select('svg').node().getBBox().height;
            parcoordsDiv.selectAll('.axis').each(function () {
                var ticks = d3.select(this).selectAll('.tick').nodes().length;
                if ((height / ticks) < 12) {
                    d3.select(this).selectAll('.tick').style('display', 'none');
                } else {
                    d3.select(this).selectAll('.tick').style('display', null);
                }
            });
        }

        function heightFitToScreen() {
            heightToggle = !heightToggle;
            resizeViz();
        }

        function widthFitToScreen() {
            widthToggle = !widthToggle;
            resizeViz();
        }

        function drillDown() {
            var filterObj = {},
                components = [],
                panels,
                hasData = false,
                alias,
                i,
                len;

            parcoordsDiv.selectAll('.selection') // grabs the brushed rectangles
                .each(function () { // loop through and find the column that has been brushed
                    var column, col, val;

                    // parcoords
                    if (globalBrushed && globalBrushed.length > 0 && d3.brushSelection(d3.select(this).nodes()[0].parentNode) !== null) {
                        // TODO this sets the filterColumn correctly but if user drills down a second time, before they 'applyFilter' it will only take account of the latest filterColumn when 'apply filter' is run
                        column = d3.select(this).nodes()[0].parentNode.parentNode.__data__;
                        if (column !== 'Count') { // TODO this is not good. there may be an actual column called 'Count'
                            col = column.replace(/ /g, '_');
                            if (!filterObj[col]) {
                                filterObj[col] = [];
                            }

                            for (i = 0, len = globalBrushed.length; i < len; i++) {
                                if (globalBrushed[i].hasOwnProperty(column)) {
                                    val = globalBrushed[i][column].replace(/ /g, '_');

                                    // TODO this is bad--if there is an actual string called 'null', it will break
                                    if (val === 'null') {
                                        val = null;
                                    }

                                    if (filterObj[col].indexOf(val) === -1) {
                                        filterObj[col].push(val);
                                    }
                                }
                            }
                        }
                    }
                });

            for (alias in filterObj) {
                if (filterObj.hasOwnProperty(alias)) {
                    if (filterObj[alias].length > 0) {
                        components.push({
                            type: 'variable',
                            components: [
                                scope.widgetCtrl.getFrame('name')
                            ]
                        }, {
                            'type': 'setFrameFilter',
                            'components': [
                                [{
                                    type: 'value',
                                    alias: alias,
                                    comparator: '==',
                                    values: filterObj[alias]
                                }]
                            ],
                            'terminal': true
                        });
                        hasData = true;
                    }
                }
            }


            if (!hasData) {
                components.push({
                    type: 'variable',
                    components: [
                        scope.widgetCtrl.getFrame('name')
                    ]
                }, {
                    'type': 'unfilterFrame',
                    'components': [],
                    'terminal': true
                });
            }

            if (components.length > 0) {
                panels = scope.widgetCtrl.getShared('panels');
                // refresh
                for (i = 0, len = panels.length; i < len; i++) {
                    components.push({
                        'type': 'refresh',
                        'components': [panels[i].widgetId],
                        'terminal': true
                    });
                }
            }

            scope.widgetCtrl.execute(components);
        }

        /**
         * @name toolUpdateProcessor
         * @param {object} toolUpdateConfig information passed from message service
         * @desc routes tools when they are used (ie add count)
         * @return {void}
         */
        function toolUpdateProcessor(toolUpdateConfig) {
            if (toolUpdateConfig) {
                if (toolUpdateConfig.fn === 'heightFitToScreen') {
                    heightFitToScreen();
                } else if (toolUpdateConfig.fn === 'widthFitToScreen') {
                    widthFitToScreen();
                }
                if (toolUpdateConfig.fn === 'drillDown') {
                    drillDown();
                }
            }
            // setData();
        }


        function update(updateData) {
            var selectedItem,
                key,
                filteredData,
                i;

            dimensions = [];

            // if its not the first time data is loaded (create button pressed 1+n times) and not drilling down (aka reset function or new data passed)
            if (updateStatus) {
                parcoords.unhighlight(parcoordsHighlightData);
                parcoordsHighlightData = [];
            }

            // create button pressed at least once
            updateStatus = true;

            // set the dimensions for the data
            for (key in updateData[0]) {
                if (key !== '$$hashKey' && key !== '$selected') {
                    dimensions.push(key);
                }
            }
            numcols = dimensions.length;

            filteredData = updateData;
            for (i = 0; i < dimensions.length; i++) {
                dimensions[i] = $filter('shortenAndReplaceUnderscores')(dimensions[i]);
            }

            parcoords
                .data(filteredData)
                .detectDimensions()
                .dimensions(dimensions);

            resizeViz();

            // we're using a timeout here to add the dblclick event because the full graph does not get
            // created right away. before we selectAll, we need the elements to be on the canvas.
            $timeout(function () {
                parcoordsDiv.selectAll('text:not(.label)')
                    .on('dblclick', function (element) {
                        var filteredEle = $filter('replaceSpaces')(element);
                        if (uriMap[filteredEle]) {
                            selectedItem = [{
                                uri: uriMap[filteredEle].uri,
                                name: element,
                                axisName: uriMap[filteredEle].header
                            }];
                            highlightSelectedItem(selectedItem);
                        }
                    });
            });

            // Array.prototype.forEach.call(document.querySelectorAll('.d3-tip'), (t) => t.parentNode.removeChild(t));
        } // end of update function

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize() {
            var resizeListener = scope.widgetCtrl.on('resize-widget', resizeViz),
                toolListener = scope.widgetCtrl.on('update-tool', toolUpdateProcessor),
                updateListener = scope.widgetCtrl.on('update-ornaments', setData),
                updateTaskListener = scope.widgetCtrl.on('update-task', setData),
                addDataListener = scope.widgetCtrl.on('added-data', setData);

            chart = ele[0].firstElementChild;
            tooltipId = 'parallel-coordinates-standard__tip__' + scope.widgetCtrl.widgetId;

            setData();

            scope.$on('$destroy', function () {
                resizeListener();
                toolListener();
                updateListener();
                updateTaskListener();
                addDataListener();
                // clear chart div
                globalBrushed = undefined;
                globalOrder = undefined;
                chart.innerHTML = '';
                d3.selectAll('#' + tooltipId).remove();
            });
        }

        initialize();
    }
}
