'use strict';

import angular from 'angular';

import './nlp-search.scss';
import {TEMPLATE_VALUES, SUGGESTION_TYPE, TEMPLATE_TYPE, TEMPLATES, AGGREGATES} from './templates';

export default angular.module('app.nlp-search.directive', [])
    .directive('nlpSearch', nlpSearchDirective);

nlpSearchDirective.$inject = ['semossCoreService', '$timeout', 'optionsService'];

function nlpSearchDirective(semossCoreService: SemossCoreService, $timeout, optionsService: OptionsService) {
    nlpSearchLink.$inject = [];

    return {
        restrict: 'E',
        template: require('./nlp-search.directive.html'),
        scope: {},
        require: ['^widget'],
        controllerAs: 'nlpSearch',
        bindToController: {},
        link: nlpSearchLink,
        controller: nlpSearchCtrl
    };

    function nlpSearchCtrl() { }

    function nlpSearchLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];

        let updateFrameListener;
        let templateDropdownEle;


        // Frame
        scope.nlpSearch.availableFrames = [];
        scope.nlpSearch.selectedFrame = '';
        scope.nlpSearch.headers = [];
        scope.nlpSearch.frameMap = {};

        // Template Dropdown
        scope.nlpSearch.selectedTemplate = '';
        scope.nlpSearch.showTemplateDropdown = false;

        // Query
        scope.nlpSearch.query = [];
        scope.nlpSearch.disableIndex = scope.nlpSearch.query.length;
        scope.nlpSearch.temporaryComponent;

        // Suggestions
        scope.nlpSearch.suggestions = [];
        scope.nlpSearch.suggestionType = SUGGESTION_TYPE.TEMPLATE;

        // Results
        scope.nlpSearch.results = [];
        scope.nlpSearch.loading = false;

        // Functions
        scope.nlpSearch.frameChanged = frameChanged;
        scope.nlpSearch.isValidQuery = isValidQuery;
        scope.nlpSearch.clearQuery = clearQuery;
        scope.nlpSearch.addComponent = addComponent;
        scope.nlpSearch.componentUpdated = componentUpdated;
        scope.nlpSearch.resetComponent = resetComponent;
        scope.nlpSearch.deleteComponent = deleteComponent;
        scope.nlpSearch.getSuggestions = getSuggestions;
        scope.nlpSearch.getInsights = getInsights;
        scope.nlpSearch.createInsight = createInsight;

        /**
         * @name clearQuery
         * @desc resets the whole query 
         */
        function clearQuery(): void {
            scope.nlpSearch.query = [];
            scope.nlpSearch.suggestionType = SUGGESTION_TYPE.TEMPLATE;
            scope.nlpSearch.results = [];
            scope.nlpSearch.temporaryComponent = '';
        }

        /**
         * @name addComponent
         * @desc called whenever a new component is added to the query
         * @param model - the component to add
         */
        function addComponent(): void{
            let model = scope.nlpSearch.selectedTemplate;
            scope.nlpSearch.selectedTemplate = '';
            templateDropdownEle.blur();
            scope.nlpSearch.suggestionType = SUGGESTION_TYPE.SUBCOMPONENT;
            let template:any = {
                    component: model,
                    query: TEMPLATES[model].query,
                    column: {
                        options: [],
                        selected: TEMPLATES[model].column == 'SINGLE' ? '' : [],
                        type: TEMPLATES[model].column,
                        showPopover: false
                    },
                    isComplete: false
                };
            if (TEMPLATES[model].operation) {
                template.operation = {
                    options: [],
                    selected: TEMPLATES[model].operation == 'SINGLE' ? '' : [],
                    type: TEMPLATES[model].operation
                }
            }
            if (TEMPLATES[model].static_operation) {
                template.static_operation = TEMPLATES[model].static_operation;
            }
            if (TEMPLATES[model].value) {
                template.value = {
                    options: [],
                    selected: TEMPLATES[model].value == 'SINGLE' ? '' : [],
                    type: TEMPLATES[model].value
                }
            }
            if (TEMPLATES[model].static_value) {
                template.static_value = TEMPLATES[model].static_value;
            }

            scope.nlpSearch.query.push(template);
            let currentIndex = scope.nlpSearch.query.length - 1;

            // If the user manually added the group, sync with select
            if (model === TEMPLATE_TYPE.GROUP) {
                scope.nlpSearch.temporaryComponent = currentIndex;
                scope.nlpSearch.query[currentIndex].omit = 'column';
                updateSelect(scope.nlpSearch.query[currentIndex], []);
            }
            


            
            // If an aggregate is added, must check to see if select and group components are present
            if (AGGREGATES.indexOf(model) > -1) {
                    //let selectPresent = false;
                    let groupPresent = false;
                for (let i = 0; i < scope.nlpSearch.query.length; i++) {
                    // if (scope.nlpSearch.query[i].component === TEMPLATE_TYPE.SELECT) {
                    //     selectPresent = true;
                    // }
                    if (scope.nlpSearch.query[i].component === TEMPLATE_TYPE.GROUP) {
                        groupPresent = true;
                    }
                }
                
                // if (!selectPresent) {
                //     //TO DO: auto add select
                // }
                if (!groupPresent) {
                    let groupTemplate = {
                        component: TEMPLATE_TYPE.GROUP,
                        query: TEMPLATES[TEMPLATE_TYPE.GROUP].query,
                        column: {
                            options: [],
                            selected: TEMPLATES[TEMPLATE_TYPE.GROUP].column == 'SINGLE' ? '' : [],
                            type: TEMPLATES[TEMPLATE_TYPE.GROUP].column
                        },
                        isComplete: true,
                        isTemporary: true,
                        omit: 'all'
                    };
                    
                    scope.nlpSearch.query.push(groupTemplate);
                    updateSelect(scope.nlpSearch.query[scope.nlpSearch.query.length - 1], []);
                    scope.nlpSearch.temporaryComponent = scope.nlpSearch.query.length - 1;
                }
            }
            getSuggestions(currentIndex, 'column');
        }

        /**
         * @name componentUpdated
         * @desc called whenever a template is updated in the query, will get the next set of suggestions
         * @param templateIndex the location of the template in the query
         */
        function componentUpdated(templateIndex: number): void {
            let template = scope.nlpSearch.query[templateIndex],
                required: string[] =  TEMPLATES[template.component].required,
                isComplete = true,
                i;

            // If the component needs a value, set the data type of the column
            if (required.indexOf('value') > -1) {
                if (template.column.selected) {
                    for (let k = 0; k < scope.nlpSearch.headers.length; k++) {
                        if (template.column.selected === scope.nlpSearch.headers[k].alias) {
                            template.dataType = scope.nlpSearch.headers[k].dataType;
                            break;
                        }
                    }
                }
            }
            // Check if all the required parts are filled in the template
            for (i = 0; i < required.length; i++) {
                if (template[required[i]].selected.length < 1) {
                    isComplete = false;
                    break;
                }
            }

            // Enforces rule that group and select must have the same columns selected
            if (template.component === TEMPLATE_TYPE.SELECT) {
                updateGroup(template, template.column.selected);
            }
            if (template.component === TEMPLATE_TYPE.GROUP) {
                // Add all group columns to select
                updateSelect(template, template.column.selected);
            }


            // If the component is complete, then get the next set of component suggestions
            // Else get the suggestions for the next subcomponent
            if (isComplete) {
                template.isComplete = true;
                if (templateIndex === scope.nlpSearch.temporaryComponent) {
                    scope.nlpSearch.temporaryComponent = '';
                    scope.nlpSearch.query[templateIndex].omit = '';
                    scope.nlpSearch.suggestionType = SUGGESTION_TYPE.TEMPLATE;
                }
                if (templateIndex + 1 == scope.nlpSearch.temporaryComponent) {
                    scope.nlpSearch.query[templateIndex + 1].omit = scope.nlpSearch.query[templateIndex + 1].omit === 'all' ? 'column' : '';
                    
                    scope.nlpSearch.suggestionType = SUGGESTION_TYPE.SUBCOMPONENT;
                    getSuggestions(templateIndex + 1, 'column');
                } else {
                    scope.nlpSearch.suggestionType = SUGGESTION_TYPE.TEMPLATE;
                    getSuggestions();
                }
                
            } else {
                template.isComplete = false;
                scope.nlpSearch.suggestionType = SUGGESTION_TYPE.SUBCOMPONENT;
                getSuggestions(templateIndex, required[i]);
            }
        }

        /**
         * @name mergeColumns
         * @desc helper function to merge columns to enforce that select and group have the same columns
         * @param col1 - array of columns
         * @param col2 - array  of columns
         */
        function mergeColumns (col1, col2): string[] {
            let merged: string[] = [];

            for (let id1 = 0; id1 < col1.length; id1++) {
                if (merged.indexOf(col1[id1]) === -1) {
                    merged.push(col1[id1]);
                }
            }

            for (let id2 = 0; id2 < col2.length; id2++) {
                if (merged.indexOf(col2[id2]) === -1) {
                    merged.push(col2[id2]);
                }
            }

            return merged;
        }
        /**
         * @name updateGroup
         * @desc updates the selected columns for the group component when the select component is updated
         * @param columns - list of selected columns
         */
        function updateGroup(template, columns): void {
            for (let i = 0; i < scope.nlpSearch.query.length; i++) {
                if (scope.nlpSearch.query[i].component === TEMPLATE_TYPE.GROUP) {
                    let old = scope.nlpSearch.query[i].column.selected,
                        merged = mergeColumns(old, columns);
                    scope.nlpSearch.query[i].column.selected = merged;
                    template.column.selected = merged;
                }
            }
        }

        /**
         * @name updateSelect
         * @desc updates the selected columns for the select component when the select component is updated
         * @param columns - list of selected columns
         */
        function updateSelect(template, columns): void {
            for (let i = 0; i < scope.nlpSearch.query.length; i++) {
                if (scope.nlpSearch.query[i].component === TEMPLATE_TYPE.SELECT) {
                    let old = scope.nlpSearch.query[i].column.selected,
                        merged = mergeColumns(old, columns);
                    scope.nlpSearch.query[i].column.selected = merged;
                    template.column.selected = merged;
                }
            }
        }

        /**
         * @name resetComponent
         * @desc for where and having components, will reset any subcomponents if an earlier subcomponent is updated
         * @param componentIndex - the index of the component to update
         * @param subcomponent  - the subcomponent that is being updated
         */
        function resetComponent (componentIndex: number, subcomponent: string): void {
            let component = scope.nlpSearch.query[componentIndex],
                template =  TEMPLATES[component.component];

            if (subcomponent == 'column' && template.operation && template.value) {
                // reset operation and value
                component.operation = {
                    options: [],
                    selected: template.operation == 'SINGLE' ? '' : [],
                    type: template.operation
                };
                component.value = {
                    options: [],
                    selected: template.value == 'SINGLE' ? '' : [],
                    type: template.value
                };
            } else if(subcomponent == 'operation' && template.value) {
                // reset value
                component.value = {
                    options: [],
                    selected: template.value == 'SINGLE' ? '' : [],
                    type: template.value
                }
            }
        }

        /**
         * @name deleteComponent
         * @desc called to delete a component in the query
         * @param index - the component to delete
         */
        function deleteComponent(index): void {
            if (index <= scope.nlpSearch.temporaryComponent) {
                scope.nlpSearch.temporaryComponent = '';
            }
            let newQuery = scope.nlpSearch.query;
            for (let i = scope.nlpSearch.query.length-1; i >= index; i--) {
                newQuery.pop();
            }
            scope.nlpSearch.query = newQuery;
            scope.nlpSearch.suggestionType = SUGGESTION_TYPE.TEMPLATE;
        }
    
        /**
         * @name getInsights
         * @desc runs the query that the user built and returns insights to build
         */
        function getInsights(): void {
            let query: any[] = [],
                message = semossCoreService.utility.random('query-pixel'),
                panelId: number = scope.widgetCtrl.getShared('panelCounter');
            panelId++;
            scope.nlpSearch.loading = true;

            for (let i = 0; i < scope.nlpSearch.query.length; i++) {
                let oldTemplate = scope.nlpSearch.query[i],
                    template: any = {
                        component: oldTemplate.query
                    };
                if (oldTemplate.column && oldTemplate.column.selected.length) {
                    template.column = oldTemplate.column.selected;
                }
                if (oldTemplate.operation && oldTemplate.operation.selected.length) {
                    template.operation = oldTemplate.operation.selected;
                }
                if (oldTemplate.static_operation) {
                    template.operation = oldTemplate.static_operation;
                }
                if (oldTemplate.value && oldTemplate.value.selected.length) {
                    template.value = oldTemplate.value.selected
                }
                if (oldTemplate.static_value) {
                    template.value = oldTemplate.static_value;
                }
                query.push(template);
            }

            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType[0];
                scope.nlpSearch.loading = false;
                if (type.indexOf('ERROR') > -1) {
                    return;
                }
                scope.nlpSearch.results = output;
                
                if (scope.nlpSearch.results.length === 1) {
                    createInsight(output[0].pixel);
                }
            });
            
            semossCoreService.emit('query-pixel', {
                commandList: [
                    {
                        type: 'Pixel',
                        components: [
                            scope.nlpSearch.selectedFrame
                        ]
                    },
                    {
                        type: 'naturalLanguageSearch',
                        components: [
                            query,
                            [],
                            false,
                            panelId
                        ],
                        terminal: true
                    }
                ],
                listeners: [],
                response: message,
                insightID: scope.widgetCtrl.insightID
            });
        }

        /**
         * @name createInsight
         * @desc creates the insight
         * @param insight - the insight to create
         */
        function createInsight(recipe): void {
            let message = semossCoreService.utility.random('execute-pixel'),
                currentPanel = scope.widgetCtrl.panelId,
                currentWidgetId = scope.widgetCtrl.widgetId;
            semossCoreService.once(message, function(response) {
                for (let i = 0; i < response.pixelReturn.length; i++) {
                    if (response.pixelReturn[i].operationType.indexOf('ERROR') > -1) {
                        return;
                    }
                }
                let panelId = response.pixelReturn[0].output.panelId,
                    widgetId = currentWidgetId.substring(0, currentWidgetId.length - currentPanel.length ) + panelId;
                optionsService.set(widgetId, 'nlpQuery', scope.nlpSearch.query)
            })
            semossCoreService.emit('execute-pixel', {
                commandList: [
                    {
                        type: 'Pixel',
                        components: [
                            recipe
                        ],
                        terminal: true
                    }
                ],
                listeners: [],
                insightID: scope.widgetCtrl.insightID,
                responseObject: {
                    response: message,
                    payload: {}
                }
            });
        }

        /**
         * @name getSuggestions
         * @desc gets the next set of suggested templates
         * @param templateIndex - the component to update (optional)
         * @param subcomponent - the subcomponent that is being updated (optional)
         */
        function getSuggestions(templateIndex?: number | undefined, subcomponent?: string | undefined): void {
            scope.nlpSearch.suggestions= [];
            let message = semossCoreService.utility.random('query-pixel'),
                query: any= [];

            // Build the query in the format that the backend expects
            for (let i = 0; i < scope.nlpSearch.query.length; i++) {
                let oldTemplate = scope.nlpSearch.query[i],
                    template: any = {
                        component: oldTemplate.query
                    };

                // If group was auto-added and the aggregate is empty, omit the whole component to get the options for the aggregate
                if (oldTemplate.omit && oldTemplate.omit === 'all') {
                    continue; 
                }
                // If group was auto-added and aggregate is filled, omit the column to get the options for the group 
                if (oldTemplate.omit && oldTemplate.omit === 'column') {
                    query.push(template);
                    continue;
                }

                if (oldTemplate.column && oldTemplate.column.selected.length) {
                    template.column = oldTemplate.column.selected;
                }
                if (oldTemplate.operation && oldTemplate.operation.selected.length) {
                    template.operation = oldTemplate.operation.selected;
                }
                if (oldTemplate.static_operation) {
                    template.operation = oldTemplate.static_operation;
                }
                if (oldTemplate.value && oldTemplate.value.selected.length) {
                    template.value = oldTemplate.value.selected
                }
                if (oldTemplate.static_value) {
                    template.value = oldTemplate.static_value;
                }
                query.push(template);
            }
            semossCoreService.once(message, function(response) {
                let output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType[0];
                if (type.indexOf('ERROR') > -1) {
                    return;
                }
                if (scope.nlpSearch.suggestionType === SUGGESTION_TYPE.SUBCOMPONENT && (templateIndex || templateIndex === 0) && subcomponent){
                    scope.nlpSearch.query[templateIndex][subcomponent].options = output.sort();
                    scope.nlpSearch.query[templateIndex][subcomponent].showPopover = true;
                    scope.nlpSearch.suggestions = [];
                    // If the template being updated is the temporary component, then update to get the next suggestions
                    if(scope.nlpSearch.temporaryComponent === templateIndex){
                        componentUpdated(templateIndex);
                    }
                    // If there is only one result, automatically set it and get the next set off suggestions
                    if (scope.nlpSearch.query[templateIndex][subcomponent].options.length === 1 && scope.nlpSearch.query[templateIndex][subcomponent].options[0] !== '?') {
                        scope.nlpSearch.query[templateIndex][subcomponent].selected = scope.nlpSearch.query[templateIndex][subcomponent].options[0];
                        scope.nlpSearch.query[templateIndex][subcomponent].showPopover = false;
                        componentUpdated(templateIndex);
                    }
                } else {
                    scope.nlpSearch.suggestions = output;
                }
                
            });
            semossCoreService.emit('query-pixel', {
                commandList: [
                    {
                        type: 'Pixel',
                        components: [
                            scope.nlpSearch.selectedFrame
                        ]
                    },
                    {
                        type:'nlsQueryHelper',
                        components: [
                            query,
                            [],
                            false

                        ],
                        terminal: true
                    }
                ],
                listeners: [],
                response: message,
                insightID: scope.widgetCtrl.insightID
            });
        }

        /**
         * @name getFrames
         * @desc gets the possible frames for the insight
         */
        function getFrames(): void {
            let frames = scope.widgetCtrl.getShared('frames') || {},
                current = scope.widgetCtrl.getFrame();
            scope.nlpSearch.availableFrames = [];
            scope.nlpSearch.frameMap = frames;
            for (let frame in frames) {
                if (frames.hasOwnProperty(frame)){
                    scope.nlpSearch.availableFrames.push(frames[frame].name);
                }
            }
            scope.nlpSearch.selectedFrame = current.name;
        }

        /**
         * @name frameChanged
         * @desc called to update the headers whenever the selected frame changes
         */
        function frameChanged(): void {
            for (let i = 0; i < scope.nlpSearch.availableFrames.length; i++) {
                if (scope.nlpSearch.availableFrames[i] === scope.nlpSearch.selectedFrame) {
                    scope.nlpSearch.headers = scope.nlpSearch.frameMap[scope.nlpSearch.availableFrames[i]].headers;
                    break;
                }
            }
        }


        /**
         * @name isValidQuery
         * @desc checks if the query is valid
         */
        function isValidQuery(): boolean {
            let hasSelect = false,
                hasAggregate = false,
                hasGroup = false;

            // Invalid if the frame has not been selected
            if (scope.nlpSearch.selectedFrame.length === 0) {
                scope.nlpSearch.errorMessage = 'You must select a frame.';
                return false;
            }

            // Invalid if the query is empty
            if (scope.nlpSearch.query.length === 0) {
                return false;
            }
            
            for (let i = 0; i < scope.nlpSearch.query.length; i++) {
                // Invalid if the components are incomplete
                if (!scope.nlpSearch.query[i].isComplete) {
                    scope.nlpSearch.disableIndex = i + 1;
                    scope.nlpSearch.errorMessage = 'All components must be complete before execution.';
                    return false;
                }
                if (scope.nlpSearch.query[i].component === TEMPLATE_TYPE.SELECT) {
                    hasSelect = true;
                }
                if (AGGREGATES.indexOf(scope.nlpSearch.query[i].component) > -1) {
                    hasAggregate = true;
                }
                if (scope.nlpSearch.query[i].component === TEMPLATE_TYPE.GROUP) {
                    hasGroup = true;
                }
            }

            scope.nlpSearch.disableIndex = scope.nlpSearch.query.length;

            // Invalid if the query does not contain select or aggregate (average, count, max, min, sum)
            if (!(hasSelect || hasAggregate)) {
                scope.nlpSearch.errorMessage = 'Your query must contain select, average, count, max, min, or sum.';
                return false;
            }

            // Invalid if the query contains an aggregate but no group
            if (hasAggregate && !hasGroup) {
                scope.nlpSearch.errorMessage = 'Your query must contain a group component.';
                return false;
            }
            scope.nlpSearch.errorMessage = '';
            return true;
        }

        /**
         * @name initialize
         * @desc called when the directive is loaded
         * @returns {void}
         */
        function initialize(): void {
            // Check the store if the widget has a previously saved query
            let storedQuery = optionsService.get(scope.widgetCtrl.widgetId, 'nlpQuery');
            if (storedQuery) {
                scope.nlpSearch.query = storedQuery;
            }
            getFrames();
            updateFrameListener = scope.widgetCtrl.on('update-frame', getFrames);

            templateDropdownEle = ele[0].querySelector('#template-dropdown');
            getSuggestions();
            $timeout(function(){
                scope.nlpSearch.showTemplateDropdown = true;
            })
            
            scope.$on('$destroy', function () {
                updateFrameListener();
            })

        }

        initialize();
    }
}
