'use strict';
/**
 * @name map-echarts.service.js
 * @desc This service configures our data into echarts data as according to the
 *       e-charts examples (one of which is: https://ecomfe.github.io/echarts-examples/public/editor.html?c=map-disk).
 *       This service functions similarly to the other e-charts services but it
 *       has two specific functions: setMapDataSeries() and constructChildObj().
 *       The first function simply loops through our structured data and converts
 *       it into the appropriate data object to display map data, while the
 *       second is a function and the first calls in order to create child objects
 *       that are populated into our map data object.
 */
angular.module('app.map.service', [])
    .factory('mapService', mapService);

mapService.$inject = ['semossCoreService'];

function mapService(semossCoreService) {
    var colorArr = [];

    function getConfig(type, data, uiOptions, keysObj, groupByInstance, groupBy) {
        colorArr = uiOptions.color;

        var finalData,
            groupedData,
            i,
            formattedData,
            boundingCoord,
            animation = customizeAnimation(uiOptions.animation),
            backgroundColorStyle = false;

        if (groupBy && groupBy.viewType === 'All Instances') {
            finalData = [];
            for (i = 0; i < groupBy.uniqueInstances.length; i++) {
                groupedData = {};
                groupedData.headers = data.headers;
                groupedData.rawHeaders = data.rawHeaders;
                groupedData.values = groupBy.tempData[groupBy.uniqueInstances[i]];
                finalData.push(formatData(groupedData, keysObj, colorArr).data);
            }
        } else {
            formattedData = formatData(data, keysObj, colorArr);
            finalData = formattedData.data;
            boundingCoord = formattedData.boundingCoord;
        }

        // Set Background Color (check for watermark)
        if (/\S/.test(uiOptions.watermark)) {
            backgroundColorStyle = {
                type: 'pattern',
                image: paintWaterMark(uiOptions.watermark),
                repeat: 'repeat'
            };
        }

        // Configure data for ECharts
        return {
            data: finalData,
            boundingCoords: getBoundingCoord(boundingCoord),
            legendHeaders: finalData.valuesNames,
            showLegend: !uiOptions.toggleLegend,
            showAnimation: animation.showAnimation,
            animationType: animation.defaultAnimationType,
            animationDelay: animation.defaultAnimationSpeed,
            animationDuration: animation.defaultAnimationDuration,
            backgroundColorStyle: backgroundColorStyle,
            groupByInstance: groupByInstance,
            options: getUiOptions(uiOptions)
        };
    }

    function getDataTableAlign(headers, keys) {
        var returnObj = {},
            tooltipArr = [],
            i,
            j;

        // Loop through keys
        for (i = 0; i < keys.length; i++) {
            // For each key find it's corresponding header position
            for (j = 0; j < headers.length; j++) {
                if (keys[i].alias === headers[j]) {
                    // If it's a tooltip just add it to our temp array
                    if (keys[i].model === 'tooltip') {
                        tooltipArr.push(j);
                        // Else put it in our returnObj
                    } else {
                        returnObj[keys[i].model] = j;
                    }
                    break;
                }
            }
        }

        // If we found tooltips add them to our returnObj
        if (tooltipArr.length > 0) {
            returnObj.tooltip = tooltipArr;
        }

        return returnObj;
    }

    /**
     * @name formatData
     * @desc puts data into ECharts data format
     * @param {Object} data - object of data in original format
     * @param {Object} keys - semoss keys
     * @param {Array} color - array of color palette
     * @returns {Object} - object of data in ECharts format
     */
    function formatData(data, keys, color) {
        var dataArray = [],
            dataTableAlign = getDataTableAlign(data.headers, keys),
            tooltipPositions = [],
            hasSize = false,
            hasColor = false,
            tempObj = {},
            valueArray,
            colorMapping,
            colorIdx,
            min,
            max,
            tempSize,
            pos,
            latArray = [],
            lonArray = [],
            boundingCoord = {},
            i,
            j;

        if (dataTableAlign.hasOwnProperty('size')) {
            tooltipPositions.push(dataTableAlign.size);
            hasSize = true;
        }

        if (dataTableAlign.hasOwnProperty('color')) {
            tooltipPositions.push(dataTableAlign.color);
            hasColor = true;
        }

        if (dataTableAlign.hasOwnProperty('tooltip')) {
            for (i = 0; i < dataTableAlign.tooltip.length; i++) {
                pos = dataTableAlign.tooltip[i];

                // Ensure that the tooltip value is not already included in what we are displaying to the user
                if (pos !== dataTableAlign.latitude && pos !== dataTableAlign.longitude && pos !== dataTableAlign.label &&
                    !_.includes(tooltipPositions, pos)) {
                    tooltipPositions.push(pos);
                } else {
                    // TODO: Make the alert bound to a widget
                    semossCoreService.emit('alert', {
                        color: 'warn',
                        text: data.headers[pos] + ' is already being displayed in the tooltip. Ignoring duplicate entry.'
                    });
                }
            }
        }

        // Grab all sizes and make them their own array
        if (hasSize) {
            valueArray = data.values.map(function (val) {
                return val[dataTableAlign.size];
            });

            min = _.min(valueArray);
            max = _.max(valueArray);
        }

        if (hasColor) {
            colorMapping = {};
            colorIdx = 0;
            for (i = 0; i < data.values.length; i++) {
                if (!colorMapping.hasOwnProperty(data.values[i][dataTableAlign.color])) {
                    colorMapping[data.values[i][dataTableAlign.color]] = color[colorIdx];
                    colorIdx++;
                    if (colorIdx >= color.length) {
                        colorIdx = colorIdx % color.length;
                    }
                }
            }
        }
        // Loop through our data
        for (i = 0; i < data.values.length; i++) {
            if (data.values[i][dataTableAlign.latitude] === null || data.values[i][dataTableAlign.longitude] === null) {
                continue;
            }
            latArray.push(data.values[i][dataTableAlign.latitude]);
            lonArray.push(data.values[i][dataTableAlign.longitude]);
            // Default sizing
            if (!hasSize) {
                tempObj = {
                    name: data.values[i][dataTableAlign.label],
                    value: [
                        data.values[i][dataTableAlign.longitude], // Longitude
                        data.values[i][dataTableAlign.latitude], // Latitude
                        80 // Default value if none provided
                    ],
                    valueMapping: {
                        label: data.headers[dataTableAlign.label],
                        longitude: data.headers[dataTableAlign.longitude],
                        latitude: data.headers[dataTableAlign.latitude]
                    }
                };
                // Custom sizing
            } else {
                tempSize = calcSize(min, max, data.values[i][dataTableAlign.size]);

                tempObj = {
                    name: data.values[i][dataTableAlign.label],
                    value: [
                        data.values[i][dataTableAlign.longitude], // Longitude
                        data.values[i][dataTableAlign.latitude], // Latitude
                        tempSize // Optional value
                    ],
                    valueMapping: {
                        label: data.headers[dataTableAlign.label],
                        longitude: data.headers[dataTableAlign.longitude],
                        latitude: data.headers[dataTableAlign.latitude]
                    }
                };
            }

            if (hasColor) {
                tempObj.color = colorMapping[data.values[i][dataTableAlign.color]];
            } else {
                tempObj.color = color[0];
            }

            // Create a tooltip sub-object if the user has passed tooltips
            // * note * we can have tooltips without having size
            if (tooltipPositions.length > 0) {
                tempObj.tooltips = {};

                for (j = 0; j < tooltipPositions.length; j++) {
                    tempObj.tooltips[data.headers[tooltipPositions[j]]] = data.values[i][tooltipPositions[j]];
                }
            }

            tempObj.hasSize = hasSize;

            // Push our final object to our dataArray
            dataArray.push(tempObj);
        }

        // Get bounding coordinates
        if (latArray.length > 1 && lonArray.length > 1) {
            boundingCoord.lat = {
                'min': Math.min.apply(null, latArray),
                'max': Math.max.apply(null, latArray)
            };

            boundingCoord.lon = {
                'min': Math.min.apply(null, lonArray),
                'max': Math.max.apply(null, lonArray)
            };
        }

        return {
            data: dataArray,
            boundingCoord: boundingCoord
        };
    }

    function calcSize(min, max, val) {
        var sizeMin = 60,
            sizeMax = 300,
            nodeSize;

        nodeSize = Math.round(((val - min) / (max - min)) * (sizeMax - sizeMin) + sizeMin);

        if (isNaN(Number(nodeSize))) {
            return sizeMin;
        }

        if (nodeSize < sizeMin) {
            return sizeMin;
        }

        if (nodeSize > sizeMax) {
            return sizeMax;
        }

        return nodeSize;
    }

    /**
     * @name getBoundingCoord
     * @desc sets the bounding coordinates
     * @param {object} param - max and min of lat and long
     * @returns {array/bool} - bounding coordinates
     */
    function getBoundingCoord(param) {
        if (param) {
            if (param.hasOwnProperty('lat') || param.hasOwnProperty('lon')) {
                return [
                    [param.lon.min, param.lat.max],
                    [param.lon.max, param.lat.min]
                ];
            }
        }

        return null;
    }

    /**
     * @name customizeAnimation
     * @desc sets the animation style
     * @param {object} param - object of animation settings
     * @returns {object} - object of animation settings
     */
    function customizeAnimation(param) {
        var animationSettings = {};
        // TODO small refactor
        if (param && param.chooseType === 'No Animation') {
            animationSettings.showAnimation = false;
        } else if (param) {
            animationSettings.showAnimation = true;
            animationSettings.defaultAnimationType = param.chooseType;
            animationSettings.defaultAnimationSpeed = param.animationSpeed;
            animationSettings.defaultAnimationDuration = param.animationDuration;
        } else {
            // default
            animationSettings.showAnimation = true;
            // TODO set as same as widget service default state
            animationSettings.defaultAnimationType = 'No Animation';
            animationSettings.defaultAnimationSpeed = 1;
            animationSettings.defaultAnimationDuration = 500;
        }
        return animationSettings;
    }

    function getUiOptions(options) {
        return options;
    }

    /**
     * @name paintWaterMark
     * @desc paints a custom watermark on the viz
     * @param {string} watermark - string of the watermark text
     * @returns {Object} - canvas details
     */
    function paintWaterMark(watermark) {
        var canvas = document.createElement('canvas'),
            ctx = canvas.getContext('2d');
        canvas.width = canvas.height = 100;
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        ctx.globalAlpha = 0.08;
        ctx.font = '20px Microsoft Yahei';
        ctx.translate(50, 50);
        ctx.rotate(-Math.PI / 4);
        if (watermark) {
            ctx.fillText(watermark, 0, 0);
        }
        return canvas;
    }


    return {
        getConfig: getConfig
    };
}
