'use strict';

import echarts from 'echarts';
import EchartsHelper from '@/widget-resources/js/echarts/echarts-helper.js';
import '@/widget-resources/js/echarts/map/world.js';
import './map-echarts.service.js';

/**
 *
 * @name map-echarts
 * @desc map-echarts chart directive for creating and visualizing a column chart
 */

export default angular.module('app.map-echarts.directive', [
    'app.map.service'
])
    .directive('mapEcharts', mapEcharts);

mapEcharts.$inject = ['semossCoreService', 'mapService'];

function mapEcharts(semossCoreService, mapService) {
    mapChartLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^widget'],
        priority: 300,
        link: mapChartLink
    };

    function mapChartLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];

        /** ************* Main Event Listeners ************************/
        var resizeListener,
            updateTaskListener,
            updateOrnamentsListener,
            addDataListener,
            modeListener,
            /** *************** ECharts ****************************/
            eChartsConfig,
            mapChart,
            num,
            clickTimer,
            destroyListeners,
            hoverTimer;

        /**
         * @name initialize
         * @desc creates the visualization on the chart div
         * @returns {void}
         */
        function initialize() {
            // bind listeners
            resizeListener = scope.widgetCtrl.on('resize-widget', resizeViz);
            updateTaskListener = scope.widgetCtrl.on('update-task', setData);
            updateOrnamentsListener = scope.widgetCtrl.on('update-ornaments', setData);
            addDataListener = scope.widgetCtrl.on('added-data', setData);
            modeListener = scope.widgetCtrl.on('update-mode', toggleMode);

            scope.$on('$destroy', destroy);

            setData();
        }

        /**
         * @name setData
         * @desc setData for the visualization and paints
         * @returns {void}
         */
        function setData() {
            var selectedLayout = scope.widgetCtrl.getWidget('view.visualization.layout'),
                individiualTools = scope.widgetCtrl.getWidget('view.visualization.tools.individual.' + selectedLayout) || {},
                sharedTools = scope.widgetCtrl.getWidget('view.visualization.tools.shared'),
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.' + selectedLayout),
                layerIndex = 0,
                data = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.data'),
                groupBy = {},
                groupedData,
                groupByInstance,
                groupByInfo = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.groupByInfo'),
                uiOptions = angular.extend(sharedTools, individiualTools),
                n;

            uiOptions.color = semossCoreService.visualization.getColorPalette(uiOptions.colorName);

            if (groupByInfo && groupByInfo.viewType) {
                if (groupByInfo.viewType === 'Individual Instance') {
                    groupBy = formatDataForGroupByIndividual(data, groupByInfo);
                    data = groupBy.data;
                    groupByInstance = groupBy.name;
                } else if (groupByInfo.viewType === 'All Instances') {
                    groupedData = formatDataForGroupByAll(data, groupByInfo);
                    groupByInfo.tempData = groupedData;
                    // Remove empty groups (if filtered)
                    groupByInfo.uniqueInstances = [];
                    for (n = 0; n < Object.keys(groupedData).length; n++) {
                        groupByInfo.uniqueInstances.push(Object.keys(groupedData)[n]);
                    }
                }
            }

            eChartsConfig = mapService.getConfig('map', data, uiOptions, keys, groupByInstance, groupByInfo);
            eChartsConfig.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.callbacks = scope.widgetCtrl.getEventCallbacks();
            eChartsConfig.comments = scope.widgetCtrl.getWidget('view.visualization.commentData');
            eChartsConfig.groupByInfo = groupByInfo;
            num = getHighlightMapsNum();

            determineResize();
            paint();
        }

        /**
         * @name formatDataForGroupByIndividual
         * @desc formats data when Group By exists
         * @param {object} data orginial data
         * @param {object} groupBy groupBy object
         * @returns {void}
         */
        function formatDataForGroupByIndividual(data, groupBy) {
            var formattedData = data,
                groupByIndex,
                name,
                i,
                instanceIdx,
                returnObj = {};

            if (groupBy.viewType === 'Individual Instance') {
                groupByIndex = data.headers.indexOf(groupBy.selectedDim);
                if (groupByIndex === -1) {
                    // return data;
                    groupByIndex = data.headers.length;
                }

                if (typeof groupBy.instanceIndex === 'string') {
                    instanceIdx = parseInt(groupBy.instanceIndex, 10);
                }
                // Create name for title
                name = groupBy.selectedDim + ' : ' + groupBy.uniqueInstances[instanceIdx];
                // Remove Group By dimension from data headers and values
                formattedData.headers.splice(groupByIndex, 1);
                formattedData.rawHeaders.splice(groupByIndex, 1);

                // Remove any added data from brush/click
                for (i = 0; i < data.values.length; i++) {
                    if (data.values[i][groupByIndex] !== groupBy.uniqueInstances[instanceIdx]) {
                        data.values.splice(i, 1);
                        i--;
                    }
                }

                for (i = 0; i < data.values.length; i++) {
                    data.values[i].splice(groupByIndex, 1);
                }
                returnObj.name = name;
                returnObj.data = data;
            }

            return returnObj;
        }

        /**
         * @name formatDataForGroupByAll
         * @desc formats data when Group By All Instances exists
         * @param {object} data orginial data
         * @param {object} groupBy groupBy object
         * @returns {void}
         */
        function formatDataForGroupByAll(data, groupBy) {
            var groupByIndex,
                i,
                n,
                dataObj = {};

            groupByIndex = data.headers.indexOf(groupBy.selectedDim);
            data.headers.splice(groupByIndex, 1);
            data.rawHeaders.splice(groupByIndex, 1);

            for (n = 0; n < groupBy.uniqueInstances.length; n++) {
                dataObj[groupBy.uniqueInstances[n]] = [];
                for (i = 0; i < data.values.length; i++) {
                    if (data.values[i][groupByIndex] === groupBy.uniqueInstances[n]) {
                        data.values[i].splice(groupByIndex, 1);
                        dataObj[groupBy.uniqueInstances[n]].push(data.values[i]);
                    }
                }

                if (dataObj[groupBy.uniqueInstances[n]].length === 0) {
                    delete dataObj[groupBy.uniqueInstances[n]];
                }
            }

            return dataObj;
        }

        /**
         * @name getHighlightMapsNum
         * @return {number} The number to fill in for the optional highlight map series option
         * @desc Returns the number of cities to highlight or none if the user hasn't selected any
         */
        function getHighlightMapsNum() {
            if (eChartsConfig.options.highlightMaps) {
                return eChartsConfig.options.highlightMaps;
            }

            return 0;
        }

        /**
         * @name paint
         * @desc paints the visualization
         * @returns {void}
         */
        function paint() {
            // var option;

            if (mapChart) {
                mapChart.clear();
                mapChart.dispose();
            }

            // TODO also think abou abstracting some of these options to variables for more customizabilty from uiOptions
            mapChart = echarts.init(ele[0].firstElementChild);

            var option = {},
                colorTheme,
                titleFontSize,
                facetLayout,
                i;

            if (eChartsConfig.options.facetHeaders && eChartsConfig.options.facetHeaders.titleFontSize) {
                titleFontSize = eChartsConfig.options.facetHeaders.titleFontSize;
            } else {
                titleFontSize = 18;
            }

            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType) {
                if (eChartsConfig.groupByInfo.viewType === 'Individual Instance') {
                    option.graphic = [];
                }
            }

            colorTheme = getColorTheme(eChartsConfig.options.mapColorTheme);
            option.backgroundColor = colorTheme.ocean;
            option.tooltip = {
                show: eChartsConfig.options.showTooltips,
                trigger: 'item',
                confine: true,
                formatter: function (info) {
                    var returnArray = [],
                        i;

                    if (info.data.value) {
                        returnArray = [
                            info.marker + String(info.data.name).replace(/_/g, ' ') + '<br>',
                            '<b>' + info.data.valueMapping.longitude + '</b>: ' + cleanValue(info.data.value[0]) + '<br>',
                            '<b>' + info.data.valueMapping.latitude + '</b>: ' + cleanValue(info.data.value[1])
                        ];

                        if (info.data.tooltips) {
                            for (i in info.data.tooltips) {
                                if (info.data.tooltips.hasOwnProperty(i)) {
                                    returnArray.push(
                                        '<br><b>' + String(i).replace(/_/g, ' ') + '</b>: ' + cleanValue(info.data.tooltips[i])
                                    );
                                }
                            }
                        }
                    }

                    return returnArray.join('');
                }
            };

            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType === 'All Instances') {
                option.geo = [];
                option.series = [];
                option.title = [];
                for (i = 0; i < eChartsConfig.groupByInfo.uniqueInstances.length; i++) {
                    facetLayout = getMapPosition(i, eChartsConfig.groupByInfo.uniqueInstances[i], eChartsConfig.options.facetHeaders);
                    option.geo.push({
                        map: 'world',
                        label: {
                            emphasis: {
                                show: false
                            }
                        },
                        roam: false,
                        itemStyle: {
                            normal: {
                                // TODO add tool widget to change color scheme
                                areaColor: colorTheme.land, // Options: #323c48 (default), #ffffff (white), #339966 (green),
                                borderColor: colorTheme.borderColor
                            },
                            emphasis: {
                                areaColor: colorTheme.landEmphasis,
                                borderWidth: 2
                            }
                        },
                        width: '500px',
                        height: '250px',
                        top: facetLayout.mapPosition.top,
                        left: facetLayout.mapPosition.left,
                        id: i + ''
                    });
                    option.series.push({
                        name: 'Scatter Points',
                        type: 'scatter',
                        coordinateSystem: 'geo',
                        animation: false,
                        data: eChartsConfig.data[i],
                        symbolSize: function (val) {
                            return val[2] / 10;
                        },
                        label: {
                            normal: {
                                formatter: function (info) {
                                    return String(info.name).replace(/_/g, ' ');
                                },
                                // position: 'left',
                                fontSize: 15,
                                show: eChartsConfig.options.displayValues
                            },
                            emphasis: {
                                position: 'right',
                                show: true
                            }
                        },
                        itemStyle: {
                            normal: {
                                color: colorFn
                            }
                        },
                        geoIndex: i,
                        id: i + ''
                    });
                    option.title.push(facetLayout.titlePosition);
                }
                option.title = option.title.concat([{
                    text: eChartsConfig.options.facetHeaders.titleName || 'All Instances of ' + eChartsConfig.groupByInfo.selectedDim.replace(/_/g, ' '),
                    top: 'top',
                    left: 'center',
                    textStyle: {
                        fontSize: titleFontSize
                    }
                }]);
            } else {
                option.geo = {
                    map: 'world',
                    label: {
                        emphasis: {
                            show: false
                        }
                    },
                    roam: true,
                    zoom: 1.15,
                    boundingCoords: eChartsConfig.boundingCoords,
                    itemStyle: {
                        normal: {
                            areaColor: colorTheme.land, // Options: #323c48 (default), #ffffff (white), #339966 (green),
                            borderColor: colorTheme.borderColor
                        },
                        emphasis: {
                            areaColor: colorTheme.landEmphasis,
                            borderWidth: 2
                        }
                    }
                };
                option.series = [{
                    name: 'Scatter Points',
                    type: 'scatter',
                    coordinateSystem: 'geo',
                    animation: false,
                    data: eChartsConfig.data,
                    symbolSize: function (val) {
                        return val[2] / 10;
                    },
                    label: {
                        normal: {
                            formatter: function (info) {
                                return String(info.name).replace(/_/g, ' ');
                            },
                            fontSize: 15,
                            show: eChartsConfig.options.displayValues
                        },
                        emphasis: {
                            position: 'right',
                            show: true
                        }
                    },
                    itemStyle: {
                        normal: {
                            color: colorFn,
                            opacity: 0.85
                        }
                    },
                    progressiveThreshold: 50000
                }
                    // TODO bring back "highlight-map" additional tool
                    // ,
                    // {
                    //     name: function () {
                    //         var returnString = '';

                    //         if (eChartsConfig.options.highlightMaps) {
                    //             returnString += 'Showing top ' + eChartsConfig.options.highlightMaps;
                    //         }

                    //         return returnString;
                    //     },
                    //     type: 'effectScatter',
                    //     coordinateSystem: 'geo',
                    //     data: eChartsConfig.data.sort(function (a, b) {
                    //         return b.value[2] - a.value[2];
                    //     }).slice(0, num),
                    //     symbolSize: function (val) {
                    //         return val[2] / 10;
                    //     },
                    //     showEffectOn: 'render',
                    //     rippleEffect: {
                    //         brushType: 'stroke'
                    //     },
                    //     hoverAnimation: true,
                    //     label: {
                    //         normal: {
                    //             formatter: function (info) {
                    //                 return String(info.name).replace(/_/g, ' ');
                    //             },
                    //             position: 'left',
                    //             show: true,
                    //             fontSize: 15 // Arbitrary default
                    //         }
                    //     },
                    //     itemStyle: {
                    //         normal: {
                    //             color: colorFn,
                    //             shadowBlur: 10,
                    //             shadowColor: '#333'
                    //         }
                    //     },
                    //     zlevel: 1
                    // }
                ];
            }
            option.textStyle = {
                fontFamily: 'Libre Franklin'
            };
            // use configuration item and data specified to show chart
            EchartsHelper.setOption(mapChart, option);

            // Add event listeners
            // TODO figure out comment / brush / eventing paradigm
            initializeEvents();
        }

        /**
         * @name getRowsAndColumns
         * @desc determines number of rows and columns for Facet All Instances
         * @param {number} instances - number of instances of selected Facet dimension
         * @param {obj} options - uiOptions
         * @returns {obj} - object of layout (rows and columns)
         */
        function getRowsAndColumns(instances, options) {
            var numRows,
                numColumns,
                numCharts = instances,
                chartContainer = ele[0].childNodes[0],
                containerHeight,
                containerWidth,
                mapWidth = 500,
                mapHeight = 250;

            // Set number of rows and columns
            if (options.facetHeaders && options.facetHeaders.numberColumns) {
                numColumns = parseInt(options.facetHeaders.numberColumns, 10);
                if (numColumns > numCharts) {
                    options.facetHeaders.numberColumns = numCharts;
                    numColumns = numCharts;
                }
                if (numColumns < 2) {
                    options.facetHeaders.numberColumns = 1;
                    numColumns = 1;
                }
                numRows = Math.ceil(numCharts / numColumns);
            } else {
                numColumns = Math.floor(chartContainer.clientWidth / (mapWidth + 50));
                if (numColumns < 2) {
                    numColumns = 1;
                }
                numRows = Math.ceil(numCharts / numColumns);
            }

            // Set Container Dimensions
            containerHeight = 100 + ((mapHeight + 50) * numRows);
            containerWidth = 25 + ((mapWidth + 25) * numColumns);
            if (containerHeight < chartContainer.clientHeight) {
                containerHeight = chartContainer.clientHeight;
            }
            if (containerWidth < chartContainer.clientWidth) {
                containerWidth = chartContainer.clientWidth;
            }

            return {
                'numRows': numRows,
                'numColumns': numColumns,
                'containerHeight': containerHeight,
                'containerWidth': containerWidth
            };
        }

        /**
         * @name getMapPosition
         * @desc sets grid dimensions based on whether or not datazoom is present
         * @param {num} idx i
         * @param {string} text title text
         * @param {obj} facetHeaders facetHeaders object
         * @returns {obj} - object of grid dimensions
         */
        function getMapPosition(idx, text, facetHeaders) {
            var numColumns = eChartsConfig.groupByInfo.numColumns,
                mapWidth = 500,
                mapHeight = 250,
                rowIndex = Math.floor(idx / numColumns),
                columnIndex = idx % numColumns,
                fontSize = 14,
                mapPosition = {},
                titlePosition = {};

            if (numColumns === 1) {
                mapPosition.left = 'center';
                titlePosition.left = 'center';
            } else {
                mapPosition.left = 25 + ((mapWidth + 25) * columnIndex) + 'px';
                titlePosition.left = 25 + (mapWidth / 2) + ((mapWidth + 25) * columnIndex) + 'px';
            }
            mapPosition.top = 100 + (mapHeight + 50) * rowIndex + 'px';

            if (typeof text === 'string') {
                titlePosition.text = text.replace(/_/g, ' ');
            } else {
                titlePosition.text = text;
            }

            titlePosition.top = 70 + (mapHeight + 50) * rowIndex + 'px';
            if (facetHeaders && facetHeaders.headerFontSize) {
                fontSize = facetHeaders.headerFontSize;
            }

            titlePosition.textAlign = 'center';
            titlePosition.textStyle = {
                fontSize: fontSize,
                fontWeight: 'bold'
            };

            return {
                'mapPosition': mapPosition,
                'titlePosition': titlePosition
            };
        }

        /**
         * @name cleanValue
         * @param {string | number} item the value to replace
         * @desc if number just returns value, otherwise removes spaces from string
         * @return {string | number} altered value
         */
        function cleanValue(item) {
            if (typeof item === 'string') {
                return item.replace(/_/g, ' ');
            } else if (typeof item === 'number') {
                return item.toLocaleString(undefined, {
                    minimumFractionDigits: 0,
                    maximumFractionDigits: 3
                });
            }
            return item;
        }

        /**
         * @name getColorTheme
         * @param {string} param selected color theme for map
         * @desc sets color for land and ocean
         * @return {obj} land and ocean color
         */
        function getColorTheme(param) {
            if (param === 'Dark') {
                return {
                    'land': '#323c48',
                    'landEmphasis': '#2a333d',
                    'ocean': '#617087',
                    'borderColor': '#111'
                };
            } else if (param === 'Light 2') {
                return {
                    'land': '#cbe5be ',
                    'landEmphasis': '#84aa70',
                    'ocean': '#cee9ff',
                    'borderColor': '#111'
                };
            } else if (param === 'Geo') {
                return {
                    'land': '#2d5236',
                    'landEmphasis': '#8fba99',
                    'ocean': '#283b72',
                    'borderColor': '#111'
                };
            }

            return {
                'land': '#f2efe9',
                'landEmphasis': '#e0dfdb',
                'ocean': '#aad3df',
                'borderColor': '#84bccc'
            };
        }

        function colorFn(info) {
            var selectedLayout = scope.widgetCtrl.getWidget('view.visualization.layout'),
                colorBy = scope.widgetCtrl.getWidget('view.visualization.colorByValue'),
                returnCBV = false;


            if (colorBy && colorBy.length > 0) {
                colorBy.forEach(function (rule) {
                    // our labels in visual panel dimensions vs their corresponding ways to access
                    // them via info arg, see switch statement below
                    // ours         theirs
                    // ====         ======
                    // label        info.data.name
                    // latitude     info.value[0]
                    // longitude    info.value[1]
                    // size         info.data.value[2]
                    // tooltips     info.data.tooltips
                    rule.valuesToColor.forEach(function (val) {
                        var coloringOn, i, tip,
                            dimensions = scope.widgetCtrl.getWidget('view.visualization.keys.' + selectedLayout),
                            cleanVal = val;

                        for (i = 0; i < dimensions.length; i++) {
                            if (rule.colorOn === dimensions[i].alias) {
                                coloringOn = dimensions[i].model;
                                break;
                            }
                        }

                        switch (coloringOn) {
                            case 'label':
                                if (info.data.name === cleanVal) {
                                    returnCBV = rule.color;
                                }
                                break;
                            case 'latitude':
                                if (info.value[0] === cleanVal) {
                                    returnCBV = rule.color;
                                }
                                break;
                            case 'longitude':
                                if (info.value[1] === cleanVal) {
                                    returnCBV = rule.color;
                                }
                                break;
                            case 'size':
                                if (info.data.value[2] === cleanVal) {
                                    returnCBV = rule.color;
                                }
                                break;
                            case 'tooltip':
                                for (tip in info.data.tooltips) {
                                    if (info.data.tooltips.hasOwnProperty(tip)) {
                                        if (info.data.tooltips[tip] === cleanVal) {
                                            returnCBV = rule.color;
                                            break;
                                        }
                                    }
                                }
                                break;
                            default:
                                scope.widgetCtrl.alert('error', 'Something went wrong coloring by value');
                        }
                    });
                });

                if (returnCBV) {
                    return returnCBV;
                }
            }

            return info.data.color;
        }

        /**
         * @name initializeEvents
         * @desc creates the event layer
         * @returns {void}
         */
        function initializeEvents() {
            if (typeof destroyListeners === 'function') {
                destroyListeners();
            }

            // it is necessary to initialize comment mode so the nodes are painted
            EchartsHelper.initializeCommentMode({
                comments: eChartsConfig.comments,
                currentMode: eChartsConfig.currentMode,
                saveCb: eChartsConfig.callbacks.commentMode.onSave
            });

            if (eChartsConfig.echartsMode) {
                mapChart.dispatchAction({
                    type: 'takeGlobalCursor',
                    key: 'brush',
                    brushOption: {
                        brushType: eChartsConfig.echartsMode
                    }
                });
            }

            if (eChartsConfig.currentMode === 'defaultMode' || eChartsConfig.currentMode === 'polygonBrushMode') {
                destroyListeners = EchartsHelper.initializeClickHoverKeyEvents(mapChart, {
                    cb: eChartsConfig.callbacks.defaultMode,
                    header: eChartsConfig.legendLabels,
                    getCurrentEvent: function () {
                        return scope.widgetCtrl.getEvent('currentEvent');
                    },
                    vizType: 'Map'
                });
            }
        }

        /**
         * @name toggleMode
         * @desc switches the jv mode to the new specified mode
         * @returns {void}
         */
        function toggleMode() {
            eChartsConfig.echartsMode = EchartsHelper.getEchartsMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
            initializeEvents();
        }

        /**
         * @name eventCallback
         * @desc click callback event
         * @param {object} event - echarts event sent back on click
         * @param {string} type - click or double click
         * @returns {void}
         */
        function eventCallback(event, type) {
            var returnObj = {
                data: {}
            };
            returnObj.data[eChartsConfig.legendLabels] = [event.name];
            eChartsConfig.callbacks.defaultMode[type](returnObj);
            clickTimer = null;
            hoverTimer = null;
        }

        /**
         * @name determineResize
         * @desc detemin parent and chart container dimensions if Facet exists
         * @param {bool} resize - whether or not container has been resized or not
         * @returns {void}
         */
        function determineResize(resize) {
            var chartContainer = ele[0].childNodes[0],
                parent = ele[0],
                facetLayout;

            parent.style.position = '';
            parent.style.top = '';
            parent.style.right = '';
            parent.style.bottom = '';
            parent.style.left = '';
            parent.style.overflowY = '';
            chartContainer.style.width = '';
            chartContainer.style.height = '';

            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType === 'All Instances') {
                facetLayout = getRowsAndColumns(eChartsConfig.groupByInfo.uniqueInstances.length, eChartsConfig.options);

                eChartsConfig.groupByInfo.numRows = facetLayout.numRows;
                eChartsConfig.groupByInfo.numColumns = facetLayout.numColumns;
                eChartsConfig.groupByInfo.containerHeight = facetLayout.containerHeight;

                parent.style.position = 'absolute';
                parent.style.top = '0';
                parent.style.right = '0';
                parent.style.bottom = '0';
                parent.style.left = '0';
                parent.style.overflowY = 'auto';
                chartContainer.style.height = '' + facetLayout.containerHeight + 'px';

                if (chartContainer.clientWidth < facetLayout.containerWidth) {
                    chartContainer.style.width = facetLayout.containerWidth + 'px';
                } else {
                    chartContainer.style.width = '';
                }
                parent.style.overflowY = 'auto';

                if (resize) {
                    paint();
                }
            }
        }

        /**
         * @name resizeViz
         * @desc reruns the jv paint function
         * @returns {void}
         */
        function resizeViz() {
            mapChart.resize();
            if (eChartsConfig.groupByInfo) {
                determineResize(true);
            }
        }

        /**
         * @name destroy
         * @desc destroys listeners and dom elements outside of the scope
         * @returns {void}
         */
        function destroy() {
            resizeListener();
            updateTaskListener();
            updateOrnamentsListener();
            addDataListener();
            modeListener();
        }

        // Start Visualization Creation
        initialize();
    }
}
