'use strict';

import echarts from 'echarts';
import EchartsHelper from '@/widget-resources/js/echarts/echarts-helper.js';
import './graph-echarts.service.js';

/**
 *
 * @name graph-echarts
 * @desc graph-echarts chart directive for creating and visualizing a column chart
 */

export default angular.module('app.graph-echarts.directive', [
    'app.graph-echarts.service'
]).directive('graphEcharts', graphEcharts);

graphEcharts.$inject = ['$timeout', 'VIZ_COLORS', 'graphEchartsService'];

function graphEcharts($timeout, VIZ_COLORS, graphEchartsService) {
    graphEchartsLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^widget', '^visualization'],
        priority: 300,
        link: graphEchartsLink
    };

    function graphEchartsLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];
        scope.visualizationCtrl = ctrl[1];

        /** ************* Main Event Listeners ************************/
        var resizeListener,
            updateTaskListener,
            updateOrnamentsListener,
            addDataListener,
            modeListener,
            /** *************** ECharts ****************************/
            eChartsConfig,
            graphChart,
            mouseUp,
            destroyListeners;


        /**
         * @name initialize
         * @desc creates the visualization on the chart div
         * @returns {void}
         */
        function initialize() {
            // bind listeners
            resizeListener = scope.widgetCtrl.on('resize-widget', resizeViz);
            updateTaskListener = scope.widgetCtrl.on('update-task', setData);
            updateOrnamentsListener = scope.widgetCtrl.on('update-ornaments', setData);
            addDataListener = scope.widgetCtrl.on('added-data', setData);
            modeListener = scope.widgetCtrl.on('update-mode', toggleMode);

            scope.$on('$destroy', function () {
                resizeListener();
                updateTaskListener();
                updateOrnamentsListener();
                addDataListener();
                modeListener();
            });

            setData();
        }

        /** Data */
        /**
         * @name setData
         * @desc setData for the visualization and paints
         * @returns {void}
         */
        function setData() {
            var active = scope.widgetCtrl.getWidget('active'),
                individiualTools = scope.widgetCtrl.getWidget('view.' + active + '.tools.individual.Graph') || {},
                sharedTools = scope.widgetCtrl.getWidget('view.' + active + '.tools.shared'),
                layerIndex = 0,
                data = scope.widgetCtrl.getWidget('view.' + active + '.tasks.' + layerIndex + '.data'),
                groupBy = {},
                groupByInstance,
                groupByInfo = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.groupByInfo'),
                colorBy = scope.widgetCtrl.getWidget('view.visualization.colorByValue'),
                uiOptions;

            uiOptions = angular.extend(sharedTools, individiualTools);
            uiOptions.colorByValue = colorBy;
            uiOptions = EchartsHelper.setColors(uiOptions, undefined, VIZ_COLORS);

            if (groupByInfo) {
                groupBy = formatDataForGroupBy(data, groupByInfo);
                data = groupBy.data;
                groupByInstance = groupBy.name;
            }

            // set the config
            eChartsConfig = graphEchartsService.getConfig(data, uiOptions, groupByInstance);
            eChartsConfig.comments = scope.widgetCtrl.getWidget('view.visualization.commentData');
            eChartsConfig.callbacks = scope.widgetCtrl.getEventCallbacks();
            eChartsConfig.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.echartsMode = EchartsHelper.getEchartsMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.groupByInfo = groupByInfo;

            paint();
        }

        /**
         * @name formatDataForGroupBy
         * @desc formats data when Group By exists
         * @param {object} data orginial data
         * @param {object} groupBy groupBy object
         * @returns {void}
         */
        function formatDataForGroupBy(data, groupBy) {
            var formattedData = data,
                groupByIndex,
                name,
                instanceIdx,
                returnObj = {};

            if (groupBy.viewType === 'Individual Instance') {
                groupByIndex = data.headers.indexOf(groupBy.selectedDim);
                if (groupByIndex === -1) {
                    // return data;
                    groupByIndex = data.headers.length;
                }

                if (typeof groupBy.instanceIndex === 'string') {
                    instanceIdx = parseInt(groupBy.instanceIndex, 10);
                }
                // Create name for title
                name = groupBy.selectedDim + ' : ' + groupBy.uniqueInstances[instanceIdx];
                // Remove Group By dimension from data headers and values
                formattedData.headers.splice(groupByIndex, 1);
                formattedData.rawHeaders.splice(groupByIndex, 1);

                // // Remove any added data from brush/click
                // for (i = 0; i < data.values.length; i++) {
                //     if (data.values[i][groupByIndex] !== groupBy.uniqueInstances[instanceIdx]) {
                //         data.values.splice(i, 1);
                //         i--;
                //     }
                // }

                // for (i = 0; i < data.values.length; i++) {
                //     data.values[i].splice(groupByIndex, 1);
                // }
                returnObj.name = name;
                returnObj.data = data;
            }

            return returnObj;
        }

        /**
         * @name paint
         * @desc paints the visualization
         * @returns {void}
         */
        function paint() {
            if (graphChart) {
                graphChart.clear();
                graphChart.dispose();
            }

            // initialize visualization
            graphChart = echarts.init(ele[0].firstElementChild);

            // add facet information
            var dataEmpty;

            if (eChartsConfig.groupByInfo) {
                dataEmpty = false;

                if (eChartsConfig.groupByInfo.viewType === 'Individual Instance') {
                    eChartsConfig.graphic = [];
                    if (dataEmpty) {
                        eChartsConfig.graphic = eChartsConfig.graphic.concat({
                            id: 'textGroup',
                            type: 'group',
                            right: 'center',
                            top: 'center',
                            children: [{
                                type: 'rect',
                                top: 'center',
                                right: 'center',
                                shape: {
                                    width: 200,
                                    height: 40
                                },
                                style: {
                                    fill: '#fff',
                                    stroke: '#999',
                                    lineWidth: 2,
                                    shadowBlur: 8,
                                    shadowOffsetX: 3,
                                    shadowOffsetY: 3,
                                    shadowColor: 'rgba(0,0,0,0.3)'
                                }
                            },
                            {
                                type: 'text',
                                right: 'center',
                                top: 'center',
                                style: {
                                    text: 'There is no data for this instance.',
                                    textAlign: 'center'
                                }
                            }
                            ]
                        });
                    }
                }
            }
            eChartsConfig.textStyle = {
                fontFamily: 'Libre Franklin'
            };

            // use configuration item and data specified to show chart
            EchartsHelper.setOption(graphChart, eChartsConfig);

            // initialize events
            initializeEvents();
        }

        /** Events */
        /**
         * @name eChartBrushed
         * @desc brush selected event from echarts
         * @param {object} e - charts event
         * @returns {void}
         */
        function eChartBrushed(e) {
            // batch = e.batch;
        }

        /**
         * @name startBrush
         * @desc sets up mouse listener and updates the batch from the event
         * @param {object} e - echarts event
         * @returns {void}
         */
        function startBrush() {
            // if (!mouseUp) {
            //     mouseUp = true;
            //     graphChart._dom.addEventListener('mouseup', executeBrushQuery);
            // }
        }

        /**
         * @name executeBrushQuery
         * @desc called on echarts brush mouseup, makes the pixel for the selected brushed data
         * @returns {void}
         */
        function executeBrushQuery() {
            // console.warn(eChartsConfig, batch);
            // var brushedValues,
            //     brushedIndecies,
            //     xLabels = graphChartConfig.xAxisConfig[0].data,
            //     brushObj = {};
            // // error handling
            // if (batch && batch[0] && batch[0].selected) {
            //     brushedIndecies = getBrushedIndecies(batch[0].selected);
            //     brushedValues = brushedIndecies.map(function (index) {
            //         if (index < xLabels.length || index >= 0) {
            //             return xLabels[index];
            //         }
            //         return 0;
            //     });
            //     brushObj.data = {};
            //     brushObj.data[config.dataTable.label] = brushedValues;
            //     // execute pixel with brushed values
            //     currentCallbacks.onBrush(brushObj);


            // TODO remove mouseUp bool.. find another less confusing way
            // batch = null;
            // mouseUp = false;
            // graphChart._dom.removeEventListener('mouseup', executeBrushQuery);
        }

        /**
         * @name initializeEvents
         * @desc creates the event layer
         * @returns {void}
         */
        function initializeEvents() {
            if (typeof destroyListeners === 'function') {
                destroyListeners();
            }

            // Context Menu
            graphChart.on('contextmenu', function (e) {
                if (e.name && e.hasOwnProperty('data') && e.data.category) {
                    scope.visualizationCtrl.setContextMenuDataFromClick(e.name, {
                        name: [e.data.category]
                    });
                }
            });
            graphChart._dom.addEventListener('contextmenu', scope.visualizationCtrl.openContextMenu);

            // it is necessary to initialize comment mode so the nodes are painted
            EchartsHelper.initializeCommentMode({
                comments: eChartsConfig.comments,
                currentMode: eChartsConfig.currentMode,
                saveCb: eChartsConfig.callbacks.commentMode.onSave
            });

            if (eChartsConfig.echartsMode) {
                graphChart.dispatchAction({
                    type: 'takeGlobalCursor',
                    key: 'brush',
                    brushOption: {
                        brushType: eChartsConfig.echartsMode
                    }
                });
            }

            if (eChartsConfig.currentMode === 'defaultMode' || eChartsConfig.currentMode === 'polygonBrushMode') {
                destroyListeners = EchartsHelper.initializeClickHoverKeyEvents(graphChart, {
                    cb: eChartsConfig.callbacks.defaultMode,
                    header: eChartsConfig.legendLabels,
                    getCurrentEvent: function () {
                        return scope.widgetCtrl.getEvent('currentEvent');
                    },
                    vizType: 'Graph'
                });
            }
        }

        /**
         * @name getBrushedIndecies
         * @desc returns the brushed indecies for the echarts selection
         * @param {array} selected - current array of echarts data
         * @returns {array} array of indecies
         */
        function getBrushedIndecies(selected) {
            return selected
                // reduce to get only indecies
                .reduce(getIndexData, [])
                // filter to be only unique values
                .filter(onlyUnique);
        }

        /**
         * @name toggleMode
         * @desc switches the jv mode to the new specified mode
         * @returns {void}
         */
        function toggleMode() {
            eChartsConfig.echartsMode = EchartsHelper.getEchartsMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
            initializeEvents();
        }

        /**
         * @name getIndexData
         * @desc continues to build the sumArray of dataIndex selections
         * @param {array} sumArray - current array of data indecies
         * @param {array} selection - current selection to iterate over
         * @returns {array} total array of indecies
         */
        function getIndexData(sumArray, selection) {
            if (Array.isArray(selection.dataIndex)) {
                sumArray.push.apply(sumArray, selection.dataIndex);
            }
            return sumArray;
        }

        /**
         * @name onlyUnique
         * @desc returns whether the value is in the self array
         * @param {string} value - value added to array
         * @param {number} index - current iteration index of self
         * @param {array} self - current array of unique values
         * @returns {boolean} - whether the value is already added to self or not
         */
        function onlyUnique(value, index, self) {
            return self.indexOf(value) === index;
        }

        /**
         * @name eChartClicked
         * @desc single click event from echarts
         * @param {object} event - echarts event sent back on click
         * @returns {void}
         */
        function eChartClicked(event) {
            // highlight
            if (event.dataType === 'node') {
                // TODO: Optimize
                graphChart.dispatchAction({
                    type: 'downplay'
                });
                graphChart.dispatchAction({
                    type: 'highlight',
                    dataIndex: event.dataIndex
                });

                dataPointSelected(event);
            }
        }


        /**
         * @name dataPointSelected
         * @param {object} event echarts event
         * @return {object} chart config
         */
        function dataPointSelected(event) {
            var selectedData;
            if (isNaN(Number(event.data.name))) {
                event.data.name = event.data.name;
            }
            selectedData = [{
                alias: event.data.category,
                instances: [event.data.name]
            }];


            $timeout(function () {
                scope.widgetCtrl.open('handle', 'traverse');
            });

            scope.widgetCtrl.emit('change-selected', {
                selected: selectedData
            });
        }


        // /**
        // * @name eChartClicked
        // * @desc single click event from echarts
        // * @param {object} event - echarts event sent back on click
        // * @returns {void}
        // */
        // function eChartClicked(event) {
        //     if (clickTimer) {
        //         clearTimeout(clickTimer);
        //         clickCallback(event, 'doubleClick');
        //     } else {
        //         clickTimer = setTimeout(clickCallback.bind(null, event, 'click'), 250);
        //     }
        // }

        // /**
        // * @name clickCallback
        // * @desc click callback event
        // * @param {string} type - click or double click
        // * @param {object} event - echarts event sent back on click
        // * @returns {void}
        // */
        // function clickCallback(event, type) {
        //     var returnObj = {
        //         data: {}
        //     };
        //     returnObj.data[eChartsConfig.legendLabels] = [event.name];
        //     if (type === 'click') {
        //         eChartsConfig.callbacks.defaultMode.onClick(returnObj);
        //     } else {
        //         eChartsConfig.callbacks.defaultMode.onDoubleClick(returnObj);
        //     }
        //     clickTimer = null;
        // }


        /**
         * @name resizeViz
         * @param {object} payload the payload from the message
         * @desc reruns the jv paint function
         * @returns {void}
         */
        function resizeViz(payload) {
            var containerEle = ele;
            // don't resize if we're just changing the menu sizing
            if (payload && payload.src === 'widget-view') {
                // we are looking for the container size...including the space the menu occupies
                while (containerEle[0].classList[0] !== 'widget-view__resizable') {
                    if (containerEle.parent()) {
                        containerEle = containerEle.parent();
                    } else {
                        containerEle = undefined;
                        console.error('cannot find the parent height and width to set for resizing graph.');
                        break;
                    }
                }

                // once we find it, we will use that size (which is the size of the panel)
                if (containerEle) {
                    graphChart.resize({
                        width: containerEle[0].clientWidth,
                        height: containerEle[0].clientHeight
                    });
                } else {
                    graphChart.resize();
                }
                return;
            }
            graphChart.resize();
            // We repaint so that we can re-center the graph viz
            paint();
        }

        // Start Visualization Creation
        initialize();
    }
}
