'use strict';

import angular from 'angular';

import visualizationUniversal from '../../core/store/visualization/visualization.js';

import './format-data-values.scss';

export default angular.module('app.format-data-values.directive', [])
    .directive('formatDataValues', formatDataValuesDirective);

formatDataValuesDirective.$inject = ['semossCoreService'];

function formatDataValuesDirective(semossCoreService) {
    formatDataValuesCtrl.$inject = [];
    formatDataValuesLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        scope: {},
        restrict: 'EA',
        require: ['^widget'],
        controllerAs: 'formatDataValues',
        bindToController: {},
        template: require('./format-data-values.directive.html'),
        controller: formatDataValuesCtrl,
        link: formatDataValuesLink
    };

    function formatDataValuesCtrl() { }

    function formatDataValuesLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];

        // variables
        scope.formatDataValues.dimensions = {
            list: [],
            selected: {}
        };

        scope.formatDataValues.formats = [];

        scope.formatDataValues.format = {
            dimension: '',
            dimensionType: '',
            model: '',
            type: 'Default',
            date: 'Default',
            delimiter: 'Default',
            prepend: '',
            append: '',
            round: 2,
            appliedString: '',
            layout: ''
        };
        scope.formatDataValues.formatOptions = semossCoreService.visualization.getCustomOptions();
        scope.formatDataValues.formatOptions.date = [
            {
                display: '1879-03-14',
                value: 'yyyy-MM-dd'
            },
            {
                display: '03/14/1879',
                value: 'MM/dd/yyyy'
            },
            {
                display: '3/14/1879',
                value: 'M/d/yyyy'
            },
            {
                display: '03/14/79',
                value: 'MM/dd/yy'
            },
            {
                display: '03/14',
                value: 'MM/dd'
            },
            {
                display: 'March 14, 1879',
                value: 'MMMMM d, yyyy'
            },
            {
                display: '14-Mar',
                value: 'dd-MMM'
            },
            {
                display: '14-Mar-79',
                value: 'dd-MMM-yy'
            },
            {
                display: '14-Mar-1879',
                value: 'dd-MMM-yyyy'
            },
            {
                display: 'Mar-79',
                value: 'MMM-yy'
            },
            {
                display: 'Friday, March 14, 1879',
                value: 'EEEEE, MMMMM d, yyyy'
            },
            {
                display: '3/14/79 1:30 PM',
                value: 'M/d/yy hh:mm a'
            },
            {
                display: '3/14/79 13:30',
                value: 'M/d/yy HH:mm'
            },
            {
                display: '3/14/79 13:30:55',
                value: 'M/d/yy HH:mm:ss'
            },
            {
                display: '3/14/79 1:30',
                value: 'M/d/yy hh:mm'
            }
        ];

        // functions
        scope.formatDataValues.updateFormatDimensions = updateFormatDimensions;
        scope.formatDataValues.updateFormatType = updateFormatType;
        scope.formatDataValues.editFormat = editFormat;
        scope.formatDataValues.removeFormat = removeFormat;
        scope.formatDataValues.resetFormat = resetFormat;
        scope.formatDataValues.executeFormat = executeFormat;

        /** Logic **/
        /**
         * @name resetPanel
         * @desc function that is resets the panel when the data changes
         */
        function resetPanel(): void {
            const active = scope.widgetCtrl.getWidget('active'),
                sharedTools = scope.widgetCtrl.getWidget('view.' + active + '.tools.shared');

            scope.formatDataValues.formats = sharedTools.formatDataValues.formats ? sharedTools.formatDataValues.formats : [];

            //get new list of dimensions in case keys have changed
            setFormatDimensions();
        }

        /**
         * @name setFormatDimensions
         * @desc set the list of dimensions available to format
         */
        function setFormatDimensions(): void {
            const tasks = scope.widgetCtrl.getWidget('view.visualization.tasks');

            //pull keys from task data to account for multiple layers
            let mapping: any = {},
                options: any = [];
            for (let taskIdx = 0, taskLen = tasks.length; taskIdx < taskLen; taskIdx++) {
                const layout = tasks[taskIdx].layout,
                    keys = tasks[taskIdx].keys[layout];

                for (let keyIdx = 0, keyLen = keys.length; keyIdx < keyLen; keyIdx++) {
                    const key = keys[keyIdx];

                    //only enable formatting for certain value dimensions depending on layout
                    if (isFormatAllowed(layout, key.model) && key.model !== 'facet') {
                        const option = {
                            alias: key.alias,
                            selector: key.header,
                            derived: key.derived,
                            math: key.math,
                            calculatedBy: key.calculatedBy,
                            groupBy: key.groupBy,
                            type: key.type,
                            dataType: key.dataType,
                            model: key.model,
                            additionalDataType: key.hasOwnProperty("additionalDataType") ? key.additionalDataType : ''
                        };

                        // add if it doesn't exist
                        if (!mapping.hasOwnProperty(option.alias)) {
                            options.push(option);

                            mapping[option.alias] = true;
                        }
                    }
                }
            }

            options.sort(function (a, b) {
                let lowerA = a.alias.toLowerCase(),
                    lowerB = b.alias.toLowerCase();

                if (lowerA > lowerB) {
                    return 1;
                } else if (lowerA === lowerB) {
                    return 0;
                }

                return -1;
            });


            // set the dimensions
            scope.formatDataValues.dimensions.list = JSON.parse(JSON.stringify(options));

            //populate dimension field so not empty when opening widget
            if (scope.formatDataValues.dimensions.list.length > 0) {
                scope.formatDataValues.dimensions.selected = JSON.parse(JSON.stringify(scope.formatDataValues.dimensions.list[0]));
                updateFormatDimensions();
            }
        }

        /**
         * @name updateFormatDimensions
         * @desc reset scope to new selected dimension
         * @returns {void}
         */
        function updateFormatDimensions(): void {
            //dont run if dimension not yet selected
            if (scope.formatDataValues.dimensions.selected.hasOwnProperty('alias')) {
                //if formats have been applied
                for (let format in scope.formatDataValues.formats) {
                    if (scope.formatDataValues.formats[format].dimension === scope.formatDataValues.dimensions.selected.alias) {
                        //if selected dimension has applied formats, populate widget with selected rules (Viz Level)
                        scope.formatDataValues.format = scope.formatDataValues.formats[format];
                        //refresh which options are disabled
                        updateFormatType();
                        return;
                    }
                }
                //if no formats applied to selected dimension, populate widget with rules based on data type (Frame Level)
                scope.formatDataValues.format = visualizationUniversal.mapFormatOpts(scope.formatDataValues.dimensions.selected);
                //refresh which options are disabled
                updateFormatType();
            }
        }

        /**
         * @name updateFormatType
         * @desc if a certain format contracts other format selections, disabled contracting dropdowns 
         */
        function updateFormatType(): void {
            if (scope.formatDataValues.format.type === 'Million' || scope.formatDataValues.format.type === 'Billion' || scope.formatDataValues.format.type === 'Trillion' || scope.formatDataValues.format.type === 'Scientific') {
                scope.formatDataValues.format.delimiter = 'Default';
            } else if (scope.formatDataValues.format.type === 'Accounting') {
                scope.formatDataValues.format.delimiter = ',';
            }
        }

        /**
         * @name editFormat
         * @desc edit the selected format from applied formats
         * @param idx - selected format
         */
        function editFormat(dimension: string): void {
            let dim;

            for (dim = 0; dim < scope.formatDataValues.dimensions.list.length; dim++) {
                if (scope.formatDataValues.dimensions.list[dim].alias === dimension) {
                    scope.formatDataValues.dimensions.selected = scope.formatDataValues.dimensions.list[dim];
                }
            }
            
            updateFormatDimensions();
        }

        /**
         * @name removeFormat
         * @desc remove the selected format from applied formats
         * @param idx - selected format
         */
        function removeFormat(idx: number): void {
            scope.formatDataValues.formats.splice(idx, 1);
            executeFormat(false);
        }

        /**
         * @name resetFormat
         * @desc reset scope to default state for new dimension selection
         */
        function resetFormat(): void {
            scope.formatDataValues.format = visualizationUniversal.mapFormatOpts(scope.formatDataValues.dimensions.selected);
        }

        /**
         * @name getApplied
         * @desc gets a string of applied formats for display in "Applied Formats" field
         * @returns {void}
         */
        function getApplied(): string {
            let applied = '';

            const format = scope.formatDataValues.format;
            if (format) {
                let firstFormat = true;

                applied += '(';
                if (format.prepend !== '') {
                    applied = applied + '\'' + format.prepend + '\' ';
                    firstFormat = false;
                }

                if (format.append !== '') {
                    if (firstFormat) {
                        applied = applied + '\'' + format.append + '\' ';
                        firstFormat = false;
                    } else {
                        applied = applied + ', ' + '\'' + format.append + '\' ';
                    }
                }
                if (format.round !== 2) {
                    var roundString = '0.'
                    for (let i = 0; i < format.round; i++) {
                        roundString = roundString + '0';
                    }
                    if (firstFormat) {
                        applied = applied + '\'' + roundString + '\' ';
                        firstFormat = false;
                    } else {
                        applied = applied + ', ' + '\'' + roundString + '\' ';
                    }
                }
                if (format.type !== 'Default') {
                    if (firstFormat) {
                        applied = applied + '\'' + format.type + '\' ';
                        firstFormat = false;
                    } else {
                        applied = applied + ', ' + '\'' + format.type + '\' ';
                    }
                }
                if (format.date !== 'Default') {
                    if (firstFormat) {
                        applied = applied + '\'' + format.date + '\' ';
                        firstFormat = false;
                    } else {
                        applied = applied + ', ' + '\'' + format.date + '\' ';
                    }
                }
                if (format.delimiter !== 'Default') {
                    let delimiterName = '';
                    if (format.delimiter === ',') {
                        delimiterName = 'Comma';
                    } else if (format.delimiter === '.') {
                        delimiterName = 'Period';
                    }

                    if (firstFormat) {
                        applied = applied + '\'' + delimiterName + '\' ';
                        firstFormat = false;
                    } else {
                        applied = applied + ', ' + '\'' + delimiterName + '\' ';
                    }
                }
                if (firstFormat) {
                    applied = applied + 'Default';
                }
                applied = applied + ')';
            }

            return applied;
        }

        /**
         * @name executeFormat
         * @desc executes the pixel with updated tools settings
         * @param apply - apply the new filter
         */
        function executeFormat(apply: boolean): void {
            if (scope.formatDataValues.format.hasOwnProperty('date') && scope.formatDataValues.format.date !== 'Default' && scope.formatDataValues.dimensions.selected.dataType !== 'DATE' && scope.formatDataValues.dimensions.selected.dataType !== 'TIMESTAMP') {
                scope.widgetCtrl.alert('error', 'Date Formats cannot be applied to non-date type fields.');
                return;
            }

            //if new format rule created
            if (apply) {
                //add format selection and string of applied formats to array of formats 
                const format = {
                    dimension: scope.formatDataValues.dimensions.selected.alias,
                    dimensionType: scope.formatDataValues.dimensions.selected.dataType,
                    model: scope.formatDataValues.dimensions.selected.model,
                    type: scope.formatDataValues.format.type || 'Default',
                    date: scope.formatDataValues.format.date || 'Default',
                    delimiter: scope.formatDataValues.format.delimiter || 'Default',
                    prepend: scope.formatDataValues.format.prepend || '',
                    append: scope.formatDataValues.format.append || '',
                    round: scope.formatDataValues.format.round,
                    appliedString: getApplied() || '(Default)',
                    layout: scope.widgetCtrl.getWidget('view.visualization.layout')
                };

                //if new format rules are applied to the same dimension, remove old rules
                for (let formatIdx = scope.formatDataValues.formats.length - 1; formatIdx >= 0; formatIdx--) {
                    if (scope.formatDataValues.formats[formatIdx].dimension === format.dimension) {
                        scope.formatDataValues.formats.splice(formatIdx, 1);
                    }
                }

                scope.formatDataValues.formats.push(format);
            }

            // execute the pixel
            scope.widgetCtrl.execute([
                {
                    'type': 'panel',
                    'components': [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    'type': 'addPanelOrnaments',
                    'components': [{
                        'tools': {
                            'shared': {
                                formatDataValues: scope.formatDataValues.formats.length > 0 ? {
                                    formats: scope.formatDataValues.formats
                                } : false
                            }
                        }
                    }],
                    'terminal': true
                },
                {
                    'type': 'panel',
                    'components': [
                        scope.widgetCtrl.panelId
                    ]
                },
                {
                    'type': 'retrievePanelOrnaments',
                    'components': ['tools'],
                    'terminal': true
                }
            ], () => { }, [scope.widgetCtrl.widgetId]);
        }

        /** Utility */
        /**
         * @name isFormatAllowed
         * @desc checks whether custom formatting should be enabled for the selected dimension based on visualization type
         * @returns formatting allowed
         */
        function isFormatAllowed(layout: string, model: string): boolean {
            //old rules to re-implement if labels should NOT be allowed
            /*if (layout === 'HeatMap') {
                if (model !== 'x' && model !== 'y') {
                    return true;
                }
            } else if (layout === 'Waterfall') {
                if (model !== 'label' && model !== 'end') {
                    return true;
                } 
            } else if (layout === 'GanttD3') {
                if (model !== 'task') {
                    return true;
                } 
            } else if (layout === 'TreeMap') {
                if (model !== 'series') {
                    return true;
                }
            } else if (layout === 'Cloud') { 
                if (model !== 'label') {
                    return true;
                }
            } else if (layout === 'Bullet') {
                if (model !== 'targetValue' && model !== 'badMarker' && model !== 'satisfactoryMarker' && model !== 'excellentMarker') {
                    return true;
                }
            } else if (layout === 'GanttD3') {
                //all gannt date dimensions should be formatted the same
                if (model !== 'task' && model !== 'end' && model !== 'milestone') {
                    return true;
                } 
            } else {
                if (model !== 'label') {
                    return true;
                }
            }*/
            if (layout === 'Waterfall') {
                if (model !== 'end') {
                    return true;
                }
            } else if (layout === 'GanttD3') {
                //TODO unsure how gantt d3 configures axis labels
                if (model !== 'progress' && model !== 'task') {
                    return true;
                }
            } else if (layout === 'Cloud') {
                if (model !== 'label') {
                    return true;
                }
            } else {
                return true;
            }

            return false;
        }

        /**
         * @name initialize
         * @desc function that is called on directive load
         */
        function initialize(): void {
            // listeners
            const formatDataValuesUpdateFrameListener = scope.widgetCtrl.on('update-frame', resetPanel),
                formatDataValuesUpdateTaskListener = scope.widgetCtrl.on('update-task', resetPanel),
                updateOrnamentsListener = scope.widgetCtrl.on('update-ornaments', resetPanel),
                addDataListener = scope.widgetCtrl.on('added-data', resetPanel);

            // cleanup
            scope.$on('$destroy', function () {
                formatDataValuesUpdateFrameListener();
                formatDataValuesUpdateTaskListener();
                updateOrnamentsListener();
                addDataListener();
            });

            resetPanel();
        }

        initialize();
    }
}
