'use strict';

import '@/widget-resources/css/d3-charts.css';
import './dendrogram-standard.scss';

import * as d3 from 'd3';

/* *
 * @name dendrogram
 * @desc Dendrogram directive for flattening and visualizing a dendrogram.
 *       Adatped from: https://bl.ocks.org/d3noob/43a860bc0024792f8803bba8ca0d5ecd
 */

export default angular.module('app.dendrogram-standard.directive', [])
    .directive('dendrogramStandard', dendrogramStandard);

dendrogramStandard.$inject = ['semossCoreService'];

function dendrogramStandard(semossCoreService) {
    dendrogramStandardLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^widget'],
        link: dendrogramStandardLink
    };

    function dendrogramStandardLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];

        /* * **************Get Chart Div *************************/
        scope.chartDiv = d3.select(ele[0].firstElementChild);

        /* * **************Main Event Listeners*************************/
        // var resizeListener = scope.widgetCtrl.on('resize-widget', resizeViz);
        var updateListener = scope.widgetCtrl.on('update-ornaments', initialize),
            toolListener = scope.widgetCtrl.on('update-tool', toolUpdateProcessor),
            updateTaskListener = scope.widgetCtrl.on('update-task', initialize),
            addDataListener = scope.widgetCtrl.on('added-data', initialize),
            resizeListener = scope.widgetCtrl.on('resize-widget', initialize),
            width, height, viz,
            svg,
            margin = {
                top: 20,
                right: 120,
                bottom: 20,
                left: 120
            },
            nodeWidthSpace = 20,
            nodeHeightSpace = 50,
            graph,
            globalTransform = 1, // Need to set to 1 first since user has not zoomed in or out
            globalHeightCount,
            heightArray = [],
            originalHeightArray = [],
            previousTransform = 1,
            increment = 1;

        function toolUpdateProcessor(toolUpdateConfig) {
            // need to invoke tool functions
            scope[toolUpdateConfig.fn](toolUpdateConfig.args);
        }

        initialize();

        // Widget Functions
        function initialize() {
            var layerIndex = 0,
                individiualTools = scope.widgetCtrl.getWidget('view.visualization.tools.individual.Dendrogram') || {},
                sharedTools = scope.widgetCtrl.getWidget('view.visualization.tools.shared'),
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.Dendrogram'),
                taskData = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.data'),
                localChartData = {},
                data;

            if (!taskData.children) {
                localChartData = {
                    chartData: semossCoreService.visualization.getTableData(taskData.headers, taskData.values, taskData.rawHeaders),
                    dataTableAlign: semossCoreService.visualization.getDataTableAlign(keys),
                    callbacks: scope.widgetCtrl.getEventCallbacks(),
                    uiOptions: angular.extend(sharedTools, individiualTools)
                };
            } else {
                localChartData.chartData = taskData;
            }

            data = JSON.parse(JSON.stringify(localChartData.chartData));

            data.uiOptions = localChartData.uiOptions;
            data.dataTableAlign = localChartData.dataTableAlign;

            if (!(data === undefined || data === null || data === {} || data === '')) {
                scope.chartDiv.selectAll('*').remove();
                var rootData;
                if (!data.specificData || _.isEmpty(data.specificData)) {
                    if (data.children) {
                        var list = [],
                            allHash = {};
                        makeTree(data.children, list);

                        if (list.length > 1) {
                            allHash.name = 'root';
                            allHash.children = list;
                        } else if (list.length === 1) {
                            allHash = list[0];
                        } else {
                            allHash = {};
                        }
                        allHash.stats = data.stats;
                        rootData = allHash;
                    } else {
                        var headers = [];
                        if (data.dataTableAlign) {
                            for (var header in data.dataTableAlign) {
                                headers.push(data.dataTableAlign[header]);
                            }
                        } else {
                            for (var header in data.headers) {
                                headers.push(data.headers[header].title);
                            }
                        }

                        if (sharedTools.showHierarchy) {
                            rootData = processHierarchy(JSON.parse(JSON.stringify(data.viewData)), data.dataTableAlign, sharedTools.showHierarchyByUpstream);
                        } else {
                            rootData = processTableData(JSON.parse(JSON.stringify(data.viewData)), headers);
                        }
                    }
                } else {
                    rootData = JSON.parse(JSON.stringify(data.specificData));
                }
                setConfig(rootData);
                update(rootData, sharedTools);

                if (_.isEmpty(rootData)) {
                    // remove all nodes if no data
                    svg.selectAll('*').remove();
                }
            } else {
                console.log('no data');
            }
        }

        function processHierarchy(data, headers, byUpstream) {
            if (byUpstream) {
                Array.prototype.diff = function (a) {
                    return this.filter(function (i) {
                        return a.indexOf(i) < 0;
                    });
                };

                var upstreamArr = [];
                var downstreamArr = [];
                data.forEach(function (a) {
                    upstreamArr.push(a.Upstream);
                    downstreamArr.push(a.Downstream);
                }, {});

                var root = upstreamArr.diff(downstreamArr);
                root = root[0];

                for (var i = 0; i < data.length; i++) {
                    if (data[i][headers.dimension] === root) {
                        var a = data.splice(i, 1); // removes the item
                        data.unshift(a[0]); // adds it back to the beginning
                        break;
                    }
                }

                var tree = {},
                    flat = {};

                var addItemToTree = function (item) {
                    var parent = flat[item[headers.dimension]],
                        child = flat[item[headers['dimension 1']]];
                    if (!parent) {
                        parent = flat[item[headers.dimension]] = {
                            name: item[headers.dimension]
                        };
                        if (!Object.keys(tree).length) tree = parent;
                    }
                    if (!child) {
                        child = flat[item[headers['dimension 1']]] = {
                            name: item[headers['dimension 1']]
                        };
                    }
                    if (!parent.children) {
                        parent.children = [];
                    }
                    parent.children.push(child);
                };
                data.forEach(addItemToTree);
                console.log(tree);

                return tree;
            }
            // console.log(JSON.stringify(data));
            data.sort(function (a, b) {
                var result;
                a = a[headers.dimension].split('.');
                b = b[headers.dimension].split('.');
                while (a.length) {
                    if (result = a.shift() - (b.shift() || 0)) {
                        return result;
                    }
                }
                return -b.length;
            });

            // After sort, get first element and split on '.' to retrieve rootNumber
            let rootNumber = data[0][headers.dimension].split('.');


            if (rootNumber.length > 1) {
                // This functions under the assumption that the root is being defined as 1.0
                // kinda specific and messy
                data[0][headers.dimension] = rootNumber[0];
            }

            var result = [];

            data.forEach(function (a) {
                var parent = a[headers.dimension].split('.').slice(0, -1).join('.');

                this[a[headers.dimension]] = {
                    id: a[headers.dimension],
                    name: a[headers['dimension 1']]
                };
                if (parent) {
                    this[parent] = this[parent] || {};
                    this[parent].children = this[parent].children || [];
                    this[parent].children.push(this[a[headers.dimension]]);
                } else {
                    result.push(this[a[headers.dimension]]);
                }
            }, {});

            return result[0];
        }

        // pass in all of the tabular data and the selected levels (dataTableAlign); we will set up the tree data according to the order of the data table align
        function processTableData(data, tableHeaders) {
            var allHash = {},
                list = [],
                rootMap = {},
                currentMap = {};

            Object.keys(data).forEach(function (i) { // all of this is to change it to a tree structure and then call makeTree to structure the data appropriately for this viz
                var count = 0;

                tableHeaders.forEach(function (header) {
                    if (!data[i][header] && data[i][header] !== 0) {
                        data[i][header.replace(/[_]/g, ' ')] = '"null"';
                    }

                    var currentValue = data[i][header.replace(/[_]/g, ' ')].toString().replace(/["]/g, ''),
                        nextMap = {};

                    if (count === 0) { // will take care of the first level and put into rootmap if it doesnt already exist in rootmap
                        currentMap = rootMap[currentValue];

                        if (!currentMap) {
                            currentMap = {};
                            rootMap[currentValue] = currentMap;
                        }

                        nextMap = currentMap;
                        count++;
                    } else {
                        nextMap = currentMap[currentValue];

                        if (!nextMap) {
                            nextMap = {};
                            currentMap[currentValue] = nextMap;
                        }

                        currentMap = nextMap;
                    }
                });
            });

            makeTree(rootMap, list);

            if (list.length > 1) {
                allHash.name = 'root';
                allHash.children = list;
            } else if (list.length === 1) {
                allHash = list[0];
            } else {
                allHash = {};
            }

            return allHash;
        }

        function makeTree(map, list) {
            var keyset = Object.keys(map),
                childSet = [];

            for (var key in keyset) {
                var childMap = map[keyset[key]],
                    dataMap = {};

                dataMap.name = keyset[key];

                if (_.isEmpty(childMap)) {
                    list.push(dataMap);
                } else {
                    dataMap.children = childSet;
                    list.push(dataMap);

                    makeTree(childMap, childSet);
                    childSet = [];
                }
            }
        }

        function zoom() {
            graph.attr('transform', d3.event.transform);
            var inverseTransform = 1 / d3.event.transform.k;
            globalTransform = inverseTransform;

            // Only apply changes if an actual zoom was triggered
            if (previousTransform !== globalTransform) {
                scope.chartDiv.selectAll('.dendrogram').attr('r', 10 * inverseTransform); // node width adjust
                graph.selectAll('text').attr('transform', 'scale(' + inverseTransform + ')');
                scope.chartDiv.selectAll('path.dendrogram-link').style('stroke-width', inverseTransform);
                scope.chartDiv.selectAll('.dendrogram').style('stroke-width', 3 * inverseTransform);
                scope.chartDiv.selectAll('legend.text').attr('transform', 'scale(1)');

                if (globalTransform > previousTransform) {
                    calcStateofTextPerLevel(-1, -1);
                } else {
                    calcStateofTextPerLevel(1, -1);
                }
            }

            previousTransform = globalTransform;
        }

        function setConfig(rootData) {
            var countHeight = 0,
                heightCount, widthCount, dimensions, clientHeight, clientWidth,
                tempMargin = {},
                countWidth = 1; // Every Dendrogram has at least one level

            heightArray = []; // reset
            originalHeightArray = []; // reset

            // The treeHeightCount function returns the number of instances in the last level of the tree
            function treeHeightCount(dataset) { // recursive funciton
                if (dataset.children) {
                    for (var i = 0; i < dataset.children.length; i++) {
                        treeHeightCount(dataset.children[i]);
                    }
                } else {
                    countHeight++;
                }
                return countHeight;
            }

            // The treeWidthCount function returns the number of levels in the tree
            function treeWidthCount(dataset) { // recursive function
                if (dataset.children) {
                    countWidth++; // If there exists a level below the current level, add 1 to the countWidth (number of levels in Dendrogram).
                    for (var i = 0; i < dataset.children.length; i++) {
                        treeWidthCount(dataset.children[i]);
                        return countWidth;
                    }
                }
            }

            /* *
             * @description recursive function that loops through a dendrogram structure
             *              and creates a heightArray where each index = level and
             *              each value at that index is the amount of objects in that level -
             *              this can later be used against id searches
             * @param {Array} childArray
             */
            function createHeightArray(childArray) {
                var tempHeight = 0,
                    returnArray = [],
                    tempObj = {},
                    data;

                for (data in childArray) {
                    tempHeight++;

                    if (childArray[data].children) {
                        for (var val in childArray[data].children) {
                            returnArray.push(childArray[data].children[val]);
                        }
                    }
                }

                tempObj.height = tempHeight;
                tempObj.visible = true;

                heightArray.push(tempObj); // Update global heightArray

                if (!_.isEmpty(returnArray)) {
                    createHeightArray(returnArray);
                }
            }

            createHeightArray(rootData.children);

            angular.copy(heightArray, originalHeightArray);

            heightCount = treeHeightCount(rootData);
            widthCount = treeWidthCount(rootData);
            dimensions = scope.chartDiv.node().getBoundingClientRect();
            width = widthCount * nodeWidthSpace;
            height = heightCount * nodeHeightSpace;
            width = width < dimensions.width ? width : dimensions.width;
            height = height < dimensions.height ? height : dimensions.height;

            clientHeight = ele[0].childNodes[0].clientHeight;
            clientWidth = ele[0].childNodes[0].clientWidth;

            tempMargin.top = 20;
            tempMargin.right = 120;
            tempMargin.bottom = 20;
            tempMargin.left = 120;

            // Center Dendrogram if smaller than window dimensions
            if (clientHeight !== 0 && clientWidth !== 0) {
                if (height > clientHeight) {
                    tempMargin.top = 20;
                } else {
                    tempMargin.top = (clientHeight - height) / 2;
                }
                if (width > clientWidth) {
                    tempMargin.left = 120;
                } else {
                    tempMargin.left = (clientWidth - width) / 2.35;
                }
            }
            // create the svg
            viz = scope.chartDiv.append('svg')
                .attr('class', 'dendrogram-svg-full-size')
                .call(
                    d3.zoom()
                        .scaleExtent([0.05, 10])
                        .on('zoom', zoom)
                )
                .on('dblclick.zoom', null);

            // This is what gets resized on zoom
            graph = viz.append('g')
                .attr('class', 'graph');

            // Set initial positioning of the dendrogram
            // Center of window
            svg = graph.append('g')
                .attr('transform', 'translate(' + tempMargin.left + ',' + tempMargin.top + ')');
            // OR
            // Top Left of window
            // svg = graph.append('g')
            //     .attr('align', 'translate(' + margin.left + ',' + margin.top + ')');

            // clear the elements inside of the directive
            svg.selectAll('*').remove();

            globalHeightCount = heightCount;
        }


        /* *
         * @description Given a zoom level, traverses our dendrogram and decides whether to Hide
         *              the text in each level or not
         * @param {int} zoom - 0 is used for either transition or click events, -1 for zoom out, 1 for zoom in
         * @param {int} changeInLevel - only relavent for zoom = 0; tells our method which level to beging traversing down from
         */
        function calcStateofTextPerLevel(zoom, changeInLevel) {
            var bbox = graph._groups[0][0].getBoundingClientRect(),
                visibleNodesInLevel = 0,
                totalNodesInLevel = 0,
                oldIncrement = 0,
                level = 0,
                i;

            increment = 1; // reset

            if (zoom === 0) {
                increment = 2;

                for (i = 0; i < changeInLevel; i++) {
                    increment += heightArray[i].height;
                }

                level = changeInLevel;

                // Go through each level
                for (level; level < heightArray.length; level++) {
                    visibleNodesInLevel = heightArray[level].height + increment;
                    totalNodesInLevel = originalHeightArray[level].height + increment;
                    oldIncrement = increment;

                    // Arbitrary cut-off, each node is of size 20 so they will touch
                    // 5 pixels within each other if cutoff is 15
                    if ((15 * (visibleNodesInLevel - oldIncrement)) > bbox.height) {
                        scope.chartDiv.selectAll('text').filter(function (d) {
                            return (d.id < (totalNodesInLevel + oldIncrement)) && (d.id >= oldIncrement);
                        })
                            .attr('opacity', 0);

                        increment = oldIncrement;
                        heightArray[level].visible = false;
                    } else {
                        scope.chartDiv.selectAll('text').filter(function (d) {
                            return (d.id < (totalNodesInLevel + oldIncrement)) && (d.id >= oldIncrement);
                        })
                            .attr('opacity', 1);

                        increment = oldIncrement;
                        heightArray[level].visible = true;
                    }

                    // We need to add 1 in the first case, otherwise increment is visibleNodesInLevel
                    if (oldIncrement === 1) {
                        increment = totalNodesInLevel + 1;
                    } else {
                        increment = totalNodesInLevel;
                    }
                }

                increment = 0; // reset
            } else if (zoom < 0) {
                for (level; level < heightArray.length; level++) {
                    if (!heightArray[level].visible) {
                        break;
                    }

                    visibleNodesInLevel = heightArray[level].height + increment;
                    totalNodesInLevel = originalHeightArray[level].height + increment;
                    oldIncrement = increment;

                    // Arbitrary cut-off, each node is of size 20 so they will touch
                    // 5 pixels within each other if cutoff is 15
                    if ((15 * (visibleNodesInLevel - oldIncrement)) > bbox.height) {
                        // Hide all of the excess text
                        scope.chartDiv.selectAll('text').filter(function (d) {
                            if (oldIncrement === 1) {
                                return (d.id <= totalNodesInLevel) && (d.id > oldIncrement);
                            }
                            return (d.id <= totalNodesInLevel) && (d.id >= oldIncrement);
                        })
                            .attr('opacity', 0);

                        increment = oldIncrement;
                        heightArray[level].visible = false;
                    } else {
                        // Show text that is not in excess
                        scope.chartDiv.selectAll('text').filter(function (d) {
                            return (d.id < totalNodesInLevel) && (d.id >= oldIncrement);
                        })
                            .attr('opacity', 1);

                        increment = oldIncrement;
                        heightArray[level].visible = true;
                    }

                    // We need to add 1 in the first case, otherwise increment is visibleNodesInLevel
                    if (oldIncrement === 1) {
                        increment = totalNodesInLevel + 1;
                    } else {
                        increment = totalNodesInLevel;
                    }
                }

                increment = 0; // reset
            } else {
                for (i in heightArray) {
                    if (!heightArray[i].visible) {
                        level = i;
                        break;
                    }
                }

                increment = 2;

                for (i = 0; i < level; i++) {
                    increment += heightArray[i].height;
                }

                // Go through each level
                for (level; level < heightArray.length; level++) {
                    visibleNodesInLevel = heightArray[level].height + increment;
                    totalNodesInLevel = originalHeightArray[level].height + increment;
                    oldIncrement = increment;

                    // Arbitrary cut-off, each node is of size 20 so they will touch
                    // 5 pixels within each other if cutoff is 15
                    if ((15 * (visibleNodesInLevel - oldIncrement)) > bbox.height) {
                        // Hide all of the excess text
                        scope.chartDiv.selectAll('text').filter(function (d) {
                            return (d.id < (totalNodesInLevel)) && (d.id >= oldIncrement);
                        })
                            .attr('opacity', 0);

                        increment = oldIncrement;
                        heightArray[level].visible = false;
                    } else {
                        // Show text that is not in excess
                        scope.chartDiv.selectAll('text').filter(function (d) {
                            return (d.id < (totalNodesInLevel)) && (d.id >= oldIncrement);
                        })
                            .attr('opacity', 1);

                        increment = oldIncrement;
                        heightArray[level].visible = true;
                    }

                    // We need to add 1 in the first case, otherwise increment is visibleNodesInLevel
                    if (oldIncrement === 1) {
                        increment = totalNodesInLevel + 1;
                    } else {
                        increment = totalNodesInLevel;
                    }
                }

                increment = 0; // reset
            }
        }

        /* *
         * @desc Takes in an id and an amount and adds that mount to the height
         *       of the current level
         * @param {int} id - id of the text
         * @param {String} amount - height change for this level
         */
        function updateLevelCount(depth, amount) {
            heightArray[depth].height = heightArray[depth].height + amount;
        }

        /* *
         * @description Initializes our dendrogram
         * @param {Object} rootData - data used to build our dendrogram
         * @oaram {Object} sharedTools - shared tools
         */
        function update(rootData, sharedTools) {
            var i = 0,
                duration = 750,
                root,
                treemap;

            previousTransform = globalTransform = 1; // reset transform vals

            // declares a tree layout and assigns the size
            treemap = d3.tree().size([height, width]);

            // Assigns parent, children, height, depth
            root = d3.hierarchy(rootData, function (d) {
                return d.children;
            });

            root.x0 = height / 2;
            root.y0 = 0;

            updateviz(root, false);
            updateSizingElements();

            /* * ***************************TOOL*FUNCTIONS***************************/
            // scope.expandOrCollapseAll = function (choice) {
            //     if (choice == 'expand') {
            //         expandAll();
            //         updateSizingElements();
            //         calcStateofTextPerLevel(0, 0);
            //     } else {
            //         collapseAll();
            //         updateSizingElements();
            //     }
            // };

            if (sharedTools.collapseAll) {
                collapseAll();
                updateSizingElements();
            } else {
                expandAll();
                updateSizingElements();
                calcStateofTextPerLevel(0, 0);
            }

            // Single Collapse of a node
            function collapse(d) {
                if (d.children) {
                    d._children = d.children;
                    d._children.forEach(collapse);
                    d.children = null;
                }
            }

            // Single Expansion of a node
            function expand(d) {
                if (d._children) {
                    d.children = d._children;
                    d.children.forEach(expand);
                    d._children = null;
                }
            }

            // Expand all nodes
            function expandAll() {
                expand(root);
                if (root.children) {
                    root.children.forEach(expand);
                }
                updateviz(root, false);
            }

            // Collapse all nodes
            function collapseAll() {
                try {
                    root.children.forEach(collapse);
                    collapse(root);
                    updateviz(root, false);
                } catch (e) {
                    console.warn('Cannot collapse all when already collapsed');
                }
            }

            /* * ***********************END*TOOL*FUNCTIONS***************************/

            function updateviz(source) {
                // Assigns the x and y position for the nodes
                var treeData = treemap(root),
                    nodes, links, node, nodeEnter, nodeUpdate,
                    nodeExit, link, transitions, linkEnter, linkUpdate, linkExit,
                    legend, legendEnter,
                    tree = scope.widgetCtrl.getWidget('view.visualization.layout'),
                    colorBy = scope.widgetCtrl.getWidget('view.visualization.colorByValue'),
                    hasRoot = false;

                // Compute the new tree layout.
                nodes = treeData.descendants();
                links = treeData.descendants().slice(1);

                // Normalize for fixed-depth.
                nodes.forEach(function (d) {
                    d.y = d.depth * 180;
                });

                //*  ***************** Nodes section ***************************

                // Update the nodes...
                node = svg.selectAll('g.dendrogram')
                    .data(nodes, function (d) {
                        return d.id || (d.id = ++i);
                    });

                // Enter any new modes at the parent's previous position.
                nodeEnter = node.enter().append('g')
                    .attr('class', 'dendrogram')
                    .attr('transform', function () {
                        return 'translate(' + source.y0 + ',' + source.x0 + ')';
                    })
                    .on('click', click);

                // Add Circle for the nodes
                nodeEnter.append('circle')
                    .attr('class', 'dendrogram')
                    .attr('r', 1e-6)
                    .style('stroke', function (d) {
                        var taskData, colorValue, chartData, layerIndex = 0;
                        taskData = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.data');

                        if (!taskData.children) {
                            chartData = semossCoreService.visualization.getTableData(taskData.headers, taskData.values, taskData.rawHeaders);
                        } else {
                            chartData = taskData;
                        }

                        if (colorBy) {
                            if (d.data.name === 'root' && d.depth === 0) {
                                hasRoot = true;
                            }
                            colorValue = getColorValue(colorBy, d, chartData, hasRoot);
                            if (colorValue) {
                                return colorValue;
                            }
                        }

                        return 'steelblue';
                    })
                    .style('fill', function (d) {
                        return d._children ? 'lightsteelblue' : '#fff';
                    });

                // Add labels for the nodes
                nodeEnter.append('text')
                    .attr('dy', '.35em')
                    .attr('x', function (d) {
                        return d.children || d._children ? -13 : 13;
                    })
                    .attr('text-anchor', function (d) {
                        return d.children || d._children ? 'end' : 'start';
                    })

                    .text(function (d) {
                        // any longer and label runs into other nodes
                        if (d.data.name && d.data.name.length >= 27) {
                            return d.data.name.substring(0, 24) + '...';
                        }

                        return d.data.name;
                    });

                // UPDATE
                nodeUpdate = nodeEnter.merge(node);

                // Transition to the proper position for the node
                nodeUpdate.transition()
                    .duration(duration)
                    .attr('transform', function (d) {
                        return 'translate(' + d.y + ',' + d.x + ')';
                    });

                // Update the node attributes and style
                nodeUpdate.select('circle.dendrogram')
                    .attr('r', 10)
                    .style('stroke', function (d) {
                        var taskData, colorValue, chartData, layerIndex = 0;
                        taskData = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.data');

                        if (!taskData.children) {
                            chartData = semossCoreService.visualization.getTableData(taskData.headers, taskData.values, taskData.rawHeaders);
                        } else {
                            chartData = taskData;
                        }

                        if (colorBy) {
                            if (d.data.name === 'root' && d.depth === 0) {
                                hasRoot = true;
                            }
                            colorValue = getColorValue(colorBy, d, chartData, hasRoot);
                            if (colorValue) {
                                return colorValue;
                            }
                        }

                        return 'steelblue';
                    })
                    .style('fill', function (d) {
                        return d._children ? 'lightsteelblue' : '#fff';
                    })
                    .attr('cursor', 'pointer');


                // Remove any exiting nodes
                nodeExit = node.exit().transition()
                    .duration(duration)
                    .attr('transform', function () {
                        return 'translate(' + source.y + ',' + source.x + ')';
                    })
                    .remove();

                // On exit reduce the node circles size to 0
                nodeExit.select('circle')
                    .attr('r', 1e-6);

                // On exit reduce the opacity of text labels
                nodeExit.select('text')
                    .style('fill-opacity', 1e-6);

                //*  ***************** links section ***************************

                // Update the links...
                link = svg.selectAll('path.dendrogram-link')
                    .data(links, function (d) {
                        return d.id;
                    });

                // Enter any new links at the parent's previous position.
                linkEnter = link.enter().insert('path', 'g')
                    .attr('class', 'dendrogram-link')
                    .attr('d', function () {
                        var o = {
                            x: source.x0,
                            y: source.y0
                        };
                        return diagonal(o, o);
                    });

                // UPDATE
                linkUpdate = linkEnter.merge(link);
                transitions = 0;

                // Transition back to the parent element position
                linkUpdate.transition()
                    .duration(duration)
                    .attr('d', function (d) {
                        return diagonal(d, d.parent);
                    })
                    .on('end', function () {
                        if (transitions === globalHeightCount - 1) {
                            calcStateofTextPerLevel(0, 0);
                        }
                        transitions++;
                    });

                // Remove any exiting links
                linkExit = link.exit().transition()
                    .duration(duration)
                    .attr('d', function () {
                        var o = {
                            x: source.x,
                            y: source.y
                        };
                        return diagonal(o, o);
                    })
                    .remove();

                // Store the old positions for transition.
                nodes.forEach(function (d) {
                    d.x0 = d.x;
                    d.y0 = d.y;
                });

                // Creates a curved (diagonal) path from parent to the child nodes
                function diagonal(s, d) {
                    return `M ${s.y} ${s.x}
                            C ${(s.y + d.y) / 2} ${s.x},
                            ${(s.y + d.y) / 2} ${d.x},
                            ${d.y} ${d.x}`;
                }

                // draw special Legend
                if (rootData.stats) {
                    legend = viz.selectAll('.legend')
                        .data(rootData.stats);

                    // enter legend g
                    legendEnter = legend.enter().append('g')
                        .attr('class', 'legend');

                    legendEnter.append('text')
                        .attr('x', function () {
                            return (margin.right / 2 + 1);
                        })
                        .attr('y', function (d, idx) {
                            return (margin.top + 15 * (idx + 1));
                        })
                        .attr('dy', '.25em')
                        .style('text-anchor', 'start')
                        .text(function (d) {
                            var statKey = _.keys(d)[0];
                            return statKey + ': ' + d[statKey];
                        });
                }
            }

            /* *
             * @name updateSizingElements
             * @desc we use this to update the d3 elements that are painted whenever a zoom is not
             *       explicitly called so that they are no painting discrepancies
             * @return {void}
             */
            function updateSizingElements() {
                scope.chartDiv.selectAll('.dendrogram').attr('r', 10 * globalTransform); // node width adjust
                scope.chartDiv.selectAll('text').attr('transform', 'scale(' + globalTransform + ')');
                scope.chartDiv.selectAll('path.dendrogram-link').style('stroke-width', globalTransform);
                scope.chartDiv.selectAll('.dendrogram').style('stroke-width', 3 * globalTransform);
            }

            function updateDown(childArray) {
                var child, depth, amount;
                for (child in childArray.children) {
                    if (childArray.children[child].children) {
                        updateDown(childArray.children[child]);
                    }
                }

                depth = childArray.children[0].depth - 1;
                amount = childArray.children.length * -1;
                updateLevelCount(depth, amount);
            }

            function updateUp(childArray) {
                var child, depth, amount;
                for (child in childArray.children) {
                    if (childArray.children[child].children) {
                        updateDown(childArray.children[child]);
                    }
                }

                depth = childArray.children[0].depth - 1;
                amount = childArray.children.length;
                updateLevelCount(depth, amount);
            }

            // Toggle children on click.
            function click(d) {
                var depth,
                    recalc = false;

                if (d.children) {
                    depth = d.children[0].depth - 1;

                    updateDown(d);
                    recalc = true;

                    d._children = d.children;
                    d.children = null;
                } else {
                    d.children = d._children;

                    if (d.children) {
                        depth = d.children[0].depth - 1;

                        updateUp(d);
                        recalc = true;
                    }
                    d._children = null;
                }
                updateviz(d, true);
                updateSizingElements();

                if (recalc) {
                    calcStateofTextPerLevel(0, depth);
                }
            }
        }
        /**
         * @name getColorValue
         * @param {object} colorByValue Semoss uiOptions.colorByValue
         * @param {object} d current data point
         * @param {object} chartData Semoss chartData
         * @param {boolean} hasRoot true if dendrogram has a root node, will effect indexing for coloring
         * @desc determines if node should be colored using colorByValue
         * @return {string | boolean} color string if node should be colored,
         *                            false otherwise 
         */
        function getColorValue(colorByValue, d, chartData, hasRoot) {
            var i, value, returnColor = false;
            if (!colorByValue) {
                return returnColor;
            }

            colorByValue.forEach(function (rule) {
                var valueToColor;
                if (hasRoot) {
                    d.data[chartData.viewHeaders[d.depth - 1]] = d.data.name;
                } else {
                    d.data[chartData.viewHeaders[d.depth]] = d.data.name;
                }

                value = d.data[rule.colorOn.replace(/_/g, ' ')];
                if (typeof value === 'string') {
                    value = value.replace(/_/g, ' ');
                }

                for (i = 0; i < rule.valuesToColor.length; i++) {
                    valueToColor = rule.valuesToColor[i];

                    if (typeof valueToColor === 'string') {
                        valueToColor = valueToColor.replace(/_/g, ' ');
                    }

                    // looks like dendro stores all values as string,
                    // so lets just typecast our values from BE as string too
                    if (String(valueToColor) === value) {
                        returnColor = rule.color;
                        break;
                    }
                }
            });

            return returnColor;
        }

        scope.$on('$destroy', function () {
            resizeListener();
            toolListener();
            updateListener();
            updateTaskListener();
            addDataListener();
            // clear chart div
            scope.chartDiv.node().innerHTML = '';
        });
    }
}
