'use strict';

import angular from 'angular';

import './dashboard-filter.scss';

import './dashboard-filter-dimensions/dashboard-filter-dimensions.directive.ts';


export default angular.module('app.dashboard-filter.directive', [
    'app.dashboard-filter.dashboard-filter-dimensions'
])
    .directive('dashboardFilter', dashboardFilterDirective);

dashboardFilterDirective.$inject = ['$timeout', 'semossCoreService'];

function dashboardFilterDirective($timeout: ng.ITimeoutService, semossCoreService: SemossCoreService) {
    dashboardFilterLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'EA',
        scope: {},
        require: ['^insight', '^widget'],
        controllerAs: 'dashboardFilter',
        bindToController: {},
        template: require('./dashboard-filter.directive.html'),
        controller: dashboardFilterCtrl,
        link: dashboardFilterLink
    };

    function dashboardFilterCtrl() { }

    function dashboardFilterLink(scope, ele, attrs, ctrl) {
        scope.insightCtrl = ctrl[0];
        scope.widgetCtrl = ctrl[1];

        interface options {
            frame: string, // name of the frame that we are filtering off of
            column: string, // name of the column that we are filtering
            type: 'checklist' | 'dropdown' | 'typeahead' | 'slider' | 'datepicker' | 'button', // type of filter to render
            multiple: boolean, // are allowing the users to select multiple options?
            applied?: false | { // special scenarios where we only apply to certain things, if false or missing, apply to all.
                frames?: string[], // array of frames to apply the filter
                panels?: string[] // array of panels to apply the filter
            },
            auto: boolean // auto run the dashboard (or click),
            dynamic: boolean, // is the filter dynamic
            // specific options
            comparator?: string | string[] | undefined, // comparator the range?
            sensitivity?: undefined | number // interval from the slder
            format?: undefined | string // format for the slider 
            rendered?: undefined | string | string[] // use the set range values
            vertical?: boolean // for the button
        }

        const DEFAULT_OPTIONS: options = {
            frame: '',
            column: '',
            type: 'checklist',
            multiple: true,
            applied: false,
            auto: true,
            dynamic: false,
            comparator: undefined,
            sensitivity: undefined,
            format: 'YYYY-MM-DD',
            rendered: undefined,
            vertical: false
        }

        let searchTimeout: ng.IPromise<void>,
            changeTimeout: ng.IPromise<void>;

        // all of the filter options
        scope.dashboardFilter.header = {}; // set later
        scope.dashboardFilter.options = {}; // set later
        scope.dashboardFilter.html = '<div class="smss-message">No Filter</div>'; // html for the filter

        scope.dashboardFilter.list = {
            search: '',// was it searched
            loading: false, // is it loading
            options: [], // options to view
            selected: [], // original selected values (checked). This can be searched or not.
            rendered: [], // rendered selected values (checked). Changed/updated model.
            renderedSelectAll: false, // what is the rendered selectAll value.
            renderedText: '', // what is the rendered text value
            offset: 0, // current location
            limit: 50, // how many more values to collect
            unrendered: 0, // count of the unrendered selected values
            total: 0, // count of the total number of options available
            canCollect: true, // flag to determine whether call should be made - infinite scroll
            delta: [] // actions performed on the values. This is the 'state
        }

        scope.dashboardFilter.range = {
            type: 'categorical', // is it a categorical slider?
            options: [], // categorical options
            rendered: '', // rendered selected value
            comparator: '', // comparator that we will use
            operator: '', // operator that we will use
            min: undefined, // minimum value
            max: undefined, // minimum value
            sensitivity: undefined, // sensitivity
        }

        scope.dashboardFilter.changeFilter = changeFilter;
        scope.dashboardFilter.applyFilter = applyFilter;
        scope.dashboardFilter.searchList = searchList;
        scope.dashboardFilter.getMoreList = getMoreList;

        /** General */
        /**
          * @name resetFilter
          * @desc reset the filter to the default state
          * @return {void}
          */
        function resetFilter(): void {
            let options: options = scope.widgetCtrl.getWidget('view.dashboard-filter.options') || {};

            // merge with the defaults
            options = angular.merge({}, DEFAULT_OPTIONS, options);

            // update the options
            scope.dashboardFilter.options = options;

            // clear out
            scope.dashboardFilter.header = {};

            // check the options
            if (!scope.dashboardFilter.options.frame) {
                scope.dashboardFilter.html = '<div class="smss-message">No Filter</div>';
                return;
            }

            const frame = scope.insightCtrl.getShared('frames.' + scope.dashboardFilter.options.frame);
            if (typeof frame === 'undefined' || !frame.hasOwnProperty('headers')) {
                scope.insightCtrl.alert('error', 'Frame is not valid. Please select a frame to get filter values from.')
                scope.dashboardFilter.html = '<div class="smss-message">No Filter</div>';
                return;
            }

            if (!scope.dashboardFilter.options.column) {
                scope.dashboardFilter.html = '<div class="smss-message">No Filter</div>';
                return;
            }

            for (let headerIdx = 0, headerLen = frame.headers.length; headerIdx < headerLen; headerIdx++) {
                if (frame.headers[headerIdx].alias === scope.dashboardFilter.options.column) {
                    scope.dashboardFilter.header = frame.headers[headerIdx];
                    break;
                }
            }

            if (!scope.dashboardFilter.header || Object.keys(scope.dashboardFilter.header).length === 0) {
                scope.insightCtrl.alert('error', 'Column is not valid. Please select a valid column to get filter values from.')
                scope.dashboardFilter.html = '<div class="smss-message">No Filter</div>';
                return;
            }


            // reset the instances
            if (scope.dashboardFilter.options.type === 'checklist' || scope.dashboardFilter.options.type === 'dropdown' || scope.dashboardFilter.options.type === 'typeahead' || scope.dashboardFilter.options.type === 'button') {
                resetList()
            } else if (scope.dashboardFilter.options.type === 'slider' || scope.dashboardFilter.options.type === 'datepicker') {
                resetRange();
            }

            // generate and set the HTML for the filter
            scope.dashboardFilter.html = generateFilter();
        }

        /**
          * @name generateFilter
          * @desc generate the HTML for the filter
          * @return {void}
          */
        function generateFilter(): string {
            if (scope.dashboardFilter.options.type === 'checklist') {
                return generateChecklist();
            } else if (scope.dashboardFilter.options.type === 'dropdown') {
                return generateDropdown();
            } else if (scope.dashboardFilter.options.type === 'typeahead') {
                return generateTypeahead();
            } else if (scope.dashboardFilter.options.type === 'slider') {
                return generateSlider();
            } else if (scope.dashboardFilter.options.type === 'datepicker') {
                return generateDatepicker();
            } else if (scope.dashboardFilter.options.type === 'button') {
                return generateButton();
            } else {
                console.warn('TODO')
            }

            return '<div class="smss-message">No Filter</div>';
        }

        /**
          * @name changeFilter
          * @param model - new model value
          * @param delta - delta for the change
          * @desc callback for when the filter changes
          * @return {void}
          */
        function changeFilter(model: any, delta: any): void {
            let apply: boolean = false;

            if (scope.dashboardFilter.options.type === 'checklist' || scope.dashboardFilter.options.type === 'dropdown') {
                apply = changeList(delta)
            } else if (scope.dashboardFilter.options.type === 'typeahead') {
                apply = changeTypeahead(model)
            } else if (scope.dashboardFilter.options.type === 'slider' || scope.dashboardFilter.options.type === 'datepicker') {
                apply = changeRange()
            } else if (scope.dashboardFilter.options.type === 'button') {
                apply = changeButton(model)
            } else {
                console.error('TODO');
            }


            if (apply) {
                if (changeTimeout) {
                    $timeout.cancel(changeTimeout);
                }

                changeTimeout = $timeout(() => {
                    applyFilter(false);
                }, 300);
            }
        }

        /**
          * @name applyFilter
          * @desc actually run the filter
          * @param unfilter - execute an unfilter?
          * @return {void}
          */
        function applyFilter(unfilter: boolean): void {
            let frames: string[] = [],
                panels: string[] = [],
                components: PixelCommand[] = [],
                refresh: string[] = [],
                widgetMapping = {},
                panelMapping = {},
                frameMapping = {};


            // all the frames by default
            const allFrames = scope.insightCtrl.getShared('frames') || {};
            for (let f in allFrames) {
                if (allFrames.hasOwnProperty(f)) {
                    frames.push(allFrames[f].name);
                }
            }

            if (scope.dashboardFilter.options.applied) {
                frames = [];
                panels = [];

                if (
                    scope.dashboardFilter.options.applied.hasOwnProperty('frames') &&
                    Array.isArray(scope.dashboardFilter.options.applied.frames)
                ) {
                    frames = scope.dashboardFilter.options.applied.frames;
                }

                if (
                    scope.dashboardFilter.options.applied.hasOwnProperty('panels') &&
                    Array.isArray(scope.dashboardFilter.options.applied.panels)
                ) {
                    panels = scope.dashboardFilter.options.applied.panels;
                }
            }

            // all the panels
            const allPanels = scope.insightCtrl.getShared('panels') || [];
            // get a mapping of panel to frame and frame to panel
            for (let panelIdx = 0, panelLen = allPanels.length; panelIdx < panelLen; panelIdx++) {
                const frame = semossCoreService.getWidget(allPanels[panelIdx].widgetId, 'frame');

                // save the panel to widget mapping for refresh
                widgetMapping[allPanels[panelIdx].panelId] = allPanels[panelIdx].widgetId;

                // save the panel to frame mapping for later use
                panelMapping[allPanels[panelIdx].panelId] = frame;

                // create a frame to widget mapping
                if (!frameMapping.hasOwnProperty(frame)) {
                    frameMapping[frame] = []
                }

                frameMapping[frame].push(allPanels[panelIdx].panelId);
            }

            // build the components
            for (let frameIdx = 0, frameLen = frames.length; frameIdx < frameLen; frameIdx++) {
                // check if the frame exist
                if (!allFrames.hasOwnProperty(frames[frameIdx])) {
                    continue;
                }

                let exists = false;
                // check if the frame header is valid
                const headers = allFrames[frames[frameIdx]].headers || [];
                for (let headerIdx = 0, headerLen = headers.length; headerIdx < headerLen; headerIdx++) {
                    if (headers[headerIdx].alias === scope.dashboardFilter.header.alias) {
                        exists = true;
                        break;
                    }
                }

                if (!exists) {
                    continue;
                }


                let filterComponent: PixelCommand[] = [];
                if (unfilter) {
                    filterComponent = buildUnfilter(frames[frameIdx], true);
                } else if (scope.dashboardFilter.options.type === 'checklist' || scope.dashboardFilter.options.type === 'dropdown') {
                    filterComponent = buildList(frames[frameIdx], true);
                } else if (scope.dashboardFilter.options.type === 'typeahead') {
                    filterComponent = buildTypeahead(frames[frameIdx], true);
                } else if (scope.dashboardFilter.options.type === 'slider' || scope.dashboardFilter.options.type === 'datepicker') {
                    filterComponent = buildRange(frames[frameIdx], true);
                } else if (scope.dashboardFilter.options.type === 'button') {
                    filterComponent = buildButton(frames[frameIdx], true);
                } else {
                    console.error('TODO');
                }

                if (filterComponent.length > 0) {
                    components = components.concat(filterComponent);

                    // add the refresh
                    refresh = refresh.concat(frameMapping[frames[frameIdx]]);
                }
            }

            for (let panelIdx = 0, panelLen = panels.length; panelIdx < panelLen; panelIdx++) {
                // check if the panel exists and the frame exists
                if (!panelMapping.hasOwnProperty(panels[panelIdx])) {
                    continue;
                }

                let exists = false;
                // check if the frame header is valid
                const headers = allFrames[panelMapping[panels[panelIdx]]].headers || [];
                for (let headerIdx = 0, headerLen = headers.length; headerIdx < headerLen; headerIdx++) {
                    if (headers[headerIdx].alias === scope.dashboardFilter.header.alias) {
                        exists = true;
                        break;
                    }
                }

                if (!exists) {
                    continue;
                }


                let filterComponent: PixelCommand[] = [];
                if (unfilter) {
                    filterComponent = buildUnfilter(panels[panelIdx], false);
                } else if (scope.dashboardFilter.options.type === 'checklist' || scope.dashboardFilter.options.type === 'dropdown') {
                    filterComponent = buildList(panels[panelIdx], false);
                } else if (scope.dashboardFilter.options.type === 'typeahead') {
                    filterComponent = buildTypeahead(panels[panelIdx], false);
                } else if (scope.dashboardFilter.options.type === 'slider' || scope.dashboardFilter.options.type === 'datepicker') {
                    filterComponent = buildRange(panels[panelIdx], false);
                } else if (scope.dashboardFilter.options.type === 'button') {
                    filterComponent = buildButton(panels[panelIdx], false);
                } else {
                    console.error('TODO');
                }

                if (filterComponent.length > 0) {
                    components = components.concat(filterComponent);

                    // add the refresh if it is not there
                    if (refresh.indexOf(panels[panelIdx]) === -1) {
                        refresh.push(panels[panelIdx])
                    }
                }
            }

            if (components.length > 0) {
                // add the refresh
                for (let refreshIdx = 0, refreshLen = refresh.length; refreshIdx < refreshLen; refreshIdx++) {
                    components.push({
                        'type': 'refresh',
                        'components': [widgetMapping[refresh[refreshIdx]]],
                        'terminal': true
                    });
                }

                scope.insightCtrl.execute(components)
            }
        }


        /** Unfilter */
        /**
         * @name buildUnfilter
         * @desc build the pixel for the unfilter
         * @param name - name to filter
         * @param
         */
        function buildUnfilter(name: string, frame: boolean): PixelCommand[] {
            let components: PixelCommand[] = [];

            const start = frame ? name : `Panel(${name})`;

            components.push(
                {
                    type: 'Pixel',
                    components: [
                        start
                    ]
                },
                {
                    type: frame ? 'unfilterFrame' : 'unfilterPanel',
                    components: [
                        scope.dashboardFilter.header.alias
                    ],
                    terminal: true
                });



            return components;
        }

        /** List */
        /**
        * @name resetList
        * @desc reset instances for the filter
        * @return {void}
        */
        function resetList(): void {
            // clear out delta
            scope.dashboardFilter.list.delta = [];

            // we want to keep the same position, so we recollect up to that point
            scope.dashboardFilter.list.limit = scope.dashboardFilter.list.offset < 50 ? 50 : scope.dashboardFilter.list.offset;
            scope.dashboardFilter.list.offset = 0;

            // if it is button get everyting
            if (scope.dashboardFilter.options.type === 'button') {
                scope.dashboardFilter.list.limit = -1;
            }

            getList(true);
        }

        /**
          * @name getList
          * @desc get instances for the filter
          * @param reset - reset the list?
          * @return {void}
          */
        function getList(reset: boolean): void {
            // get the data based on the options

            // is it loading?
            if (scope.dashboardFilter.list.loading) {
                return;
            }

            scope.dashboardFilter.list.loading = true;

            const callback = (response: PixelReturnPayload) => {
                const output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType[0];

                scope.dashboardFilter.list.loading = false;

                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                // set options and selected
                if (reset) {
                    scope.dashboardFilter.list.options = output.options;
                    scope.dashboardFilter.list.selected = output.selectedValues;
                } else {
                    scope.dashboardFilter.list.options = scope.dashboardFilter.list.options.concat(output.options);
                    scope.dashboardFilter.list.selected = scope.dashboardFilter.list.selected.concat(output.selectedValues);
                }

                // see if you have all of the selected
                scope.dashboardFilter.list.limit = 50 // set the limit to 50;
                scope.dashboardFilter.list.offset = output.limit + output.offset; // this is the new offset
                scope.dashboardFilter.list.total = output.totalCount;
                scope.dashboardFilter.list.unrendered = output.selectedCount - scope.dashboardFilter.list.selected.length;

                // can you collect more
                scope.dashboardFilter.list.canCollect = (output.limit === output.options.length);

                renderList();
            }

            scope.insightCtrl.meta([
                {
                    type: 'variable',
                    components: [
                        scope.dashboardFilter.options.frame
                    ],
                    meta: true
                },
                {
                    type: 'getFrameFilterState',
                    components: [
                        scope.dashboardFilter.header.alias,
                        scope.dashboardFilter.list.search,
                        scope.dashboardFilter.list.limit,
                        scope.dashboardFilter.list.offset,
                        scope.dashboardFilter.options.dynamic,
                        scope.dashboardFilter.options.applied.hasOwnProperty('panels') && Array.isArray(scope.dashboardFilter.options.applied.panels) ? scope.dashboardFilter.options.applied.panels : undefined
                    ],
                    terminal: true
                }], callback, []);
        }

        /**
          * @name changeList
          * @desc the list has changed, update the values
          * @return boolean if the filter should be applied
          */
        function changeList(delta: { type: 'all' | 'none' | 'add' | 'remove' | 'replace', value: string[] | string }): boolean {
            if (delta.type === 'replace') {
                scope.dashboardFilter.list.delta = [{
                    type: 'replace',
                    search: scope.dashboardFilter.list.search,
                    added: [],
                    removed: [],
                    replaced: [delta.value]
                }];
            } else {
                let stateIdx = - 1; // the state that we are adding/removing from

                // find the old state
                for (let deltaIdx = scope.dashboardFilter.list.delta.length - 1; deltaIdx >= 0; deltaIdx--) {
                    if (scope.dashboardFilter.list.delta[deltaIdx].search === scope.dashboardFilter.list.search) {
                        stateIdx = deltaIdx;
                        break;
                    } else {
                        break; // state has to be the same
                    }
                }

                if (stateIdx === -1) {
                    scope.dashboardFilter.list.delta.push({
                        type: 'ignore',
                        search: scope.dashboardFilter.list.search,
                        added: [],
                        removed: [],
                        replaced: undefined
                    });

                    stateIdx = scope.dashboardFilter.list.delta.length - 1;
                };

                if (delta.type === 'all') {
                    // update the new state
                    scope.dashboardFilter.list.delta[stateIdx].type = 'all'
                    scope.dashboardFilter.list.delta[stateIdx].added = [];
                    scope.dashboardFilter.list.delta[stateIdx].removed = [];
                    scope.dashboardFilter.list.delta[stateIdx].replaced = undefined;
                } else if (delta.type === 'none') {
                    // update the new state
                    scope.dashboardFilter.list.delta[stateIdx].type = 'none'
                    scope.dashboardFilter.list.delta[stateIdx].added = [];
                    scope.dashboardFilter.list.delta[stateIdx].removed = [];
                    scope.dashboardFilter.list.delta[stateIdx].replaced = undefined;
                } else if (delta.type === 'add') {
                    // has it previously been removed? If so, remove it from that array
                    const removedIdx = scope.dashboardFilter.list.delta[stateIdx].removed.indexOf(delta.value);
                    if (removedIdx > -1) {
                        scope.dashboardFilter.list.delta[stateIdx].removed.splice(removedIdx, 1);
                    }

                    // check if it hasn't been already added. If it hasn't, add it to the added
                    const addedIdx = scope.dashboardFilter.list.delta[stateIdx].removed.indexOf(delta.value);
                    if (addedIdx === -1) {
                        scope.dashboardFilter.list.delta[stateIdx].added.push(delta.value);
                    }


                    scope.dashboardFilter.list.delta[stateIdx].replaced = undefined;
                } else if (delta.type === 'remove') {
                    // has it previously been added? If so, added it from that array
                    const addedIdx = scope.dashboardFilter.list.delta[stateIdx].added.indexOf(delta.value);
                    if (addedIdx > -1) {
                        scope.dashboardFilter.list.delta[stateIdx].added.splice(addedIdx, 1);
                    }

                    // check if it hasn't been already removed. If it hasn't, add it to the removed
                    const removedIdx = scope.dashboardFilter.list.delta[stateIdx].removed.indexOf(delta.value);
                    if (removedIdx === -1) {
                        scope.dashboardFilter.list.delta[stateIdx].removed.push(delta.value);
                    }

                    scope.dashboardFilter.list.delta[stateIdx].replaced = undefined;
                }
            }


            if (scope.dashboardFilter.options.auto || (delta.type === 'all' || delta.type === 'none')) {
                return true;
            }

            renderList();

            return false;
        }

        /**
          * @name renderList
          * @desc render instances for the filter based on the delta
          * @return {void}
          */
        function renderList(): void {
            // render the state
            scope.dashboardFilter.list.rendered = JSON.parse(JSON.stringify(scope.dashboardFilter.list.selected));

            // ideally the all/none should do a regex ?like to render all of the values in the options
            // TODO: implement a ?like search to update the selected values
            for (let deltaIdx = 0, deltaLen = scope.dashboardFilter.list.delta.length; deltaIdx < deltaLen; deltaIdx++) {
                // if (scope.dashboardFilter.list.delta[deltaIdx].type === 'all') {
                //     scope.dashboardFilter.list.rendered = JSON.parse(JSON.stringify(scope.dashboardFilter.list.options));
                // } else if (scope.dashboardFilter.list.delta[deltaIdx].type === 'none') {
                //     scope.dashboardFilter.list.rendered = [];
                // } else {
                //     scope.dashboardFilter.list.rendered = JSON.parse(JSON.stringify(scope.dashboardFilter.list.selected));
                // }

                TODO: // implement a ?like search to update the selected values


                // remove extra
                for (let removedIdx = 0, removedLen = scope.dashboardFilter.list.delta[deltaIdx].removed.length; removedIdx < removedLen; removedIdx++) {
                    let renderedIdx = scope.dashboardFilter.list.rendered.indexOf(scope.dashboardFilter.list.delta[deltaIdx].removed[removedIdx]);
                    if (renderedIdx !== -1) {
                        scope.dashboardFilter.list.rendered.splice(renderedIdx, 1);
                    }
                }

                // add extra
                for (let addedIdx = 0, addedLen = scope.dashboardFilter.list.delta[deltaIdx].added.length; addedIdx < addedLen; addedIdx++) {
                    let renderedIdx = scope.dashboardFilter.list.rendered.indexOf(scope.dashboardFilter.list.delta[deltaIdx].removed[addedIdx]);
                    if (renderedIdx === -1) {
                        scope.dashboardFilter.list.rendered.push(scope.dashboardFilter.list.delta[deltaIdx].added[addedIdx])
                    }
                }

                // if it is replace
                if (typeof scope.dashboardFilter.list.delta[deltaIdx].replaced !== "undefined") {
                    scope.dashboardFilter.list.rendered = JSON.parse(JSON.stringify(scope.dashboardFilter.list.delta[deltaIdx].replaced));
                }
            }

            // check the select all
            // select all is based on unsearched values
            // math is simple. 
            // rendered + unrendered selected === total
            scope.dashboardFilter.list.renderedSelectAll = (scope.dashboardFilter.list.rendered.length + scope.dashboardFilter.list.unrendered === scope.dashboardFilter.list.total);

            // this is just a string for the dashboard toggle (probably will change this)
            scope.dashboardFilter.list.renderedText = scope.dashboardFilter.list.rendered.join(', ');
        }

        /**
         * @name searchList
         * @desc search instances based on the search term
         * @param search - search termlist
         */
        function searchList(search: string): void {
            if (searchTimeout) {
                $timeout.cancel(searchTimeout);
            }
            searchTimeout = $timeout(() => {
                // is it loading?
                if (scope.dashboardFilter.list.loading) {
                    return;
                }

                //update the options
                scope.dashboardFilter.list.search = search || '';
                scope.dashboardFilter.list.offset = 0;
                scope.dashboardFilter.list.limit = 50;
                scope.dashboardFilter.list.canCollect = true;

                getList(true);

                $timeout.cancel(searchTimeout);
            }, 500);
        }

        /**
         * @name getMoreList
         * @desc get the next set of instances
         */
        function getMoreList(): void {
            // is it loading? can you collect more?
            if (scope.dashboardFilter.list.loading || !scope.dashboardFilter.list.canCollect) {
                return;
            }

            getList(false);
        }

        /**
         * @name buildList
         * @desc build the pixel for the list
         * @param name - name to filter
         * @param
         */
        function buildList(name: string, frame: boolean): PixelCommand[] {
            let components: PixelCommand[] = [];

            const start = frame ? name : `Panel(${name})`
            for (let deltaIdx = 0, deltaLen = scope.dashboardFilter.list.delta.length; deltaIdx < deltaLen; deltaIdx++) {
                if (scope.dashboardFilter.list.delta[deltaIdx].type === 'replace') {
                    components.push(
                        {
                            type: 'Pixel',
                            components: [
                                start
                            ]
                        },
                        {
                            type: frame ? 'setFrameFilter' : 'setPanelFilter',
                            components: [
                                [
                                    {
                                        type: 'value',
                                        alias: scope.dashboardFilter.header.alias,
                                        comparator: '==',
                                        values: scope.dashboardFilter.list.delta[deltaIdx].replaced
                                    }
                                ]
                            ],
                            terminal: true
                        });
                } else {
                    if (scope.dashboardFilter.list.delta[deltaIdx].type === 'all') {
                        components.push(
                            {
                                type: 'Pixel',
                                components: [
                                    start
                                ]
                            },
                            {
                                type: frame ? 'addFrameFilter' : 'addPanelFilter',
                                components: [
                                    [
                                        {
                                            type: 'value',
                                            alias: scope.dashboardFilter.header.alias,
                                            comparator: '?like',
                                            values: scope.dashboardFilter.list.delta[deltaIdx].search
                                        }
                                    ]
                                ],
                                terminal: true
                            }
                        )
                    } else if (scope.dashboardFilter.list.delta[deltaIdx].type === 'none') {
                        components.push(
                            {
                                type: 'Pixel',
                                components: [
                                    start
                                ]
                            },
                            {
                                type: frame ? 'addFrameFilter' : 'addPanelFilter',
                                components: [
                                    [
                                        {
                                            type: 'value',
                                            alias: scope.dashboardFilter.header.alias,
                                            comparator: '?nlike',
                                            values: scope.dashboardFilter.list.delta[deltaIdx].search
                                        }
                                    ]
                                ],
                                terminal: true
                            }
                        )
                    }

                    if (scope.dashboardFilter.list.delta[deltaIdx].removed.length > 0) {
                        components.push(
                            {
                                type: 'Pixel',
                                components: [
                                    start
                                ]
                            },
                            {
                                type: frame ? 'addFrameFilter' : 'addPanelFilter',
                                components: [
                                    [
                                        {
                                            type: 'value',
                                            alias: scope.dashboardFilter.header.alias,
                                            comparator: '!=',
                                            values: scope.dashboardFilter.list.delta[deltaIdx].removed
                                        }
                                    ]
                                ],
                                terminal: true
                            }
                        )
                    }

                    if (scope.dashboardFilter.list.delta[deltaIdx].added.length > 0) {
                        components.push(
                            {
                                type: 'Pixel',
                                components: [
                                    start
                                ]
                            },
                            {
                                type: frame ? 'addFrameFilter' : 'addPanelFilter',
                                components: [
                                    [
                                        {
                                            type: 'value',
                                            alias: scope.dashboardFilter.header.alias,
                                            comparator: '==',
                                            values: scope.dashboardFilter.list.delta[deltaIdx].added
                                        }
                                    ]
                                ],
                                terminal: true
                            }
                        )
                    }
                }
            }

            return components;
        }

        /** Range */
        /**
          * @name resetRange
          * @desc reset the range for the filter
          * @return {void}
          */
        function resetRange(): void {
            let components: PixelCommand[] = [];

            // this is not merged b/c I feel like it is easier to read

            // check the type based on dataType
            if (scope.dashboardFilter.header.dataType === 'DATE') {
                scope.dashboardFilter.range.type = 'date';
            } else if (scope.dashboardFilter.header.dataType === 'NUMBER') {
                scope.dashboardFilter.range.type = 'numerical';
            } else {
                scope.dashboardFilter.range.type = 'categorical';
            }

            // validate the comparator
            if (scope.dashboardFilter.range.type === 'categorical') {
                if (scope.dashboardFilter.options.comparator === '==' || scope.dashboardFilter.options.comparator === '!=') {
                    scope.dashboardFilter.range.comparator = scope.dashboardFilter.options.comparator;
                } else {
                    scope.dashboardFilter.range.comparator = '=='
                }

                scope.dashboardFilter.range.operator = '';
            } else {
                if (scope.dashboardFilter.options.multiple) {
                    if (
                        Array.isArray(scope.dashboardFilter.options.comparator) &&
                        scope.dashboardFilter.options.comparator.length === 2 &&

                        // validate all of the pairs
                        (
                            (
                                ['>', '>='].indexOf(scope.dashboardFilter.options.comparator[0]) > -1 &&
                                ['<', '<='].indexOf(scope.dashboardFilter.options.comparator[1]) > -1
                            ) ||
                            (
                                ['<', '<='].indexOf(scope.dashboardFilter.options.comparator[0]) > -1 &&
                                ['>', '>='].indexOf(scope.dashboardFilter.options.comparator[1]) > -1
                            )
                        )

                    ) {
                        scope.dashboardFilter.range.comparator = scope.dashboardFilter.options.comparator
                    } else {
                        scope.dashboardFilter.range.comparator = ['>=', '<='];
                    }

                    // set the operator
                    if (
                        ['>', '>='].indexOf(scope.dashboardFilter.range.comparator[0]) > -1 &&
                        ['<', '<='].indexOf(scope.dashboardFilter.range.comparator[1]) > -1
                    ) {
                        scope.dashboardFilter.range.operator = 'AND';
                    } else {
                        scope.dashboardFilter.range.operator = 'OR';
                    }
                } else {
                    if (
                        ['>', '>=', '<', '<=', '==', '!='].indexOf(scope.dashboardFilter.options.comparator) > -1
                    ) {
                        scope.dashboardFilter.range.comparator = scope.dashboardFilter.options.comparator
                    } else {
                        scope.dashboardFilter.range.comparator = '>=';
                    }

                    scope.dashboardFilter.range.operator = '';
                }
            }


            // build the pixel
            if (scope.dashboardFilter.range.type === 'categorical') {
                components.push(
                    {
                        type: 'variable',
                        components: [
                            scope.dashboardFilter.options.frame
                        ],
                        meta: true
                    },
                    {
                        type: 'getFrameFilterState',
                        components: [
                            scope.dashboardFilter.header.alias,
                            '',
                            -1,
                            0,
                            scope.dashboardFilter.options.dynamic,
                            scope.dashboardFilter.options.applied.hasOwnProperty('panels') && Array.isArray(scope.dashboardFilter.options.applied.panels) ? scope.dashboardFilter.options.applied.panels : undefined
                        ],
                        terminal: true
                    });
            } else {
                components.push(
                    {
                        type: 'variable',
                        components: [
                            scope.dashboardFilter.options.frame
                        ],
                        meta: true
                    },
                    {
                        type: 'getFrameFilterRange',
                        components: [
                            scope.dashboardFilter.header.alias,
                            scope.dashboardFilter.options.dynamic,
                            scope.dashboardFilter.options.applied.hasOwnProperty('panels') && Array.isArray(scope.dashboardFilter.options.applied.panels) ? scope.dashboardFilter.options.applied.panels : undefined
                        ],
                        terminal: true
                    });
            }

            if (components.length === 0) {
                return;
            }


            const callback = (response: PixelReturnPayload) => {
                const output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType[0];

                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                if (scope.dashboardFilter.range.type === 'categorical') {
                    scope.dashboardFilter.range.options = output.options;

                    if (scope.dashboardFilter.options.multiple) {
                        scope.dashboardFilter.range.rendered = output.selectedValues || [];
                    } else {
                        scope.dashboardFilter.range.rendered = output.selectedValues[0] || "";
                    }
                } else {
                    scope.dashboardFilter.range.min = output.minMax.absMin
                    scope.dashboardFilter.range.max = output.minMax.absMax

                    if (typeof scope.dashboardFilter.options.rendered !== 'undefined') {
                        scope.dashboardFilter.range.rendered = scope.dashboardFilter.options.rendered;
                    } else if (scope.dashboardFilter.options.multiple) {
                        scope.dashboardFilter.range.rendered = [output.minMax.min, output.minMax.max];
                    } else if (scope.dashboardFilter.range.comparator === '<' || scope.dashboardFilter.range.comparator === '<=') { // only use the max if it is greater than
                        scope.dashboardFilter.range.rendered = output.minMax.max;
                    } else {
                        scope.dashboardFilter.range.rendered = output.minMax.min;
                    }
                }
            }

            scope.insightCtrl.meta(components, callback, []);
        }

        /**
          * @name changeRange
          * @desc range has changed, update the value
          * @return boolean if the filter should be applied
          */
        function changeRange(): boolean {
            // noop
            if (scope.dashboardFilter.options.auto) {
                return true;
            }

            return false;
        }

        /**
         * @name buildRange
         * @desc build the pixel for a range
         * @param name - name to filter
         * @param
         */
        function buildRange(name: string, frame: boolean): PixelCommand[] {
            let components: PixelCommand[] = [];

            const start = frame ? name : `Panel(${name})`;

            if (scope.dashboardFilter.range.type === 'categorical') {
                components.push(
                    {
                        type: 'Pixel',
                        components: [
                            start
                        ]
                    },
                    {
                        type: frame ? 'setFrameFilter' : 'setPanelFilter',
                        components: [
                            [
                                {
                                    type: 'value',
                                    alias: scope.dashboardFilter.header.alias,
                                    comparator: scope.dashboardFilter.range.comparator,
                                    values: scope.dashboardFilter.range.rendered
                                }
                            ]
                        ],
                        terminal: true
                    }
                );
            } else if (scope.dashboardFilter.options.multiple) {
                components.push(
                    {
                        type: 'Pixel',
                        components: [
                            start
                        ]
                    },
                    {
                        type: frame ? 'setFrameFilter' : 'setPanelFilter',
                        components: [
                            [
                                {
                                    type: 'value',
                                    alias: scope.dashboardFilter.header.alias,
                                    comparator: scope.dashboardFilter.range.comparator[0],
                                    values: scope.dashboardFilter.range.rendered[0],
                                    operator: scope.dashboardFilter.range.operator
                                },
                                {
                                    type: 'value',
                                    alias: scope.dashboardFilter.header.alias,
                                    comparator: scope.dashboardFilter.range.comparator[1],
                                    values: scope.dashboardFilter.range.rendered[1]
                                }
                            ]
                        ],
                        terminal: true
                    }
                );
            } else {
                components.push(
                    {
                        type: 'Pixel',
                        components: [
                            start
                        ]
                    },
                    {
                        type: frame ? 'setFrameFilter' : 'setPanelFilter',
                        components: [
                            [
                                {
                                    type: 'value',
                                    alias: scope.dashboardFilter.header.alias,
                                    comparator: scope.dashboardFilter.range.comparator,
                                    values: scope.dashboardFilter.range.rendered
                                }
                            ]
                        ],
                        terminal: true
                    }
                );
            }

            // this is helpful so the slider does not keep shifting
            let options: options = scope.widgetCtrl.getWidget('view.dashboard-filter.options') || {};
            options = angular.merge({}, options, {
                rendered: scope.dashboardFilter.range.rendered
            });

            components.push(
                {
                    'type': 'panel',
                    'components': [
                        scope.widgetCtrl.panelId
                    ],
                    'meta': true
                },
                {
                    'type': 'setPanelView',
                    'components': [
                        'dashboard-filter',
                        options
                    ],
                    'terminal': true
                }
            )

            return components;
        }



        /** Checklist */
        /**
          * @name generateChecklist
          * @desc generate the HTML for a checklist filter
          * @return {void}
          */
        function generateChecklist(): string {
            return `
            <div class="dashboard-filter__checklist__holder">
                <smss-btn class="smss-btn--icon" title="Unfilter" ng-click="dashboardFilter.applyFilter(true)">
                    <img src="${require('images/unfilter-slash.svg')}" alt="unfilter"/>
                </smss-btn>
            </div>
            <smss-checklist 
                class="dashboard-filter__checklist__list ${scope.dashboardFilter.options.auto ? '' : 'dashboard-filter__checklist__list--button'}"
                model="dashboardFilter.list.rendered"
                options="dashboardFilter.list.options"
                loading="dashboardFilter.list.loading"
                search="dashboardFilter.searchList(search)"
                scroll="dashboardFilter.getMoreList();"
                change="dashboardFilter.changeFilter(model, delta)"
                searchable
                ${scope.dashboardFilter.options.multiple ? 'multiple' : ''}
                ${scope.dashboardFilter.options.multiple ? 'quickselect' : ''}
                ${scope.dashboardFilter.options.multiple ? 'select-all="dashboardFilter.list.renderedSelectAll"' : ''}>
            </smss-checklist>
            ${scope.dashboardFilter.options.auto ? '' : `
            <div class="smss-action">
                <smss-btn class="smss-btn--primary dashboard-filter__apply" title="Apply Filter" ng-click="dashboardFilter.applyFilter(false)">Apply</smss-btn>
            </div>`}
           `
        }

        /** Dropdown */
        /**
          * @name generateDropdown
          * @desc generate the HTML for a dropdown filter
          * @return {void}
          */
        function generateDropdown(): string {
            return `
            <div class="smss-clear">
                <smss-btn class="smss-left smss-btn--left smss-btn--right smss-btn--icon" title="Unfilter" ng-click="dashboardFilter.applyFilter(true)">
                    <img src="${require('images/unfilter-slash.svg')}" alt="unfilter"/>
                </smss-btn>
                <div class="smss-left dashboard-filter__input ${scope.dashboardFilter.options.auto ? '' : 'dashboard-filter__input--button'}">
                    <div class="dashboard-filter__dropdown__toggle"
                        smss-popover>
                        <div class="dashboard-filter__dropdown__toggle__text" title="{{dashboardFilter.list.renderedText}}">
                            {{dashboardFilter.list.renderedText}}
                        </div>
                        <i class="dashboard-filter__dropdown__toggle__icon"></i>

                        <smss-popover-content class="dashboard-filter__dropdown__list"
                            position="['SW', 'SE', 'NW','NE']"
                            closeable="false"
                            width="auto">
                            <smss-checklist 
                                model="dashboardFilter.list.rendered"
                                options="dashboardFilter.list.options"
                                loading="dashboardFilter.list.loading"
                                search="dashboardFilter.searchList(search)"
                                scroll="dashboardFilter.getMoreList();"
                                change="dashboardFilter.changeFilter(model, delta)"
                                searchable
                                ${scope.dashboardFilter.options.multiple ? 'multiple' : ''}
                                ${scope.dashboardFilter.options.multiple ? 'quickselect' : ''}
                                ${scope.dashboardFilter.options.multiple ? 'select-all="dashboardFilter.list.renderedSelectAll"' : ''}>
                            </smss-checklist>
                        </smss-popover-content>
                    </div>
                </div>
                ${scope.dashboardFilter.options.auto ? '' : `
                <smss-btn class="smss-left smss-btn--primary smss-btn--right dashboard-filter__apply" title="Apply Filter" ng-click="dashboardFilter.applyFilter(false)">
                    Apply
                </smss-btn>
                `}
            </div>`
        }

        /** Typeahead */
        /**
          * @name generateTypeahead
          * @desc generate the HTML for a typeahead filter
          * @return {void}
          */
        function generateTypeahead(): string {
            return `
            <div class="smss-clear">
                <smss-btn class="smss-left smss-btn--left smss-btn--right smss-btn--icon" title="Unfilter" ng-click="dashboardFilter.applyFilter(true)">
                    <img src="${require('images/unfilter-slash.svg')}" alt="unfilter"/>
                </smss-btn>
                <div class="smss-left dashboard-filter__input ${scope.dashboardFilter.options.auto ? '' : 'dashboard-filter__input--button'}">
                    <smss-typeahead
                        model="dashboardFilter.list.search"
                        loading="dashboardFilter.list.loading"
                        options="dashboardFilter.list.options"
                        change="dashboardFilter.changeFilter(model)"
                        scroll="dashboardFilter.getMoreList();">
                    </smss-typeahead>
                </div>
                ${scope.dashboardFilter.options.auto ? '' : `
                <smss-btn class="smss-left smss-btn--primary smss-btn--right dashboard-filter__apply" title="Apply Filter" ng-click="dashboardFilter.applyFilter(false)">
                    Apply
                </smss-btn>
                `}
            </div>`
        }


        /**
          * @name changeTypeahead
          * @desc typeahead has changed, update the value
          * @return boolean if the filter should be applied
          */
        function changeTypeahead(model: string): boolean {
            // apply will cause a new search
            if (scope.dashboardFilter.options.auto) {
                // set the search but do not actually search
                scope.dashboardFilter.list.search = model;

                return true;
            }

            searchList(model);

            return false
        }

        /**
         * @name buildTypeahead
         * @desc build the pixel for the typeahead
         * @param name - name to filter
         * @param
         */
        function buildTypeahead(name: string, frame: boolean): PixelCommand[] {
            let components: PixelCommand[] = [];

            const start = frame ? name : `Panel(${name})`;

            components.push(
                {
                    type: 'Pixel',
                    components: [
                        start
                    ]
                },
                {
                    type: frame ? 'setFrameFilter' : 'setPanelFilter',
                    components: [
                        [
                            {
                                type: 'value',
                                alias: scope.dashboardFilter.header.alias,
                                comparator: '?like',
                                values: scope.dashboardFilter.list.search
                            }
                        ]
                    ],
                    terminal: true
                });

            return components;
        }

        /** Slider */
        /**
          * @name generateSlider
          * @desc generate the HTML for a slider filter
          * @return {void}
          */
        function generateSlider(): string {
            let invert = false;

            // set invert based on the comparator
            if (scope.dashboardFilter.range.type === 'categorical') {
                if (scope.dashboardFilter.range.comparator === '!=') {
                    invert = true;
                } else {
                    invert = false;
                }
            } else {
                if (scope.dashboardFilter.options.multiple) {
                    if (
                        (scope.dashboardFilter.range.comparator[0] === '<' ||
                            scope.dashboardFilter.range.comparator[0] === '<=') &&
                        (scope.dashboardFilter.range.comparator[1] === '>' ||
                            scope.dashboardFilter.range.comparator[1] === '>=')) {
                        invert = true;
                    } else {
                        invert = false;
                    }
                } else {
                    if (
                        scope.dashboardFilter.range.comparator === '<' ||
                        scope.dashboardFilter.range.comparator === '<='
                    ) {
                        invert = true;
                    } else {
                        invert = false;
                    }
                }
            }


            return `
            <div class="smss-clear">
                <smss-btn class="smss-left smss-btn--left smss-btn--right smss-btn--icon dashboard-filter__clear--slider" title="Unfilter" ng-click="dashboardFilter.applyFilter(true)">
                    <img src="${require('images/unfilter-slash.svg')}" alt="unfilter"/>
                </smss-btn>
                <div class="smss-left dashboard-filter__input ${scope.dashboardFilter.options.auto ? '' : 'dashboard-filter__input--button'}">
                    <smss-slider
                        model="dashboardFilter.range.rendered"
                        ${scope.dashboardFilter.range.type === 'categorical' ? 'options="dashboardFilter.range.options"' : ''}
                        change="dashboardFilter.changeFilter(model)"
                        ${scope.dashboardFilter.range.type === 'numerical' ? `min="dashboardFilter.range.min"` : ''}
                        ${scope.dashboardFilter.range.type === 'numerical' ? `max="dashboardFilter.range.max"` : ''}
                        ${scope.dashboardFilter.range.type === 'numerical' && typeof scope.dashboardFilter.options.sensitivity !== 'undefined' ? `sensitivity="dashboardFilter.options.sensitivity"` : ''}
                        ${scope.dashboardFilter.range.type === 'date' ? `start="dashboardFilter.range.min"` : ''}
                        ${scope.dashboardFilter.range.type === 'date' ? `end="dashboardFilter.range.max"` : ''}
                        ${scope.dashboardFilter.range.type === 'date' && typeof scope.dashboardFilter.options.format !== 'undefined' ? `format="'${scope.dashboardFilter.options.format}'"` : ''}
                        ${scope.dashboardFilter.options.multiple ? 'multiple' : ''}
                        ${invert ? 'invert' : ''}
                        ${scope.dashboardFilter.range.type}>
                    </smss-slider>
                </div>
                ${scope.dashboardFilter.options.auto ? '' : `
                <smss-btn class="smss-left smss-btn--primary smss-btn--right dashboard-filter__apply dashboard-filter__apply--slider" title="Apply Filter" ng-click="dashboardFilter.applyFilter(false)">
                    Apply
                </smss-btn>
                `}
            </div>`
        }

        /**Datepicker */
        /**
          * @name generateDatepicker
          * @desc generate the HTML for a datepicker filter
          * @return {void}
          */
        function generateDatepicker(): string {
            let text = '';
            if (scope.dashboardFilter.options.multiple) {
                //noop
            } else {
                if (
                    scope.dashboardFilter.range.comparator === '<' ||
                    scope.dashboardFilter.range.comparator === '<='
                ) {
                    text = 'Before:';
                } else if (
                    scope.dashboardFilter.range.comparator === '>' ||
                    scope.dashboardFilter.range.comparator === '>='
                ) {
                    text = 'After:';
                } else if (
                    scope.dashboardFilter.range.comparator === '=='
                ) {
                    text = 'On:';
                } else if (
                    scope.dashboardFilter.range.comparator === '!='
                ) {
                    text = 'Not on:';
                }
            }
            return `
            <div class="smss-clear">
                <smss-btn class="smss-left smss-btn--left smss-btn--right smss-btn--icon" title="Unfilter" ng-click="dashboardFilter.applyFilter(true)">
                    <img src="${require('images/unfilter-slash.svg')}" alt="unfilter"/>
                </smss-btn>
                <div class="smss-left smss-clear dashboard-filter__input ${scope.dashboardFilter.options.auto ? '' : 'dashboard-filter__input--button'}">
                    ${scope.dashboardFilter.options.multiple ? `
                    <smss-date-picker 
                        class="smss-left dashboard-filter__datepicker__picker dashboard-filter__datepicker__picker--short"
                        model="dashboardFilter.range.rendered[0]"
                        change="dashboardFilter.changeFilter(model)"
                        ${typeof scope.dashboardFilter.options.format !== 'undefined' ? `format="'${scope.dashboardFilter.options.format}'"` : ''}>
                    </smss-date-picker>
                    <div class="smss-left smss-text smss-center dashboard-filter__datepicker__text">
                        <span class="smss-small"> to </span>
                    </div>
                    <smss-date-picker 
                        class="smss-left dashboard-filter__datepicker__picker dashboard-filter__datepicker__picker--short"
                    model="dashboardFilter.range.rendered[1]"
                        change="dashboardFilter.changeFilter(model)"
                        ${typeof scope.dashboardFilter.options.format !== 'undefined' ? `format="'${scope.dashboardFilter.options.format}'"` : ''}>
                    </smss-date-picker>
                    ` : `
                    <div class="smss-left smss-text dashboard-filter__datepicker__text">
                        <span class="smss-small"> ${text} </span>
                    </div>
                    <smss-date-picker 
                        class="smss-left dashboard-filter__datepicker__picker"
                        model="dashboardFilter.range.rendered"
                        change="dashboardFilter.changeFilter(model)"
                        ${typeof scope.dashboardFilter.options.format !== 'undefined' ? `format="'${scope.dashboardFilter.options.format}'"` : ''}>
                    </smss-date-picker>
                    `}
                </div>
                ${scope.dashboardFilter.options.auto ? '' : `
                <smss-btn class="smss-left smss-btn--primary smss-btn--right dashboard-filter__apply" title="Apply Filter" ng-click="dashboardFilter.applyFilter(false)">
                    Apply
                </smss-btn>
                `}
            </div>`
        }


        /** Button */
        /**
          * @name generateButton
          * @desc generate the HTML for a button filter
          * @return {void}
          */
        function generateButton(): string {
            return `
            <div class="smss-clear dashboard-filter__button ${scope.dashboardFilter.options.vertical ? 'dashboard-filter__button--vertical' : ''}">
                <smss-btn class="smss-left smss-btn--icon smss-btn--right" title="Unfilter" ng-click="dashboardFilter.applyFilter(true)">
                    <img src="${require('images/unfilter-slash.svg')}" alt="unfilter"/>
                </smss-btn>
                <div class="smss-left dashboard-filter__button__scroller"> 
                    <smss-btn class="smss-btn--right smss-btn--bordered dashboard-filter__button__scroller__button"
                            ng-repeat="opt in dashboardFilter.list.options track by $index" 
                            ng-click="dashboardFilter.changeFilter(opt)"
                            ng-class="{'dashboard-filter__button__scroller__button--selected': dashboardFilter.list.rendered.indexOf(opt) > -1}"
                            title="Filter by {{opt}}">
                        {{opt}}
                    </smss-btn>
                </div>
            </div>
            ${scope.dashboardFilter.options.auto ? '' : `
            <div class="smss-action">
                <smss-btn class="smss-btn--primary dashboard-filter__apply" title="Apply Filter" ng-click="dashboardFilter.applyFilter(false)">Apply</smss-btn>
            </div>`}
            `
        }


        /**
          * @name changeButton
          * @desc typeahead has changed, update the value
          * @return boolean if the filter should be applied
          */
        function changeButton(model: string): boolean {
            if (scope.dashboardFilter.options.multiple) {
                const idx = scope.dashboardFilter.list.rendered.indexOf(model);

                // remove it if it is already there
                if (idx > -1) {
                    scope.dashboardFilter.list.rendered.splice(idx, 1);
                } else {
                    scope.dashboardFilter.list.rendered.push(model);
                }
            } else {
                scope.dashboardFilter.list.rendered = [model];
            }

            if (scope.dashboardFilter.options.auto) {
                return true;
            }

            return false;
        }

        /**
         * @name buildButton
         * @desc build the pixel for the button
         * @param name - name to filter
         * @param
         */
        function buildButton(name: string, frame: boolean): PixelCommand[] {
            let components: PixelCommand[] = [];

            const start = frame ? name : `Panel(${name})`;


            // special scenario where nothing is selected
            if (scope.dashboardFilter.list.rendered.length === 0) {
                components.push(
                    {
                        type: 'Pixel',
                        components: [
                            start
                        ]
                    },
                    {
                        type: frame ? 'addFrameFilter' : 'addPanelFilter',
                        components: [
                            [
                                {
                                    type: 'value',
                                    alias: scope.dashboardFilter.header.alias,
                                    comparator: '?nlike',
                                    values: ''
                                }
                            ]
                        ],
                        terminal: true
                    }
                )
            } else {
                components.push(
                    {
                        type: 'Pixel',
                        components: [
                            start
                        ]
                    },
                    {
                        type: frame ? 'setFrameFilter' : 'setPanelFilter',
                        components: [
                            [
                                {
                                    type: 'value',
                                    alias: scope.dashboardFilter.header.alias,
                                    comparator: '==',
                                    values: scope.dashboardFilter.list.rendered
                                }
                            ]
                        ],
                        terminal: true
                    });
            }

            return components;
        }

        /**
         * @name initialize
         * @desc called when the directive is loaded
         */
        function initialize(): void {
            // append listeners
            const
                updateViewListener = scope.insightCtrl.on('update-view', resetFilter),
                updateFrameListener = scope.insightCtrl.on('update-frame', resetFilter),
                updateFrameFilterListener = scope.insightCtrl.on('update-frame-filter', resetFilter);

            scope.$on('$destroy', () => {
                updateViewListener();
                updateFrameListener();
                updateFrameFilterListener();
            });

            resetFilter();
        }

        initialize();
    }
}
