'use strict';
/**
 * @name choropleth-echarts.service.js
 * @desc This service configures our data into echarts data as according to the
 *       e-charts examples (one of which is: https://ecomfe.github.io/echarts-examples/public/editor.html?c=choropleth-disk).
 *       This service functions similarly to the other e-charts services but it
 *       has two specific functions: setchoroplethDataSeries() and constructChildObj().
 *       The first function simply loops through our structured data and converts
 *       it into the appropriate data object to display choropleth data, while the
 *       second is a function and the first calls in order to create child objects
 *       that are populated into our choropleth data object.
 */

import EchartsHelper from '@/widget-resources/js/echarts/echarts-helper.js';


angular.module('app.choropleth.service', [])
    .factory('choroplethService', choroplethService);

choroplethService.$inject = ['semossCoreService'];

function choroplethService(semossCoreService) {
    var statePos,
        valuePos,
        abbreviationMapping,
        invertedMapping,
        misspelledMapping;

    function getConfig(type, data, uiOptions, keys, keysObj, groupByInstance) {
        var tempDataObject,
            animation = customizeAnimation(uiOptions.animation),
            minMax;

        if (uiOptions.chloroType === 'State') {
            abbreviationMapping = EchartsHelper.getStateAbbreviationMapping();
            invertedMapping = _.invert(abbreviationMapping);
            misspelledMapping = createMisspelledMapping();
        } else if (uiOptions.chloroType === 'Country') {
            abbreviationMapping = EchartsHelper.getCountryAbbreviationMapping();
            invertedMapping = _.invert(abbreviationMapping);
            misspelledMapping = createMisspelledMapping();
        }

        tempDataObject = formatData(data, keysObj, uiOptions.chloroType, keys);
        minMax = setMinMax(tempDataObject);

        // Configure data for ECharts
        return {
            data: tempDataObject,
            legendHeaders: tempDataObject.valuesNames,
            showLegend: !uiOptions.toggleLegend,
            legendLabels: setLegendHeaders(data.headers),
            showAnimation: animation.showAnimation,
            animationType: animation.defaultAnimationType,
            animationDelay: animation.defaultAnimationSpeed,
            animationDuration: animation.defaultAnimationDuration,
            backgroundColorStyle: getBackgroundColorStyle(uiOptions.watermark),
            min: minMax.min,
            max: minMax.max,
            keys: keysObj,
            groupByInstance: groupByInstance,
            options: getUiOptions(uiOptions)
        };
    }

    function setLegendHeaders(data) {
        let i = data.indexOf('id');
        data.splice(i, 1);

        return data;
    }

    function formatData(data, keysObj, type, keys) {
        var dataArray = [],
            dataObj,
            tempName,
            headerPosition,
            abbreviationArray = [],
            formattingArray = [],
            errorArray = [],
            lowercaseComparison = '',
            nameStr,
            cleanedName = '',
            tipIndices = [],
            i, tipIndex;

        for (i = 0; i < data.headers.length; i++) {
            if (data.headers[i] === keysObj.label.replace(/ /g, '_')) {
                statePos = i;
            } else if (data.headers[i] === keysObj.value.replace(/ /g, '_')) {
                valuePos = i;
            }
        }

        keys.forEach(function (key) {
            if (key.model.indexOf('tooltip') > -1) {
                for (i = 0; i < data.headers.length; i++) {
                    if (data.headers[i] === key.alias) {
                        tipIndices.push(i);
                    }
                }
            }
        });

        for (i = 0; i < data.values.length; i++) {
            dataObj = {
                tooltip: {}
            };
            if (tipIndices.length > 0) {
                for (tipIndex = 0; tipIndex < tipIndices.length; tipIndex++) {
                    headerPosition = tipIndices[tipIndex];
                    dataObj.tooltip[data.headers[headerPosition]] = data.values[i][headerPosition];
                }
            }
            nameStr = data.values[i][statePos].toString();
            if (type === 'Counties' || type === 'Regions' || type === 'Zip') {
                dataObj.name = formatName(type, nameStr);
                dataObj.value = data.values[i][valuePos];
            } else {
                cleanedName = nameStr.replace(/_/g, ' ');
                if (abbreviationMapping.hasOwnProperty(cleanedName)) {
                    tempName = abbreviationMapping[nameStr];
                    abbreviationArray.push(nameStr);
                } else if (!invertedMapping.hasOwnProperty(cleanedName)) {
                    lowercaseComparison = cleanedName.toLowerCase().replace(/ /g, '').replace(/_/g, '');

                    if (misspelledMapping.hasOwnProperty(lowercaseComparison)) {
                        tempName = misspelledMapping[lowercaseComparison];
                        formattingArray.push(nameStr);
                    } else {
                        errorArray.push(nameStr);
                        continue;
                    }
                } else {
                    tempName = cleanedName; // GeoJSON doesn't have underscores in names
                }
                dataObj.name = tempName;
                dataObj.value = data.values[i][valuePos];
            }

            dataArray.push(dataObj);
        }

        if (errorArray.length > 0 || formattingArray.length > 0) {
            // TODO: Make the alert bound to a widget
            semossCoreService.emit('alert', {
                color: 'warn',
                text: 'Please use either ' + type + ' abbreviations or proper ' + type + ' names.'
            });
        }

        return dataArray;
    }

    function formatName(type, value) {
        if (type === 'Counties') {
            return value.charAt(0).toUpperCase() + value.toLocaleLowerCase().slice(1);
        }
        return value;
    }

    function setMinMax(data) {
        var
            valueArray = data.map(function (val) {
                return val.value;
            }),
            min,
            max;

        min = _.min(valueArray);
        max = _.max(valueArray);

        return {
            min: min,
            max: max
        };
    }

    function createMisspelledMapping() {
        var finalMapping = {},
            i;

        for (i in invertedMapping) {
            if (invertedMapping.hasOwnProperty(i)) {
                finalMapping[i.toLocaleLowerCase().replace(/ /g, '')] = i;
            }
        }

        return finalMapping;
    }

    /**
     * @name customizeAnimation
     * @desc sets the animation style
     * @param {object} param - object of animation settings
     * @returns {object} - object of animation settings
     */
    function customizeAnimation(param) {
        var animationSettings = {};
        // TODO small refactor
        if (param && param.chooseType === 'No Animation') {
            animationSettings.showAnimation = false;
        } else if (param) {
            animationSettings.showAnimation = true;
            animationSettings.defaultAnimationType = param.chooseType;
            animationSettings.defaultAnimationSpeed = param.animationSpeed;
            animationSettings.defaultAnimationDuration = param.animationDuration;
        } else {
            // default
            animationSettings.showAnimation = true;
            // TODO set as same as widget service default state
            animationSettings.defaultAnimationType = 'No Animation';
            animationSettings.defaultAnimationSpeed = 1;
            animationSettings.defaultAnimationDuration = 500;
        }
        return animationSettings;
    }

    function getUiOptions(options) {
        return options;
    }

    /**
     * @name getBackgroundColorStyle
     * @desc customize the background style of the canvas
     * @param {string} watermark - string of the watermark text
     * @returns {Object} - canvas details
     */
    function getBackgroundColorStyle(watermark) {
        if (/\S/.test(watermark)) {
            return {
                type: 'pattern',
                image: paintWaterMark(watermark),
                repeat: 'repeat'
            };
        }

        return false;
    }

    /**
     * @name paintWaterMark
     * @desc paints a custom watermark on the viz
     * @param {string} watermark - string of the watermark text
     * @returns {Object} - canvas details
     */
    function paintWaterMark(watermark) {
        var canvas = document.createElement('canvas'),
            ctx = canvas.getContext('2d');
        canvas.width = canvas.height = 100;
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        ctx.globalAlpha = 0.08;
        ctx.font = '20px Microsoft Yahei';
        ctx.translate(50, 50);
        ctx.rotate(-Math.PI / 4);
        if (watermark) {
            ctx.fillText(watermark, 0, 0);
        }
        return canvas;
    }

    return {
        getConfig: getConfig
    };
}
