'use strict';

import echarts from 'echarts';
import EchartsHelper from '@/widget-resources/js/echarts/echarts-helper.js';

import './choropleth-echarts.service.js';

export default angular.module('app.choropleth-echarts.directive', [
    'app.choropleth.service'
]).directive('choroplethEcharts', choroplethEcharts);

choroplethEcharts.$inject = ['semossCoreService', 'choroplethService'];

function choroplethEcharts(semossCoreService, choroplethService) {
    choroplethChartLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^widget'],
        priority: 300,
        link: choroplethChartLink
    };

    function choroplethChartLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];
        /** ************* Main Event Listeners ************************/
        var resizeListener,
            updateTaskListener,
            updateOrnamentsListener,
            addDataListener,
            /** *************** ECharts ****************************/
            eChartsConfig,
            choroplethChart,
            clickTimer,
            hoverTimer;

        /**
         * @name initialize
         * @desc creates the visualization on the chart div
         * @returns {void}
         */
        function initialize() {
            // bind listeners
            resizeListener = scope.widgetCtrl.on('resize-widget', resizeViz);
            updateTaskListener = scope.widgetCtrl.on('update-task', setData);
            updateOrnamentsListener = scope.widgetCtrl.on('update-ornaments', setData);
            addDataListener = scope.widgetCtrl.on('added-data', setData);

            scope.$on('$destroy', destroy);

            setData();
        }

        /**
         * @name setData
         * @desc setData for the visualization and paints
         * @returns {void}
         */
        function setData() {
            var layerIndex = 0,
                selectedLayout = scope.widgetCtrl.getWidget('view.visualization.layout'),
                individiualTools = scope.widgetCtrl.getWidget('view.visualization.tools.individual.' + selectedLayout) || {},
                sharedTools = scope.widgetCtrl.getWidget('view.visualization.tools.shared'),
                keys = scope.widgetCtrl.getWidget('view.visualization.keys.' + selectedLayout),
                data = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.data'),
                groupBy = {},
                groupByInstance,
                groupByInfo = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.groupByInfo'),
                uiOptions = angular.extend(sharedTools, individiualTools);

            if (groupByInfo && groupByInfo.viewType) {
                if (groupByInfo.viewType === 'Individual Instance') {
                    groupBy = formatDataForGroupByIndividual(data, groupByInfo);
                    data = groupBy.data;
                    groupByInstance = groupBy.name;
                }
            }

            // TODO add comments
            // comments: scope.widgetCtrl.getWidget('view.visualization.commentData'),
            eChartsConfig = choroplethService.getConfig('choropleth', data, uiOptions, keys, semossCoreService.visualization.getDataTableAlign(keys), groupByInstance);
            eChartsConfig.currentMode = EchartsHelper.getCurrentMode(scope.widgetCtrl.getMode('selected'));
            eChartsConfig.callbacks = scope.widgetCtrl.getEventCallbacks();
            eChartsConfig.comments = scope.widgetCtrl.getWidget('view.visualization.commentData');
            eChartsConfig.groupByInfo = groupByInfo;

            paint();
        }

        /**
         * @name formatDataForGroupByIndividual
         * @desc formats data when Group By exists
         * @param {object} data orginial data
         * @param {object} groupBy groupBy object
         * @returns {void}
         */
        function formatDataForGroupByIndividual(data, groupBy) {
            var formattedData = data,
                groupByIndex,
                name,
                i,
                instanceIdx,
                returnObj = {};

            groupByIndex = data.headers.indexOf(groupBy.selectedDim);
            if (groupByIndex === -1) {
                // return data;
                groupByIndex = data.headers.length;
            }

            if (typeof groupBy.instanceIndex === 'string') {
                instanceIdx = parseInt(groupBy.instanceIndex, 10);
            }
            // Create name for title
            name = groupBy.selectedDim + ' : ' + groupBy.uniqueInstances[instanceIdx];
            // Remove Group By dimension from data headers and values
            formattedData.headers.splice(groupByIndex, 1);
            formattedData.rawHeaders.splice(groupByIndex, 1);

            // Remove any added data from brush/click
            for (i = 0; i < data.values.length; i++) {
                if (data.values[i][groupByIndex] !== groupBy.uniqueInstances[instanceIdx]) {
                    data.values.splice(i, 1);
                    i--;
                }
            }

            for (i = 0; i < data.values.length; i++) {
                data.values[i].splice(groupByIndex, 1);
            }
            returnObj.name = name;
            returnObj.data = data;


            return returnObj;
        }

        /**
         * @name paint
         * @desc paints the visualization
         * @returns {void}
         */
        function paint() {
            if (choroplethChart) {
                choroplethChart.clear();
                choroplethChart.dispose();
            }

            choroplethChart = echarts.init(ele[0].firstElementChild);

            var option = {},
                mapOptions = {},
                folderName,
                fileName;

            if (eChartsConfig.groupByInfo && eChartsConfig.groupByInfo.viewType) {
                if (eChartsConfig.groupByInfo.viewType === 'Individual Instance') {
                    option.graphic = [];
                }
            }

            if (eChartsConfig.backgroundColorStyle) {
                option.backgroundColor = eChartsConfig.backgroundColorStyle;
            }
            option.tooltip = {
                show: eChartsConfig.options.showTooltips,
                trigger: 'item',
                confine: true,
                formatter: function (info) {
                    var returnArray = [];

                    if (!info.data) {
                        return '';
                    }

                    if (info.data.name) {
                        returnArray.push('<b>' + eChartsConfig.keys.label + '</b>: ' + cleanValue(info.data.name));
                    }

                    if (info.data.value) {
                        returnArray.push('<br><b>' + eChartsConfig.keys.value + '</b>: ' + cleanValue(info.data.value));
                    }

                    if (info.data.tooltip) {
                        Object.keys(info.data.tooltip).forEach(function (header) {
                            var tip = info.data.tooltip[header];
                            returnArray.push(
                                '<br><b>' + cleanValue(header) + '</b>: ' + cleanValue(tip)
                            );
                        });
                    }

                    return returnArray.join('');
                }
            };
            if (typeof eChartsConfig.max !== 'undefined' && typeof eChartsConfig.min !== 'undefined') {
                option.visualMap = customizeLegend(eChartsConfig.options.heatmapLegend, eChartsConfig.max, eChartsConfig.min, eChartsConfig.options.heatBuckets, eChartsConfig.options.heatmapColor, eChartsConfig.options.heatRange);
            }
            option.textStyle = {
                fontFamily: 'Libre Franklin'
            };
            option.series = [{
                type: 'map',
                roam: true,
                itemStyle: {
                    emphasis: {
                        areaColor: null,
                        shadowOffsetX: 0,
                        shadowOffsetY: 0,
                        shadowBlur: 20,
                        borderWidth: 0,
                        shadowColor: 'rgba(0, 0, 0, 0.5)',
                        label: {
                            show: true,
                            formatter: function (info) {
                                return typeof info.name === 'string' ? info.name.replace(/_/, ' ') : info.name;
                            }
                        }
                    }
                },
                data: eChartsConfig.data
            }];

            if (eChartsConfig.options.chloroType === 'State') {
                // TODO check if county data exists
                option.series[0].map = 'USA';
                option.series[0].textFixed = {
                    Alaska: [20, 20]
                };
                folderName = 'choropleth';
                fileName = 'usa';
                mapOptions = {
                    Alaska: {
                        left: -131,
                        top: 25,
                        width: 15
                    },
                    Hawaii: {
                        left: -110,
                        top: 24,
                        width: 5
                    },
                    'Puerto Rico': {
                        left: -76,
                        top: 26,
                        width: 2
                    }
                };
            } else if (eChartsConfig.options.chloroType === 'Counties') {
                option.series[0].map = 'usaCounties';
                option.series[0].textFixed = {
                    Alaska: [20, -20]
                };
                folderName = 'choropleth';
                fileName = 'usaCounties';
            } else if (eChartsConfig.options.chloroType === 'Regions') {
                option.series[0].map = 'usaRegions';
                option.series[0].textFixed = {};
                folderName = 'choropleth';
                fileName = 'usaRegions';
            } else if (eChartsConfig.options.chloroType === 'Zip') {
                option.series[0].map = 'usaZips';
                option.series[0].textFixed = {};
            } else {
                option.series[0].map = 'world';
                option.series[0].textFixed = {};
                folderName = 'map';
                fileName = 'world';
            }
            import( /* webpackChunkName: "data/[request]" */ `../../widget-resources/js/echarts/${folderName}/${fileName}.json`).then((module) => {
                var mapJson = module.default,
                    mapName = option.series[0].map;
                echarts.registerMap(mapName, mapJson, mapOptions);

                // use configuration item and data specified to show chart
                EchartsHelper.setOption(choroplethChart, option);

                // Add event listeners
                // TODO figure out comment / brush / eventing paradigm
                initializeEvents();
            }).catch((err) => {
                console.log('Error loading map data.', err);
            });
        }

        function cleanValue(item) {
            if (typeof item === 'string') {
                return item.replace(/_/g, ' ');
            } else if (typeof item === 'number') {
                return item.toLocaleString(undefined, {
                    minimumFractionDigits: 0,
                    maximumFractionDigits: 3
                });
            }
            return item;
        }

        /**
         * @name customizeLegend
         * @desc sets the legend type
         * @param {bool} legendType - legend type defined by user
         * @param {number} maxValue - max value of heat dimension
         * @param {number} minValue - min value of heat dimension
         * @param {num} heatBuckets - number of buckets
         * @param {array} heatmapColor - selected heat color palette
         * @param {object} heatRange - selected heat range
         * @returns {object} - object of visual map settings
         */
        function customizeLegend(legendType, maxValue, minValue, heatBuckets, heatmapColor, heatRange) {
            var visualMap = {};
            visualMap.type = legendType;

            if (heatRange) {
                if (heatRange.min.show && typeof heatRange.min.value !== 'undefined') {
                    visualMap.min = heatRange.min.value;
                } else {
                    visualMap.min = minValue;
                }
                if (heatRange.max.show && typeof heatRange.max.value !== 'undefined') {
                    visualMap.max = heatRange.max.value;
                } else {
                    visualMap.max = maxValue;
                }
            } else {
                visualMap.min = minValue;
                visualMap.max = maxValue;
            }

            visualMap.calculable = true;
            visualMap.orient = 'vertical';
            visualMap.left = '2%';
            visualMap.bottom = '60px';
            // visualMap.inRange = { color: ['#313695', '#4575b4', '#74add1', '#abd9e9', '#e0f3f8', '#ffffbf', '#fee090', '#fdae61', '#f46d43', '#d73027', '#a50026'] };

            if (typeof heatmapColor !== 'undefined') {
                visualMap.inRange = {
                    color: heatmapColor
                };
            } else {
                visualMap.inRange = {
                    color: ['#313695', '#4575b4', '#74add1', '#abd9e9', '#e0f3f8', '#ffffbf', '#fee090', '#fdae61', '#f46d43', '#d73027', '#a50026']
                };
            }

            visualMap.text = ['High', 'Low'];
            if (legendType === 'continuous') {
                visualMap.itemHeight = 120;
            }
            if (legendType === 'piecewise') {
                visualMap.splitNumber = heatBuckets || 5;
            }

            return visualMap;
        }

        /**
         * @name initializeEvents
         * @desc creates the event layer
         * @returns {void}
         */
        function initializeEvents() {
            /** ******************************************************/
            // Event: On single click open new tab with wikipedia page of selected label
            choroplethChart.on('click', eChartClicked);
            choroplethChart.on('mouseover', eChartMouse);
            choroplethChart.on('mouseout', eChartMouseOut);
            choroplethChart._dom.addEventListener('mouseout', mouseOut);
            // choroplethChart.on('dblclick', eChartClicked);

            // it is necessary to initialize comment mode so the nodes are painted
            EchartsHelper.initializeCommentMode({
                comments: eChartsConfig.comments,
                currentMode: eChartsConfig.currentMode,
                saveCb: eChartsConfig.callbacks.commentMode.onSave
            });

            // EchartsHelper.initializeBrush
        }

        /**
         * @name eChartClicked
         * @desc single click event from echarts
         * @param {object} event - echarts event sent back on click
         * @returns {void}
         */
        function eChartClicked(event) {
            if (clickTimer) {
                clearTimeout(clickTimer);
                eventCallback(event, 'onDoubleClick');
            } else {
                clickTimer = setTimeout(eventCallback.bind(null, event, 'onClick'), 250);
            }
        }

        /**
         * @name eChartMouse
         * @desc onHover event for echarts
         * @param {object} event - echarts event sent back on hover
         * @returns {void}
         */
        function eChartMouse(event) {
            if (hoverTimer) {
                clearTimeout(hoverTimer);
            }
            hoverTimer = setTimeout(eventCallback.bind(null, event, 'onHover'), 2000);
            // console.log(event);
        }

        /**
         * @name eChartMouseOut
         * @desc offHover event for echarts
         * @param {object} event - echarts event sent back on offHover
         * @returns {void}
         */
        function eChartMouseOut(event) {
            var currentEvent = scope.widgetCtrl.getEvent('currentEvent');
            if (currentEvent.type === 'onHover') {
                eventCallback(event, 'onMouseOut');
            }
        }

        /**
         * @name mouseOut
         * @desc clears timers on mouse out of canvas
         * @returns {void}
         */
        function mouseOut() {
            clearTimeout(hoverTimer);
        }

        /**
         * @name eventCallback
         * @desc click callback event
         * @param {object} event - echarts event sent back on click
         * @param {string} type - click or double click
         * @desc sends event data to event service, also checks if underlying data is abbreviated so proper data is sent
         * @returns {void}
         */
        function eventCallback(event, type) {
            var returnObj = {
                    data: {}
                },
                layerIndex = 0,
                values = scope.widgetCtrl.getWidget('view.visualization.tasks.' + layerIndex + '.data.values'),
                countryMapping = EchartsHelper.getCountryAbbreviationMapping(),
                stateMapping = EchartsHelper.getStateAbbreviationMapping(),
                statePos, i, actualVal = event.name;
            if (typeof values[0][0] === 'string') {
                statePos = 0;
            } else {
                statePos = 1;
            }

            if (stateMapping.hasOwnProperty(values[0][statePos])) {
                for (i = 0; i < Object.keys(stateMapping).length; i++) {
                    if (stateMapping[Object.keys(stateMapping)[i]] === event.name) {
                        actualVal = Object.keys(stateMapping)[i];
                    }
                }
            }

            if (countryMapping.hasOwnProperty(values[0][statePos])) {
                for (i = 0; i < Object.keys(countryMapping).length; i++) {
                    if (countryMapping[Object.keys(countryMapping)[i]] === event.name) {
                        actualVal = Object.keys(countryMapping)[i];
                    }
                }
            }

            returnObj.data[eChartsConfig.legendLabels] = [actualVal];
            eChartsConfig.callbacks.defaultMode[type](returnObj);
            clickTimer = null;
            hoverTimer = null;
        }

        /**
         * @name resizeViz
         * @desc reruns the jv paint function
         * @returns {void}
         */
        function resizeViz() {
            choroplethChart.resize();
        }

        /**
         * @name destroy
         * @desc destroys listeners and dom elements outside of the scope
         * @returns {void}
         */
        function destroy() {
            resizeListener();
            updateTaskListener();
            updateOrnamentsListener();
            addDataListener();
        }

        // Start Visualization Creation
        initialize();
    }
}
