'use strict';

import EchartsHelper from '@/widget-resources/js/echarts/echarts-helper.js';

export default angular.module('app.bar.service', [])
    .factory('barLineService', barLineService);

barLineService.$inject = ['semossCoreService'];

function barLineService(semossCoreService) {
    var valuesMapping = {};

    function getConfig(type, data, uiOptions, colorBy, groupByInstance, groupBy, keys) {
        valuesMapping = getValuesMapping(keys, data.headers);

        var tempDataObject,
            xAxisConfig,
            yAxisConfig,
            reverseYAxis = uiOptions.yReversed,
            animation = customizeAnimation(uiOptions.animation),
            numberOfDataSeries,
            groupedData,
            temp,
            i,
            finalData,
            valuesNames,
            labelName,
            extremes = findMaxMin(data, groupBy);

        if (groupBy && groupBy.viewType === 'All Instances') {
            tempDataObject = [];
            xAxisConfig = [];
            yAxisConfig = [];
            finalData = [];

            for (i = 0; i < groupBy.uniqueInstances.length; i++) {
                groupedData = {};
                groupedData.headers = data.headers;
                groupedData.rawHeaders = data.rawHeaders;
                groupedData.values = groupBy.tempData[groupBy.uniqueInstances[i]];
                tempDataObject.push(formatData(groupedData));
                xAxisConfig.push(configureXAxis(uiOptions, tempDataObject[i].labelData, tempDataObject[i].labelName, groupBy, i));
                yAxisConfig.push(setYaxis(uiOptions, reverseYAxis, tempDataObject[i].yLabel, groupBy, i, extremes));
                finalData.push(setEChartsDataSeries(type, tempDataObject[i], uiOptions, colorBy, i));
            }

            if (tempDataObject[0]) {
                numberOfDataSeries = tempDataObject[0].valuesData.length || 0;
                valuesNames = tempDataObject[0].valuesNames;
                labelName = tempDataObject[0].labelName;
            } else {
                numberOfDataSeries = 0;
                valuesNames = [];
                labelName = '';
            }
        } else {
            tempDataObject = formatData(data);
            xAxisConfig = configureXAxis(uiOptions, tempDataObject.labelData, tempDataObject.labelName, groupBy, 0);
            yAxisConfig = setYaxis(uiOptions, reverseYAxis, tempDataObject.yLabel, groupBy, 0, {});
            animation = customizeAnimation(uiOptions.animation);
            temp;
            finalData = setEChartsDataSeries(type, tempDataObject, uiOptions, colorBy);
            numberOfDataSeries = tempDataObject.valuesData.length || 0;
        }

        // Flip Axis
        if (uiOptions.rotateAxis) {
            temp = yAxisConfig;
            yAxisConfig = xAxisConfig;
            xAxisConfig = temp;
        }

        // Configure data for ECharts
        return {
            data: finalData,
            xAxisConfig: xAxisConfig,
            yAxisConfig: yAxisConfig,
            legendHeaders: tempDataObject.valuesNames.concat(tempDataObject.lineNames) || valuesNames,
            legendLabels: tempDataObject.labelName || labelName,
            legendData: tempDataObject.labelData || [],
            showLegend: uiOptions.toggleLegend,
            showAnimation: animation.showAnimation,
            animationType: animation.defaultAnimationType,
            animationDelay: animation.defaultAnimationSpeed,
            animationDuration: animation.defaultAnimationDuration,
            axisPointer: uiOptions.axisPointer,
            barWidth: customizeBarWidth(uiOptions.editBarWidth, numberOfDataSeries),
            backgroundColorStyle: getBackgroundColorStyle(uiOptions.watermark),
            groupByInstance: groupByInstance,
            options: uiOptions
        };
    }

    /**
     * @name getValueMapping
     * @desc loop through keys and grab value dimension and tooltip info
     * @param {object} keys semoss keys
     * @param {object} headers data headers
     * @returns {void}
     */
    function getValuesMapping(keys, headers) {
        var key,
            mappingByDimension = {},
            mappingByModel = {};

        mappingByModel.value = [];
        mappingByModel.tooltip = [];
        mappingByModel.lines = [];

        for (key in keys) {
            if (keys.hasOwnProperty(key)) {
                mappingByDimension[keys[key].alias] = headers.indexOf(keys[key].alias);
                if (keys[key].model === 'value') {
                    mappingByModel.value.push(headers.indexOf(keys[key].alias));
                } else if (keys[key].model === 'lines') {
                    mappingByModel.lines.push(headers.indexOf(keys[key].alias));
                } else if (keys[key].model === 'tooltip') {
                    mappingByModel.tooltip.push(headers.indexOf(keys[key].alias));
                } else {
                    mappingByModel[keys[key].model] = headers.indexOf(keys[key].alias);
                }
            }
        }
        return {
            mappingByDimension: mappingByDimension,
            mappingByModel: mappingByModel
        };
    }

    /**
     * @name formatData
     * @desc puts data into ECharts data format
     * @param {Object} data - object of data in original format
     * @returns {Object} - object of data in ECharts format
     */
    function formatData(data) {
        var eChartsDataObject = {},
            labelIdx = valuesMapping.mappingByModel.label,
            i, n, j,
            temp,
            valueData,
            lineData,
            tooltipIdx;

        // X-Axis Title
        eChartsDataObject.labelName = data.headers[labelIdx];
        // X-Axis Labels
        eChartsDataObject.labelData = [];
        // Y-Axis Labels (Series)
        eChartsDataObject.valuesNames = [];
        // Y-Axis Data
        eChartsDataObject.valuesData = [];
        // Y-Axis Labels (Series)
        eChartsDataObject.lineNames = [];
        // Y-Axis Data
        eChartsDataObject.lineData = [];
        // Tooltip data
        eChartsDataObject.tooltipData = [];
        eChartsDataObject.tooltipHeaders = [];

        for (j = 0; j < valuesMapping.mappingByModel.tooltip.length; j++) {
            eChartsDataObject.tooltipHeaders.push(data.headers[valuesMapping.mappingByModel.tooltip[j]]);
        }

        for (i = 0; i < data.values.length; i++) {
            eChartsDataObject.labelData.push(data.values[i][labelIdx]);
            tooltipIdx = [];
            for (j = 0; j < valuesMapping.mappingByModel.tooltip.length; j++) {
                tooltipIdx.push(data.values[i][valuesMapping.mappingByModel.tooltip[j]]);
            }
            eChartsDataObject.tooltipData.push(tooltipIdx);
        }

        for (n = 0; n < valuesMapping.mappingByModel.value.length; n++) {
            eChartsDataObject.yLabel = '';
            if (n === 0) {
                eChartsDataObject.yLabel = cleanValue(data.headers[valuesMapping.mappingByModel.value[n]]);
            }
            eChartsDataObject.valuesNames.push(data.headers[valuesMapping.mappingByModel.value[n]]);
            valueData = [];
            for (i = 0; i < data.values.length; i++) {
                valueData.push(data.values[i][valuesMapping.mappingByModel.value[n]]);
            }
            eChartsDataObject.valuesData.push(valueData);
        }

        for (n = 0; n < valuesMapping.mappingByModel.lines.length; n++) {
            eChartsDataObject.lineNames.push(data.headers[valuesMapping.mappingByModel.lines[n]]);
            lineData = [];
            for (i = 0; i < data.values.length; i++) {
                lineData.push(data.values[i][valuesMapping.mappingByModel.lines[n]]);
            }
            eChartsDataObject.lineData.push(lineData);
        }

        return eChartsDataObject;
    }

    /**
     * @name configureXAxis
     * @desc defines settings for x-axis
     * @param {Object} uiOptions - uiOptions
     * @param {Array} data - array of labels on the x-axis
     * @param {string} axisName - tempDataObject.labelName (x-axis label name)
     * @param {Object} groupBy - object of groupBy info
     * @param {num} idx - groupBy instance index
     * @returns {array} - array of x-axis settings
     */
    function configureXAxis(uiOptions, data, axisName, groupBy, idx) {
        var xAxisConfig = [],
            axisTitle,
            showAxisValues,
            nameGap = 25,
            showAxisLine,
            showAxisTicks,
            settings = uiOptions.editXAxis,
            grid = uiOptions.editGrid.x,
            flipAxis = uiOptions.rotateAxis,
            fontSize = uiOptions.fontSize ? uiOptions.fontSize.substring(0, uiOptions.fontSize.indexOf('p')) : 12,
            fontColor = uiOptions.fontColor || 'black';

        if (settings) {
            if (settings.title.show) {
                axisTitle = settings.title.name;
            } else {
                axisTitle = null;
            }
            showAxisValues = settings.values;
            nameGap = settings.nameGap;
            showAxisLine = showAxisTicks = settings.line;
        } else {
            axisTitle = axisName.replace(/_/g, ' ');
            showAxisValues = true;
            showAxisLine = showAxisTicks = true;
        }

        xAxisConfig.push({
            type: 'category',
            data: data,
            axisTick: {
                show: showAxisTicks,
                alignWithLabel: true
            },
            splitLine: {
                show: grid
            },
            axisLabel: {
                show: showAxisValues,
                rotate: settings.rotate || 0,
                formatter: function (value) {
                    return EchartsHelper.formatLabel(value, settings.format);
                },
                fontSize: fontSize,
                color: fontColor
            },
            name: axisTitle,
            nameLocation: 'center',
            nameGap: nameGap,
            nameTextStyle: {
                fontWeight: 'bold',
                fontSize: fontSize,
                color: fontColor
            },
            gridIndex: idx,
            axisLine: {
                show: showAxisLine
            }
        });
        if (flipAxis) {
            xAxisConfig[0].nameLocation = 'end';
            if (axisTitle.length > 10 && axisTitle.length < 15) {
                xAxisConfig[0].nameTextStyle.padding = [0, 0, 0, ((axisTitle.length * 10) / 4)];
            } else if (axisTitle.length >= 15) {
                xAxisConfig[0].nameTextStyle.padding = [0, 0, 0, ((axisTitle.length * 10) / 2)];
            }
        }

        return xAxisConfig;
    }

    /**
     * @name setYAxis
     * @desc sets configuration of y-axis (min, max, and inverse bool)
     * @param {Object} options - uiOptions
     * @param {bool} reverse - boolean of whether or not to reverse y-axis
     * @param {Array} yLabel - default label for y axis
     * @param {Object} groupBy - object of groupBy info
     * @param {num} idx - groupBy instance index
     * @param {Object} extremes - max and min of group by all instances
     * @returns {Array} - array of object of y-axis configuration
     */
    function setYaxis(options, reverse, yLabel, groupBy, idx, extremes) {
        var yAxis,
            axisTitle,
            showAxisValues,
            showAxisLine,
            showAxisTicks,
            yMin = null,
            yMax = null,
            nameGap = 25,
            grid = options.editGrid.y,
            flipAxis = options.rotateAxis,
            fontSize = options.fontSize ? options.fontSize.substring(0, options.fontSize.indexOf('p')) : 12,
            fontColor = options.fontColor || 'black';

        if (groupBy && groupBy.viewType === 'All Instances' && !options.toggleStack) {
            yMax = Math.ceil(extremes.max);
            if (extremes.min < 0) {
                yMin = Math.floor(extremes.min);
            }
        }

        if (options.editYAxis) {
            if (options.editYAxis.title.show) {
                axisTitle = options.editYAxis.title.name;
            } else {
                axisTitle = '';
            }
            nameGap = options.editYAxis.nameGap;
            showAxisValues = options.editYAxis.values;
            showAxisLine = showAxisTicks = options.editYAxis.line;
            if (options.editYAxis.min) {
                if (options.editYAxis.min.show) {
                    yMin = options.editYAxis.min.value;
                }
            }
            if (options.editYAxis.max) {
                if (options.editYAxis.max.show) {
                    yMax = options.editYAxis.max.value;
                }
            }
        } else {
            axisTitle = yLabel;
            showAxisValues = true;
            showAxisLine = showAxisTicks = true;
        }

        yAxis = [{
            min: yMin || null,
            max: yMax || null,
            name: axisTitle,
            nameTextStyle: {
                fontWeight: 'bold',
                fontSize: fontSize,
                color: fontColor
            },
            nameGap: nameGap,
            inverse: reverse,
            gridIndex: idx,
            type: 'value',
            splitLine: {
                show: grid
            },
            axisLabel: {
                show: showAxisValues,
                formatter: function (value) {
                    return EchartsHelper.formatLabel(value, options.editYAxis.format);
                },
                fontSize: fontSize,
                color: fontColor,
                rotate: options.editYAxis.rotate || 0
            },
            axisLine: {
                show: showAxisLine
            },
            axisTick: {
                show: showAxisTicks
            }
        }];

        if (flipAxis) {
            yAxis[0].nameLocation = 'center';
            yAxis[0].nameGap = 25;
        } else if (reverse) {
            yAxis[0].nameLocation = 'start';
        } else {
            yAxis[0].nameLocation = 'end';
        }

        if (axisTitle && !flipAxis) {
            if (axisTitle.length > 10 && axisTitle.length < 15) {
                yAxis[0].nameTextStyle.padding = [0, 0, 0, ((axisTitle.length * 10) / 4)];
            } else if (axisTitle.length >= 15) {
                yAxis[0].nameTextStyle.padding = [0, 0, 0, ((axisTitle.length * 10) / 2)];
            }
        }

        return yAxis;
    }

    /**
    * @name findMaxMin
    * @desc if Group By All Instances exists, find the max and min value for the y-axis
    * @param {Array} data - semoss data
    * @param {Object} groupBy - group by data

    * @returns {obj} max and min values
    */
    function findMaxMin(data, groupBy) {
        var extremes = {},
            dataArray = [],
            i, n;

        if (groupBy && groupBy.viewType === 'All Instances') {
            for (i = 0; i < data.values.length; i++) {
                for (n = 0; n < valuesMapping.mappingByModel.value.length; n++) {
                    dataArray.push(data.values[i][valuesMapping.mappingByModel.value[n]]);
                }
            }
            extremes.min = Math.min.apply(null, dataArray);
            extremes.max = Math.max.apply(null, dataArray);

            if (extremes.min > 0) {
                extremes.min = 0;
            }
        }
        return extremes;
    }

    /**
     * @name customizeAnimation
     * @desc sets the animation style
     * @param {object} param - object of animation settings
     * @returns {object} - object of animation settings
     */
    function customizeAnimation(param) {
        var animationSettings = {};
        // TODO small refactor
        if (param && param.chooseType === 'No Animation') {
            animationSettings.showAnimation = false;
        } else if (param) {
            animationSettings.showAnimation = true;
            animationSettings.defaultAnimationType = param.chooseType;
            animationSettings.defaultAnimationSpeed = param.animationSpeed;
            animationSettings.defaultAnimationDuration = param.animationDuration;
        } else {
            // default
            animationSettings.showAnimation = true;
            // TODO set as same as widget service default state
            animationSettings.defaultAnimationType = 'No Animation';
            animationSettings.defaultAnimationSpeed = 1;
            animationSettings.defaultAnimationDuration = 500;
        }
        return animationSettings;
    }

    /**
     * @name setEChartsDataSeries
     * @desc creates data series object to define ECharts viz
     * @param {string} type - chart type
     * @param {object} data - object of data in ECharts format created from formData function
     * @param {object} options - ui options
     * @param {object} colorBy - user-defined Color By Value rule(s)
     * @param {num} idx - groupBy instance index
     * @returns {Object} - object of ECharts data series
     */
    function setEChartsDataSeries(type, data, options, colorBy, idx) {
        var dataObject = {},
            seriesArray = [],
            i, j,
            markLineIdx = 0,
            markAreaIdx = 0,
            barSpecificConfig = barSpecific(type, data, options, idx);

        for (i = 0; i < data.valuesNames.length; i++) {
            dataObject.name = data.valuesNames[i];
            dataObject.type = type;
            dataObject.itemStyle = {
                normal: {}
            };
            if (options.barImage) {
                if (options.barImage.hasOwnProperty('borderRadius')) {
                    dataObject.itemStyle.normal.barBorderRadius = options.barImage.borderRadius || 0;
                }

                if (options.barImage.hasOwnProperty('symbol') && options.barImage.symbol !== 'Default (Bar)') {
                    dataObject.type = 'pictorialBar';
                    dataObject.symbolRepeat = options.barImage.repeat;
                    dataObject.symbolSize = options.barImage.symbolSize + '%';
                    dataObject.symbolClip = options.barImage.clip;
                }
            }
            dataObject.data = configureDataValue(data, options, i, colorBy, options.barImage, dataObject.name);
            Object.assign(dataObject, barSpecificConfig.barSettings);

            dataObject.xAxisIndex = idx;
            dataObject.yAxisIndex = idx;
            // Animation
            dataObject.animation = false;

            // Toggle Shadow
            if (options.toggleShadow === true) {
                dataObject.itemStyle.normal.shadowBlur = 50;
                dataObject.itemStyle.normal.shadowColor = 'rgba(0, 0, 0, 0.5)';
            }

            seriesArray.push(dataObject);
            dataObject = {};
        }

        for (i = 0; i < data.lineNames.length; i++) {
            dataObject = {
                name: data.lineNames[i],
                type: 'line',
                data: data.lineData[i],
                xAxisIndex: idx,
                yAxisIndex: idx
            };
            seriesArray.push(dataObject);
        }

        // Adds Trendline, and Threshold Line for Bar, Line, and Area
        for (j = 0; j < barSpecificConfig.additionalDataObjects.length; j++) {
            seriesArray.push(barSpecificConfig.additionalDataObjects[j]);
        }

        if (options.markLine) {
            markLineIdx = seriesArray.length;
            seriesArray[markLineIdx] = {
                type: 'line',
                name: 'markLine',
                xAxisIndex: idx,
                yAxisIndex: idx,
                data: [],
                markLine: {
                    data: []
                }
            };
            for (i = 0; i < options.markLine.lines.length; i++) {
                seriesArray[markLineIdx].markLine.data.push({
                    symbol: 'circle',
                    name: options.markLine.lines[i].label.name,
                    lineStyle: {
                        color: options.markLine.lines[i].color,
                        type: options.markLine.lines[i].style
                    },
                    label: {
                        show: options.markLine.lines[i].label.show,
                        position: 'end',
                        formatter: function (info) {
                            return info.name;
                        }
                    },
                    [options.rotateAxis ? 'xAxis' : 'yAxis']: options.markLine.lines[i].value
                });
            }
        }

        if (options.markArea) {
            markAreaIdx = seriesArray.length;
            seriesArray[markAreaIdx] = {
                type: 'line',
                name: 'markArea',
                xAxisIndex: idx,
                yAxisIndex: idx,
                data: [],
                markArea: {
                    data: []
                }
            };
            for (i = 0; i < options.markArea.areas.length; i++) {
                seriesArray[markAreaIdx].markArea.data.push([{
                    [options.rotateAxis ? 'xAxis' : 'yAxis']: options.markArea.areas[i].y.start,
                    itemStyle: {
                        color: options.markArea.areas[i].color,
                        opacity: options.markArea.areas[i].opacity
                    },
                    name: options.markArea.areas[i].label.name,
                    label: {
                        show: options.markArea.areas[i].label.show,
                        color: options.markArea.areas[i].color
                    }
                },
                {
                    [options.rotateAxis ? 'xAxis' : 'yAxis']: options.markArea.areas[i].y.end,
                    itemStyle: {
                        color: options.markArea.areas[i].color,
                        opacity: options.markArea.areas[i].opacity
                    }
                }
                ]);
            }
        }

        return seriesArray;
    }

    /**
     * @name configureDataValue
     * @param {object} data echarts data
     * @param {object} options Semoss ui options
     * @param {number} i index of for loop from parent function
     * @param {number} colorBy index of the data to highlight
     * @param {object} barImage uiOptions.barImage
     * @desc builds the data values objects for series
     * @return {array} array of data values objects
     */
    function configureDataValue(data, options, i, colorBy, barImage, seriesName) {
        return data.valuesData[i].map(function (value, idx) {
            var valueObj = {},
                prop, j, item, labelIndex;

            valueObj.value = value;

            if (i === 0 && valuesMapping.mappingByModel.tooltip.length > 0) {
                valueObj.tooltip = [];
                for (j = 0; j < data.tooltipData[idx].length; j++) {
                    valueObj.tooltip.push({
                        'header': data.tooltipHeaders[j],
                        'value': data.tooltipData[idx][j]
                    });
                }
            }

            if (options.highlight) {
                // check all properties in our highlight data
                for (prop in options.highlight.data) {
                    if (options.highlight.data.hasOwnProperty(prop)) {
                        // if x-axis label is equal to the property we are coloring
                        if (data.labelName === prop) {
                            options.highlight.data[prop].forEach(function (hiliteValue) {
                                if (data.labelData.indexOf(hiliteValue) === idx) {
                                    valueObj.itemStyle = {
                                        normal: {
                                            borderColor: '#000',
                                            borderWidth: 2
                                        }
                                    };
                                }
                            });
                        }
                    }
                }
            }

            if (options.label) {
                for (item in options.label) {
                    if (options.label.hasOwnProperty(item)) {
                        if (data.labelName === item) {
                            options.label[item].forEach(function (labelValue) {
                                if (data.labelData.indexOf(labelValue) === idx) {
                                    valueObj.label = {
                                        show: true,
                                        formatter: function (obj) {
                                            return cleanValue(obj.value);
                                        }
                                    };
                                }
                            });
                        }
                    }
                }
            }

            // Color by Value
            if (colorBy && colorBy.length > 0) {
                data.labelData = data.labelData.map(function (label) {
                    return label;
                });

                colorBy.forEach(function (rule) {
                    if (data.labelName === rule.colorOn) {
                        rule.valuesToColor.forEach(function (name) {
                            var cleanName = name;
                            labelIndex = data.labelData.indexOf(cleanName);
                            if (idx === labelIndex) {
                                if (
                                    (data.valuesNames.length > 1 && rule.valuesColumn === seriesName) ||
                                    data.valuesNames.length === 1 ||
                                    rule.colorOn === rule.valuesColumn
                                ) {
                                    if (valueObj.hasOwnProperty('itemStyle') && valueObj.itemStyle.hasOwnProperty('normal')) {
                                        valueObj.itemStyle.normal.color = rule.color;
                                    } else {
                                        valueObj.itemStyle = {
                                            normal: {
                                                color: rule.color
                                            }
                                        };
                                    }
                                }
                            }
                        });
                    }
                });
            }

            if (barImage.symbol && barImage.symbol !== 'Default (Bar)') {
                valueObj.symbol = barImage.symbol;
            }

            return valueObj;
        });
    }

    /**
     * @name barSpecific
     * @desc defines bar-specific settings for chart (also used for line and area)
     * @param {object} type - chart type (bar or line)
     * @param {object} data - object of data in ECharts format created from formData function
     * @param {object} options - uiOptions
     * @param {num} idx - groupBy instance index
     * @returns {object} object of bar settings
     */
    function barSpecific(type, data, options, idx) {
        var barSettings = {},
            newDataObject = {},
            dataArray = [],
            i;
        if (type === 'bar') {
            barSettings.barGap = '5%';
        }

        // Toggle Stack
        if (options.toggleStack) {
            barSettings.stack = 'toggleStack' + idx;
        }

        // TODO think about abstracting the colors #0000F and rgb out to the top of the file for easier customizability
        // Toggle Extremes
        if (options.toggleExtremes === true) {
            barSettings.markPoint = {
                silent: true,
                data: [{
                    type: 'max',
                    name: 'Max',
                    label: {
                        normal: {
                            color: '#00000F'
                        }
                    }
                },
                {
                    type: 'min',
                    name: 'Min',
                    label: {
                        normal: {
                            color: '#00000F'
                        }
                    }
                }
                ],
                label: {
                    formatter: function (info) {
                        return cleanValue(info.value);
                    }
                }
            };
        }

        // Customize Bar Label
        barSettings.label = {
            normal: {
                show: options.displayValues,
                color: options.customizeBarLabel.fontColor || 'black',
                formatter: function (obj) {
                    return cleanValue(obj.value);
                },
                position: options.customizeBarLabel.position || 'top',
                rotate: options.customizeBarLabel.rotate || 0,
                align: options.customizeBarLabel.align || 'center',
                verticalAlign: 'middle',
                fontFamily: options.customizeBarLabel.fontFamily || 'sans-serif',
                fontSize: options.customizeBarLabel.fontSize || 12,
                fontWeight: options.customizeBarLabel.fontWeight || 'normal'
            }
        };

        // Toggle Average
        if (options.toggleAverage) {
            barSettings.markLine = {
                silent: true,
                data: [{
                    type: 'average',
                    name: 'Average',
                    label: {
                        normal: {
                            show: true,
                            position: 'end'
                        }
                    }
                }],
                label: {
                    formatter: function (info) {
                        return cleanValue(info.value);
                    }
                }
            };
        }

        // Toggle Trendline
        if (options.toggleTrendline && options.toggleTrendline !== 'No Trendline') {
            for (i = 0; i < data.valuesNames.length; i++) {
                newDataObject.name = data.valuesNames[i];
                newDataObject.type = 'line';
                newDataObject.data = data.valuesData[i];
                newDataObject.xAxisIndex = idx;
                newDataObject.yAxisIndex = idx;
                if (options.toggleTrendline === 'Smooth') {
                    newDataObject.smooth = true;
                }
                if (options.toggleTrendline === 'Exact') {
                    newDataObject.smooth = false;
                }
                if (options.toggleTrendline === 'Step (start)') {
                    newDataObject.step = 'start';
                }
                if (options.toggleTrendline === 'Step (middle)') {
                    newDataObject.step = 'middle';
                }
                if (options.toggleTrendline === 'Step (end)') {
                    newDataObject.step = 'end';
                }

                newDataObject.lineStyle = {
                    normal: {
                        color: '#000000',
                        opacity: '0.3'
                    }
                };
                dataArray.push(newDataObject);
                newDataObject = {};
            }
        }

        // Threshold Line
        // if (options.thresholds !== 'none') {
        //     for (i = 0; i < Object.keys(options.thresholds).length; i++) {
        //         newDataObject.type = 'line';
        //         newDataObject.markLine = {
        //             label: {
        //                 normal: {
        //                     show: true,
        //                     position: 'middle',
        //                     formatter: options.thresholds[i].thresholdName
        //                 }
        //             },
        //             lineStyle: {
        //                 normal: {
        //                     color: options.thresholds[i].thresholdColor,
        //                     width: '2',
        //                     type: 'dashed'
        //                 }
        //             },
        //             data: [
        //                 {
        //                     yAxis: options.thresholds[i].thresholdPretty
        //                 }
        //             ]
        //         };

        //         dataArray.push(newDataObject);
        //         newDataObject = {};
        //     }
        // }

        return {
            barSettings: barSettings,
            additionalDataObjects: dataArray
        };
    }

    function cleanValue(item) {
        if (typeof item === 'string') {
            return item.replace(/_/g, ' ');
        } else if (typeof item === 'number') {
            return item.toLocaleString(undefined, {
                minimumFractionDigits: 0,
                maximumFractionDigits: 3
            });
        }
        return item;
    }

    /**
     * @name customizeBarWidth
     * @desc sets the bar with
     * @param {object} param - object of whether or not to use default width (yes or no) and bar width
     * @param {integer} numberOfDataSeries The number of data series on our bar chart
     * @returns {string} customized bar width
     */
    function customizeBarWidth(param, numberOfDataSeries) {
        if (numberOfDataSeries === 0) {
            return 'null';
        }

        var maxBarWidth = ((100 / numberOfDataSeries) - Math.min(1, (numberOfDataSeries - 1))).toFixed(2),
            barWidth;

        if (param && param.defaultWidth !== 'Yes' && Math.abs(param.barWidth) <= maxBarWidth) {
            barWidth = param.barWidth + '%';
        } else if (param && param.defaultWidth !== 'Yes' && Math.abs(param.barWidth) > maxBarWidth) {
            barWidth = maxBarWidth + '%';

            // TODO: Make the alert bound to a widget
            semossCoreService.emit('alert', {
                color: 'warn',
                text: 'Bar width currently cannot be greater than ' + maxBarWidth + '%! Setting bar width to ' + maxBarWidth + '%.'
            });
        } else {
            barWidth = 'null';
        }

        return barWidth;
    }

    /**
     * @name getBackgroundColorStyle
     * @desc customize the background style of the canvas
     * @param {string} watermark - string of the watermark text
     * @returns {Object} - canvas details
     */
    function getBackgroundColorStyle(watermark) {
        if (/\S/.test(watermark)) {
            return {
                type: 'pattern',
                image: paintWaterMark(watermark),
                repeat: 'repeat'
            };
        }

        return false;
    }

    /**
     * @name paintWaterMark
     * @desc paints a custom watermark on the viz
     * @param {string} watermark - string of the watermark text
     * @returns {Object} - canvas details
     */
    function paintWaterMark(watermark) {
        var canvas = document.createElement('canvas'),
            ctx = canvas.getContext('2d');

        canvas.width = canvas.height = 100;
        ctx.textAlign = 'center';
        ctx.textBaseline = 'middle';
        ctx.globalAlpha = 0.08;
        ctx.font = '20px Microsoft Yahei';
        ctx.translate(50, 50);
        ctx.rotate(-Math.PI / 4);
        if (watermark) {
            ctx.fillText(watermark, 0, 0);
        }

        return canvas;
    }

    return {
        getConfig: getConfig
    };
}
