'use strict';

import './add.scss';

import jsep from 'jsep';

/**
 * @name add
 * @desc add tabular data - accessible through the left panel
 */
export default angular.module('app.add.directive', [])
    .directive('add', addDirective);

addDirective.$inject = ['semossCoreService'];

function addDirective(semossCoreService) {
    addCtrl.$inject = [];
    addLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'EA',
        scope: {},
        require: ['^widget', '?^pipelineComponent'],
        controllerAs: 'add',
        bindToController: {},
        template: require('./add.directive.html'),
        controller: addCtrl,
        link: addLink
    };

    function addCtrl() {}

    function addLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];
        scope.pipelineComponentCtrl = ctrl[1];

        scope.add.PIPELINE = scope.pipelineComponentCtrl !== null;

        var queryElement,
            keys,
            updateSelectedListener,
            updateTaskListener;

        scope.add.name = '';
        scope.add.query = '';
        scope.add.temporalToggle = true;

        scope.add.validateName = validateName;
        scope.add.canSubmit = canSubmit;
        scope.add.submitFormula = submitFormula;
        scope.add.cancel = cancel;
        scope.add.loadPreview = loadPreview;

        /**
         * @name getFrame
         * @param {string} accessor - how do we want to access the frame?
         * @returns {*} frame options
         */
        function getFrame(accessor) {
            if (scope.add.PIPELINE) {
                return scope.pipelineComponentCtrl.getComponent(accessor ? ('parameters.SOURCE.value.' + accessor) : 'parameters.SOURCE.value');
            }

            return scope.widgetCtrl.getFrame(accessor);
        }
        /**
         * @name validateName
         * @desc validate the name
         * @returns {void}
         */
        function validateName() {
            var cleanedName = String(scope.add.name).replace(/ /g, '_'),
                i,
                len;

            for (i = 0, len = keys.length; i < len; i++) {
                if (cleanedName === keys[i].alias) {
                    // is it alraedy there?
                    scope.widgetCtrl.alert('warn', 'Name cannot exist in the Grid');
                }
            }
        }


        /**
         * @name parseFormula
         * @param {object} leaf - leaf of the parse tree
         * @param {array} headers - array of selected headers (from the tree)
         * @desc updates the formula value
         * @returns {object} cleaned tree
         */
        function parseFormula(leaf, headers) {
            var temp,
                temp2,
                i,
                len,
                j,
                len2;

            if (!leaf) {
                return {
                    headers: headers,
                    formula: ''
                };
            }

            if (leaf.body) {
                throw new Error('Cannot parse compound expressions');
            }


            if (leaf.type === 'BinaryExpression') {
                temp = parseFormula(leaf.left, headers);
                temp2 = parseFormula(leaf.right, headers);

                return {
                    headers: headers,
                    formula: '(' + temp.formula + ' ' + leaf.operator + ' ' + temp2.formula + ')'
                };
            } else if (leaf.type === 'UnaryExpression') {
                temp = parseFormula(leaf.argument, headers);
                return {
                    headers: headers,
                    formula: '(' + leaf.operator + ' ' + temp.formula + ')'
                };
            } else if (leaf.type === 'Literal') {
                return {
                    headers: headers,
                    formula: leaf.raw
                };
            } else if (leaf.type === 'Identifier') {
                for (i = 0, len = keys.length; i < len; i++) {
                    if (leaf.name === keys[i].alias) {
                        // is it alraedy there?
                        for (j = 0, len2 = headers.length; j < len2; j++) {
                            if (headers[j].alias === keys[i].alias) {
                                return {
                                    headers: headers,
                                    formula: keys[i].alias
                                };
                            }
                        }

                        headers.push(keys[i]);

                        return {
                            headers: headers,
                            formula: keys[i].alias
                        };
                    }
                }

                throw new Error('Formula is invalid. Check Header');
            } else if (leaf.type === 'CallExpression') {
                temp2 = '';
                len = leaf.arguments.length;
                for (i = 0; i < len; i++) {
                    temp = parseFormula(leaf.arguments[i], headers);
                    temp2 += temp.formula + ', ';
                }

                if (len > 0) {
                    temp2 = temp2.replace(/,(\s+)?$/, '');
                }

                temp2 = leaf.callee.name + '(' + temp2 + ')';

                return {
                    headers: headers,
                    formula: temp2
                };
            }

            return {
                headers: headers,
                formula: ''
            };
        }

        /**
         * @name canSubmit
         * @desc validate the name
         * @returns {boolean} is the query valid
         */
        function canSubmit() {
            var cleanedName,
                i,
                len;

            if (!scope.add.query) {
                return false;
            }

            if (!scope.add.name) {
                return false;
            }

            cleanedName = String(scope.add.name).replace(/ /g, '_');

            for (i = 0, len = keys.length; i < len; i++) {
                if (cleanedName === keys[i].alias) {
                    return false;
                }
            }

            return true;
        }

        /**
         * @name submitFormula
         * @param {boolean} preview if true, load preview for pipeline
         * @desc submits the formula
         * @returns {void}
         */
        function submitFormula(preview) {
            var parseTree,
                cleanedTree,
                pixelComponents = [],
                formulaSelectComponent = [],
                formulaJoinComponent = [],
                cleanedName = String(scope.add.name).replace(/ /g, '_'),
                headerIdx,
                headerLen,
                taskOptionsComponent,
                selectComponent,
                params,
                i,
                len,
                firstHeader;

            if (scope.add.PIPELINE) {
                firstHeader = getFrame('headers')[0];
            } else {
                firstHeader = scope.widgetCtrl.getWidget('view.' + scope.widgetCtrl.getWidget('active') + '.keys.Grid')[0];
            }

            // Remove Equals sign if it exists as first character
            if (scope.add.query.charAt(0) === '=') {
                scope.add.query = scope.add.query.substr(1);
            }

            // try to make the tree
            try {
                parseTree = jsep(scope.add.query);
            } catch (err) {
                scope.widgetCtrl.alert('warn', err.message);
                return;
            }

            // try to get the formula
            try {
                cleanedTree = parseFormula(parseTree, []);
            } catch (err) {
                scope.widgetCtrl.alert('warn', err.message);
                return;
            }

            // create components to add
            formulaSelectComponent.push({
                selector: cleanedTree.formula,
                alias: cleanedName
            });

            if (cleanedTree.headers) {
                if (cleanedTree.headers.length === 0) {
                    cleanedTree.headers.push(firstHeader);
                }

                headerLen = cleanedTree.headers.length;
                for (headerIdx = 0; headerIdx < headerLen; headerIdx++) {
                    formulaSelectComponent.push(cleanedTree.headers[headerIdx]);

                    formulaJoinComponent.push({
                        'fromColumn': cleanedTree.headers[headerIdx].alias,
                        'joinType': 'inner',
                        'toColumn': cleanedTree.headers[headerIdx].alias
                    });
                }
            }

            pixelComponents.push({
                type: 'frame',
                components: [
                    getFrame('name')
                ]
            }, {
                'type': 'select2',
                'components': [
                    formulaSelectComponent
                ]
            });

            if (headerLen > 0) {
                pixelComponents.push({
                    'type': 'merge',
                    'components': [
                        formulaJoinComponent,
                        getFrame('name')
                    ],
                    'terminal': true
                });
            }

            if (scope.add.PIPELINE) {
                params = buildParams(
                    [{
                        type: 'select2',
                        components: [formulaSelectComponent]
                    },
                    {
                        type: 'merge',
                        components: [
                            formulaJoinComponent,
                            getFrame('name')
                        ],
                        terminal: true
                    }
                    ]
                );

                if (preview) {
                    scope.pipelineComponentCtrl.previewComponent(params);
                } else {
                    scope.pipelineComponentCtrl.executeComponent(params, {});
                }

                return;
            }

            taskOptionsComponent = {};
            taskOptionsComponent[scope.widgetCtrl.panelId] = {
                'layout': 'Grid',
                'alignment': {
                    'label': []
                }
            };

            selectComponent = [];

            for (i = 0, len = keys.length; i < len; i++) {
                // add in the select component if not already added
                selectComponent.push({
                    alias: keys[i].alias
                });

                // add to the view component
                taskOptionsComponent[scope.widgetCtrl.panelId].alignment.label.push(keys[i].alias);
            }

            // add the new one in
            selectComponent.push({
                alias: cleanedName
            });

            taskOptionsComponent[scope.widgetCtrl.panelId].alignment.label.push(cleanedName);

            pixelComponents.push({
                type: 'frame',
                components: [
                    getFrame('name')
                ]
            }, {
                type: 'select2',
                components: [
                    selectComponent
                ]
            }, {
                type: 'with',
                components: [
                    scope.widgetCtrl.panelId
                ]
            }, {
                type: 'format',
                components: ['table']
            }, {
                type: 'taskOptions',
                components: [
                    taskOptionsComponent
                ]
            }, {
                type: 'collect',
                components: [scope.widgetCtrl.getOptions('limit')],
                terminal: true
            });

            scope.widgetCtrl.execute(pixelComponents, function (response) {
                var outputIdx,
                    hasErrors = false;

                for (outputIdx = 0; outputIdx < response.pixelReturn.length; outputIdx++) {
                    if (response.pixelReturn[outputIdx].operationType.indexOf('ERROR') > -1) {
                        hasErrors = true;
                        break;
                    }
                }

                if (hasErrors) {
                    // already handled in store service
                    // scope.widgetCtrl.alert('error', 'An error occurred while running the forumla.');
                } else {
                    scope.widgetCtrl.alert('success', 'Successfully added ' + cleanedName + '.');
                }
            });
        }


        /**
         * @name setData
         * @desc set the data
         * @return {void}
         */
        function setData() {
            var active = active = scope.widgetCtrl.getWidget('active'),
                newKeys = scope.widgetCtrl.getWidget('view.' + active + '.keys.Grid');

            if (scope.add.PIPELINE) {
                newKeys = scope.pipelineComponentCtrl.getComponent('parameters.SOURCE.value.headers');
            }
            if (keys !== newKeys.length) {
                scope.add.name = 'Col' + newKeys.length;
            }
            scope.add.query = '';

            keys = newKeys;
        }

        /**
         * @name buildParams
         * @param {array} components - pixel components to generate formula
         * @desc builds params for pipeline
         * @return {object} the params and their values
         */
        function buildParams(components) {
            var params = {
                SOURCE: {
                    name: scope.pipelineComponentCtrl.getComponent('parameters.SOURCE.value.name')
                }
            };

            if (canSubmit()) {
                params.QUERY = semossCoreService.pixel.build(components).slice(0, -1);
            }

            return params;
        }

        /**
         * @name cancel
         * @desc closes pipeline component
         * @return {void}
         */
        function cancel() {
            scope.pipelineComponentCtrl.closeComponent();
        }

        /**
         * @name loadPreview
         * @desc loads preview
         * @return {void}
         */
        function loadPreview() {
            if (canSubmit()) {
                submitFormula('preview');
            } else {
                scope.pipelineComponentCtrl.previewComponent(buildParams());
            }
        }

        /**
         * @name initialize
         * @desc initializes the module
         * @return {void}
         */
        function initialize() {
            var srcComponent;
            if (scope.add.PIPELINE) {
                srcComponent = scope.pipelineComponentCtrl.getComponent('parameters.SOURCE.value');
                if (!srcComponent) {
                    scope.pipelineComponentCtrl.closeComponent();
                    return;
                }

                loadPreview();
            }
            queryElement = ele[0].querySelector('#add__query');

            // add listeners
            updateSelectedListener = scope.widgetCtrl.on('update-selected', function () {
                var selected = scope.widgetCtrl.getSelected('selected');
                if (selected.length > 0) {
                    scope.add.query += selected[0].alias;
                    queryElement.focus();
                }
            });

            updateTaskListener = scope.widgetCtrl.on('update-task', setData);

            // when directive ends, make sure to clean out excess listeners and dom elements outside of the scope
            scope.$on('$destroy', function () {
                console.log('destroying formula....');
                updateSelectedListener();
                updateTaskListener();
            });

            setData();

            queryElement.focus();
        }

        initialize();
    }
}
