function Utility() {
    let API = {
        freeze: freeze,
        convert: convert,
        filter: filter,
        isEqual: isEqual,
        isEmpty: isEmpty,
        isDefined: isDefined,
        isDate: isDate,
        isLeapYear: isLeapYear,
        toDisplay: toDisplay,
        toValue: toValue,
        toSearch: toSearch,
        toDate: toDate,
        toDateString: toDateString,
        constants: {
            month: [{
                name: 'January',
                abbreviation: 'Jan',
                length: 31
            },
            {
                name: 'February',
                abbreviation: 'Feb',
                length: 28
            },
            {
                name: 'March',
                abbreviation: 'Mar',
                length: 31
            },
            {
                name: 'April',
                abbreviation: 'Apr',
                length: 30
            },
            {
                name: 'May',
                abbreviation: 'May',
                length: 31
            },
            {
                name: 'June',
                abbreviation: 'Jun',
                length: 30
            },
            {
                name: 'July',
                abbreviation: 'Jul',
                length: 31
            },
            {
                name: 'August',
                abbreviation: 'Aug',
                length: 31
            },
            {
                name: 'September',
                abbreviation: 'Sep',
                length: 30
            },
            {
                name: 'October',
                abbreviation: 'Oct',
                length: 31
            },
            {
                name: 'November',
                abbreviation: 'Nov',
                length: 30
            },
            {
                name: 'December',
                abbreviation: 'Dec',
                length: 31
            }
            ],
            week: [{
                name: 'Sunday',
                abbreviation: 'Su'
            },
            {
                name: 'Monday',
                abbreviation: 'Mo'
            },
            {
                name: 'Tuesday',
                abbreviation: 'Tu'
            },
            {
                name: 'Wednesday',
                abbreviation: 'We'
            },
            {
                name: 'Thursday',
                abbreviation: 'Th'
            },
            {
                name: 'Friday',
                abbreviation: 'Fr'
            },
            {
                name: 'Saturday',
                abbreviation: 'Sa'
            }
            ]
        }
    };

    /*** Utility */
    /**
     * @name freeze
     * @param {*} a  - frozen
     * @desc makes the passed in item immutable via stringification
     * @returns {*} frozen item
     */
    function freeze(a) {
        // non primative
        if (typeof a === 'object') {
            return JSON.parse(JSON.stringify(a));
        }

        // primative
        return a;
    }

    /**
     * @name convert
     * @param {*} raw - raw value
     * @param {*} model - model to grab the value of (in format of a.b.c)
     * @desc traverse based on the model and convert the value
     * @returns {*} converted value
     */
    function convert(raw, model) {
        let convertedRaw = freeze(raw),
            convertedModel = freeze(model); // needs to be immutable

        if (convertedModel && convertedRaw) {
            if (typeof convertedModel === 'string') {
                convertedModel = convertedModel.split('.'); //convert to array so we can traverse over the proerty
            }

            for (let aIdx = 0, aLen = convertedModel.length; aIdx < aLen; aIdx++) {
                if (!convertedRaw) {
                    break;
                }

                convertedRaw = convertedRaw[convertedModel[aIdx]];
            }
        }

        return convertedRaw;
    }

    /**
     * @name filter
     * @param {array} raw - array value
     * @param {string} term - term to compare it against
     * @param {*} model - model to grab the value of (in format of a.b.c)
     * @desc filter the raw values and traverse based on the term
     * @returns {*} converted value
     */
    function filter(raw, term, model) {
        if (typeof term !== 'string' || term.length === 0) {
            return freeze(raw);
        }

        const convertedTerm = toSearch(term);
        return raw.filter((val) => {
            let displayed = convert(val, model);

            // object check
            if (typeof displayed === 'object') {
                if (isDate(displayed)) {
                    displayed = displayed.toString();
                } else {
                    return false;
                }
            }

            displayed = toSearch(displayed);
            if (displayed.indexOf(convertedTerm) > -1) {
                return true;
            }

            return false;
        });
    }


    /** Validity */
    /**
     * @name isEqual
     * @param {*} a - a value to compare
     * @param {*} b - b value to compare
     * @desc check if two values are equal
     * @returns {boolean} - equality check
     */
    function isEqual(a, b) {
        // TODO: Put in a proper equality check
        return JSON.stringify(a) === JSON.stringify(b);
    }

    /**
     * @name isEmpty
     * @param {*} a  - checked for emptiness
     * @desc checks whether an item is empty or not
     * @returns {boolean} truthy value if the passed in item is empty or not
     */
    function isEmpty(a) {
        // check if null or undefined
        if (a === null || a === undefined) {
            return true;
        }

        // string check
        if (typeof a === 'string') {
            return !a.length;
        }

        // object check
        if (typeof a === 'object') {
            // array check
            if (Array.isArray(a)) {
                return !a.length;
            }

            // object check
            for (let i in a) {
                if (a.hasOwnProperty(i)) {
                    return false;
                }
            }

            return JSON.stringify(a) === JSON.stringify({});
        }

        return false;
    }

    /**
     * @name isDefined
     * @param {*} a  - checked if it is defined
     * @desc checks whether an item is defined
     * @returns {boolean} truthy value if the passed in item is defined
     */
    function isDefined(a) {
        return !(a === undefined);
    }

    /**
     * @name isDate
     * @param {Date} d  - date to check
     * @desc check if passed in value is a valid date
     * @returns {boolean} truthy value if the passed in item is a valid date object
     */
    function isDate(d) {
        return !(Object.prototype.toString.call(d) !== '[object Date]' || isNaN(d.getTime()));
    }

    /**
     * @name isLeapYear
     * @param {number} y  - year to check
     * @desc check if passed in value is a leap year
     * @returns {boolean} truthy value if the passed in item is a leap year
     */
    function isLeapYear(y) {
        return ((y % 4 === 0) && (y % 100 !== 0)) || (y % 400 === 0);
    }

    /*** Transformation */
    /**
     * @name toDisplay
     * @desc  format a string into a display value
     * @param {string} raw - raw value
     * @returns {date} return a cleaned display string
     */
    function toDisplay(raw) {
        if (raw === null) {
            return 'null';
        }

        return String(raw).replace(/_/g, ' ');
    }

    /**
     * @name toValue
     * @desc  format a string into a value value
     * @param {string} raw - raw value
     * @returns {date} return a cleaned value string
     */
    function toValue(raw) {
        if (raw === 'null') {
            return null;
        }

        return raw;
    }

    /**
     * @name toSearch
     * @desc  format a string into a search value
     * @param {string} raw - raw value
     * @returns {date} return a cleaned search string
     */
    function toSearch(raw) {
        return String(raw).replace(/ /g, '_').toUpperCase();
    }

    /**
     * @name toDate
     * @desc  format a string into a date
     * @param {string} raw - raw value
     * @param {*} format - format to convert into
     * @returns {date} return a date object
     */
    function toDate(raw, format) {
        let date = {
            year: undefined,
            month: undefined,
            day: undefined
        };

        if (raw && typeof raw === 'string') {
            let today = new Date(),
                tokens = {
                    year: {
                        'YYYY': (chunk) => {
                            let val = Number(chunk);

                            if (typeof val !== 'number') {
                                return undefined;
                            }

                            return val;
                        },
                        'YY': (chunk) => {
                            let val = Number(String(today.getFullYear()).substr(0, 2) + chunk);

                            if (typeof val !== 'number') {
                                return undefined;
                            }

                            return val;
                        }
                    },
                    month: {
                        'MMMM': (chunk) => {
                            for (let monthIdx = 0, monthLength = API.constants.month.length; monthIdx < monthLength; monthIdx++) {
                                if (API.constants.month[monthIdx].name === chunk) {
                                    return monthIdx;
                                }
                            }

                            return undefined;
                        },
                        'MMM': (chunk) => {
                            for (let monthIdx = 0, monthLength = API.constants.month.length; monthIdx < monthLength; monthIdx++) {
                                if (API.constants.month[monthIdx].abbreviation === chunk) {
                                    return monthIdx;
                                }
                            }

                            return undefined;
                        },
                        'MM': (chunk) => {
                            let val = Number(chunk);

                            if (typeof val !== 'number') {
                                return undefined;
                            }

                            return val - 1;
                        }
                    },
                    day: {
                        'DD': (chunk) => {
                            let val = Number(chunk);

                            if (typeof val !== 'number') {
                                return undefined;
                            }

                            return val;
                        }
                    }
                };

            for (let tokenGroup in tokens) {
                if (tokens.hasOwnProperty(tokenGroup)) {
                    for (let token in tokens[tokenGroup]) {
                        if (tokens[tokenGroup].hasOwnProperty(token)) {
                            let idx = format.indexOf(token);
                            if (idx !== -1) {
                                date[tokenGroup] = tokens[tokenGroup][token](raw.substring(idx, idx + token.length));
                                break;
                            }
                        }
                    }
                }
            }
        }

        if (date.year === undefined || date.month === undefined || date.day === undefined) {
            return undefined;
        }

        return new Date(date.year, date.month, date.day);
    }

    /**
     * @name getHrs
     * @desc  get string hours for 12 hr clocked
     * @param {date} raw - raw value
     * @returns {string} return a dateString
     */
    function getHrs(raw) {
        let hours = raw.getHours();

        if (hours > 12) {
            return String(hours - 12);
        } else if (hours === 0) {
            return '12';
        }
        return String(hours);
    }

    /**
     * @name toDateString
     * @desc  format a date into a dateString
     * @param {date} raw - raw value
     * @param {*} format - format to convert into
     * @returns {string} return a dateString
     */
    function toDateString(raw, format) {
        let tokens = {
                'YYYY': String(raw.getFullYear()),
                'yyyy': String(raw.getFullYear()),
                'yy': String(raw.getFullYear()).substr(-2),
                'MMMM': API.constants.month[raw.getMonth()].name,
                'MMMMM': API.constants.month[raw.getMonth()].name,
                'MMM': API.constants.month[raw.getMonth()].abbreviation,
                'MM': (raw.getMonth() >= 9 ? String(raw.getMonth() + 1) : ('0' + String(raw.getMonth() + 1))),
                'm': String(raw.getMonth() + 1),
                'M': String(raw.getMonth() + 1),
                'DDDD': API.constants.week[raw.getDay()].name,
                'EEEEE': API.constants.week[raw.getDay()].name,
                'DDD': API.constants.week[raw.getDay()].abbreviation,
                'EEE': API.constants.week[raw.getDay()].abbreviation,
                'DD': (raw.getDate() >= 10 ? String(raw.getDate()) : ('0' + String(raw.getDate()))),
                'dd': (raw.getDate() >= 10 ? String(raw.getDate()) : ('0' + String(raw.getDate()))),
                'd': String(raw.getDate()),
                'HH': (raw.getHours() >= 10 ? String(raw.getHours()) : ('0' + String(raw.getHours()))),
                'hh': getHrs(raw),
                'mm': (raw.getMinutes() >= 10 ? String(raw.getMinutes()) : ('0' + String(raw.getMinutes()))),
                'ss': (raw.getSeconds() >= 10 ? String(raw.getSeconds()) : ('0' + String(raw.getSeconds()))),
                //'SSSS': String(raw.getMilliseconds()),
                'a': (raw.getHours() >= 12 ? 'PM' : 'AM'),
                '-': '-',
                '/': '/',
                ' ': ' ',
                ',': ',',
                ':': ':'
            },
            remaining = format,
            formatted = '';


        while (remaining.length > 0) {
            //go reverse
            let char = remaining.length;
            for (; char >= 0; char--) {
                if (tokens.hasOwnProperty(remaining.substr(0, char))) {
                    break;
                }
            }

            if (char >= 0) {
                formatted += tokens[remaining.substr(0, char)];
                remaining = remaining.slice(char);
            } else {
                formatted += remaining.substr(0, 1);
                remaining = remaining.slice(1);
            }
        }

        return formatted;
    }

    return API;
}

export default Utility();
