import Utility from '../utility.js';

import './smss-multiselect.scss';

export default angular.module('smss-style.multiselect', []).directive('smssMultiselect', smssMultiselectDirective);

smssMultiselectDirective.$inject = ['$timeout', '$filter'];

function smssMultiselectDirective($timeout, $filter) {
    smssMultiselectCtrl.$inject = [];
    smssMultiselectLink.$inject = ['scope', 'ele', 'attrs'];

    return {
        restrict: 'E',
        template: require('./smss-multiselect.directive.html'),
        scope: {},
        bindToController: {
            model: '=',
            options: '=',
            disabled: '=?ngDisabled',
            display: '@?',
            value: '@?',
            change: '&?',
            scroll: '&?',
            loading: '=',
            filter: '=?',
            search: '&?',
            mouseover: '&?',
            mouseleave: '&?'
        },
        replace: true,
        controllerAs: 'smssMultiselect',
        controller: smssMultiselectCtrl,
        compile: smssMultiselectCompile
    };

    function smssMultiselectCompile(tElement, tAttributes) {
        var popoverEle = tElement[0].querySelector('smss-popover-content');
        // att the special attributes
        if (tAttributes.hasOwnProperty('body')) {
            popoverEle.setAttribute('body', !(tAttributes.body === 'false'));
        }

        return smssMultiselectLink;
    }

    function smssMultiselectCtrl() {}

    function smssMultiselectLink(scope, ele, attrs) {
        var multiselectTriggerEle,
            multiselectScrollEle,
            multiselectScrollScrollerEle,
            multiselectSearchEle,
            multiselectLineEle,
            optionHeight,
            filtered,
            currentScroll,
            previousScroll,
            searchTimeout;

        scope.smssMultiselect.nowrap = false;
        scope.smssMultiselect.showMultiselect = showMultiselect;
        scope.smssMultiselect.hideMultiselect = hideMultiselect;
        scope.smssMultiselect.searchMultiselect = searchMultiselect;
        scope.smssMultiselect.searchMultiselectKeydown = searchMultiselectKeydown;
        scope.smssMultiselect.keyupMultiselect = keyupMultiselect;
        scope.smssMultiselect.keydownMultiselect = keydownMultiselect;
        scope.smssMultiselect.changeMultiselect = changeMultiselect;
        scope.smssMultiselect.setView = setView;
        scope.smssMultiselect.isSelected = isSelected;
        scope.smssMultiselect.getValue = getValue;

        /**
         * @name renderMultiselectList
         * @desc called when the multiselect model changes
         * @returns {void}
         */
        function renderMultiselectList() {
            if (scope.smssMultiselect.open && multiselectScrollEle && multiselectScrollScrollerEle) {
                var scrollTop = multiselectScrollScrollerEle.scrollTop, //find position of scroll
                    height = multiselectScrollEle.offsetHeight,
                    start = Math.floor((scrollTop - height * 2) / optionHeight), //find start
                    end = Math.floor((scrollTop + height * 2) / optionHeight), //we will render extra
                    i,
                    len;

                //set listStyle
                scope.smssMultiselect.listStyle = {
                    'height': (filtered.length * optionHeight + 'px')
                };

                start = start > 0 ? start : 0;
                end = filtered.length > end ? end : filtered.length;

                //create painted
                scope.smssMultiselect.painted = filtered.slice(start, end);

                //push values down
                scope.smssMultiselect.styles = [];
                for (i = 0, len = scope.smssMultiselect.painted.length; i < len; i++) {
                    scope.smssMultiselect.styles.push({
                        'top': ((start + i) * optionHeight + 'px')
                    });
                }
            }
        }


        /**
         * @name showMultiselect
         * @desc multiselect is opened
         * @param {ele} contentEle - contentEle
         * @returns {void}
         */
        function showMultiselect(contentEle) {
            multiselectScrollEle = contentEle.querySelector('#smss-scroll');
            multiselectScrollScrollerEle = contentEle.querySelector('#smss-scroll-scroller');

            multiselectScrollScrollerEle.addEventListener('scroll', scrollMultiselect);

            renderMultiselectList();
            $timeout(function () {
                if (multiselectSearchEle) {
                    multiselectSearchEle.focus();
                }
            });
        }


        /**
         * @name hideMultiselect
         * @desc multiselect is closed
         * @returns {void}
         */
        function hideMultiselect() {
            if (multiselectSearchEle) {
                multiselectSearchEle.focus();
            }
        }

        /**
         * @name searchMultiselect
         * @desc called when a search is executed
         * @returns {void}
         */
        function searchMultiselect() {
            if (scope.smssMultiselect.search) {
                if (searchTimeout) {
                    $timeout.cancel(searchTimeout);
                }
                searchTimeout = $timeout(function () {
                    scope.smssMultiselect.search({
                        search: scope.smssMultiselect.searchTerm
                    });
                    $timeout.cancel(searchTimeout);
                }, 500);
            } else {
                filtered = Utility.filter(scope.smssMultiselect.options, scope.smssMultiselect.searchTerm, scope.smssMultiselect.display);
                renderMultiselectList();
            }
        }


        /**
         * @name searchMultiselectKeydown
         * @param {event} $event - DOM event
         * @desc menu navigation - handles keydown events for search element
         * @returns {void}
         */
        function searchMultiselectKeydown($event) {
            var selectedEle;
            if (!scope.smssMultiselect.open && $event.keyCode !== 13) {
                scope.smssMultiselect.open = true;
            }
            if ($event.keyCode === 40) { //down
                $event.preventDefault();

                if (multiselectScrollEle) {
                    selectedEle = multiselectScrollEle.querySelectorAll('.smss-multiselect__popover__list__option');
                    if (selectedEle[0]) {
                        selectedEle[0].focus();
                    }
                }
            } else if ($event.keyCode === 9) { // tab
                scope.smssMultiselect.open = false;
            } else if ($event.keyCode === 13 && $event.target.value.length > 0 && scope.smssMultiselect.model.indexOf($event.target.value) === -1) { // enter
                scope.smssMultiselect.model.push($event.target.value);
                scope.smssMultiselect.searchTerm = '';
            }
        }

        /**
         * @name keyupMultiselect
         * @param {event} $event - angular event
         * @param {*} opt - selected option
         * @desc key up event for select
         * @returns {void}
         */
        function keyupMultiselect($event, opt) {
            if ($event.keyCode === 13) { // enter
                changeMultiselect(opt);
            }
        }

        /**
         * @name keydownMultiselect
         * @param {event} $event - angular event
         * @desc key down event for navigation
         * @returns {void}
         */
        function keydownMultiselect($event) {
            var nextElement, prevElement;
            //only if there is a focused element
            if ($event.keyCode === 40) { //down
                $event.preventDefault();

                nextElement = $event.target.nextElementSibling;
                if (nextElement) {
                    nextElement.focus();
                }
            } else if ($event.keyCode === 38) { //up
                $event.preventDefault();

                prevElement = $event.target.previousElementSibling;
                if (prevElement) {
                    prevElement.focus();
                } else if (multiselectSearchEle) {
                    multiselectSearchEle.focus();
                }
            } else if ($event.keyCode === 13) { //enter
                $event.preventDefault();
            } else if ($event.keyCode === 9) { //tab
                scope.smssMultiselect.open = false;
            }
        }

        /**
         * @name changeMultiselect
         * @param {*} opt - selected option
         * @desc called when a new option is selected
         * @returns {void}
         */
        function changeMultiselect(opt) {
            var delta,
                value = getValue(opt);

            if (isSelected(value)) {
                delta = {
                    type: 'remove',
                    value: value
                };
                scope.smssMultiselect.model = scope.smssMultiselect.model.filter((x) => x !== value);
            } else {
                delta = {
                    type: 'add',
                    value: value
                };
                scope.smssMultiselect.model.push(value);
            }

            $timeout(function () {
                if (scope.smssMultiselect.change) {
                    scope.smssMultiselect.change({
                        model: scope.smssMultiselect.model,
                        delta: delta,
                        value: value
                    });
                }
                scope.smssMultiselect.open = false;
            });
            scope.smssMultiselect.searchTerm = '';
        }

        /**
         * @name clearMultiselect
         * @param {*} opt - selected option
         * @desc called when a new option is selected
         * @returns {void}
         */
        function clearMultiselect() {
            scope.smssMultiselect.model = [];

            $timeout(function () {
                if (scope.smssMultiselect.change) {
                    scope.smssMultiselect.change({
                        model: scope.smssMultiselect.model,
                        delta: {
                            type: 'replace',
                            value: []
                        },
                        value: []
                    });
                }
                scope.smssMultiselect.open = false;
            });

            scope.smssMultiselect.searchTerm = '';
        }

        /**
         * @name getValue
         * @desc called to get the actual model value
         * @param {*} opt - model backing this option
         * @returns {*} value
         */
        function getValue(opt) {
            var value = opt;
            if (scope.smssMultiselect.value && opt.hasOwnProperty(scope.smssMultiselect.value)) {
                value = Utility.convert(opt, scope.smssMultiselect.value);
            }
            return value;
        }
        /**
         * @name scroll
         * @desc called when the multiselect is scrolled
         * @returns {void}
         */
        function scrollMultiselect() {
            renderMultiselectList();

            if (scope.smssMultiselect.open && multiselectScrollEle && multiselectScrollScrollerEle) {
                //make sure scroll is reverse
                currentScroll = multiselectScrollScrollerEle.scrollTop + multiselectScrollScrollerEle.offsetHeight;
                if (currentScroll > parseInt(scope.smssMultiselect.listStyle.height, 10) * 0.75 && currentScroll > previousScroll) {
                    if (scope.smssMultiselect.scroll && !scope.smssMultiselect.loading) {
                        scope.smssMultiselect.scroll();
                    }
                }

                previousScroll = currentScroll;

                scope.$apply();
            }
        }

        function getFullObject(value) {
            var obj = scope.smssMultiselect.options.filter((x) => x[scope.smssMultiselect.value] === value)[0];
            if (obj) {
                return obj;
            }
            return value;
        }

        /**
         * @name setView
         * @desc called to set the view
         * @param {*} opt - model backing this option
         * @returns {string} view
         */
        function setView(opt) {
            var value = opt,
                view,
                obj = getFullObject(opt);
            if (scope.smssMultiselect.display) {
                if (obj.hasOwnProperty(scope.smssMultiselect.display)) {
                    value = Utility.convert(obj, scope.smssMultiselect.display);
                } else {
                    value = obj;
                }
            }
            view = Utility.toDisplay(value);

            if (scope.smssMultiselect.filter) {
                view = $filter(scope.smssMultiselect.filter)(view);
            }

            return view;
        }

        /**
         * @name isSelected
         * @param {*} opt - selected option
         * @desc called to set the view
         * @returns {void}
         */
        function isSelected(opt) {
            return scope.smssMultiselect.model.indexOf(opt) > -1;
        }

        /**
         * @name keyupMultiselectTrigger
         * @param {event} $event - DOM event
         * @desc key up event for the trigger
         * @returns {void}
         */
        function keyupMultiselectTrigger($event) {
            if ($event.keyCode === 27) { // esc
                clearMultiselect();
            }
        }

        /**
         * @name addFocus
         * @desc called to add focus class to the line
         * @returns {void}
         */
        function addFocus() {
            multiselectLineEle.classList.add('smss-multiselect__popover__line--focus');
        }

        /**
         * @name removeFocus
         * @desc called to remove focus class from the line
         * @returns {void}
         */
        function removeFocus() {
            multiselectLineEle.classList.remove('smss-multiselect__popover__line--focus');
        }
        /**
         * @name initialize
         * @desc Called when the directive is loaded
         * @returns {void}
         */
        function initialize() {
            scope.smssMultiselect.searchTerm = '';
            scope.smssMultiselect.painted = [];
            scope.smssMultiselect.styles = [];


            if (attrs.hasOwnProperty('display') && attrs.display) {
                scope.smssMultiselect.display = attrs.display.split('.');
            }

            if (attrs.hasOwnProperty('value') && attrs.value) {
                scope.smssMultiselect.value = attrs.value.split('.');
            }

            if (attrs.hasOwnProperty('nowrap')) {
                scope.smssMultiselect.nowrap = true;
            }

            if (attrs.hasOwnProperty('placeholder')) {
                scope.smssMultiselect.placeholder = attrs.placeholder;
            }

            if (!Array.isArray(scope.smssMultiselect.options)) {
                scope.smssMultiselect.options = [];
            }

            if (!Array.isArray(scope.smssMultiselect.model)) {
                scope.smssMultiselect.model = [];
            }


            // get eles
            multiselectTriggerEle = ele[0].querySelector('#smss-multiselect__trigger');
            multiselectLineEle = ele[0].querySelector('#smss-multiselect__line');
            multiselectSearchEle = ele[0].querySelector('#smss-multiselect__input');

            multiselectTriggerEle.addEventListener('keyup', keyupMultiselectTrigger);
            multiselectSearchEle.addEventListener('focus', addFocus);
            multiselectSearchEle.addEventListener('blur', removeFocus);

            let computedStyles = getComputedStyle(ele[0]);
            optionHeight = computedStyles.getPropertyValue('font-size').match(/\d+/g) * 2;

            filtered = Utility.freeze(scope.smssMultiselect.options);

            //set the view and update after the digest is complete
            $timeout(function () {
                if (!Array.isArray(scope.smssMultiselect.options)) {
                    scope.smssMultiselect.options = [];
                }
                filtered = Utility.freeze(scope.smssMultiselect.options);

                renderMultiselectList();

                scope.$watch('smssMultiselect.searchTerm', function (newValue, oldValue) {
                    if (!angular.equals(newValue, oldValue)) {
                        if (newValue.length > 0) {
                            var item = newValue;
                            if (oldValue.length > 0) {
                                scope.smssMultiselect.options.shift();
                            }
                            if (scope.smssMultiselect.display && scope.smssMultiselect.value) {
                                item = {
                                    [scope.smssMultiselect.display]: newValue,
                                    [scope.smssMultiselect.value]: newValue
                                };
                            }
                            scope.smssMultiselect.options.unshift(item);
                        } else {
                            scope.smssMultiselect.options.shift();
                        }
                        renderMultiselectList();
                    }
                });

                // Adding focus/blur listeners to each tag when it's added
                scope.$watch(function () {
                    return multiselectTriggerEle.querySelectorAll('.smss-tag').length;
                }, function (newValue, oldValue) {
                    if (!angular.equals(newValue, oldValue)) {
                        let newEle = multiselectTriggerEle.querySelectorAll('.smss-tag')[newValue - 1];
                        if (newEle) {
                            newEle.addEventListener('focus', addFocus);
                            newEle.addEventListener('blur', removeFocus);
                        }
                    }
                });

                scope.$watchCollection('smssMultiselect.options', function (newValue, oldValue) {
                    if (!angular.equals(newValue, oldValue)) {
                        // apply the search only if it's local search within the directive
                        if (!Array.isArray(scope.smssMultiselect.options)) {
                            scope.smssMultiselect.options = [];
                        }

                        filtered = Utility.freeze(scope.smssMultiselect.options);
                        if (scope.smssMultiselect.searchTerm && !scope.smssMultiselect.search) {
                            searchMultiselect();
                        } else {
                            renderMultiselectList();
                        }
                    }
                });

                scope.$watch(function () {
                    let style = getComputedStyle(ele[0]);
                    return style.getPropertyValue('font-size');
                }, function (newValue, oldValue) {
                    if (!angular.equals(newValue, oldValue)) {
                        optionHeight = newValue.match(/\d+/g) * 2;
                        renderMultiselectList();
                    }
                });

                scope.$on('$destroy', function () {
                    multiselectSearchEle.removeEventListener('focus', addFocus);
                    multiselectSearchEle.removeEventListener('blur', removeFocus);
                    var tags = multiselectTriggerEle.querySelectorAll('.smss-tag'),
                        i;
                    for (i = 0; i < tags.length; i++) {
                        tags[i].removeEventListener('focus', addFocus);
                        tags[i].removeEventListener('blur', removeFocus);
                    }
                });
            });
        }

        initialize();
    }
}
