import Utility from '../utility.js';

/**
 * @name smss-date-picker.directive.js
 * @desc smss-date-picker field
 */
export default angular.module('smss-style.date-picker', [])
    .directive('smssDatePicker', smssDatePicker);

import './smss-date-picker.scss';

smssDatePicker.$inject = ['$timeout'];

function smssDatePicker($timeout) {
    smssDatepickerCompile.$inject = ['tElement', 'tAttributes'];
    smssDatePickerLink.$inject = ['scope', 'ele', 'attrs'];

    return {
        restrict: 'E',
        template: require('./smss-date-picker.directive.html'),
        scope: {
            model: '=',
            format: '=?',
            disabled: '=?ngDisabled',
            change: '&'
        },
        transclude: true,
        replace: true,
        compile: smssDatepickerCompile
    };

    function smssDatepickerCompile(tElement, tAttributes) {
        var popoverEle = tElement[0].querySelector('smss-popover-content');
        // att the special attributes
        if (tAttributes.hasOwnProperty('body')) {
            popoverEle.setAttribute('body', !(tAttributes.body === 'false'));
        }

        return smssDatePickerLink;
    }

    function smssDatePickerLink(scope, ele, attrs) {
        var toggleEle;

        scope.day = undefined;
        scope.month = undefined;
        scope.year = undefined;
        scope.level = 'day';
        scope.title = '';
        scope.items = [];
        scope.week = Utility.constants.week;
        scope.renderedMonth = undefined;
        scope.renderedYear = undefined;

        scope.showPicker = showPicker;
        scope.hidePicker = hidePicker;
        scope.previousLevel = previousLevel;
        scope.exitLevel = exitLevel;
        scope.nextLevel = nextLevel;
        scope.enterLevel = enterLevel;

        /**Picker */
        /**
         * @name showPicker
         * @desc build (or rebuild) the picker whenever activated.
         * @returns {void}
         */
        function showPicker() {
            buildPicker();
            renderLevel();
        }

        /**
         * @name hidePicker
         * @desc called when the picker is closed
         * @returns {void}
         */
        function hidePicker() {
            if (toggleEle) {
                toggleEle.focus();
            }
        }

        /**
         * @name buildPicker
         * @desc build (or rebuild) the picker whenever activated.
         * @returns {void}
         */
        function buildPicker() {
            var val = scope.model;

            // convert to a date object
            if (val) {
                val = Utility.toDate(val, scope.format);

                if (!Utility.isDate(val)) { // try to convert it to a date
                    val = new Date(val);
                }
            }

            // set the variables
            if (Utility.isDate(val)) {
                scope.day = val.getDate();
                scope.month = val.getMonth();
                scope.year = val.getFullYear();
                scope.renderedMonth = scope.month;
                scope.renderedYear = scope.year;

                scope.computedShortDate = Utility.toDateString(val, 'MM/DD/YYYY');
                scope.computedFullDate = Utility.toDateString(val, 'MMMM DD, YYYY');
            } else {
                val = new Date();

                scope.day = '';
                scope.month = '';
                scope.year = '';
                scope.renderedMonth = val.getMonth();
                scope.renderedYear = val.getFullYear();
                scope.computedShortDate = '';
                scope.computedFullDate = '';
            }

            if (scope.computedFullDate) {
                scope.showPlaceholder = false;
            } else {
                scope.showPlaceholder = true;
            }
        }

        /**
         * @name renderLevel
         * @desc set the items for the current level
         * @returns {void}
         */
        function renderLevel() {
            var currentMonthObj,
                firstDateObj,
                firstDateDay,
                previousMonth,
                previousYear,
                previousMonthObj,
                previousMonthLength,
                b,
                bLen,
                currentMonthLength,
                d,
                nextMonth,
                nextYear,
                nextMonthObj,
                mIdx,
                mLen,
                min,
                max,
                y;

            //TODO: validate w/ minimum + maximum date ranges
            scope.items = [];

            if (scope.level === 'day') {
                currentMonthObj = Utility.constants.month[scope.renderedMonth];

                scope.title = currentMonthObj.abbreviation + ' ' + scope.renderedYear;

                //add any buffer days
                firstDateObj = new Date(scope.renderedYear, scope.renderedMonth, 1);
                firstDateDay = firstDateObj.getDay();

                if (firstDateDay > 0) {
                    // get the previously rendered month
                    if (scope.renderedMonth === 0) {
                        previousMonth = Utility.constants.month.length - 1;
                        previousYear = scope.renderedYear - 1;
                    } else {
                        previousMonth = scope.renderedMonth - 1;
                        previousYear = scope.renderedYear;
                    }

                    previousMonthObj = Utility.constants.month[previousMonth];
                    previousMonthLength = previousMonthObj.length;

                    // leapyear check
                    if (previousMonth === 1 && Utility.isLeapYear(previousYear)) {
                        previousMonthLength++;
                    }

                    for (
                        b = previousMonthLength - firstDateDay + 1; b <= previousMonthLength; b++
                    ) {
                        scope.items.push({
                            title: previousMonthObj.name + ' ' + b + ', ' + previousYear,
                            display: b,
                            year: previousYear,
                            month: previousMonth,
                            day: b
                        });
                    }
                }

                // add the days of the current rendered month
                // leapyear check
                currentMonthLength = currentMonthObj.length;
                if (
                    scope.renderedMonth === 1 &&
                    Utility.isLeapYear(scope.renderedYear)
                ) {
                    currentMonthLength++;
                }

                for (d = 1; d <= currentMonthLength; d++) {
                    scope.items.push({
                        title: currentMonthObj.name + ' ' + d + ', ' + scope.renderedYear,
                        display: d,
                        year: scope.renderedYear,
                        month: scope.renderedMonth,
                        day: d
                    });
                }

                if (scope.items.length % 7 > 0) {
                    // get the next month

                    if (scope.renderedMonth === 11) {
                        nextMonth = 0;
                        nextYear = scope.renderedYear + 1;
                    } else {
                        nextMonth = scope.renderedMonth + 1;
                        nextYear = scope.renderedYear;
                    }

                    nextMonthObj = Utility.constants.month[nextMonth];
                    for (b = 1, bLen = 7 - scope.items.length % 7; b <= bLen; b++) {
                        scope.items.push({
                            title: nextMonthObj.name + ' ' + b + ', ' + nextYear,
                            display: b,
                            year: nextYear,
                            month: nextMonth,
                            day: b
                        });
                    }
                }
            } else if (scope.level === 'month') {
                scope.title = scope.renderedYear;

                for (
                    mIdx = 0, mLen = Utility.constants.month.length; mIdx < mLen; mIdx++
                ) {
                    scope.items.push({
                        title: Utility.constants.month[mIdx].name + ' ' + scope.renderedYear,
                        display: Utility.constants.month[mIdx].abbreviation,
                        year: scope.renderedYear,
                        month: mIdx
                    });
                }
            } else if (scope.level === 'year') {
                scope.title = '';
                min = scope.renderedYear - 5;
                max = min + 11;

                for (y = min; min <= y && y <= max; y++) {
                    scope.items.push({
                        title: y,
                        display: y,
                        year: y
                    });
                }
            }
        }
        /**
         * @name nextLevel
         * @desc go to the next part of the current level and set the items
         * @returns {void}
         */
        function nextLevel() {
            var renderedMonth = scope.renderedMonth,
                renderedYear = scope.renderedYear;

            if (scope.level === 'day') {
                if (renderedMonth === 11) {
                    renderedMonth = 0;
                    renderedYear++;
                } else {
                    renderedMonth++;
                }
            } else if (scope.level === 'month') {
                renderedYear++;
            } else if (scope.level === 'year') {
                renderedYear += 12;
            }

            scope.renderedMonth = renderedMonth;
            scope.renderedYear = renderedYear;

            renderLevel();
        }
        /**
         * @name previousLevel
         * @desc go to the previous part of the current level and set the items
         * @returns {void}
         */
        function previousLevel() {
            var renderedMonth = scope.renderedMonth,
                renderedYear = scope.renderedYear;

            if (scope.level === 'day') {
                if (renderedMonth === 0) {
                    renderedMonth = Utility.constants.month.length - 1;
                    renderedYear--;
                } else {
                    renderedMonth--;
                }
            } else if (scope.level === 'month') {
                renderedYear--;
            } else if (scope.level === 'year') {
                renderedYear -= 12;
            }

            scope.renderedMonth = renderedMonth;
            scope.renderedYear = renderedYear;

            renderLevel();
        }
        /**
         * @name exitLevel
         * @desc exit the current level and set the items
         * @returns {void}
         */
        function exitLevel() {
            if (scope.level === 'year') {
                return;
            }

            if (scope.level === 'day') {
                scope.day = 1;
                scope.level = 'month';
            } else if (scope.level === 'month') {
                scope.month = 0;
                scope.level = 'year';
            }

            renderLevel();
        }

        /**
         * @name enterLevel
         * @desc select the day, and the month, and enter the next level
         * @param {number} year - selected year
         * @param {number} month - selected month
         * @param {number} day - selected day
         * @returns {void}
         */
        function enterLevel(year, month, day) {
            scope.year = year;
            scope.month = month;
            scope.day = day;
            scope.renderedYear = year;
            scope.renderedMonth = month;

            if (scope.level === 'day') {
                if (
                    Utility.isDefined(scope.month) &&
                    Utility.isDefined(scope.day) &&
                    Utility.isDefined(scope.year)
                ) {
                    var value = new Date(scope.year, scope.month, scope.day);

                    if (Utility.isDate(value)) {
                        value = Utility.toDateString(value, scope.format);

                        changeDatepicker(value);
                    }
                }

                scope.open = false;
                return;
            }

            if (scope.level === 'month') {
                scope.level = 'day';
            } else if (scope.level === 'year') {
                scope.level = 'month';
            }

            renderLevel();
        }

        /**
         * @name changeDatepicker
         * @param {*} value - selected value
         * @desc trigger a change in the model
         * @returns {void}
         */
        function changeDatepicker(value) {
            scope.model = value;

            $timeout(function () {
                if (scope.change) {
                    scope.change({
                        model: scope.model,
                        delta: {
                            type: 'replace',
                            value: value
                        }
                    });
                }

                scope.open = false;
            });
        }

        /**
         * @name keyupToggle
         * @param {event} $event - DOM event
         * @desc key up event for the toggle
         * @returns {void}
         */
        function keyupToggle($event) {
            if ($event.keyCode === 27) { // esc
                changeDatepicker('');
            }
        }

        /**Utility */

        /**
         * @name initialize
         * @desc called when the directive is loaded
         * @returns {void}
         */
        function initialize() {
            var initializeTimeout;

            // get the toggle
            toggleEle = ele[0].querySelector('#smss-date-picker__toggle');

            toggleEle.addEventListener('keyup', keyupToggle);

            if (attrs.hasOwnProperty('compact')) {
                scope.compact = true;
            }

            if (typeof scope.format === 'undefined' || !scope.format) {
                scope.format = 'YYYY-MM-DD';
            }

            if (attrs.hasOwnProperty('placeholder')) {
                scope.placeholder = attrs.placeholder;
            }

            //set the view and update after the digest is complete
            initializeTimeout = $timeout(function () {
                // set the inital one
                buildPicker();

                scope.$watch('model', function (newValue, oldValue) {
                    if (!angular.equals(newValue, oldValue)) {
                        buildPicker();
                    }
                });

                scope.$watch('format', function (newValue, oldValue) {
                    if (!angular.equals(newValue, oldValue)) {
                        scope.format = newValue;

                        buildPicker();
                    }
                });

                if (attrs.hasOwnProperty('autofocus')) {
                    if (toggleEle) {
                        toggleEle.focus();
                    }
                }

                scope.$on('$destroy', function () {});

                $timeout.cancel(initializeTimeout);
            });
        }

        initialize();
    }
}
