import Driver from 'driver.js';
import angular from 'angular';

export default angular.module('app.tour.service', [])
    .factory('tourService', tourService);

tourService.$inject = ['$stateParams', '$timeout', 'messageService', 'workspaceService'];

function tourService(
    $timeout: ng.ITimeoutService,
    $stateParams,
    messageService: MessageService,
    workspaceService: WorkspaceService
) {
    // TODO
    // customize steps when in app home page, work space with menu closed, specific menu when menu is open
    const
        /** Public */
        /** Private */
        _state: any = {},
        _actions = {
            /**
             * @name start-tour
             * @desc listen when a widget is closed
             * @param payload - payload of the message
             */
            'start-tour': (payload: any): void => {
                const driver = new Driver({
                    padding: 0
                });

                if (driver) {
                    _state.driver = driver;
                    getCurrentState(payload);
                }
            },
            /**
             * @name open-app
             * @desc listen when an app is opened
             */
            'open-app': (): void => {
                if (_state.driver) {
                    _state.driver.reset();
                }
            },
            /**
             * @name changed-workspace-menu
             * @desc listen when menu state is changed
             */
            'changed-workspace-menu': (): void => {
                if (_state.driver) {
                    _state.driver.reset();
                }
            },
            /**
             * @name new-sheet
             * @desc listen when a sheet is added
             */
            'new-worksheet': (): void => {
                if (_state.driver) {
                    _state.driver.reset();
                }
            },
            /**
             * @name select-sheet
             * @desc listen when a sheet is selected
             */
            'select-worksheet': (): void => {
                if (_state.driver) {
                    _state.driver.reset();
                }
            },
            /**
             * @name change-widget-tab
             * @desc listen when a widget tab is selected
             */
            'change-widget-tab': (): void => {
                if (_state.driver) {
                    _state.driver.reset();
                }
            },
            /**
             * @name open-app
             * @desc listen when an app is opened
             */
            'open': (): void => {
                if (_state.driver) {
                    _state.driver.reset();
                }
            }
        };

    /** Public */
    /**
     * @name initialize
     * @desc called when the module is loaded
     */
    function initialize(): void {
        // register the selected to force conformity
        for (let a in _actions) {
            if (_actions.hasOwnProperty(a)) {
                messageService.on(a, _actions[a]);
            }
        }
    }

    /**
     * @name getCurrentState
     * @param payload - where tour was called from
     * @desc defines the steps for the store based on current state
     */
    function getCurrentState(payload: { selectedState: string, selectedApp: string }): void {
        let steps: TourStep[] = [];

        if (payload.selectedState === 'home.landing') {
            steps = getTourForLandingPage();
        } else if (payload.selectedState === 'home.admin') {
            steps = getTourForAdminPage();
        } else if (payload.selectedState === 'home.app') {
            steps = getTourForAppHomePage();
        } else if (payload.selectedState === 'home.build') {
            steps = getTourForVisualization();
        }

        initializeTour(steps);
    }

    /**
     * @name getTourForLandingPage
     * @desc get steps for home layout
     */
    function getTourForLandingPage(): TourStep[] {
        const steps: TourStep[] = [];

        steps.push({
            element: '#home__semoss-tour-home-logo',
            popover: {
                title: 'Welcome to SEMOSS!',
                description: 'Click here to return back to the SEMOSS home page at any time.',
                position: 'bottom'
            }
        }, {
            element: '#landing__semoss-tour-landing-container',
            popover: {
                title: 'Your Apps',
                description: 'Here are all of your saved apps (data + insights). Click on one to see the app\'s insights.',
                position: 'top'
            }
        }, {
            element: '#home-nav__item--import',
            popover: {
                title: 'Import',
                description: 'Click here to upload new data and create an app.',
                position: 'bottom'
            }
        }, {
            element: '#home-nav__item--new-insight',
            popover: {
                title: 'New Insight',
                description: 'Create a new insight using new data or existing data from a saved app.',
                position: 'left'
            }
        }, {
            element: '#home__topbar__search',
            popover: {
                title: 'Global Search',
                description: 'Search throughout all of your apps for a specific insight no matter where you are.',
                position: 'bottom'
            }
        }, {
            element: '#home-login-tab',
            popover: {
                title: 'Log In',
                description: 'Easily log into SEMOSS through different social platforms.',
                position: 'left'
            }
        }, {
            element: '#home__semoss-tour-home-help',
            popover: {
                title: 'SEMOSS Help',
                description: 'Click here if you ever get stuck. You can access the User Guide, email the SEMOSS Help Desk, and go through a guided tour (specific to your current state).',
                position: 'left'
            }
        });

        return steps;
    }

    /**
     * @name getTourForAdminPage
     * @desc get steps for settings tab
     * @returns  array of steps
     */
    function getTourForAdminPage(): TourStep[] {
        const steps: TourStep[] = [];
        steps.push({
            element: '#admin__semoss-tour-app-permissions',
            popover: {
                title: 'App Permissions',
                description: 'Here are all of your apps and the access you have to them. Control which apps to hide/show on the home page.',
                position: 'bottom'
            }
        }, {
            element: '#admin__semoss-tour-admin-permissions',
            popover: {
                title: 'Admin Settings',
                description: 'Add users and edit their access to apps and insights.',
                position: 'bottom'
            }
        }, {
            element: '#admin__semoss-tour-admin-themes',
            popover: {
                title: 'Admin Themes',
                description: 'Customize the look and feel of your SEMOSS by creating a new theme for you and your users.',
                position: 'bottom'
            }
        }, {
            element: '#admin__semoss-tour-social',
            popover: {
                title: 'Social',
                description: 'Link social accounts by defining your social properties.',
                position: 'bottom'
            }
        });

        return steps;
    }

    /**
     * @name getTourForAppHomePage
     * @desc get steps for app home page layout
     * @returns  array of steps
     */
    function getTourForAppHomePage(): TourStep[] {
        const steps: TourStep[] = [];

        steps.push({
            element: '#app-home-insights__scroll',
            popover: {
                title: 'Your App\'s Insights',
                description: 'Here are all of the insights that belong to this app. An insight can be a single visualization, a dashbord, a form, etc.)',
                position: 'top'
            }
        }, {
            element: '#app-home-insights__semoss-tour-users',
            popover: {
                title: 'App Users',
                description: 'Here are the users of this app and their role (author, editor, or viewer)',
                position: 'right'
            }
        }, {
            element: '#app-home-insights__semoss-tour-app-info',
            popover: {
                title: 'Edit App',
                description: 'Here you can add/edit your app\'s image and description',
                position: 'top'
            }
        }, {
            element: '#app-home__semoss-tour-Data',
            popover: {
                title: 'Your Metamodel',
                description: 'Here is the metamodel of your data, which defines the structure of your data. Select an item to see the the instances within (to the right).',
                position: 'bottom'
            }
        }, {
            element: '#app-home__semoss-tour-Collaboration',
            popover: {
                title: 'Collaborate',
                description: 'Here you can post or sync your app using Github (pending app persmissions).',
                position: 'bottom'
            }
        }, {
            element: '#app-home__semoss-tour-Scheduler',
            popover: {
                title: 'Schedule a Job',
                description: 'The Scheduler allows you to execute a specific task (job) at a defined time. You can organize defined jobs into a workflow.',
                position: 'bottom'
            }
        }, {
            element: '#app-home__semoss-tour-Settings',
            popover: {
                title: 'App Settings',
                description: 'Here you can delete and edit individual insights.',
                position: 'bottom'
            }
        }
        );

        return steps;
    }

    /**
     * @name getTourForVisualization
     * @desc get steps for visualization layout
     * @param sheetId selected sheet id
     */
    function getTourForVisualization(): TourStep[] {
        const insightID = $stateParams.insight;

        //check if it is valid
        if (!insightID || typeof workspaceService.getWorkspace(insightID) === 'undefined') {
            return [];
        }

        const menuOpen = workspaceService.getWorkspace(insightID, 'menu.open')

        if (menuOpen) {
            // Customize steps for selected widget
            return [{
                element: '#widget-menu__semoss-tour-enrich',
                popover: {
                    title: 'Add New Data',
                    description: 'Click here to bring in new data to your frame.',
                    position: 'left'
                }
            }, {
                element: '#widget-menu__semoss-tour-view',
                popover: {
                    title: 'View Your Data',
                    description: 'Click here to visualize your data by choosing from over 30 visualization types.',
                    position: 'left'
                }
            }, {
                element: '#widget-menu__semoss-tour-clean',
                popover: {
                    title: 'Clean Your Data',
                    description: 'Click here to manipulate your dirty data.',
                    position: 'left'
                }
            }, {
                element: '#widget-menu__semoss-tour-analytics',
                popover: {
                    title: 'Analyze your Data',
                    description: 'Click here to run specific analytics routines on your data.',
                    position: 'left'
                }
            },
            {
                element: '#widget-menu__semoss-tour-collaboration',
                popover: {
                    title: 'Collaborate',
                    description: 'Click here to post or sync your insight using Github (pending app persmissions).',
                    position: 'left'
                }
            }, {
                element: '#widget-menu__semoss-tour-share',
                popover: {
                    title: 'Share or Export Data',
                    description: 'Click here to share your data with Tableau, export your data, or download an image of your panel.',
                    position: 'left'
                }
            }, {
                element: '#widget-menu__semoss-tour-settings',
                popover: {
                    title: 'Customize Panel Settings',
                    description: 'Click here to customize the settings (panel name, background color, etc.) for your selected panel.',
                    position: 'left'
                }
            }, {
                element: '#widget-menu__semoss-tour-menu',
                popover: {
                    title: 'More Tools',
                    description: 'Click here to access additional widgets.',
                    position: 'left'
                }
            }];
        } else {
            return [{
                element: '#workspace__main__top__icon--menu',
                popover: {
                    title: 'Open/Close Menu',
                    description: 'Click here to open or close the menu. The menu allows you to customize how you want to visualize, clean, analyze, or federate your data.',
                    position: 'left'
                }
            }, {
                element: '#sheet__semoss-tour-settings',
                popover: {
                    title: 'Sheet Settings',
                    description: 'Customize the look and feel of your sheet here.',
                    position: 'left'
                }
            }, {
                element: '#sheet__semoss-tour-save',
                popover: {
                    title: 'Save Sheet',
                    description: 'Save your entire sheet as an individual insight here.',
                    position: 'left'
                }
            },
            {
                element: '#sheet__semoss-tour-filter',
                popover: {
                    title: 'Filter Data',
                    description: 'Add and customize a sheet filter to apply to your data.',
                    position: 'left'
                }
            }, {
                element: '#sheet__semoss-tour-layout',
                popover: {
                    title: 'Panel Layout',
                    description: 'Customize the position and size of your panels within your sheet by using these layouts.',
                    position: 'left'
                }
            }, {
                element: '#workbook__navigation',
                popover: {
                    title: 'Your Open Sheets',
                    description: 'Jump between all of your open sheets. A sheet can have one or multiple panels.',
                    position: 'top'
                }
            }, {
                element: '#workbook__navigation--new-sheet',
                popover: {
                    title: 'Add New Sheet',
                    description: 'Open a new, clean sheet.',
                    position: 'top'
                }
            }, {
                element: '#book__semoss-tour-presentation-mode',
                popover: {
                    title: 'Presentation Mode',
                    description: 'Expand your sheet when presenting your insights.',
                    position: 'top-left',
                    offset: -300
                }
            }];
        }

        return []
    }

    /**
     * @name initializeTour
     * @desc start tour or redirect to user guide
     * @param steps - tour steps defined for the users current state
     */
    function initializeTour(steps: TourStep[]): void {
        if (steps.length > 0 && _state.driver) {
            _state.driver.defineSteps(steps);
            $timeout(function () {
                _state.driver.start();
            }, 50);
        } else {
            window.open('http://semoss.org/userguide/', '_blank');
        }
    }

    /** Private */
    return {
        initialize: initialize,
        getCurrentState: getCurrentState
    };
}
