import angular from 'angular';
import SuccessFn from './SuccessFn';

/**
 * @name monolith.service.js
 * @desc service used to interface with the Monolith REST API
 */
export default angular.module('app.monolith.service', [])
    .factory('monolithService', monolithService)
    .factory('_', () => window._);

monolithService.$inject = ['CONFIG', 'ENDPOINT', '$http', '$q', 'messageService', '$location'];

function monolithService(CONFIG, ENDPOINT: EndPoint, $http: ng.IHttpService, $q: ng.IQService, messageService: MessageService, $location) {
    const THEN_ARGS = [
        (response: any): SuccessFn => successReturnFn(response, 'data'),
        errorReturnFn
    ],
    GENERIC_SUCCESS = <T>(response: T): T => response;
    enum METHODS {
        GET = 'GET',
        POST = 'POST'
    };

    /**
     * @name errorReturnFn
     * @param err - the error sent back from the server
     * @param loggedout - if we get 302 from logged out, everyone has been logged out
     * @desc used for callback function when there is an error received from the server
     * @return rejected promise
     */
    function errorReturnFn(err: any, loggedout?: boolean): ng.IPromise<never> {
        messageService.emit('session-reset');

        if (err.status === 302) {
            if (loggedout) {
                // if every account has been logged out, we get a redirect and cannot
                // clean up CONFIG in security service
                CONFIG.loggedIn = false;
                CONFIG.logins = {};
            }
            var headers = err.headers();
            if (headers.redirect) {
                if (navigator.userAgent.indexOf('Tableau') === -1) {
                    window.location.replace(headers.redirect);
                }
            }

            messageService.emit('clear-loading');
            return $q.reject();
        }

        return $q.reject(err);
    }

    /**
     * @name successReturnFn
     * @param response the response from the BE
     * @param accessor what to access in the response
     * @desc returns the successful response
     * @returns the response
     */
    function successReturnFn(response: any, accessor?: string): any {
        let convertedAccessor: string[],
            convertedRoot = response;

        messageService.emit('session-reset');
        if (accessor) {
            convertedAccessor = accessor.split('.');
            for (let i = 0, len = convertedAccessor.length; i < len; i++) {
                // move to the next key in the path, reset the root
                if (typeof convertedRoot[convertedAccessor[i]] === 'undefined') {
                    return false;
                }
                convertedRoot = convertedRoot[convertedAccessor[i]];
            }
        }

        return convertedRoot;
    }

    /** ****Searching for and Retrieving Insights (Solr Calls)******/
    /**
     * @name getSearchInsightsResults
     * @param searchInputConfig - configuration object for the search. TODO describe config inputs
     * @desc Returns insights from Solr
     * @return $http promise
     */
    function getSearchInsightsResults(searchInputConfig: { searchString: string, filterData: any, limit: number, offset: number}): ng.IPromise<any> {
        let postData = 'searchString=' + encodeURIComponent(searchInputConfig.searchString);

        postData += '&filterData=' + encodeURIComponent(JSON.stringify(searchInputConfig.filterData));
        postData += '&limit=' + encodeURIComponent(JSON.stringify(searchInputConfig.limit));
        postData += '&offset=' + encodeURIComponent(JSON.stringify(searchInputConfig.offset));

        return HTTPReq('/api/engine/central/context/getSearchInsightsResults', postData, METHODS.POST)
            .then(...THEN_ARGS);
    }

    /**
     * @name uploadFile
     * @param files - list of file to upload
     * @param insightId - insight id used for messaging
     * @param appId - do we want to load this directly into an engine folder?
     * @param path - load it directly to a path?
     * @desc xray upload
     * @returns return message
     */
    function uploadFile(files: any, insightId: string, appId?: string, path?: string): ng.IPromise<any> {
        let url = '/api/uploadFile/baseUpload',
            param = '',
            fd = new FormData();

        if (insightId || appId || path) {
            if (insightId) {
                if (param.length > 0) {
                    param += '&';
                }
                param += 'insightId=' + insightId;
            }

            if (appId) {
                if (param.length > 0) {
                    param += '&';
                }
                param += 'appId=' + appId;
            }

            if (path) {
                if (param.length > 0) {
                    param += '&';
                }
                param += 'path=' + path;
            }

            param = '?' + param;
        }


        url += param;

        if (typeof files === 'object') {
            for (let i = 0; i < files.length; i++) {
                fd.append('file', files[i].file);
            }
        } else {
            // pasted data
            fd.append('file', files);
        }

        return HTTPReq(url, fd, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name checkHeaders
     * @param data the data that will be sent to the BE to check the headers for flat upload
     * @desc this call will check the headers of a flat upload for both excel and csv
     * @return $http promise
     */
    function checkHeaders(data: any): ng.IPromise<any> {
        let postData = '';

        postData += 'uploadType=' + encodeURIComponent(data.uploadType);
        postData += '&userHeaders=' + encodeURIComponent(JSON.stringify(data.userHeaders));

        return HTTPReq('/api/uploadFile/headerCheck', postData, METHODS.POST)
            .then(...THEN_ARGS);
    }

    /**
     * @name runPixel
     * @param insightID id for the insight
     * @param input pixel to run
     * @param sessionInfo jsessionid and Secret
     * @desc this call will run the pixel query
     * @returns {promise} promise of data from BE
     */
    function runPixel(insightID: string, input: string, sessionInfo?: any): ng.IPromise<any> {
        let url = '/api/engine/runPixel',
            postData = 'expression=' + encodeURIComponent(input);

        if (sessionInfo && sessionInfo.jsessionId && sessionInfo.secret && sessionInfo.hash && sessionInfo.insightId) {
            url += '?JSESSIONID=' + encodeURIComponent(sessionInfo.jsessionId);
            url += '&s=' + encodeURIComponent(sessionInfo.secret);
            url += '&hash=' + encodeURIComponent(sessionInfo.hash);
            url += '&i=' + encodeURIComponent(sessionInfo.insightId);
        }

        if (insightID) {
            postData += '&insightId=' + encodeURIComponent(insightID);
        }

        return HTTPReq(url, postData, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name getInsightState
     * @param {string} insightId insight id to get the state for
     * @desc get the dashboard state of this insight id
     * @returns {promise} promise of the data from BE
     */
    function getInsightState(insightId: string): ng.IPromise<any> {
        const url = '/api/share/i-' + encodeURIComponent(insightId) + '/getInsightState';

        return HTTPReq(url, '', METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name getPipeline
     * @param insightID id for the insight
     * @param expression pixel to get pipeline for
     * @desc gets pipeline AST reconstruct a pipeline from a pixel
     * @returns promise of data from BE
     */
    function getPipeline(insightID: string, expression: string): ng.IPromise<any> {
        const data = `expression=${encodeURIComponent(expression)}&insightId=${encodeURIComponent(insightID)}`;

        return HTTPReq('/api/engine/getPipeline', data, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name createUser
     * @desc this call will run createUser endpoint
     * @param name, name of new user
     * @param username, username of new user
     * @param email, email of new user
     * @param password, password of new user
     * @returns $http promise
     */
    function createUser(name: string, username: string, email: string, password: string): ng.IPromise<any> {
        const create: string = 'name=' + encodeURIComponent(name) + '&username=' + encodeURIComponent(username) +
            '&email=' + encodeURIComponent(email) + '&password=' + encodeURIComponent(password);

        return HTTPReq('/api/auth/createUser', create, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name updateUser
     * @desc this call will run updateUser endpoint
     * @param userId, id of updating user
     * @param name, name of updating user
     * @param userEmail, email of updating user
     * @param userPassword, password of updating user
     * @param userAdmin, true or false
     * @returns $http promise
     */
    function updateUser(userId: string, name: string, userEmail: string, userPassword: string, userAdmin: string, type: string, username: string): ng.IPromise<any> {
        const updateUserInfo: string = 'user=' + encodeURIComponent(JSON.stringify({
                id: userId,
                name: name,
                email: userEmail,
                password: userPassword,
                admin: userAdmin,
                type: type,
                username: username
            }));

        return HTTPReq('/api/authorization/editUser', updateUserInfo, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name addNewUser
     * @desc this call will run addNewUser endpoint
     * @param userId, id of updating user
     * @param userAdmin, true or false
     * @returns $http promise
     */
    function addNewUser(userId: string, userAdmin: string): ng.IPromise<any> {
        let newUserInfo = 'userId=' + encodeURIComponent(userId) + '&admin=',
            isAdminBool = userAdmin ? userAdmin : false;

        newUserInfo += encodeURIComponent(isAdminBool);

        return HTTPReq('/api/authorization/admin/registerUser', newUserInfo, METHODS.POST)
            .then(...THEN_ARGS);
    }

    /**
     * @name loginUser
     * @desc this call will run login endpoint
     * @param username of login user
     * @param password of login user
     * @returns $http promise
     */
    function loginUser(username: string, password: string): ng.IPromise<any> {
        const loginInfo = 'username=' + encodeURIComponent(username) + '&password=' + encodeURIComponent(password);

        return HTTPReq('/api/auth/login', loginInfo, METHODS.POST)
            .then(
                (response: any): SuccessFn => {
                    window.localStorage.setItem('smssusername', response.data.username);
                    window.localStorage.setItem('smssname', response.data.name);

                    return successReturnFn(response, 'data');
                },
                errorReturnFn
            );
    }

    /**
     * @name logout
     * @desc this call will run logout endpoint
     * @returns $http promise
     */
    function logoutUser(): ng.IPromise<any> {
        return HTTPReq('/api/auth/logout', '', METHODS.GET)
            .then(
                (response: any): SuccessFn => {
                    window.localStorage.removeItem('smssusername');
                    window.localStorage.removeItem('smssname');

                    return successReturnFn(response, 'data');
                },
                errorReturnFn
            );
    }

    /**
     * @name loginsAllowed
     * @desc this call will return which method will be used for user login i.e. OAuth or native
     * @returns $http promise
     */
    function loginsAllowed(): ng.IPromise<any> {
        return HTTPReq('/api/auth/loginsAllowed', '', METHODS.GET).then(...THEN_ARGS);
    }

    /**
     * @name console
     * @desc this call will run the pixel query
     * @param id - id to grab the message for
     * @returns $http promise
     */
    function console(id: string): ng.IPromise<any> {
        const postData = 'jobId=' + encodeURIComponent(id);

        return HTTPReq('/api/engine/console', postData, METHODS.POST).then(...THEN_ARGS);
    }


    /**
     * @name downloadFile
     * @desc this call will create a get to download a file
     * @param insightID - insightID
     * @param fileKey - id for the file to grab
     */
    function downloadFile(insightID: string, fileKey: string): void {
        let url = ENDPOINT.URL + '/api/engine/downloadFile?insightId=' + insightID + '&fileKey=' + encodeURIComponent(fileKey),
            link: HTMLAnchorElement;

        link = document.createElement('a');
        link.href = url;
        link.target = '_blank';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }

    /**
     * @name uploadAppImage
     * @desc this call will run the pixel query
     * @param id - id to upload for
     * @param file - file to upload
     * @returns $http promise
     */
    function uploadAppImage(id: string, file: File): ng.IPromise<any> {
        const fd = new FormData();

        fd.append('app', id);
        fd.append('file', file);

        return HTTPReq('/api/images/appImage/upload', fd, METHODS.POST, false)
            .then(...THEN_ARGS);
    }

    /**
     * @name deleteAppImage
     * @param id app id to delete image for
     * @desc delete app image
     * @returns $http promise
     */
    function deleteAppImage(id: string): ng.IPromise<any> {
        let postData = '';

        postData += 'appId=' + encodeURIComponent(id);

        return HTTPReq('/api/images/appImage/delete', postData, METHODS.POST).then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name uploadInsightImage
     * @desc this call will run the pixel query
     * @param app - the app of the insight
     * @param id - id to upload for
     * @param file - file to upload
     * @returns $http promise
     */
    function uploadInsightImage(app: string, id: string, file: string): ng.IPromise<any> {
        const fd = new FormData();

        fd.append('app', app);
        fd.append('insightId', id);
        fd.append('file', file);

        return HTTPReq('/api/images/insightImage/upload', fd, METHODS.POST, false)
            .then(...THEN_ARGS);
    }
    
    /**
     * @name deleteAppImage
     * @param appId app id to delete image for
     * @param insightId the insight id
     * @desc delete insight image
     * @returns $http promise
     */
    function deleteInsightImage(appId: string, insightId: string): ng.IPromise<any> {
        let postData = '';

        postData += 'appId=' + encodeURIComponent(appId);
        postData += '&insightId=' + encodeURIComponent(insightId);

        return HTTPReq('/api/images/insightImage/delete', postData, METHODS.POST).then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name backendConfig
     * @desc get the  configuration settings from BE
     * @returns $http promise
     */
    function backendConfig(): ng.IPromise<any> {
        let url: string = '/api/config',
            param: string,
            urlParams = $location.search(),
            customParams,
            customParamStr: string = '';

        // additional params that serve to add query params
        if (urlParams && urlParams.hasOwnProperty('customParams')) {
            customParams = JSON.parse(urlParams.customParams)
            for (param in customParams) {
                if (!customParamStr) { // first query param so start with ?
                    customParamStr += '?';
                } else {
                    customParamStr += '&';
                }

                customParamStr += param; // ?sencha
                customParamStr += '='; // ?sencha=
                customParamStr += customParams[param]; // ?sencha=davyzhang
            }

            if (customParamStr) {
                url += customParamStr;
            }
        }

        return HTTPReq(url, '', METHODS.GET)
            .then(
                (response: any): any => {
                    let key: string;
                    for (key in response.data) {
                        if (response.data.hasOwnProperty(key)) {
                          CONFIG[key] = response.data[key];
                        }
                    }
                    // message out to indicate config came back
                    messageService.emit('initialize-config');
                    return response.data;
                },
                errorReturnFn
            );
    }

    /**
     * @name isAdmin
     * @desc checks to see if user is admin
     * @returns {object} $http promise
     */
    function isAdmin(): ng.IPromise<any> {
        return HTTPReq('/api/authorization/admin/isAdminUser', '', METHODS.GET).then(...THEN_ARGS);
    }

    /**
     * @name getAllDBUsers
     * @desc to list all user information for admin
     * @returns $http promise
     */
    function getAllDBUsers(): ng.IPromise<any> {
        return HTTPReq('/api/authorization/getAllDbUsers', '', METHODS.GET).then(...THEN_ARGS);
    }

    /**
     * @name getAllUserDbs
     * @desc to list all user information for admin
     * @returns $http promise
     */
    function getAllUserDbs(userId: string): ng.IPromise<any> {
        const id = 'userId=' + encodeURIComponent(userId);

        return HTTPReq('/api/auth/admin/app/getAllUserApps', id, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name removeAdminUser
     * @desc this call will remove a user
     * @param userId, id of user
     * @returns $http promise
     */
    function removeAdminUser(userId: string): ng.IPromise<any> {
        const dbId = 'userId=' + encodeURIComponent(userId);

        return HTTPReq('/api/authorization/admin/deleteUser', dbId, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name getUserInformation
     * @desc this call will return all users for a given search result
     * @param searchInput what to search for
     * @returns $http promise
     */
    function getUserInformation(searchInput: any): ng.IPromise<any> {
        const url = '/api/authorization/searchForUser?searchTerm=' + searchInput;

        return HTTPReq(url, '', METHODS.GET).then(...THEN_ARGS);
    }

    /**
     * @name getInsightUsers
     * @param appId the app id the insight belongs to
     * @param insightId the id of the insight
     * @desc get the list of users for this insight
     * @returns $http promise
     */
    function getInsightUsers(appId: string, insightId: string): ng.IPromise<any> {
        return HTTPReq('/api/auth/insight/getInsightUsers', '', METHODS.GET, { appId, insightId })
            .then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name getAdminInsightUsers
     * @param appId the app id the insight belongs to
     * @param insightId the id of the insight
     * @desc get the list of users for this insight
     * @returns $http promise
     */
    function getAdminInsightUsers(appId: string, insightId: string): ng.IPromise<any> {
        return HTTPReq('/api/auth/admin/insight/getInsightUsers', '', METHODS.GET, { appId, insightId })
            .then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name getUserAppPermission
     * @param appId the appId to get user permission for
     * @desc get permission of current user
     * @returns http promise
     */
    function getUserAppPermission(appId: string): ng.IPromise<any> {
        return HTTPReq('/api/auth/app/getUserAppPermission', '', METHODS.GET, { appId })
            .then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name getAppUsers
     * @param admin if admin initiated the call
     * @param appId the id of app
     * @desc get the existing users and their permissions for this app
     * @returns $http promise
     */
    function getAppUsers(admin: boolean, appId: string): ng.IPromise<any> {
        let url = '/api/auth/';

        if (admin) {
            url += 'admin/';
        }

        url += 'app/getAppUsers';

        return HTTPReq(url, '', METHODS.GET, { appId }).then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name addAppUserPermissions
     * @param admin came from admin or not
     * @param appId id of app
     * @param id id of user
     * @param permission permission to give
     * @desc add the user permissions
     * @returns $http promise
     */
    function addAppUserPermission(admin: boolean, appId: string, id: string, permission: string): ng.IPromise<any> {
        let url = '/api/auth/',
            postData = '';
        if (admin) {
            url += 'admin/';
        }

        url += 'app/addAppUserPermission';

        postData += 'appId=' + encodeURIComponent(appId);
        postData += '&id=' + encodeURIComponent(id);
        postData += '&permission=' + encodeURIComponent(permission);

        return HTTPReq(url, postData, METHODS.POST).then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name editAppUserPermissions
     * @param admin admin true or false
     * @param appId id of app
     * @param id id of user
     * @param permission permission to give
     * @desc edit the user permissions
     * @returns $http promise
     */
    function editAppUserPermission(admin: boolean, appId: string, id: string, permission: string): ng.IPromise<any> {
        let url = '/api/auth/',
            postData = '';

        if (admin) {
            url += 'admin/';
        }

        url += 'app/editAppUserPermission';

        postData += 'appId=' + encodeURIComponent(appId);
        postData += '&id=' + encodeURIComponent(id);
        postData += '&permission=' + encodeURIComponent(permission);

        return HTTPReq(url, postData, METHODS.POST).then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name removeAppUserPermissions
     * @param admin true or false
     * @param appId id of app
     * @param id id of user
     * @desc remove the user permissions
     * @returns $http promise
     */
    function removeAppUserPermission(admin: boolean, appId: string, id: string): ng.IPromise<any> {
        let url = '/api/auth/',
            postData = '';

        if (admin) {
            url += 'admin/';
        }

        url += 'app/removeAppUserPermission';

        postData += 'appId=' + encodeURIComponent(appId);
        postData += '&id=' + encodeURIComponent(id);

        return HTTPReq(url, postData, METHODS.POST).then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name setAppGlobal
     * @param admin whether made from admin call
     * @param appId id of app
     * @param global global value
     * @desc set the global value to true/false
     * @returns $http promise
     */
    function setAppGlobal(admin: boolean, appId: string, global: string): ng.IPromise<any> {
        let url = '/api/auth/',
            postData = '';

        if (admin) {
            url += 'admin/';
        }

        url += 'app/setAppGlobal';

        postData += 'appId=' + encodeURIComponent(appId);
        postData += '&public=' + encodeURIComponent(global);

        return HTTPReq(url, postData, METHODS.POST).then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name setAppVisibility
     * @param appId id of app
     * @param visibility visibility value
     * @desc set the visibility value to true/false
     * @returns $http promise
     */
    function setAppVisibility(appId: string, visibility: number): ng.IPromise<any> {
        let url = '/api/auth/',
            postData = '';

        url += 'app/setAppVisibility';

        postData += 'appId=' + encodeURIComponent(appId);
        postData += '&visibility=' + encodeURIComponent(visibility);

        return HTTPReq(url, postData, METHODS.POST).then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name getApps
     * @param admin whether made from admin call
     * @desc get all the apps with their permissions
     * @returns $http promise
     */
    function getApps(admin: boolean): ng.IPromise<any> {
        let url = '/api/auth/';

        if (admin) {
            url += 'admin/';
        }

        url += 'app/getApps';

        return HTTPReq(url, '', METHODS.GET, {}).then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name getUserInsightPermission
     * @param appId the app id the insight belongs to
     * @param insightId the id of the insight
     * @desc get the user's permission for this insight
     * @returns $http promise
     */
    function getUserInsightPermission(appId: string, insightId: string): ng.IPromise<any> {
        const params = {
            appId,
            insightId
        };

        return HTTPReq('/api/auth/insight/getUserInsightPermission', '', METHODS.GET, params)
            .then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name addInsightUserPermission
     * @param appId the appId
     * @param insightId the insight id to add permission to
     * @param id the user id
     * @param permission the permission to give
     * @desc add permission to an insight
     * @returns the promise to return
     */
    function addInsightUserPermission(appId: string, insightId: string, id: string, permission: string): ng.IPromise<any> {
        let postData = '';

        postData += 'appId=' + appId;
        postData += '&insightId=' + insightId;
        postData += '&id=' + id;
        postData += '&permission=' + permission;

        return HTTPReq('/api/auth/insight/addInsightUserPermission', postData, METHODS.POST)
            .then(...THEN_ARGS);
    }

    /**
     * @name addAdminInsightUserPermission
     * @param appId the appId
     * @param insightId the insight id to add permission to
     * @param id the user id
     * @param permission the permission to give
     * @desc add permission to an insight
     * @returns the promise to return
     */
    function addAdminInsightUserPermission(appId: string, insightId: string, id: string, permission: string): ng.IPromise<any> {
        let postData = '';

        postData += 'appId=' + appId;
        postData += '&insightId=' + insightId;
        postData += '&id=' + id;
        postData += '&permission=' + permission;

        return HTTPReq('/api/auth/admin/insight/addInsightUserPermission', postData, METHODS.POST)
            .then(...THEN_ARGS);
    }

    /**
     * @name editInsightUserPermission
     * @param appId the appId
     * @param insightId the insight id to edit permission for
     * @param id the user id
     * @param permission the permission to edit
     * @desc edit permission of an insight
     * @returns the promise to return
     */
    function editInsightUserPermission(appId: string, insightId: string, id: string, permission: string): ng.IPromise<any> {
        let postData = '';

        postData += 'appId=' + appId;
        postData += '&insightId=' + insightId;
        postData += '&id=' + id;
        postData += '&permission=' + permission;

        return HTTPReq('/api/auth/insight/editInsightUserPermission', postData, METHODS.POST)
            .then(...THEN_ARGS );
    }

    /**
     * @name editAdminInsightUserPermission
     * @param appId the appId
     * @param insightId the insight id to edit permission for
     * @param id the user id
     * @param permission the permission to edit
     * @desc edit permission of an insight
     * @returns the promise to return
     */
    function editAdminInsightUserPermission(appId: string, insightId: string, id: string, permission: string): ng.IPromise<any> {
        let postData = '';

        postData += 'appId=' + appId;
        postData += '&insightId=' + insightId;
        postData += '&id=' + id;
        postData += '&permission=' + permission;

        return HTTPReq('/api/auth/admin/insight/editInsightUserPermission', postData, METHODS.POST)
            .then(...THEN_ARGS);
    }

    /**
     * @name removeInsightUserPermission
     * @param appId the appId
     * @param insightId the insight id to remove permission from
     * @param id the user id
     * @desc remove permission from an insight
     * @returns the promise to return
     */
    function removeInsightUserPermission(appId: string, insightId: string, id: string): ng.IPromise<any> {
        let postData = '';

        postData += 'appId=' + appId;
        postData += '&insightId=' + insightId;
        postData += '&id=' + id;

        return HTTPReq('/api/auth/insight/removeInsightUserPermission', postData, METHODS.POST)
            .then(...THEN_ARGS);
    }

    /**
     * @name removeAdminInsightUserPermission
     * @param appId the appId
     * @param insightId the insight id to remove permission from
     * @param id the user id
     * @desc remove permission from an insight
     * @returns the promise to return
     */
    function removeAdminInsightUserPermission(appId: string, insightId: string, id: string): ng.IPromise<any> {
        let postData = '';

        postData += 'appId=' + appId;
        postData += '&insightId=' + insightId;
        postData += '&id=' + id;

        return HTTPReq('/api/auth/admin/insight/removeInsightUserPermission', postData, METHODS.POST)
            .then(...THEN_ARGS);
    }

    /**
     * @name setInsightGlobal
     * @param appId the app id
     * @param insightId the insight id
     * @param isPublic the boolean to set it to
     * @desc sets the insight public/private
     * @returns promise
     */
    function setInsightGlobal(appId: string, insightId: string, isPublic: boolean): ng.IPromise<any> {
        let postData = '';

        postData += 'appId=' + appId;
        postData += '&insightId=' + insightId;
        postData += '&isPublic=' + isPublic;

        return HTTPReq('/api/auth/insight/setInsightGlobal', postData, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name setAdminInsightGlobal
     * @param appId the app id
     * @param insightId the insight id
     * @param isPublic the boolean to set it to
     * @desc sets the insight public/private
     * @returns promise
     */
    function setAdminInsightGlobal(appId: string, insightId: string, isPublic: boolean): ng.IPromise<any> {
        let postData = '';

        postData += 'appId=' + appId;
        postData += '&insightId=' + insightId;
        postData += '&isPublic=' + isPublic;

        return HTTPReq('/api/auth/admin/insight/setInsightGlobal', postData, METHODS.POST)
            .then(...THEN_ARGS);
    }

    /**
     * @name getAdminAppInsights
     * @param appId the app id to get insights for
     * @desc get the insights for app id for the admin
     * @returns Promise
     */
    function getAdminAppInsights(appId: string): ng.IPromise<any> {
        const params = {
            appId
        };

        return HTTPReq('/api/auth/admin/insight/getAppInsights', '', METHODS.GET, params)
            .then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name deleteAdminInsight
     * @param appId the add id insight belongs to
     * @param insightId the list of ids to delete
     * @desc delete the insight as an admin
     * @returns Promise
     */
    function deleteAdminInsight(appId: string, insightId: string[]): ng.IPromise<any> {
        let postData = '';

        postData += 'appId=' + appId;
        postData += '&insightId=' + JSON.stringify(insightId);

        return HTTPReq('/api/auth/admin/insight/deleteAppInsights', postData, METHODS.POST)
            .then(...THEN_ARGS);
    }

    /**
     * @name getAllEngines
     * @returns all engines
     * @desc Get all engines
     */
    function getAllEngines(): ng.IPromise<any> {
        return HTTPReq('/api/engine/all', '', METHODS.GET).then(GENERIC_SUCCESS, errorReturnFn);
    }

    /**
     * @name loginProperties
     * @desc get the current login properties
     * @returns the login properties
     */
    function loginProperties(): ng.IPromise<any> {
        return HTTPReq('/api/auth/loginProperties', '', METHODS.GET).then(...THEN_ARGS);
    }

    /**
     * @name modifyLoginProperties
     * @param provider the provider to modify
     * @param properties the properties to modify
     * @desc modify the properties for this provider
     * @returns response
     */
    function modifyLoginProperties(provider: string, properties: any): ng.IPromise<any> {
        let url = '/api/auth/modifyLoginProperties/' + provider,
            postData = '';

        postData += 'modifications=' + JSON.stringify(properties);

        return HTTPReq(url, postData, METHODS.POST).then(...THEN_ARGS);
    }
    /**
     * @name activeLogins
     * @desc get a list of things that are currently loggedIn
     * @returns response
     */
    function activeLogins(): ng.IPromise<any> {
        return HTTPReq('/api/auth/logins', '', METHODS.GET).then(...THEN_ARGS);
    }

    /**
     * @name login
     * @param provider the provider to log into
     * @desc log into a specific provider
     * @returns response
     */
    function login(provider: string): ng.IPromise<any> {
        const url = '/api/auth/login/' + provider;

        return HTTPReq(url, '', METHODS.GET).then(...THEN_ARGS);
    }

    /**
     * @name logout
     * @param provider name of provider...google/facebook/etc.
     * @desc logout from the provider
     * @returns response
     */
    function logout(provider: string): ng.IPromise<any> {
        const url = '/api/auth/logout/' + provider;

        return HTTPReq(url, '', METHODS.GET).then(...THEN_ARGS);
    }

    /**
     * @name getUserInfo
     * @param provider the name of the provider
     * @desc gets the userInfo
     * @returns user info
     */
    function getUserInfo(provider: string): ng.IPromise<any> {
        const url = '/api/auth/userinfo/' + provider;

        return HTTPReq(url, '', METHODS.GET).then(...THEN_ARGS);
    }

    /**
     * @name saveFilesInInsightAsDb
     * @param insightID - insightID to check
     * @param engineName - new DB name
     * @dsec Creates a db before saving an insight
     * @returns response data
     */
    function saveFilesInInsightAsDb(insightID: string, engineName: string): ng.IPromise<any> {
        const url = '/api/engine/i-' + insightID + '/saveFilesInInsightAsDb',
            postData = 'engineName=' + encodeURIComponent(engineName);

        return HTTPReq(url, postData, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name getAdminThemes
     * @desc this call will get the active theme
     * @return $http promise
     */
    function getAdminThemes(): ng.IPromise<any> {
        return HTTPReq('/api/themes/getAdminThemes', '', METHODS.GET).then(...THEN_ARGS);
    }

    /**
     * @name createAdminTheme
     * @param data the data that will be sent to the BE to define a theme
     * @desc this call will create a new theme defined by the admin
     * @return $http promise
     */
    function createAdminTheme(data: { name: string, json: any, isActive: boolean }): ng.IPromise<any> {
        let postData = '';

        postData += 'name=' + encodeURIComponent(data.name);
        postData += '&json=' + encodeURIComponent(JSON.stringify(data.json));
        postData += '&isActive=' + encodeURIComponent(data.isActive);

        return HTTPReq('/api/themes/createAdminTheme', postData, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name editAdminTheme
     * @param data the data that will be sent to the BE to define a theme
     * @desc this call will edit an existing admin theme
     * @return $http promise
     */
    function editAdminTheme(data: { id: string, name: string, json: any, isActive: boolean }): ng.IPromise<any> {
        let postData = '';

        postData += '&id=' + encodeURIComponent(data.id);
        postData += '&name=' + encodeURIComponent(data.name);
        postData += '&json=' + encodeURIComponent(JSON.stringify(data.json));
        postData += '&isActive=' + encodeURIComponent(data.isActive);

        return HTTPReq('/api/themes/editAdminTheme', postData, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name deleteAdminTheme
     * @param data the data that will be sent to the BE to define a theme
     * @desc this call will delete an existing admin theme
     * @return $http promise
     */
    function deleteAdminTheme(data: { id: string }): ng.IPromise<any> {
        const postData = 'id=' + encodeURIComponent(data.id);

        return HTTPReq('/api/themes/deleteAdminTheme', postData, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name setActiveAdminTheme
     * @param data the data that will be sent to the BE to set the admin theme
     * @desc this call will set an existing admin theme
     * @return $http promise
     */
    function setActiveAdminTheme(data: { id: string }): ng.IPromise<any> {
        const postData = 'id=' + encodeURIComponent(data.id);

        return HTTPReq('/api/themes/setActiveAdminTheme', postData, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name setAllAdminThemesInactive
     * @desc this call will set all admin themes inactive
     * @return $http promise
     */
    function setAllAdminThemesInactive(): ng.IPromise<any> {
        return HTTPReq('/api/themes/setAllAdminThemesInactive', '', METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name setCookie
     * @param insightId - insight id
     * @param secret - user defined secret
     * @desc sets cookie to allow user to share session
     * @return $http promise
     */
    function setCookie(insightId: string, secret: string): ng.IPromise<any> {
        let postData = 'i=' + insightId + '&s=' + secret;
        
        return HTTPReq('/api/auth/cookie', postData, METHODS.POST).then(...THEN_ARGS);
    }

    /**
     * @name invalidateSession
     * @desc invalidates the current session
     * @return $http promise
     */
    function invalidateSession() {
        return HTTPReq('/api/session/invalidateSession', '', METHODS.GET).then(...THEN_ARGS);
    }

    /**
     * @name HTTPReq
     * @param route - url endpoint
     * @param data - data for POSTs
     * @param method GET | POST
     * @param param get request params
     * @desc helper function for http reqs
     * @return promise
     */
    function HTTPReq(route: string, data: string | FormData, method: string, param?: any): ng.IPromise<any> {
        const url = ENDPOINT.URL + route,
            cache = false;
        let headers: { 'Content-Type': string | undefined } = {
                    'Content-Type': 'application/x-www-form-urlencoded'
                },
            params;
        if (typeof data !== 'string') {
            headers = {
                'Content-Type': undefined
            };
        }
        if (param) {
            params = param;
        }

        return $http({
            url,
            data,
            method,
            cache,
            headers,
            params
        });
    }

    return {
        getSearchInsightsResults: getSearchInsightsResults,
        uploadFile: uploadFile,
        checkHeaders: checkHeaders,
        runPixel: runPixel,
        getInsightState: getInsightState,
        getPipeline: getPipeline,
        createUser: createUser,
        updateUser: updateUser,
        addNewUser: addNewUser,
        loginUser: loginUser,
        logoutUser: logoutUser,
        console: console,
        downloadFile: downloadFile,
        uploadAppImage: uploadAppImage,
        deleteAppImage: deleteAppImage,
        uploadInsightImage: uploadInsightImage,
        deleteInsightImage: deleteInsightImage,
        activeLogins: activeLogins,
        login: login,
        logout: logout,
        getUserInfo: getUserInfo,
        backendConfig: backendConfig,
        isAdmin: isAdmin,
        getAllEngines: getAllEngines,
        removeAdminUser: removeAdminUser,
        getUserInformation: getUserInformation,
        getAllDBUsers: getAllDBUsers,
        getAllUserDbs: getAllUserDbs,

        // auth/app
        getAppUsers: getAppUsers,
        getUserAppPermission: getUserAppPermission,
        addAppUserPermission: addAppUserPermission,
        editAppUserPermission: editAppUserPermission,
        removeAppUserPermission: removeAppUserPermission,
        setAppGlobal: setAppGlobal,
        setAppVisibility: setAppVisibility,
        getApps: getApps,

        // insights permissions
        getInsightUsers: getInsightUsers,
        getAdminInsightUsers: getAdminInsightUsers,
        getUserInsightPermission: getUserInsightPermission,
        addInsightUserPermission: addInsightUserPermission,
        addAdminInsightUserPermission: addAdminInsightUserPermission,
        editInsightUserPermission: editInsightUserPermission,
        editAdminInsightUserPermission: editAdminInsightUserPermission,
        removeInsightUserPermission: removeInsightUserPermission,
        removeAdminInsightUserPermission: removeAdminInsightUserPermission,
        setInsightGlobal: setInsightGlobal,
        setAdminInsightGlobal: setAdminInsightGlobal,
        getAdminAppInsights: getAdminAppInsights,
        deleteAdminInsight: deleteAdminInsight,
        saveFilesInInsightAsDb: saveFilesInInsightAsDb,
        loginsAllowed: loginsAllowed,
        loginProperties: loginProperties,
        modifyLoginProperties: modifyLoginProperties,
        getAdminThemes: getAdminThemes,
        createAdminTheme: createAdminTheme,
        editAdminTheme: editAdminTheme,
        deleteAdminTheme: deleteAdminTheme,
        setActiveAdminTheme: setActiveAdminTheme,
        setAllAdminThemesInactive: setAllAdminThemesInactive,
        setCookie: setCookie,
        invalidateSession: invalidateSession
    };
}