import angular from 'angular';

import Resizable from '../../utility/resizable.ts';
import Utility from '../../utility/utility.js';
import { PANEL_TYPES } from '../../constants.js';

import './workspace.scss';

import './workspace-save/workspace-save.directive.ts';

export default angular.module('app.workspace.directive', [
    'app.workspace.workspace-save'
]).directive('workspace', workspaceDirective);

workspaceDirective.$inject = ['$timeout'];

function workspaceDirective($timeout) {
    workspaceCtrl.$inject = ['$scope'];
    workspaceLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^insight'],
        scope: {},
        controller: workspaceCtrl,
        controllerAs: 'workspace',
        bindToController: {},
        template: require('./workspace.directive.html'),
        link: workspaceLink,
        replace: true
    };

    function workspaceCtrl() { }

    function workspaceLink(scope, ele, attrs, ctrl) {
        let resizeTimer,
            workspaceMainEle: HTMLElement,
            workspaceMenuEle: HTMLElement,
            workspaceTerminalEle: HTMLElement,
            workspaceInputEle: HTMLElement;

        // TODO: build switching here (between pipeline, workbook, and presentation)
        scope.insightCtrl = ctrl[0];

        scope.workspace.name = {
            open: false,
            visible: '',
            updated: ''
        };

        scope.workspace.save = {
            open: false
        };

        scope.workspace.presentation = false;

        scope.workspace.sheets = [];
        scope.workspace.menu = {};
        scope.workspace.terminal = {};

        scope.workspace.openName = openName;
        scope.workspace.closeName = closeName;
        scope.workspace.openSave = openSave;
        scope.workspace.closeSave = closeSave;
        scope.workspace.newSheet = newSheet;
        scope.workspace.selectSheet = selectSheet;
        scope.workspace.openMenu = openMenu;
        scope.workspace.toggleTerminal = toggleTerminal;
        scope.workspace.addPanel = addPanel;
        scope.workspace.togglePresentation = togglePresentation;
        scope.workspace.getContent = getContent;
        scope.workspace.getMenuContent = getMenuContent;

        /**Name */
        /**
         * @name openName
         * @desc open the name
         */
        function openName($event: JQueryEventObject): void {
            if (scope.workspace.presentation || scope.workspace.name.open) {
                return;
            }
            // Only open on click or on enter
            if ($event.type === 'click' || ($event.type === 'keypress' && $event.keyCode === 13)) {
                scope.workspace.name.updated = scope.workspace.name.visible;
                scope.workspace.name.open = true;

                workspaceInputEle.focus();
            }
        }

        /**
         * @name closeName
         * @desc open the name
         * @param save - save the name change?
         */
        function closeName(save: boolean): void {
            if (save) {
                scope.insightCtrl.emit('change-insight-name', {
                    name: scope.workspace.name.updated
                });
            }
            scope.workspace.name.open = false;
        }


        /**Save */
        /**
         * @name openSave
         * @desc open the save
         */
        function openSave(): void {
            scope.workspace.save.open = true;
        }

        /**
         * @name closeSave
         * @desc open the save
         */
        function closeSave(): void {
            scope.workspace.save.open = false;
        }

        /**Sheet */
        /**
         * @name newSheet
         * @desc called when a sheet is added
         */
        function newSheet(): void {
            let counter = scope.insightCtrl.getWorkbook('worksheetCounter');

            // add one
            counter++

            // emit
            scope.insightCtrl.execute([
                {
                    type: 'Pixel',
                    components: [`AddSheet("${counter}")`],
                    terminal: true
                }
            ]);
        }

        /**
         * @name selectSheet
         * @param sheetId - sheetId to select
         * @desc called when a sheet is selected
         */
        function selectSheet(sheetId: string): void {
            scope.insightCtrl.emit('select-worksheet', {
                sheetId: sheetId
            });
        }
        /**Menu */
        /**
         * @name openMenu
         * @desc open the side menu
         */
        function openMenu(): void {
            const menu = Utility.freeze(scope.workspace.menu);

            // already open
            if (menu.open) {
                return;
            }

            // open it
            menu.open = true;

            scope.insightCtrl.emit('change-workspace-menu', {
                options: menu
            });
        }

        /** Terminal */
        /**
         * @name toggleTerminal
         * @desc toggle the terminal open or closed
         */
        function toggleTerminal(): void {
            if (scope.workspace.terminal.open) {
                scope.insightCtrl.emit('close-workspace-terminal');
            } else {
                scope.insightCtrl.emit('open-workspace-terminal');
            }
        }

        /**
        * @name addPanel
        * @desc add a panel to the current sheet
        * @param $event - DOM event
        * @param view - view for panel to add
        * @param menu - open the menu?
        */
        function addPanel($event: KeyboardEvent, view: string, visualize: boolean): void {
            let panelId: number = scope.insightCtrl.getShared('panelCounter'),
                worksheet = scope.insightCtrl.getWorkbook('worksheet'),
                pixelComponents: any = [];

            // increment
            panelId++;

            if ($event && ($event.metaKey || $event.ctrlKey || $event.keyCode === 17 || $event.keyCode === 224)) {
                worksheet = scope.insightCtrl.getWorkbook('worksheetCounter') + 1;

                pixelComponents.push({
                    type: 'Pixel',
                    components: [`AddSheet("${worksheet}")`],
                    terminal: true
                });
            }

            pixelComponents = pixelComponents.concat([
                {
                    type: 'addPanel',
                    components: [panelId, worksheet],
                    terminal: true
                },
                {
                    type: 'panel',
                    components: [panelId]
                },
                {
                    type: 'addPanelConfig',
                    components: [{
                        type: PANEL_TYPES.GOLDEN
                    }],
                    terminal: true
                },
                {
                    type: 'panel',
                    components: [panelId]
                },
                {
                    type: 'addPanelEvents',
                    components: [{
                        onSingleClick: {
                            Unfilter: [{
                                panel: '',
                                query: '<encode>(<Frame> | UnfilterFrame(<SelectedColumn>));</encode>',
                                options: {},
                                refresh: false,
                                default: true,
                                disabledVisuals: ['Grid', 'Sunburst'],
                                disabled: false
                            }]
                        },
                        onBrush: {
                            Filter: [{
                                panel: '',
                                query: '<encode>if((IsEmpty(<SelectedValues>)),(<Frame> | UnfilterFrame(<SelectedColumn>)), (<Frame> | SetFrameFilter(<SelectedColumn>==<SelectedValues>)));</encode>',
                                options: {},
                                refresh: false,
                                default: true,
                                disabled: false
                            }]
                        }
                    }],
                    terminal: true
                },
                {
                    type: 'panel',
                    components: [panelId]
                },
                {
                    type: 'retrievePanelEvents',
                    components: [],
                    terminal: true
                },
                {
                    type: 'panel',
                    components: [panelId]
                },
                {
                    type: 'setPanelView',
                    components: [view],
                    terminal: true
                }
            ]);

            // open the menu
            let callback = () => { };
            if (visualize) {
                callback = () => {
                    // semossCoreService.emit('change-widget-tab', {
                    //     widgetId: $scope.widget.widgetId,
                    //     tab: 'view'
                    // });

                    // open the menu
                    openMenu();
                };
            }

            // execute the pixel
            scope.insightCtrl.execute(pixelComponents, callback);
        }

        /**
        * @name togglePresentation
        * @desc called to toggle presentation mode
        */
        function togglePresentation(): void {
            const presentation = !scope.workspace.presentation;

            scope.insightCtrl.emit('change-presentation', {
                presentation: presentation
            });
        }

        /** Updates */
        /**
         * @name updateName
         * @desc called to update when the name changes
         */
        function updateName(): void {
            if (scope.workspace.name.open) {
                closeName(false);
            }

            scope.workspace.name.visible = scope.insightCtrl.getShared('insight.name');
        }

        /**
         * @name updateWorkbook
         * @desc called when the sheet information changes
         */
        function updateWorkbook(): void {
            const book = scope.insightCtrl.getWorkbook();

            // set the selected sheet
            scope.workspace.sheets = [];
            for (let sheetId in book.worksheets) {
                if (book.worksheets.hasOwnProperty(sheetId)) {
                    if (sheetId !== book.worksheet) {
                        scope.workspace.sheets.push({
                            sheetId: book.worksheets[sheetId].sheetId,
                            sheetLabel: book.worksheets[sheetId].sheetLabel
                        });
                    }
                }
            }

            scope.workspace.sheet = {
                sheetId: book.worksheets.hasOwnProperty(book.worksheet) ? book.worksheets[book.worksheet].sheetId : -1,
                sheetLabel: book.worksheets.hasOwnProperty(book.worksheet) ? book.worksheets[book.worksheet].sheetLabel : ''
            };
        }

        /**
         * @name updateMenu
         * @desc called to update when the menu information changes
         */
        function updateMenu(): void {
            scope.workspace.menu = scope.insightCtrl.getWorkspace('menu');

            // set the menu size
            if (scope.workspace.menu.open) {
                workspaceMainEle.style.right = `${100 - scope.workspace.menu.current}%`;
                workspaceMenuEle.style.left = scope.workspace.menu.current + '%';
                workspaceMenuEle.style.width = `${100 - scope.workspace.menu.current}%`;
            } else {
                workspaceMainEle.style.right = '0';
                workspaceMenuEle.style.left = '100%';
                workspaceMenuEle.style.width = '0%';
            }
        }

        /**
         * @name updateTerminal
         * @desc called to update when the menu information changes
         */
        function updateTerminal(): void {
            scope.workspace.terminal = scope.insightCtrl.getWorkspace('terminal');

            // set menu components
            if (scope.workspace.terminal.open && scope.workspace.terminal.view === 'inline') {
                workspaceMenuEle.style.bottom = `${100 - scope.workspace.terminal.current}%`;
                workspaceMainEle.style.bottom = `${100 - scope.workspace.terminal.current}%`;
            } else {
                workspaceMenuEle.style.bottom = '0';
                workspaceMainEle.style.bottom = '0';
            }

            // set the terminal size
            if (scope.workspace.terminal.open && (scope.workspace.terminal.view === 'overlay' || scope.workspace.terminal.view === 'inline')) {
                workspaceTerminalEle.style.display = '';
                workspaceTerminalEle.style.top = scope.workspace.terminal.current + '%';
                workspaceTerminalEle.style.height = `${100 - scope.workspace.terminal.current}%`;
            } else {
                workspaceTerminalEle.style.display = 'none';
                workspaceTerminalEle.style.top = '100%';
                workspaceTerminalEle.style.height = '0%';
            }
        }


        /**
         * @name updatedPresentation
         * @desc called when the presentation information changes
         */
        function updatedPresentation(): void {
            scope.workspace.presentation = scope.insightCtrl.getWorkspace('presentation');

            // close everything
            if (scope.workspace.presentation) {
                closeName(false);
                closeSave();
            }
        }


        /** Getters */
        /**
         * @name getContent
         * @desc get the content to paint
         * @returns html to render
         */
        function getContent(): string {
            return '<workbook></workbook>';
        }

        /**
         * @name getMenuContent
         * @desc get the side content to paint
         * @returns html to render
         */
        function getMenuContent(): string {
            if (scope.workspace.terminal.open && scope.workspace.terminal.view === 'side') {
                return `<terminal ng-if="workspace.terminal.open && (workspace.terminal.view === 'side')" location="workspace"></terminal>`
            }

            const sheetId = scope.insightCtrl.getWorkbook('worksheet'),
                panelId = scope.insightCtrl.getWorksheet(sheetId, 'panel');

            if (panelId) {
                const widgetType = scope.insightCtrl.getShared('type');

                if (widgetType === 'insight') {
                    const widgetId = 'SMSSWidget' + scope.insightCtrl.insightID + '___' + panelId;
                    return `<widget widget-id="${widgetId}"><widget-menu></widget-menu></widget>`;
                }

                return '<widget-placeholder><widget-placeholder>';
            }

            return '<div style="position: absolute; left: 50%; top: 50%; transform:translate(-50%, -50%);">No panel is selected. Please select one.</div>';
        }


        /** Initialize */
        /**
         * @name initialize
         * @desc initializes the workspace directive
         * @returns {void}
         */
        function initialize(): void {
            let menuResizable: any,
                terminalResizable: any,
                changedNameListener: () => {},
                addedWorksheetListener: () => {},
                closedWorksheetListener: () => {},
                selectedWorksheetListener: () => {},
                updatedWorksheetListener: () => {},
                changedWorkspaceMenuListener: () => {},
                updatedWorkspaceTerminalListener: () => {},
                savedInsightListener: () => {},
                updatedPresentationListener: () => {};

            // get elements          
            workspaceMainEle = ele[0].querySelector('#workspace__main');
            workspaceMenuEle = ele[0].querySelector('#workspace__menu');
            workspaceTerminalEle = ele[0].querySelector('#workspace__terminal');
            workspaceInputEle = ele[0].querySelector('#workspace__main__top__list__item__group__content__input');


            // set up the resizable
            menuResizable = Resizable({
                available: ['W'],
                unit: '%',
                content: workspaceMenuEle,
                container: ele[0],
                restrict: {
                    minimumWidth: '20%',
                    maximumWidth: '70%'
                },
                on: function (top, left, height, width) {
                    // trigger the digest if we wait (to repaint)
                    if (resizeTimer) {
                        $timeout.cancel(resizeTimer);
                    }

                    resizeTimer = $timeout(function () {
                        $timeout.cancel(resizeTimer);
                    }, 500);
                },
                stop: function (top, left, height, width) {
                    const menu = Utility.freeze(scope.workspace.menu);

                    workspaceMainEle.style.right = `${100 - left}%`;

                    menu.dragged = left;

                    scope.insightCtrl.emit('change-workspace-menu', {
                        options: menu
                    });

                    // trigger the digest again (to repaint)
                    if (resizeTimer) {
                        $timeout.cancel(resizeTimer);
                    }

                    resizeTimer = $timeout(function () {
                        $timeout.cancel(resizeTimer);
                    });
                }
            });

            terminalResizable = Resizable({
                available: ['N'],
                unit: '%',
                content: workspaceTerminalEle,
                container: ele[0],
                restrict: {
                    minimumHeight: '20%',
                    maximumHeight: '90%'
                },
                on: function (top, left, height, width) {
                    // trigger the digest if we wait (to repaint)
                    if (resizeTimer) {
                        $timeout.cancel(resizeTimer);
                    }

                    resizeTimer = $timeout(function () {
                        $timeout.cancel(resizeTimer);
                    }, 500);
                },
                stop: function (top, left, height, width) {
                    const terminal = Utility.freeze(scope.workspace.terminal);

                    if (terminal.view === 'inline') {
                        workspaceMenuEle.style.bottom = `${100 - top}%`;
                        workspaceMainEle.style.bottom = `${100 - top}%`;
                    }

                    scope.insightCtrl.emit('drag-workspace-terminal', {
                        dragged: top
                    });

                    // trigger the digest again (to repaint)
                    if (resizeTimer) {
                        $timeout.cancel(resizeTimer);
                    }

                    resizeTimer = $timeout(function () {
                        $timeout.cancel(resizeTimer);
                    });
                }
            });

            // register listeners
            changedNameListener = scope.insightCtrl.on('changed-insight-name', updateName);
            addedWorksheetListener = scope.insightCtrl.on('added-worksheet', updateWorkbook);
            closedWorksheetListener = scope.insightCtrl.on('closed-worksheet', updateWorkbook);
            selectedWorksheetListener = scope.insightCtrl.on('selected-worksheet', updateWorkbook);
            updatedWorksheetListener = scope.insightCtrl.on('updated-worksheet', updateWorkbook);
            changedWorkspaceMenuListener = scope.insightCtrl.on('changed-workspace-menu', updateMenu);
            updatedWorkspaceTerminalListener = scope.insightCtrl.on('updated-workspace-terminal', updateTerminal);
            savedInsightListener = scope.insightCtrl.on('saved-insight', updateName);
            updatedPresentationListener = scope.insightCtrl.on('updated-presentation', updatedPresentation);

            scope.$on('$destroy', function () {
                console.log('destroying workspace....');
                changedNameListener();
                addedWorksheetListener
                closedWorksheetListener();
                selectedWorksheetListener();
                updatedWorksheetListener();
                changedWorkspaceMenuListener();
                updatedWorkspaceTerminalListener();
                savedInsightListener();
                updatedPresentationListener();

                menuResizable.destroy();
                terminalResizable.destroy();
            });

            updateName();
            updateWorkbook();
            updateMenu();
            updateTerminal();
            updatedPresentation();
        }

        initialize();
    }
}