import angular from 'angular';
import { LayoutManager } from './LayoutManager';
import { PANEL_TYPES } from '../../constants.js';

import './worksheet.scss';

import './worksheet-settings/worksheet-settings.directive.ts';

export default angular.module('app.worksheet.directive', [
    'app.worksheet.worksheet-settings'
])
    .directive('worksheet', worksheetDirective);

worksheetDirective.$inject = ['$compile', '$timeout'];

function worksheetDirective($compile, $timeout) {
    worksheetCtrl.$inject = [];
    worksheetLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: ['^insight'],
        template: require('./worksheet.directive.html'),
        scope: {},
        controller: worksheetCtrl,
        controllerAs: 'worksheet',
        bindToController: {
            sheetId: '@'
        },
        link: worksheetLink,
        replace: true
    };

    function worksheetCtrl() { }

    function worksheetLink(scope, ele, attrs, ctrl) {
        let worksheetEle,
            worksheetGoldenEle,
            layoutManager,
            widgetScope = {},
            changeTimeout;

        scope.insightCtrl = ctrl[0];

        scope.worksheet.PANEL_TYPES = PANEL_TYPES;

        scope.worksheet.panels = {};

        scope.worksheet.popover = {
            save: false
        };

        scope.worksheet.onChange = onChange;
        scope.worksheet.selectPanel = selectPanel;
        scope.worksheet.closePanel = closePanel;
        scope.worksheet.clonePanel = clonePanel;

        /*** Panel */
        /**
         * @name onChange
         * @param panelId - panel id
         * @param changes - what has changed in the config
         * @desc when user moves or resizes a panel, this updates the necessary data structure
         */
        function onChange(panelId: string, changes: any): void {
            const config = scope.worksheet.panels[panelId].config;

            for (let change in changes) {
                if (changes.hasOwnProperty(change)) {
                    config[change] = changes[change];
                }
            }

            scope.insightCtrl.emit('cache-panel', {
                insightID: scope.insightCtrl.insightID,
                panelId: panelId,
                panel: {
                    config: config
                }
            });
        }

        /**
         * @name selectPanel
         * @param panelId to select
         * @desc select the panel
         */
        function selectPanel(panelId: string): void {
            const presentation = scope.insightCtrl.getWorkspace('presentation');

            // we allow selecting only when it isn't in presentation mode
            if (presentation) {
                return;
            }

            scope.insightCtrl.emit('select-panel', {
                sheetId: scope.worksheet.sheetId,
                panelId: panelId
            });
        }

        /**
         * @name closePanel
         * @param panelId - id of panel to remove
         * @desc close the panel
         */
        function closePanel(panelId: string): void {
            scope.insightCtrl.execute([
                {
                    type: 'closePanel',
                    components: [panelId],
                    terminal: true
                }
            ]);
        }

        /**
         * @name clonePanel
         * @param panelId - id to clone
         * @desc clone the panel
         */
        function clonePanel(event: JQueryEventObject, panelId: string): void {
            let newCloneId: number = scope.insightCtrl.getShared('panelCounter'),
                openNewSheet: boolean = event.ctrlKey;
            // increment
            newCloneId++;

            if (openNewSheet) {
                let sheetId = scope.insightCtrl.getWorkbook('worksheetCounter');

                // add one
                sheetId++

                scope.insightCtrl.execute([
                    {
                        type: 'Pixel',
                        components: [`AddSheet("${sheetId}")`],
                        terminal: true
                    },
                    {
                        type: 'clonePanel',
                        components: [panelId, newCloneId, sheetId],
                        terminal: true
                    }
                ]);
            } else {
                scope.insightCtrl.execute([
                    {
                        type: 'panel',
                        components: [panelId]
                    },
                    {
                        type: 'clonePanel',
                        components: [newCloneId],
                        terminal: true
                    }
                ]);
            }
        }

        /**
         * @name changePanel
         * @param
         * @desc the panel has changed
         */
        function changePanel(): void {
            if (changeTimeout) {
                $timeout.cancel(changeTimeout);
            }

            // debounce
            changeTimeout = $timeout(function () {
                $timeout.cancel(changeTimeout);
            }, 100);
        }

        /**
         * @name toggleMaximize
         * @param panelStatus - the current panel status to set to
         * @desc set the panel status
         * @returns {void}
         */
        function toggleMaximize(panelId: string, panelStatus: string): void {
            scope.worksheet.panels[panelId].config.panelstatus = panelStatus;
            // update the changes
            onChange(panelId, {
                panelstatus: panelStatus
            });
        }

        /*** Layout */
        /**
         * @name initializeLayout
         * @desc initializes the worksheet layout
         */
        function initializeLayout(): void {
            const golden = scope.insightCtrl.getWorkbook(`worksheets.${scope.worksheet.sheetId}.golden`) || {};

            layoutManager = new LayoutManager({
                ele: worksheetGoldenEle,
                content: golden.content || [],
                compile: (container, state) => {
                    const widgetId = `SMSSWidget${scope.insightCtrl.insightID}___${state.panelId}`;
                    container
                        .getElement()
                        .html(`<widget widget-id="${widgetId}"><widget-view></widget-view></widget>`);

                    // timeout so that it is appended to the DOM
                    setTimeout(() => {
                        // destroy the scope
                        if (widgetScope.hasOwnProperty(state.panelId)) {
                            widgetScope[state.panelId].$destroy();
                        }

                        widgetScope[state.panelId] = scope.$new();

                        // compile the element
                        $compile(container.getElement())(widgetScope[state.panelId])
                    });
                },
                destroy: (state) => {
                    if (widgetScope.hasOwnProperty(state.panelId)) {
                        widgetScope[state.panelId].$destroy();
                    }
                },
                events: {
                    select: (panelId: string) => {
                        selectPanel(panelId);

                        // need to digest
                        $timeout();
                    },
                    close: closePanel,
                    clone: clonePanel,
                    change: changePanel,
                    toggleMaximize: toggleMaximize
                }
            });

            // add a resize listener
            scope.$watch(function () {
                return ele[0].offsetHeight + '-' + ele[0].offsetWidth;
            }, function (newValue, oldValue) {
                if (!angular.equals(newValue, oldValue)) {
                    layoutManager.resize();
                }
            });

            updatedWorksheet();
            selectedPanel();
            updatedPresentation();
        }

        /**
         * @name saveWorksheet
         * @desc sets flag to let golden layout know not to drop insights when components are destroyed
         * called when switching sheets
         */
        function saveWorksheet(): void {
            // use this method because it will return undefined if the worksheet is not there
            const worksheet = scope.insightCtrl.getWorkbook(`worksheets.${scope.worksheet.sheetId}`);

            if (!worksheet) {
                return;
            }

            // this check is here because it can be deleted....
            if (worksheet) {
                let golden = {};

                // if one panel is golden, use this layout
                for (let panelId in scope.worksheet.panels) {
                    if (
                        scope.worksheet.panels.hasOwnProperty(panelId) &&
                        scope.worksheet.panels[panelId].config.type === PANEL_TYPES.GOLDEN
                    ) {
                        golden = layoutManager.getConfig();
                        break;
                    }
                }

                scope.insightCtrl.emit('cache-worksheet', {
                    golden: golden,
                    sheetId: scope.worksheet.sheetId
                });
            }
        }


        /*** Listeners */
        /**
         * @name updatedWorksheet
         * @desc update the worksheet
         */
        function updatedWorksheet(): void {
            const worksheet = scope.insightCtrl.getWorksheet(scope.worksheet.sheetId),
                backgroundColor = worksheet && worksheet.backgroundColor ? worksheet.backgroundColor : '';

            if (worksheetEle.style.backgroundColor !== backgroundColor) {
                worksheetEle.style.backgroundColor = backgroundColor;
            }

            // update the config
            layoutManager.render(worksheet.golden);

            //TODO: what happens when we close from a different sheet? This should probably be done in the service.............
            // remove the extra golden panels
            const addedPanels = layoutManager.getPanels();
            for (let panelIdx = 0, panelLen = addedPanels.length; panelIdx < panelLen; panelIdx++) {
                // the panel exists and is golden, so it should be part of the layout
                if (
                    worksheet.panels.hasOwnProperty(addedPanel[panelIdx]) &&
                    worksheet.panels[addedPanel[panelIdx]].config.type === PANEL_TYPES.GOLDEN
                ) {
                    continue;
                }

                // it was removed previously (or is not golden), so we don't need it
                layoutManager.removePanel(addedPanel[panelIdx]);
            }

            //add the panels
            for (let panelId in worksheet.panels) {
                if (worksheet.panels.hasOwnProperty(panelId)) {
                    // add  the layout if it isn't there
                    if (worksheet.panels[panelId].config.type === PANEL_TYPES.GOLDEN) {
                        if (addedPanels.indexOf(panelId) === -1) {
                            layoutManager.addPanel(panelId, worksheet.panels[panelId].panelLabel, worksheet.panels[panelId].config);
                        } else {
                            // update the label (might have changed)
                            layoutManager.updatePanelLabel(panelId, worksheet.panels[panelId].panelLabel);

                            // update the options (might have)
                            layoutManager.updatePanelConfig(panelId, worksheet.panels[panelId].config);
                        }
                    }

                    // save it to the variable
                    scope.worksheet.panels[panelId] = worksheet.panels[panelId];
                }
            }

            if (worksheet.hideHeaders) {
                layoutManager.hideHeaders();
            } else {
                layoutManager.showHeaders();
            }

            if (worksheet.hideBorders) {
                layoutManager.hideSplitter();

                // set it now, it won't repain
                if (worksheet.borderSize) {
                    layoutManager.resizeSplitter(worksheet.borderSize);
                }
            } else {
                // set it now, it will repaint when shown
                if (worksheet.borderSize) {
                    layoutManager.resizeSplitter(worksheet.borderSize);
                }

                layoutManager.showSplitter();
            }
        }

        /**
         * @name addedPanel
         * @desc add a panel
         * @param payload {insightID, sheetId, panelId}
         */
        function addedPanel(payload: {
            insightID: string,
            sheetId: string,
            panelId: string
        }): void {
            // check if it is correct
            if (payload.sheetId !== scope.worksheet.sheetId) {
                return;
            }

            const panel = scope.insightCtrl.getWorksheet(scope.worksheet.sheetId, `panels.${payload.panelId}`);

            // add to the layout
            if (panel.config.type === PANEL_TYPES.GOLDEN) {
                layoutManager.addPanel(payload.panelId, panel.panelLabel, panel.config);
            }

            // save it
            scope.worksheet.panels[payload.panelId] = panel;
        }

        /**
         * @name closedPanel
         * @desc close a panel
         * @param payload {insightID, sheetId, panelId}
         */
        function closedPanel(payload: {
            insightID: string,
            sheetId: string,
            panelId: string
        }): void {
            // check if it is correct
            if (payload.sheetId !== scope.worksheet.sheetId) {
                return;
            }

            // see if the panel is golden and remove it if is
            if (scope.worksheet.panels[payload.panelId].config.type === PANEL_TYPES.GOLDEN) {
                layoutManager.removePanel(payload.panelId);
            }

            delete scope.worksheet.panels[payload.panelId];
        }


        /**
         * @name updatePanel
         * @desc updates panel information
         * @param payload {insightID, sheetId, panelId}
         */
        function updatePanel(payload: { panelId: string, sheetId: string }): void {
            // check if it is correct
            if (payload.sheetId !== scope.worksheet.sheetId) {
                return;
            }

            // see if the panel exists
            if (!scope.worksheet.panels[payload.panelId]) {
                return;
            }

            const panel = scope.insightCtrl.getPanel(payload.sheetId, payload.panelId);
            // see what changed and upate accordintly

            // check if the type has changed!
            if (scope.worksheet.panels[payload.panelId].config.type !== panel.config.type) {
                // it is now golden, add it to the layout
                if (panel.config.type === PANEL_TYPES.GOLDEN) {
                    layoutManager.addPanel(payload.panelId, panel.panelLabel, panel.config);
                } else if (scope.worksheet.panels[payload.panelId].config.type === PANEL_TYPES.GOLDEN) { // not golden, if it previously was remove it
                    layoutManager.removePanel(payload.panelId);
                }

                // type has changed we need to redo the highlighting
                if (payload.panelId === scope.worksheet.selectedPanel) {
                    if (panel.config.type === PANEL_TYPES.GOLDEN) {
                        layoutManager.addHighlight(payload.panelId)
                    } else {
                        layoutManager.removeHighlight();
                    }
                }
            } else {
                let changes: any = {};
                // check if the title has changed
                if (scope.worksheet.panels[payload.panelId].panelLabel !== panel.panelLabel) {
                    if (panel.config.type === PANEL_TYPES.GOLDEN) {
                        layoutManager.updatePanelLabel(payload.panelId, panel.panelLabel);
                    }
                }

                // check if the background has changed
                if (scope.worksheet.panels[payload.panelId].config.backgroundColor !== panel.config.backgroundColor) {
                    if (panel.config.type === PANEL_TYPES.GOLDEN) {
                        changes.backgroundColor = panel.config.backgroundColor
                    }
                }

                // check if the opacity has changed
                if (scope.worksheet.panels[payload.panelId].config.opacity !== panel.config.opacity) {
                    if (panel.config.type === PANEL_TYPES.GOLDEN) {
                        changes.opacity = panel.config.opacity
                    }
                }

                if (Object.keys(changes).length > 0) {
                    layoutManager.updatePanelConfig(payload.panelId, changes);
                }
            }
            // update
            scope.worksheet.panels[payload.panelId] = panel;
        }

        /**
         * @name selectedPanel
         * @param update the selected panel
         */
        function selectedPanel() {
            scope.worksheet.selectedPanel = scope.insightCtrl.getWorksheet(scope.worksheet.sheetId, 'panel');

            // add the highlight if its active
            if (
                typeof scope.worksheet.selectedPanel !== 'undefined' &&
                scope.worksheet.panels[scope.worksheet.selectedPanel] &&
                scope.worksheet.panels[scope.worksheet.selectedPanel].config.type === PANEL_TYPES.GOLDEN) {
                layoutManager.addHighlight(scope.worksheet.selectedPanel)
            } else {
                layoutManager.removeHighlight();
            }
        }


        /**
         * @name updatedPresentation
         * @desc called when the presentation information changes
         */
        function updatedPresentation(): void {
            const presentation = scope.insightCtrl.getWorkspace('presentation');

            //only change controls if it is different
            //TODO: modify golden-layout to disable different events
            if (presentation) {
                layoutManager.disable();
            } else {
                layoutManager.enable();
            }
        }

        /**
         * @name updatePanelHighlight
         * @desc highlight the panel
         */
        function updatePanelHighlight(payload: any): void {
            layoutManager.updatePanelHighlight(payload);
        }

        /** Utility */

        /**
         * @name initialize
         * @desc initializes the worksheet directive
         */
        function initialize(): void {
            let updatedWorksheetListener: () => {},
                addedPanelListener: () => {},
                closedPanelListener: () => {},
                updatedPanelListener: () => {},
                resetPanelListener: () => {},
                updateViewListener: () => {},
                selectedPanelListener: () => {},
                saveWorksheetListener: () => {},
                updatedPresentationListener: () => {},
                updatePanelHighlightListener: () => {},
                windowResizeListener;


            worksheetEle = ele[0];
            worksheetGoldenEle = ele[0].querySelector('#worksheet__content__golden')

            // register listeners
            updatedWorksheetListener = scope.insightCtrl.on('updated-worksheet', updatedWorksheet)
            addedPanelListener = scope.insightCtrl.on('added-panel', addedPanel);
            closedPanelListener = scope.insightCtrl.on('closed-panel', closedPanel);
            updatedPanelListener = scope.insightCtrl.on('updated-panel', updatePanel);
            resetPanelListener = scope.insightCtrl.on('reset-panel', updatePanel);
            updateViewListener = scope.insightCtrl.on('update-view', updatePanel);
            selectedPanelListener = scope.insightCtrl.on('selected-panel', selectedPanel);
            saveWorksheetListener = scope.insightCtrl.on('save-worksheet', saveWorksheet);
            updatedPresentationListener = scope.insightCtrl.on('updated-presentation', updatedPresentation);
            updatePanelHighlightListener = scope.insightCtrl.on('highlight-panel', updatePanelHighlight);
            windowResizeListener = function () {
                scope.$apply();
            }

            // trigger a digest when window resizes so golden layout can resize correctly
            window.addEventListener('resize', windowResizeListener);

            scope.$on('$destroy', function () {
                console.log('destroying worksheet....');
                updatedWorksheetListener();
                addedPanelListener();
                closedPanelListener();
                updatedPanelListener();
                resetPanelListener();
                updateViewListener();
                saveWorksheetListener();
                selectedPanelListener();
                updatedPresentationListener();
                updatePanelHighlightListener();
                window.removeEventListener('resize', windowResizeListener);
                // save before we destroy it
                saveWorksheet();

                if (layoutManager) {
                    layoutManager.destroy();
                }
            });

            // initialize the layout
            initializeLayout();
        }

        initialize();
    }
}
