'use strict';

import angular from 'angular';

import './worksheet-settings.scss';
import { WORKSHEET } from '../../../constants.js';

export default angular.module('app.worksheet.worksheet-settings', [])
    .directive('worksheetSettings', worksheetSettingsDirective);


worksheetSettingsDirective.$inject = ['semossCoreService', 'VIZ_COLORS'];

function worksheetSettingsDirective(semossCoreService, VIZ_COLORS) {
    worksheetSettingsCtrl.$inject = [];
    worksheetSettingsLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        require: [],
        template: require('./worksheet-settings.directive.html'),
        scope: {
            insightCtrl: '=?'
        },
        controller: worksheetSettingsCtrl,
        controllerAs: 'worksheetSettings',
        bindToController: {
            close: '&?'
        },
        link: worksheetSettingsLink
    };

    function worksheetSettingsCtrl() { }

    function worksheetSettingsLink(scope, ele, attrs, ctrl) {
        // scope.insightCtrl = ctrl[0];

        // Variables
        scope.worksheetSettings.golden = true;
        scope.worksheetSettings.theme = VIZ_COLORS.COLOR_SEMOSS;
        scope.worksheetSettings.sheetId = undefined;
        scope.worksheetSettings.sheetLabel = '';
        scope.worksheetSettings.config = {
            backgroundColor: ''
        };

        scope.worksheetSettings.save = save;

        // Functions

        /**
         * @name save
         * @desc save the changes
         */
        function save(): void {
            let commandList: PixelCommand[] = [],
                changes = {};

            const sheetId = scope.insightCtrl.getWorkbook('worksheet'),
                worksheet = scope.insightCtrl.getWorkbook('worksheets.' + sheetId);


            // If the name has not changed, no need to run the pixel (or if it isn't set)
            if (scope.worksheetSettings.sheetLabel !== worksheet.sheetLabel || !worksheet.sheetLabel) {
                commandList.push({
                    type: 'Pixel',
                    components: [
                        `Sheet("${sheetId}") | SetSheetLabel(sheetLabel = ["${scope.worksheetSettings.sheetLabel}"])`
                    ],
                    terminal: true
                });
            }

            // see which options are changed
            for (let option in scope.worksheetSettings.config) {
                if (
                    scope.worksheetSettings.config.hasOwnProperty(option) &&
                    worksheet.hasOwnProperty(option) &&
                    JSON.stringify(scope.worksheetSettings.config[option]) !== JSON.stringify(worksheet[option])
                ) {
                    changes[option] = scope.worksheetSettings.config[option];
                }
            }

            if (Object.keys(changes).length > 0) {
                commandList.push({
                    type: 'Pixel',
                    components: [
                        `Sheet("${sheetId}") | SetSheetConfig(sheetConfig = [${JSON.stringify(changes)}])`
                    ],
                    terminal: true
                });
            }

            if (commandList.length === 0) {
                scope.insightCtrl.alert('warn', 'No options selected.')
                return;
            }

            scope.insightCtrl.execute(commandList, () => {
                scope.worksheetSettings.close();
            });
        }

        /** Updates */
        /**
         * @name updateWorksheet
         * @desc called when the sheet information changes
         */
        function updateWorksheet(): void {
            // save the workspace to get the proper layout
            semossCoreService.workspace.saveWorkspace(scope.insightCtrl.insightID);

            const sheetId = scope.insightCtrl.getWorkbook('worksheet');
            const worksheet = scope.insightCtrl.getWorkbook('worksheets.' + sheetId);

            scope.worksheetSettings.sheetLabel = worksheet.sheetLabel;
            scope.worksheetSettings.config = {
                backgroundColor: worksheet.backgroundColor,
                hideHeaders: worksheet.hideHeaders,
                hideBorders: worksheet.hideBorders,
                borderSize: worksheet.borderSize
            }

            // show golden?
            if (worksheet.hasOwnProperty('golden') &&
                JSON.stringify(worksheet.golden) !== JSON.stringify(WORKSHEET.golden)) {
                scope.worksheetSettings.golden = true;
            } else {
                scope.worksheetSettings.golden = false;
            }
        }

        /** Initialize */
        /**
         * @name initialize
         * @desc initializes the worksheet-settings directive
         */
        function initialize(): void {
            let updateLayoutListener: () => {},
                addedPanelListener: () => {},
                closedPanelListener: () => {},
                updatedPanelListener: () => {},
                resetPanelListener: () => {};

            // register listeners
            updateLayoutListener = semossCoreService.on('updated-worksheet', updateWorksheet);
            addedPanelListener = scope.insightCtrl.on('added-panel', updateWorksheet);
            closedPanelListener = scope.insightCtrl.on('closed-panel', updateWorksheet);
            updatedPanelListener = scope.insightCtrl.on('updated-panel', updateWorksheet);
            resetPanelListener = scope.insightCtrl.on('reset-panel', updateWorksheet);

            // cleanup
            scope.$on('$destroy', function () {
                console.log('destroying worksheet-settings....');
                updateLayoutListener();
                addedPanelListener();
                closedPanelListener();
                updatedPanelListener();
                resetPanelListener();
            });

            // initialize the worksheet-settings
            updateWorksheet();
        }

        initialize();
    }
}
