'use strict';

/**
 * @name widgetTabCollaboration
 * @desc widgetTabCollaboration - accessible as the context menu widget
 */
export default angular.module('app.widget-tab.widget-tab-collaboration', [])
    .directive('widgetTabCollaboration', widgetTabCollaborationDirective);

import './widget-tab-collaboration.scss';

widgetTabCollaborationDirective.$inject = ['ENDPOINT', '$timeout', 'semossCoreService'];

function widgetTabCollaborationDirective(ENDPOINT, $timeout, semossCoreService) {
    widgetTabCollaborationCtrl.$inject = [];
    widgetTabCollaborationLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        scope: {},
        require: ['^widget'],
        controllerAs: 'widgetTabCollaboration',
        bindToController: {},
        template: require('./widget-tab-collaboration.directive.html'),
        controller: widgetTabCollaborationCtrl,
        link: widgetTabCollaborationLink
    };

    function widgetTabCollaborationCtrl() {

    }

    function widgetTabCollaborationLink(scope, ele, attrs, ctrl) {
        scope.widgetCtrl = ctrl[0];

        var logoutListener,
            loginListener;

        // variables
        scope.widgetTabCollaboration.accordionSettings = {
            engine: {
                height: 100,
                disabled: false
            },
            repo: {
                height: 0,
                disabled: true
            },
            settings: {
                height: 0,
                disabled: true
            },
            sync: {
                height: 0,
                disabled: true
            }
        };

        scope.widgetTabCollaboration.engines = {
            options: [],
            selected: '',
            selectedName: '',
            isGit: false
        };

        scope.widgetTabCollaboration.repos = {
            valid: false,
            options: [],
            selected: '',
            custom: false,
            user: scope.widgetTabCollaboration.credentials && scope.widgetTabCollaboration.credentials.username ? scope.widgetTabCollaboration.credentials.username : '',
            name: ''
        };

        scope.widgetTabCollaboration.members = [];
        scope.widgetTabCollaboration.member = {
            selected: '',
            options: []
        };

        scope.widgetTabCollaboration.files = {
            options: [],
            selected: []
        };

        scope.widgetTabCollaboration.databaseSync = false;

        scope.widgetTabCollaboration.validation = {
            open: false,
            valid: false,
            type: '',
            name: '',
            title: ''
        };


        // functions
        scope.widgetTabCollaboration.login = login;
        scope.widgetTabCollaboration.getGit = getGit;
        scope.widgetTabCollaboration.toggleRepo = toggleRepo;
        scope.widgetTabCollaboration.selectRepo = selectRepo;
        scope.widgetTabCollaboration.validateRepo = validateRepo;
        scope.widgetTabCollaboration.createRepo = createRepo;
        scope.widgetTabCollaboration.deleteRepo = deleteRepo;
        scope.widgetTabCollaboration.dropRepo = dropRepo;
        scope.widgetTabCollaboration.getTeam = getTeam;
        scope.widgetTabCollaboration.searchMember = searchMember;
        scope.widgetTabCollaboration.addMember = addMember;
        scope.widgetTabCollaboration.removeMember = removeMember;
        scope.widgetTabCollaboration.getStatus = getStatus;
        scope.widgetTabCollaboration.pull = pull;
        scope.widgetTabCollaboration.syncAll = syncAll;
        scope.widgetTabCollaboration.syncSpecific = syncSpecific;
        scope.widgetTabCollaboration.checkValidation = checkValidation;
        scope.widgetTabCollaboration.confirmValidation = confirmValidation;


        /** Default */
        /**
         * @name resetCollaboration
         * @desc checks if collaboration is enabled
         * @returns {void}
         */

        function resetCollaboration() {
            semossCoreService.getCredentials('github').then(function (response) {
                // check if the user is logged in
                if (!response.name) {
                    setAuth();
                    return;
                }

                // logged in
                $timeout(function () {
                    scope.widgetTabCollaboration.credentials = {
                        username: response.name
                    };
                    scope.widgetTabCollaboration.authenticated = true;

                    resetEngines();
                });
            });
        }

        /** Auth */
        /**
         * @name setAuth
         * @desc set the accordion for an unauthenticated
         * @returns {void}
         */
        function setAuth() {
            scope.widgetTabCollaboration.authenticated = false;
        }

        /**
         * @name login
         * @desc function that is called to login
         * @returns {void}
         */
        function login() {
            var message = semossCoreService.utility.random('login');
            semossCoreService.once(message, function (response) {
                if (response.success) {
                    resetCollaboration();
                }
            });
            scope.widgetCtrl.emit('oauth-login', {
                provider: 'github',
                message: message
            });
        }

        /** Engine */
        /**
         * @name resetEngines
         * @desc function gets the list of engine options (and sets the default)
         * @returns {void}
         */
        function resetEngines() {
            var callback;

            // clear
            scope.widgetTabCollaboration.engines = {
                options: [],
                selected: '',
                selectedName: '',
                isGit: false
            };

            scope.widgetTabCollaboration.repos = {
                valid: false,
                options: [],
                selected: '',
                custom: false,
                user: scope.widgetTabCollaboration.credentials && scope.widgetTabCollaboration.credentials.username ? scope.widgetTabCollaboration.credentials.username : '',
                name: ''
            };

            scope.widgetTabCollaboration.members = [];
            scope.widgetTabCollaboration.member = {
                selected: '',
                options: []
            };

            scope.widgetTabCollaboration.files = {
                options: [],
                selected: []
            };

            scope.widgetTabCollaboration.databaseSync = false;

            scope.widgetTabCollaboration.validation = {
                open: false,
                valid: false,
                type: '',
                name: '',
                title: ''
            };

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType,
                    appId = scope.widgetCtrl.getShared('insight.app_id'),
                    recipe,
                    regEx,
                    match,
                    i,
                    len;

                if (type.indexOf('ERROR') === -1) {
                    scope.widgetTabCollaboration.engines.options = [];

                    for (i = 0, len = output.length; i < len; i++) {
                        scope.widgetTabCollaboration.engines.options.push({
                            'display': String(output[i].app_name).replace(/_/g, ' '),
                            'value': output[i].app_id,
                            'image': semossCoreService.app.generateAppImageURL(output[i].app_id)
                        });
                    }

                    // set the selected option
                    if (appId) {
                        getGit(appId);
                        return;
                    }

                    recipe = scope.widgetCtrl.getShared('steps');
                    for (i = 0, len = recipe.length; i < len; i++) {
                        // regex is looking for something between Database (  and the next ,
                        // so basically the first argument to Database
                        regEx = /Database\s*\(\s*database\s*=\s*\[\s*\"([a-zA-Z0-9-]+)+\"\s*]\s*\)/g;
                        match = recipe[i].expression.match(regEx); // regEx.exec(recipe[i].expression);
                        if (match) {
                            getGit(match[0].replace(regEx, '$1'));
                            return;
                        }
                    }

                    if (scope.widgetTabCollaboration.engines.options.length > 0) {
                        getGit(scope.widgetTabCollaboration.engines.options[0].value);
                        return;
                    }

                    scope.widgetTabCollaboration.accordionSettings = {
                        engine: {
                            height: 100,
                            disabled: false
                        },
                        repo: {
                            height: 0,
                            disabled: true
                        },
                        settings: {
                            height: 0,
                            disabled: true
                        },
                        sync: {
                            height: 0,
                            disabled: true
                        }
                    };
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            scope.widgetCtrl.meta([{
                meta: true,
                type: 'getDatabaseList',
                components: [],
                terminal: true
            }], callback);
        }

        /**
         * @name getGit
         * @desc function gets the git status based on the selected engine
         * @param {string} appId - selected appId
         * @returns {void}
         */
        function getGit(appId) {
            var callback,
                appIdx,
                currentAppName = '';

            for (appIdx = 0; appIdx < scope.widgetTabCollaboration.engines.options.length; appIdx++) {
                if (scope.widgetTabCollaboration.engines.options[appIdx].value === appId) {
                    currentAppName = scope.widgetTabCollaboration.engines.options[appIdx].display;
                }
            }

            scope.widgetTabCollaboration.engines.selected = appId;
            scope.widgetTabCollaboration.engines.selectedName = currentAppName;

            // clear
            scope.widgetTabCollaboration.repos = {
                valid: false,
                options: [],
                selected: '',
                custom: false,
                user: scope.widgetTabCollaboration.credentials && scope.widgetTabCollaboration.credentials.username ? scope.widgetTabCollaboration.credentials.username : '',
                name: ''
            };

            scope.widgetTabCollaboration.members = [];
            scope.widgetTabCollaboration.member = {
                selected: '',
                options: []
            };

            scope.widgetTabCollaboration.files = {
                options: [],
                selected: []
            };

            scope.widgetTabCollaboration.databaseSync = false;

            scope.widgetTabCollaboration.validation = {
                open: false,
                valid: false,
                type: '',
                name: '',
                title: ''
            };

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    scope.widgetTabCollaboration.engines.isGit = output;
                    if (scope.widgetTabCollaboration.engines.isGit) {
                        getRepos(scope.widgetTabCollaboration.engines.selected, false);
                    } else {
                        customRepo();
                    }
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            scope.widgetCtrl.meta(
                [{
                    meta: true,
                    type: 'isGit',
                    components: [
                        scope.widgetTabCollaboration.engines.selected
                    ],
                    terminal: true
                }],
                callback);
        }

        /** Repos */
        /**
         * @name toggleRepo
         * @desc toggle the mode for repo creation
         * @returns {void}
         */
        function toggleRepo() {
            scope.widgetTabCollaboration.repos.custom = !scope.widgetTabCollaboration.repos.custom;

            if (scope.widgetTabCollaboration.repos.custom) {
                customRepo();
            } else {
                getRepos(scope.widgetTabCollaboration.engines.selected, scope.widgetTabCollaboration.repos.selected);
            }
        }

        /**
         * @name customRepo
         * @desc customRepo
         * @returns {void}
         */
        function customRepo() {
            scope.widgetTabCollaboration.repos.custom = true;

            scope.widgetTabCollaboration.accordionSettings = {
                engine: {
                    height: 20,
                    disabled: false
                },
                repo: {
                    height: 80,
                    disabled: false
                },
                settings: {
                    height: 0,
                    disabled: true
                },
                sync: {
                    height: 0,
                    disabled: true
                }
            };
        }

        /**
         * @name getRepos
         * @desc function gets the list of repos options based on the selected engine
         * @param {string} engine - selected engine
         * @param {string} selected - use a selected repo?
         * @returns {void}
         */
        function getRepos(engine, selected) {
            var callback;

            scope.widgetTabCollaboration.engines.selected = engine;

            // clear
            scope.widgetTabCollaboration.repos = {
                valid: false,
                options: [],
                selected: '',
                custom: false,
                user: scope.widgetTabCollaboration.credentials && scope.widgetTabCollaboration.credentials.username ? scope.widgetTabCollaboration.credentials.username : '',
                name: ''
            };

            scope.widgetTabCollaboration.members = [];
            scope.widgetTabCollaboration.member = {
                selected: '',
                options: []
            };

            scope.widgetTabCollaboration.files = {
                options: [],
                selected: []
            };

            scope.widgetTabCollaboration.databaseSync = false;

            scope.widgetTabCollaboration.validation = {
                open: false,
                valid: false,
                type: '',
                name: '',
                title: ''
            };

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    scope.widgetTabCollaboration.repos.options = output;

                    if (selected) {
                        selectRepo(selected);
                    } else if (scope.widgetTabCollaboration.repos.options.length > 0) {
                        selectRepo(scope.widgetTabCollaboration.repos.options[0].name);
                    } else {
                        customRepo();
                    }
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            // get remote repos
            scope.widgetCtrl.meta([{
                meta: true,
                type: 'listAppRemotes',
                components: [
                    scope.widgetTabCollaboration.engines.selected
                ],
                terminal: true
            }], callback);
        }

        /**
         * @name selectRepo
         * @desc select a repo and get the associated information
         * @param {string} repo - location of the remote repo
         * @returns {void}
         */
        function selectRepo(repo) {
            scope.widgetTabCollaboration.repos.selected = repo;

            getTeam();
            getStatus();

            scope.widgetTabCollaboration.accordionSettings = {
                engine: {
                    height: 0,
                    disabled: false
                },
                repo: {
                    height: 20,
                    disabled: false
                },
                settings: {
                    height: 40,
                    disabled: false
                },
                sync: {
                    height: 40,
                    disabled: false
                }
            };
        }


        /**
         * @name validateRepo
         * @desc is the repo valid?
         * @returns {void}
         */
        function validateRepo() {
            var newRepo = scope.widgetTabCollaboration.repos.user + '/' + scope.widgetTabCollaboration.repos.name,
                i, len;

            if (scope.widgetTabCollaboration.repos.user.length === 0) {
                scope.widgetTabCollaboration.repos.valid = false;
                return;
            }

            if (scope.widgetTabCollaboration.repos.name.length === 0) {
                scope.widgetTabCollaboration.repos.valid = false;
                return;
            }

            for (i = 0, len = scope.widgetTabCollaboration.repos.options.length; i < len; i++) {
                if (scope.widgetTabCollaboration.repos.options[i].name === newRepo) {
                    scope.widgetTabCollaboration.repos.valid = false;
                    return;
                }
            }
            scope.widgetTabCollaboration.repos.valid = true;
        }
        /**
         * @name createRepo
         * @desc create a new repo
         * @param {boolean} valid - able to proceed with irreversible actions?
         * @returns {void}
         */
        function createRepo(valid) {
            var newRepo,
                callback;

            validateRepo();

            if (!scope.widgetTabCollaboration.repos.valid) {
                return;
            }

            newRepo = scope.widgetTabCollaboration.repos.user + '/' + scope.widgetTabCollaboration.repos.name;

            // list remote
            if (scope.widgetTabCollaboration.databaseSync && !valid) {
                scope.widgetTabCollaboration.validation.type = 'createRepo';
                scope.widgetTabCollaboration.validation.name = '';
                scope.widgetTabCollaboration.validation.title = 'Enter app name (' + scope.widgetTabCollaboration.engines.selectedName + ') to confirm';
                scope.widgetTabCollaboration.validation.valid = false;
                scope.widgetTabCollaboration.validation.open = true;
                return;
            }

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;
                if (type.indexOf('ERROR') === -1) {
                    scope.widgetCtrl.alert('success', 'Successfully created ' + newRepo);
                    scope.widgetTabCollaboration.engines.isGit = true;
                    getRepos(scope.widgetTabCollaboration.engines.selected, newRepo);
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            // list remote
            scope.widgetCtrl.meta([{
                meta: true,
                type: 'initAppRepo',
                components: [
                    scope.widgetTabCollaboration.engines.selected,
                    newRepo,
                    valid
                ],
                terminal: true
            }], callback);
        }

        /**
         * @name deleteRepo
         * @desc delete a repo
         * @param {boolean} valid - able to proceed with irreversible actions?
         * @returns {void}
         */
        function deleteRepo(valid) {
            var callback;

            // list remote
            if (!valid) {
                scope.widgetTabCollaboration.validation.type = 'deleteRepo';
                scope.widgetTabCollaboration.validation.name = '';
                scope.widgetTabCollaboration.validation.title = 'Enter app name (' + scope.widgetTabCollaboration.engines.selectedName + ') to confirm';
                scope.widgetTabCollaboration.validation.valid = false;
                scope.widgetTabCollaboration.validation.open = true;
                return;
            }

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;
                if (type.indexOf('ERROR') === -1) {
                    scope.widgetCtrl.alert('success', 'Successfully deleted ' + scope.widgetTabCollaboration.repos.selected);
                    getGit(scope.widgetTabCollaboration.engines.selected);
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            // list remote
            scope.widgetCtrl.meta([{
                meta: true,
                type: 'deleteAppRepo',
                components: [
                    scope.widgetTabCollaboration.engines.selected,
                    scope.widgetTabCollaboration.repos.selected
                ],
                terminal: true
            }], callback);
        }

        /**
         * @name dropRepo
         * @desc drop a remote repo
         * @param {boolean} valid - able to proceed with irreversible actions?
         * @returns {void}
         */
        function dropRepo(valid) {
            var callback;

            // list remote
            if (!valid) {
                scope.widgetTabCollaboration.validation.type = 'dropRepo';
                scope.widgetTabCollaboration.validation.name = '';
                scope.widgetTabCollaboration.validation.title = 'Enter app name (' + scope.widgetTabCollaboration.engines.selectedName + ') to confirm';
                scope.widgetTabCollaboration.validation.valid = false;
                scope.widgetTabCollaboration.validation.open = true;
                return;
            }

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;
                if (type.indexOf('ERROR') === -1) {
                    scope.widgetCtrl.alert('success', 'Successfully dropped ' + scope.widgetTabCollaboration.repos.selected);
                    getGit(scope.widgetTabCollaboration.engines.selected);
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            scope.widgetCtrl.meta([{
                meta: true,
                type: 'dropAppRepo',
                components: [
                    scope.widgetTabCollaboration.engines.selected,
                    scope.widgetTabCollaboration.repos.selected
                ],
                terminal: true
            }], callback);
        }


        /** Team */
        /**
         * @name getTeam
         * @desc get a team associated with a remote repo
         * @returns {void}
         */
        function getTeam() {
            var callback;

            // clear
            scope.widgetTabCollaboration.members = [];
            scope.widgetTabCollaboration.member = {
                selected: '',
                options: []
            };

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;
                if (type.indexOf('ERROR') === -1) {
                    scope.widgetTabCollaboration.members = output;
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            scope.widgetCtrl.meta([{
                meta: true,
                type: 'listAppCollaborators',
                components: [
                    scope.widgetTabCollaboration.repos.selected
                ],
                terminal: true
            }], callback);
        }

        /**
         * @name searchMember
         * @desc search a team member to your app
         * @param {string} search - search ter
         * @returns {void}
         */
        function searchMember(search) {
            var callback;

            // clear
            scope.widgetTabCollaboration.member.options = [];
            if (search) {
                scope.widgetTabCollaboration.member.loading = true;
            } else {
                scope.widgetTabCollaboration.member.loading = false;
            }

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;
                scope.widgetTabCollaboration.member.loading = false;

                if (type.indexOf('ERROR') === -1) {
                    scope.widgetTabCollaboration.member.options = output;
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            // list remote
            scope.widgetCtrl.meta([{
                meta: true,
                type: 'searchAppCollaborator',
                components: [
                    search
                ],
                terminal: true
            }], callback, []);
        }

        /**
         * @name addMember
         * @desc add a team member to your app
         * @returns {void}
         */
        function addMember() {
            var callback;

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    scope.widgetCtrl.alert('success', 'Successfully invited ' + scope.widgetTabCollaboration.member.selected[0] + ' to ' + scope.widgetTabCollaboration.repos.selected);
                    getTeam();
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            // list remote
            scope.widgetCtrl.meta([{
                meta: true,
                type: 'addAppCollaborator',
                components: [
                    scope.widgetTabCollaboration.repos.selected,
                    scope.widgetTabCollaboration.member.selected[0]
                ],
                terminal: true
            }], callback);
        }

        /**
         * @name removeMember
         * @desc removes a team member from your app
         * @param {string} removedMember - member to remove from your app
         * @returns {void}
         */
        function removeMember(removedMember) {
            var callback;

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    scope.widgetCtrl.alert('success', 'Successfully removed ' + removedMember + ' from ' + scope.widgetTabCollaboration.repos.selected);
                    getTeam();
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            scope.widgetCtrl.meta([{
                meta: true,
                type: 'removeAppCollaborator',
                components: [
                    scope.widgetTabCollaboration.repos.selected,
                    removedMember
                ],
                terminal: true
            }], callback);
        }

        /** Sync Functions */
        /**
         * @name getStatus
         * @desc gets the current status of the app
         * @returns {void}
         */
        function getStatus() {
            var callback;
            // clear
            scope.widgetTabCollaboration.files = {
                options: [],
                selected: []
            };

            scope.widgetTabCollaboration.databaseSync = false;

            scope.widgetTabCollaboration.validation = {
                open: false,
                valid: false,
                type: '',
                name: '',
                title: ''
            };

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    scope.widgetTabCollaboration.files = {
                        options: JSON.parse(JSON.stringify(output)),
                        selected: JSON.parse(JSON.stringify(output))
                    };
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            scope.widgetCtrl.meta([{
                meta: true,
                type: 'gitStatus',
                components: [
                    scope.widgetTabCollaboration.engines.selected
                ],
                terminal: true
            }], callback);
        }

        /**
         * @name pull
         * @desc pull your app
         * @returns {void}
         */
        function pull() {
            var callback;

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    scope.widgetCtrl.alert('success', 'Successfully pulled from ' + scope.widgetTabCollaboration.repos.selected);
                    getStatus();
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            scope.widgetCtrl.meta([{
                meta: true,
                type: 'syncApp',
                components: [
                    scope.widgetTabCollaboration.engines.selected,
                    scope.widgetTabCollaboration.repos.selected,
                    false,
                    scope.widgetTabCollaboration.databaseSync
                ],
                terminal: true
            }], callback);
        }

        /**
         * @name syncAll
         * @desc sync your app
         * @param {boolean} valid - able to proceed with irreversible actions?
         * @returns {void}
         */
        function syncAll(valid) {
            var callback;

            // list remote
            if (scope.widgetTabCollaboration.databaseSync && !valid) {
                scope.widgetTabCollaboration.validation.type = 'syncAll';
                scope.widgetTabCollaboration.validation.name = '';
                scope.widgetTabCollaboration.validation.title = 'Enter app name (' + scope.widgetTabCollaboration.engines.selectedName + ') to confirm';
                scope.widgetTabCollaboration.validation.valid = false;
                scope.widgetTabCollaboration.validation.open = true;
                return;
            }

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    scope.widgetCtrl.alert('success', 'Successfully synced with ' + scope.widgetTabCollaboration.repos.selected);
                    getStatus();
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            scope.widgetCtrl.meta([{
                meta: true,
                type: 'syncApp',
                components: [
                    scope.widgetTabCollaboration.engines.selected,
                    scope.widgetTabCollaboration.repos.selected,
                    true,
                    valid
                ],
                terminal: true
            }], callback);
        }

        /**
         * @name syncSpecific
         * @desc sync specific portions of your app
         * @param {boolean} valid - able to proceed with irreversible actions?
         * @returns {void}
         */
        function syncSpecific(valid) {
            var files = [],
                i, len, callback;

            if (scope.widgetTabCollaboration.databaseSync && !valid) {
                scope.widgetTabCollaboration.validation.type = 'syncSpecific';
                scope.widgetTabCollaboration.validation.name = '';
                scope.widgetTabCollaboration.validation.title = 'Enter app name ( ' + scope.widgetTabCollaboration.engines.selectedName + ' ) to confirm';
                scope.widgetTabCollaboration.validation.valid = false;
                scope.widgetTabCollaboration.validation.open = true;
                return;
            }

            for (i = 0, len = scope.widgetTabCollaboration.files.selected.length; i < len; i++) {
                files.push(scope.widgetTabCollaboration.files.selected[i].fileLoc);
            }

            if (files.length === 0) {
                return;
            }

            // register message to come back to
            callback = function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType;

                if (type.indexOf('ERROR') === -1) {
                    scope.widgetCtrl.alert('success', output);
                    getStatus();
                } else {
                    scope.widgetCtrl.alert('error', output);
                }
            };

            scope.widgetCtrl.meta([{
                meta: true,
                type: 'syncAppFilesO',
                components: [
                    scope.widgetTabCollaboration.engines.selected,
                    scope.widgetTabCollaboration.repos.selected,
                    true,
                    files,
                    valid
                ],
                terminal: true
            }], callback);
        }

        /**
         * @name checkValidation
         * @desc sync with a specific database
         * @returns {void}
         */
        function checkValidation() {
            var cleanedName = scope.widgetTabCollaboration.validation.name.replace(/ /g, '_');
            scope.widgetTabCollaboration.validation.valid = (cleanedName === scope.widgetTabCollaboration.engines.selectedName);
        }

        /**
         * @name confirmValidation
         * @desc confirm validation and trigger appropriate function
         * @returns {void}
         */
        function confirmValidation() {
            checkValidation();

            if (!scope.widgetTabCollaboration.validation.valid) {
                return;
            }

            if (scope.widgetTabCollaboration.validation.type === 'syncAll') {
                syncAll(true);
            } else if (scope.widgetTabCollaboration.validation.type === 'syncSpecific') {
                syncSpecific(true);
            } else if (scope.widgetTabCollaboration.validation.type === 'createRepo') {
                createRepo(true);
            } else if (scope.widgetTabCollaboration.validation.type === 'dropRepo') {
                dropRepo(true);
            } else if (scope.widgetTabCollaboration.validation.type === 'deleteRepo') {
                deleteRepo(true);
            }
        }

        /**
         * @name initialize
         * @desc function that is called on directive load
         * @returns {void}
         */
        function initialize() {
            logoutListener = scope.widgetCtrl.on('oauth-logout-success', resetCollaboration);
            loginListener = scope.widgetCtrl.on('oauth-login-success', resetCollaboration);

            // cleanup
            scope.$on('$destroy', function () {
                logoutListener();
                loginListener();
                console.log('DESTROY');
            });

            resetCollaboration();
        }

        initialize();
    }
}
