import angular from 'angular';
import Movable from '../../utility/movable.ts';
import Resizable from '../../utility/resizable.ts';
import {
    PANEL_TYPES
} from '../../constants.js';

import './panel.scss';

/**
 * @name panel
 * @desc Panels have multiple distinct types as well as subtypes. The type of a panel effects
 * its behavior, whereas a subtype modifies the type in either a cosmetic or functional way.
 *
 * The types of floating panels are:
 * 1. Floating/Standard: The default SEMOSS floating panel. They are the most flexibile panel type when it
 * comes to user customization. In the latest version of SEMOSS, these panels are known as floating,
 * but in previous version they were called standard.
 * 2. Collapsible (deprecated): These panels can be open and closed. When they are closed, only the panel
 * header is available for interaction. Additionally, the active widget's directive is destroyed
 * when the panel is closed (ie it will not receive messages/update!!)
 * 3. Menu (deprecated): In older versions of Semoss, menu panels took on the full height of the
 * worksheet, and stacked to the left.
 *
 * The subytypes of floating panels are. This is old:
 * 1. Unfilter: The unfilter subtype is used with floating panels. When a panel has this subtype
 * it will be a small panel that can be clicked to remove all frame filters.
 * 2. Slider: The slider subtype is used with collapsible panels. When a panel has this subtype,
 * the collapsible panel will only have a height large enough for the slider type dashboard filter.
 *
 * Panels can be sized and positioned as a percentage of the worksheet, or a specific pixel amount.
 * The config is sent in AddPanelConfig. Panel positioning properties are top, left, height and width,
 * and the values are strings (ie '5%' or '250px'). When the panel receives a message to update its config
 * save change is called to cache the new values for the panel. Afterward, when updatePanelPosition is
 * called, These values are parsed, and the appropriate style is applied to the panel.
 *
 * TODO:
 * 1. Fixed panel positioning. Previously in SEMOSS a user could fix a panel so it would always be
 * in the same spot on the screen. In order to do that, the top, left, bottom, and right values
 * of the worksheet were calculated into the position of the panel, and its css position property
 * was set to fixed. In order to move and resize the panel, its css position property as well as
 * the top, right, bottom, left properties are reset (so absolute and to the previous positioning values)
 * This allows Movable.js and Resizable.js to properly handle the panel. Once user interaction is done
 * the fixed positioning is set on the panel again. The code for this is still in the directive,
 * but needs to be updated to reflect the sheet sizing. Additionally, it might be better to have a parent
 * container for fixed panels, and a chid container with regular panels that has a scroll.
 *
 * 2. Refactor, refactor, refactor! It is a bit confusing that there is previousConverted, rendered,
 * and previous global variables to calculate panel position. In addition to this, positioning is
 * bound to the controller on init. Reorganizing that to be be simpler will reduce a large amount of
 * complexity in this file.
 */
export default angular.module('app.panel.directive', [])
    .directive('panel', panelDirective);

panelDirective.$inject = ['$timeout', 'semossCoreService'];

function panelDirective($timeout: ng.ITimeoutService, semossCoreService: SemossCoreService) {
    panelCtrl.$inject = [];
    panelLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        template: require('./panel.directive.html'),
        scope: {},
        controller: panelCtrl,
        controllerAs: 'panel',
        require: ['^insight', '^worksheet'],
        bindToController: {
            sheetId: '=',
            panelId: '='
        },
        transclude: true,
        link: panelLink
    };

    function panelCtrl() { }

    function panelLink(scope, ele, attrs, ctrl) {
        scope.insightCtrl = ctrl[0];
        scope.worksheetCtrl = ctrl[1];

        const PIXELS = 'px';

        var panelEle: any,
            panelHolderEle: any,
            containerEle: any,
            panelHeaderDragEle: any,
            panelHeaderInputEle: any,
            movable: any,
            resizable: any,
            previousConvertedHeight: string | number,
            previousConvertedWidth: string | number,
            resizeTimer: ng.IPromise<void>;

        scope.panel.TYPES = PANEL_TYPES;
        scope.panel.type = null;
        scope.panel.panelstatus = 'normalized';
        scope.panel.edit = {
            open: false,
            name: ''
        };
        scope.panel.collapsibleOpen = false;

        scope.panel.hidePanel = hidePanel;
        scope.panel.getContent = getContent;
        scope.panel.togglePanelStatus = togglePanelStatus;
        scope.panel.onPanelKeyDown = onPanelKeyDown;
        scope.panel.unfilterCol = unfilterCol;
        scope.panel.showEditWidgetName = showEditWidgetName;
        scope.panel.hideEditWidgetName = hideEditWidgetName;

        /**
         * @name getContent
         * @desc gets the content of a widget to the layout
         */
        function getContent(): string {
            if (
                scope.panel.type === scope.panel.TYPES.FLOATING ||
                scope.panel.type === scope.panel.TYPES.MENU ||
                (scope.panel.type === scope.panel.TYPES.COLLAPSIBLE && scope.panel.collapsibleOpen)
            ) {
                const widgetId = `SMSSWidget${scope.insightCtrl.insightID}___${scope.panel.panelId}`;
                return `<widget widget-id="${widgetId}"><widget-view></widget-view></widget>`;
            }

            return '';
        }


        /** Panel */

        /**
        * @name resetPanel
         * @desc reset the panel based on the options
        * @return {void}
        */
        function resetPanel(): void {
            const config = scope.insightCtrl.getPanel(scope.panel.sheetId, scope.panel.panelId, 'config');

            // this sets the type
            setOldPanel();

            // hide borders
            scope.panel.hideBorders = semossCoreService.workbook.getPanel(scope.insightCtrl.insightID, scope.panel.sheetId, 'hideBorders');

            // update label
            scope.panel.panelLabel = scope.insightCtrl.getPanel(scope.panel.sheetId, scope.panel.panelId, 'panelLabel');

            // set the panel status
            scope.panel.panelstatus = config.panelstatus;

            // if it has a zIndex, use it
            panelEle.style.zIndex = config.zIndex;

            // set the height and width
            panelEle.style.top = typeof config.top === "number" ? `${config.top}%` : config.top;
            panelEle.style.left = typeof config.left === "number" ? `${config.left}%` : config.left;
            panelEle.style.height = typeof config.height === "number" ? `${config.height}%` : config.height;
            panelEle.style.width = typeof config.width === "number" ? `${config.width}%` : config.width;

            // if it has a backgroundColor, use it
            if (config.backgroundColor) {
                panelEle.style.backgroundColor = config.backgroundColor;
            }

            // update the opacity
            if (config.opacity || config.opacity === 0) {
                panelEle.style.opacity = String((config.opacity || 100) / 100);
            }
        }

        /**
         * @name setOldPanel
         * @desc set the widget data
         */
        function setOldPanel(): void {
            let panel = scope.insightCtrl.getPanel(scope.panel.sheetId, scope.panel.panelId);

            const widgetId = `SMSSWidget${scope.insightCtrl.insightID}___${scope.panel.panelId}`;
            scope.panel.isFilter = semossCoreService.getWidget(widgetId, 'active') === 'DashboardFilter';

            // use the old one if it is there, otherwise use the config.type
            if (panel.type === scope.panel.TYPES.FLOATING) {
                scope.panel.type = scope.panel.TYPES.FLOATING;
            } else if (panel.type) {
                scope.panel.type = scope.panel.TYPES.COLLAPSIBLE;
            } else {
                scope.panel.type = panel.config.type;
            }

            /// use the old one if it is there
            if (panel.subType) {
                scope.panel.subType = panel.subType;
            } else if (panel.config && panel.config.subType) {
                scope.panel.subType = panel.config.subType;
            }
        }


        /**
         * @name cachePanel
         * @desc cache the new postion of the panels (and update the store)
         * @param top - top position of the panel (in %)
         * @param left - left position of the panel (in %)
         * @param height - height the panel (in %)
         * @param width - width of the panel (in %)
         * @return {void}
         */
        function cachePanel(top: number, left: number, height: number, width: number): void {
            const config = scope.insightCtrl.getPanel(scope.panel.sheetId, scope.panel.panelId, 'config');

            let changes:
                {
                    top?: string,
                    left?: string,
                    height?: string,
                    width?: string
                } = {},
                extracted: [number, string],
                value: number,
                unit: string,
                updated: number; // updated value

            if (scope.panel.panelstatus === 'maximized') {
                console.warn('why was this triggered?')
                return;
            }

            // first we have to convert the position to the appropriate unit (based on the ending of the unit in the)
            // check the top
            extracted = extractUnit(config.top);
            value = extracted[0];
            unit = extracted[1] || '%'
            updated = convertUnits(top, '%', unit, 'height');
            if (value !== updated) {
                changes.top = updated + unit;
            }

            // check the left
            extracted = extractUnit(config.left);
            value = extracted[0];
            unit = extracted[1] || '%'
            updated = convertUnits(left, '%', unit, 'width');
            if (value !== updated) {
                changes.left = updated + unit;
            }

            // check the height
            extracted = extractUnit(config.height);
            value = extracted[0];
            unit = extracted[1] || '%'
            updated = convertUnits(height, '%', unit, 'height');
            if (value !== updated) {
                changes.height = updated + unit;
            }

            // check the width
            extracted = extractUnit(config.width);
            value = extracted[0];
            unit = extracted[1] || '%'
            updated = convertUnits(width, '%', unit, 'width');
            if (value !== updated) {
                changes.width = updated + unit;
            }

            if (Object.keys(changes).length > 0) {
                scope.worksheetCtrl.onChange(scope.panel.panelId, changes);
            }
        }

        /**
         * @name togglePanelStatus
         * @desc toggle the panelstatus of the panel
         * @param panelstatus - new panelstatus to set
         */
        function togglePanelStatus(panelstatus: string): void {
            let old = scope.panel.panelstatus;

            if (scope.panel.panelstatus === panelstatus) {
                scope.panel.panelstatus = 'normalized';
            } else {
                scope.panel.panelstatus = panelstatus;
            }

            if (old !== scope.panel.panelstatus) {
                scope.worksheetCtrl.onChange(scope.panel.panelId, {
                    panelstatus: scope.panel.panelstatus
                });
            }
        }

        /**
         * @name onPanelKeyDown
         * @desc trigger an action when a key is pressed
         * @param $event - DOM event
         */
        function onPanelKeyDown($event: KeyboardEvent): void {
            if (scope.panel.presentation) {
                return;
            }

            if ($event && ($event.metaKey || $event.ctrlKey)) {
                if ($event.keyCode === 38) { // up
                    sizePanel('up');
                } else if ($event.keyCode === 40) { // down
                    sizePanel('down');
                }
            }
        }

        /**
         * @name sizePanel
         * @desc size the panel
         * @param {string} dir - direction to size to
         */
        function sizePanel(dir: string): void {
            const config = scope.insightCtrl.getPanel(scope.panel.sheetId, scope.panel.panelId, 'config');

            let changes: {
                height: number | string,
                width: number | string,
                panelstatus?: string
            } = {
                height: '',
                width: ''
            },
                extracted: [number, string],
                delta = 0;

            if (scope.panel.panelstatus === 'maximized') {
                return;
            }

            if (dir === 'up') {
                delta = 5;
            } else if (dir === 'down') {
                delta = -5;
            }

            extracted = extractUnit(config.height);
            extracted[0] += delta;
            if (extracted[0] < 10) {
                extracted[0] = 10;
            }

            if (config.height !== extracted[0] + extracted[1]) {
                changes.height = extracted[0] + extracted[1];
            }

            extracted = extractUnit(config.width);
            extracted[0] += delta;
            if (extracted[0] < 10) {
                extracted[0] = 10;
            }

            if (config.height !== extracted[0] + extracted[1]) {
                changes.height = extracted[0] + extracted[1];
            }


            if (Object.keys(changes).length > 0) {
                scope.worksheetCtrl.onChange(scope.panel.panelId, changes);

                // repaint it
                resetPanel();
            }
        }


        /** Collapse */
        /**
         * @name collapsibleMouseDown
         * @param event - MouseEvent
         * @desc listener added to collapsible panels for checking when drag ele has mousedown
         */
        function collapsibleMouseDown(event: MouseEvent): void {
            if (scope.panel.type === scope.panel.TYPES.COLLAPSIBLE) {
                scope.panel.pos = {
                    x: event.clientX,
                    y: event.clientY
                };
            }
        }

        /**
         * @name collapsibleMouseUp
         * @param event - MouseEvent
         * @desc listener added to collapsible panels for checking when drag ele has mouseup
         */
        function collapsibleMouseUp(event: MouseEvent): void {
            if (
                scope.panel.type === scope.panel.TYPES.COLLAPSIBLE &&
                scope.panel.pos.x === event.clientX &&
                scope.panel.pos.y === event.clientY
            ) {
                $timeout(function () {
                    toggleCollapse();
                });
            }
        }

        /**
         * @name toggleCollapse
         * @desc toggles the boolean that determines if a collapsible panel is open or closed
         */
        function toggleCollapse(): void {

            scope.panel.collapsibleOpen = !scope.panel.collapsibleOpen;

            if (!scope.panel.collapsibleOpen) {
                let lastStandard,
                    history = scope.insightCtrl.getWorksheet(scope.panel.sheetId, 'history'),
                    panels = scope.insightCtrl.getWorksheet(scope.panel.sheetId, 'panels');

                for (let i = history.length - 1; i > -1; i--) {
                    let panel = panels[history[i]];
                    if (panel.config.type === scope.panel.TYPES.GOLDEN || (panel.config.type === scope.panel.TYPES.FLOATING && !panel.subType)) {
                        lastStandard = history[i];
                        break;
                    }
                }

                if (typeof lastStandard !== 'undefined') {
                    scope.worksheetCtrl.selectPanel(lastStandard);
                }
            }
        }

        /**
         * @name closeCollapsible
         * @param event - MouseEvent
         * @desc closes panel when user clicks outside of it
         */
        function closeCollapsible(event: MouseEvent): void {
            if (scope.panel.collapsibleOpen && !panelEle.contains(event.target)) {
                $timeout(function () {
                    toggleCollapse();
                });
            }
        }

        /**
         * @name unfilterCol
         * @desc unfilters the dashboard filters instances
         */
        function unfilterCol(): void {
            const widgetId = `SMSSWidget${scope.insightCtrl.insightID}___${scope.panel.panelId}`;

            scope.panel.unfiltering = true;
            const instances = semossCoreService.getWidget(widgetId, 'view.DashboardFilter.dashboardFilter.instances');
            instances.forEach((instance) => {
                semossCoreService.emit('execute-pixel', {
                    insightID: semossCoreService.getWidget(widgetId, 'insightID'),
                    commandList: [{
                        type: 'variable',
                        components: [
                            semossCoreService.getWidget(widgetId, 'frame')
                        ]
                    },
                    {
                        'type': 'unfilterFrame',
                        'components': [instance],
                        'terminal': true
                    }
                    ]
                });
            });
        }


        /** Name */
        /**
          * @name showEditWidgetName
          * @desc show the edited widget name
          */
        function showEditWidgetName(): void {
            if (scope.panel.presentation || scope.panel.type === scope.panel.TYPES.COLLAPSIBLE) {
                return;
            }
            scope.panel.edit = {
                open: true,
                name: scope.panel.panelLabel
            };

            panelHeaderInputEle = panelEle.querySelector('#panel__holder__header__input');

            $timeout(function () {
                panelHeaderInputEle.focus();
                panelHeaderInputEle.select();
            });
        }

        /**
         * @name hideEditWidgetName
         * @desc hide the edited widget name
         * @param save - save the changes
         */
        function hideEditWidgetName(save: boolean): void {
            scope.panel.edit.open = false;

            if (save) {
                const labelOverride = semossCoreService.workbook.getPanel(scope.insightCtrl.insightID, scope.panel.sheetId, scope.panel.panelId, 'labelOverride');
                // If the name has not changed, no need to run the pixel (or if it isn't set)
                if (scope.panel.edit.name === scope.panel.panelLabel && labelOverride) {
                    return;
                }

                semossCoreService.emit('execute-pixel', {
                    insightID: scope.insightCtrl.insightID,
                    commandList: [{
                        type: 'panel',
                        components: [
                            scope.panel.panelId
                        ]
                    },
                    {
                        type: 'setPanelLabel',
                        components: [
                            scope.panel.edit.name
                        ],
                        terminal: true
                    }
                    ]
                });
            }
        }

        /** Utility */
        /**
         * @name extractUnit
         * @param value - value to extract
         * @returns the number and unit for the value
         */
        function extractUnit(value: string | number): [number, string] {
            let unit: string = '';

            // convert value to a string
            value = String(value);

            if (!value) {
                return [0, unit];
            }

            // remove white space
            value = value.trim();

            // check endings
            if (value.slice(-2) === 'px') {
                value = value.slice(0, -2);
                unit = 'px';
            } else if (value.slice(-1) === '%') {
                value = value.slice(0, -1);
                unit = '%';
            }

            return [Number(value), unit];
        }

        /**
         * @name convertUnits
         * @desc convert to the appropriate units
         * @param value - value to convert
         * @param from - unit to convert to
         * @param to - unit to convert to
         * @param type - type of conversion
         */
        function convertUnits(value: number, from: string, to: string, type: string): number {
            if (from === to) { // same unit
                return value;
            }

            if (from === '%' && to === 'px') {
                const containerBoundingClientRect = containerEle.getBoundingClientRect();
                if (type === 'height') {
                    return value / 100 * containerBoundingClientRect.height;
                } else if (type === 'width') {
                    return value / 100 * containerBoundingClientRect.width;
                }
            } else if (from === 'px' && to === '%') {
                const containerBoundingClientRect = containerEle.getBoundingClientRect();
                if (type === 'height') {
                    return value / containerBoundingClientRect.height * 100;
                } else if (type === 'width') {
                    return value / containerBoundingClientRect.width * 100;
                }
            }

            // don't have a conversion for it .... yet
            return value;
        }
        /** Update */
        /**
         * @name updatePanel
         * @desc call to update the panel
         * @param payload - {panelId}
         */
        function updatePanel(payload: { panelId: string }): void {
            if (scope.panel.panelId === payload.panelId) {
                resetPanel();
            }
        }

        /**
         * @name updateHighlight
         * @param payload 9data for checking if widget is correct and if panel should highlight
         * @desc called to update when the layout changes
         * @desc resets and updates panel
         */
        function updateHighlight(payload: { panelId: string, highlight: boolean }): void {
            if (payload.panelId === scope.panel.panelId) {
                if (payload.highlight) {
                    panelHolderEle.style.cssText = 'outline: 2px solid #278DD3; outline: 2px solid var(--color-highlight)';
                } else {
                    panelHolderEle.style.cssText = 'outline: inherit';
                }
            } else {
                panelHolderEle.style.cssText = 'outline: inherit';
            }
        }

        /**
         * @name updateSelectedPanel
         * @desc reset the position when the window or container resizes
         */
        function updateSelectedPanel(): void {
            const selectedPanel = semossCoreService.workbook.getWorksheet(scope.insightCtrl.insightID, scope.panel.sheetId, 'panel');

            scope.panel.selected = scope.panel.panelId === selectedPanel;
        }

        /**
         * @name updatePresentation
         * @desc called when the presentation information changes
         */
        function updatePresentation(): void {
            scope.panel.presentation = scope.insightCtrl.getWorkspace('presentation');

            if (scope.panel.presentation) {
                // stop editing
                hideEditWidgetName(false);

                // set the z-index to the saved value

                if (movable) {
                    movable.disable();
                }

                if (resizable) {
                    resizable.disable();
                }
            } else {
                if (movable) {
                    movable.enable();
                }

                if (resizable) {
                    resizable.enable();
                }
            }
        }

        /**
         * @name hidePanel
         * @desc checks to see if we need to hide the panel
         * @returns {boolean} true/false
         */
        function hidePanel(): boolean {
            // assumption is panel is floating and not maximized, so no need to check those
            let hide = false,
                sheet = semossCoreService.workbook.getWorksheet(scope.insightCtrl.insightID, scope.panel.sheetId),
                panelId: string;

            // look at other panels in the sheet, check to see if any of them are maximized. if so, return return.
            for (panelId in sheet.panels) {
                if (sheet.panels.hasOwnProperty(panelId)) {
                    // don't need to look at itself
                    if (sheet.panels[panelId].panelId === scope.panel.panelId) {
                        continue;
                    }

                    if (sheet.panels[panelId].config && sheet.panels[panelId].config.panelstatus === 'maximized') {
                        hide = true;
                        break;
                    }
                }
            }

            return hide;
        }

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize(): void {
            let updatedWorksheetListener: () => void,
                selectedPanelListener: () => void,
                updatedPanelListener: () => void,
                highlightPanelListener: () => void,
                resetPanelListener: () => void,
                updatedPresentationListener: () => void;

            // get the variables
            panelEle = ele[0].firstElementChild;
            containerEle = panelEle.parentElement.parentElement;
            panelHolderEle = panelEle.querySelector('#panel__holder');
            if (semossCoreService.workbook.getPanel(scope.insightCtrl.insightID, scope.panel.sheetId, scope.panel.panelId, 'subType') !== scope.panel.TYPES.UNFILTER) {
                panelHeaderDragEle = panelEle.querySelector('#panel__holder__header__drag');
            } else {
                panelHeaderDragEle = panelEle.querySelector('#panel__holder__content');
            }

            // drag
            document.addEventListener('click', closeCollapsible);
            panelHeaderDragEle.addEventListener('mousedown', collapsibleMouseDown);
            panelHeaderDragEle.addEventListener('mouseup', collapsibleMouseUp);

            updatedWorksheetListener = scope.insightCtrl.on('updated-worksheet', resetPanel)
            selectedPanelListener = scope.insightCtrl.on('selected-panel', updateSelectedPanel);
            updatedPanelListener = scope.insightCtrl.on('updated-panel', updatePanel);
            resetPanelListener = scope.insightCtrl.on('reset-panel', updatePanel);
            highlightPanelListener = semossCoreService.on('highlight-panel', updateHighlight);
            updatedPresentationListener = scope.insightCtrl.on('updated-presentation', updatePresentation);

            // add the listeners
            scope.$on('$destroy', function () {
                // drag
                document.removeEventListener('click', closeCollapsible);
                panelHeaderDragEle.removeEventListener('mousedown', collapsibleMouseDown);
                panelHeaderDragEle.removeEventListener('mouseup', collapsibleMouseUp);

                if (movable) {
                    movable.destroy();
                }

                if (resizable) {
                    resizable.destroy();
                }

                updatedWorksheetListener();
                selectedPanelListener();
                updatedPanelListener();
                resetPanelListener();
                highlightPanelListener();
                updatedPresentationListener();
            });

            // add the resize events
            resizable = Resizable({
                container: containerEle,
                content: panelEle,
                unit: '%',
                restrict: {
                    minimumHeight: "28px",
                    minimumWidth: "36px"
                },
                start: function () {
                    scope.worksheetCtrl.selectPanel(scope.panel.panelId);

                    // trigger digest
                    $timeout();
                },
                on: function () {
                    if (resizeTimer) {
                        $timeout.cancel(resizeTimer);
                    }

                    resizeTimer = $timeout(function () {
                        $timeout.cancel(resizeTimer);
                    }, 500);
                },
                stop: function (top, left, height, width) {
                    if (resizeTimer) {
                        $timeout.cancel(resizeTimer);
                    }

                    cachePanel(top, left, height, width);

                    // trigger digest
                    $timeout();
                }
            });


            // add the drag events
            movable = Movable({
                container: containerEle,
                handle: panelHeaderDragEle,
                content: panelEle,
                unit: '%',
                start: function () {
                    // trigger digest
                    $timeout();
                },
                on: function () { },
                stop: function (top, left) {
                    const config = scope.insightCtrl.getPanel(scope.panel.sheetId, scope.panel.panelId, 'config');

                    // make sure it's a number ;)
                    cachePanel(top, left, extractUnit(config.height)[0], extractUnit(config.width)[0]);

                    // trigger digest
                    $timeout();
                }
            });

            resetPanel();
            updateSelectedPanel()
            updatePresentation();

            // scroll into view
            ele[0].firstElementChild.scrollIntoView();
        }



        initialize();
    }
}
