'use strict';

import angular from 'angular';
import jsPlumb from 'jsplumb';
import panzoom from 'panzoom';
import Movable from '../../utility/movable.ts';

import './metamodel.scss';

import './metamodel-tables/metamodel-tables.directive.js';
import './metamodel-relationships/metamodel-relationships.directive.js';
import './metamodel-table/metamodel-table.directive.js';
import './metamodel-column/metamodel-column.directive.js';

export default angular.module('app.metamodel.directive', [
    'app.metamodel.metamodel-tables',
    'app.metamodel.metamodel-relationships',
    'app.metamodel.metamodel-table',
    'app.metamodel.metamodel-column'
]).directive('metamodel', metamodelDirective);


metamodelDirective.$inject = ['$compile', '$timeout'];

function metamodelDirective($compile: ng.ICompileService, $timeout: ng.ITimeoutService) {
    metamodelCtrl.$inject = [];
    metamodelLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        template: require('./metamodel.directive.html'),
        scope: {},
        controllerAs: 'metamodel',
        bindToController: {
            metamodel: '=',
            type: '=',
            edit: '&?'
        },
        controller: metamodelCtrl,
        link: metamodelLink
    };


    function metamodelCtrl() { }

    function metamodelLink(scope, ele, attrs, ctrl) {
        let metamodelGraphEle: HTMLElement,
            metamodelScope,
            plumbing,
            zoom,
            watch;

        scope.metamodel.searched = '';
        scope.metamodel.movable = {};
        scope.metamodel.searchMetamodel = searchMetamodel;
        scope.metamodel.editMetamodel = editMetamodel;

        /** Metamodel Functions */
        /**
         * @name drawMetamodel
         * @desc initialize the metamodel
         */
        function drawMetamodel(): void {
            scope.metamodel.loading = true;

            // destroy it before creating it
            destroyMetamodel();

            let html = '';


            // generate the html
            // add the table
            html += '<div>';
            for (let table in scope.metamodel.metamodel.tables) {
                if (scope.metamodel.metamodel.tables.hasOwnProperty(table)) {
                    html += generateTable(scope.metamodel.metamodel.tables[table].table);
                }
            }
            html += '</div>';

            // create a new scope
            metamodelScope = scope.$new();

            // mount and compile
            metamodelGraphEle.appendChild(angular.element(html)[0]);
            $compile(metamodelGraphEle)(metamodelScope);


            // store all of the eles and add the movable
            let eles = {};
            for (let table in scope.metamodel.metamodel.tables) {
                if (scope.metamodel.metamodel.tables.hasOwnProperty(table)) {
                    eles[scope.metamodel.metamodel.tables[table].table] = metamodelGraphEle.querySelector(`#metamodel__graph__table--${scope.metamodel.metamodel.tables[table].table}`);

                    // add movable
                    scope.metamodel.movable[scope.metamodel.metamodel.tables[table].table] = Movable({
                        handle: eles[scope.metamodel.metamodel.tables[table].table],
                        content: eles[scope.metamodel.metamodel.tables[table].table],
                        container: metamodelGraphEle,
                        restrict: {
                            top: -Infinity,
                            right: -Infinity,
                            bottom: -Infinity,
                            left: -Infinity
                        },
                        on: () => {
                            //noop
                        },
                        stop: (top: string, left: string) => {
                            // save in the data
                            removeWatch();

                            // if no position, add it
                            if (!scope.metamodel.metamodel.tables[table].hasOwnProperty('position')) {
                                scope.metamodel.metamodel.tables[table].position = {
                                    top: 0,
                                    left: 0
                                }
                            }

                            scope.metamodel.metamodel.tables[table].position.top = top;
                            scope.metamodel.metamodel.tables[table].position.left = left;


                            plumbing.revalidate(eles[scope.metamodel.metamodel.tables[table].table])

                            // new digest
                            $timeout(() => {
                                addWatch();
                            })
                        }
                    });
                }
            }

            // add edges
            for (let edgeIdx = 0, edgeLen = scope.metamodel.metamodel.relationships.length; edgeIdx < edgeLen; edgeIdx++) {
                plumbing.connect({
                    source: eles[scope.metamodel.metamodel.relationships[edgeIdx].fromTable],
                    target: eles[scope.metamodel.metamodel.relationships[edgeIdx].toTable],
                    detachable: false,
                    anchor: "AutoDefault",
                    endpoint: 'Blank',
                    connectionsDetachable: false,
                    maxConnections: -1,
                    connector: [
                        'Flowchart',
                        {
                            cssClass: 'metamodel__graph__edge__connector'
                        }
                    ]
                });
            }

            // now add the watch
            addWatch();

            scope.metamodel.loading = false;
        }

        /**
         * @name destroyMetamodel
         * @desc destroy the metamodel
         */
        function destroyMetamodel(): void {
            // remove watch
            removeWatch();

            // remove connections
            if (plumbing) {
                plumbing.reset()
            }

            // remove movable
            for (let id in scope.metamodel.movable) {
                if (scope.metamodel.movable.hasOwnProperty(id)) {
                    scope.metamodel.movable[id].destroy();
                }
            }

            // destroy the old scope
            if (metamodelScope) {
                metamodelScope.$destroy();
            }

            // remove the eles
            if (metamodelGraphEle) {
                while (metamodelGraphEle.firstChild) {
                    if (metamodelGraphEle.lastChild) {
                        metamodelGraphEle.removeChild(metamodelGraphEle.lastChild);
                    }
                }
            }
        }

        /**
         * @name searchMetamodel
         * @desc search the metamodel
         */
        function searchMetamodel(): void {
            if (metamodelGraphEle) {
                let tables = metamodelGraphEle.querySelectorAll<HTMLElement>('[metamodel-alias]') || [];
                const len = tables.length;
                if (len > 0) {
                    let searchString = scope.metamodel.searched || '';
                    searchString = searchString.toUpperCase().replace(/ /g, '_');

                    for (let i = 0; i < len; i++) {
                        // clear the old
                        let temp = tables[i].getAttribute('metamodel-alias') || '';
                        temp = temp.toUpperCase().replace(/ /g, '_');

                        if (temp.indexOf(searchString) === -1 || !searchString) {
                            tables[i].style.backgroundColor = 'transparent';
                        } else {
                            tables[i].style.backgroundColor = '#fff9e9';
                        }
                    }
                }
            }
        }

        /**
         * @name editMetamodel
         * @desc search the metamodel
         * @param {string} type - type of edit (table, column)
         * @param {string} options - options to save
         */
        function editMetamodel(type: string, options: any): void {
            if (scope.metamodel.edit) {
                scope.metamodel.edit({
                    type: type,
                    options: options
                })
            }
        }

        /** Table */
        /**
         * @name generateTable
         * @param id - id for the table to create
         * @desc generates a label for the selected table
         * @return  the html for the table
         */
        function generateTable(id: string): string {
            const table = scope.metamodel.metamodel.tables[id];

            let labelHolder = '';
            labelHolder += `<div id="metamodel__graph__table--${table.table}"
                            class="metamodel__graph__table"
                            style="top:${table.position && table.position.hasOwnProperty('top') ? table.position.top : 0}px;left:${table.position && table.position.hasOwnProperty('left') ? table.position.left : 0}px">`;
            labelHolder +=
                `<div class="metamodel__graph__table__item metamodel__graph__table__item--border smss-clear"
                    title="${table.alias}"
                    metamodel-alias="${table.alias}">
                <div class="metamodel__graph__table__item__icon smss-text smss-left">
                    <i class="fa fa-table"></i>
                </div>
                <div class="metamodel__graph__table__item__text metamodel__graph__table__item__text--wide smss-text smss-left">
                    ${table.alias}
                </div>
                ${ scope.metamodel.edit ? `<smss-btn class="smss-left smss-btn--icon" title="Edit Table: ${table.alias}" ng-click="metamodel.editMetamodel('table', {'table':'${table.table}'})">
                    <i class="fa fa-edit"></i>
                </smss-btn>` : ''}
            </div>`;

            // column list
            for (let column in table.columns) {
                if (table.columns.hasOwnProperty(column)) {
                    labelHolder +=
                        `<div class="metamodel__graph__table__item smss-clear"
                        title="${table.columns[column].alias}"
                        metamodel-alias="${table.columns[column].alias}">
                        <div class="metamodel__graph__table__item__icon smss-text smss-left">
                            ${table.columns[column].isPrimKey ? '<i class="fa fa-key"></i>' : ''}
                        </div>
                        <div class="metamodel__graph__table__item__text smss-text smss-left">
                            ${table.columns[column].alias}
                        </div>
                        <div class="metamodel__graph__table__item__icon smss-text smss-left">
                            ${table.columns[column].type === 'STRING' ? '<i class="fa fa-font"></i>' : ''}
                            ${table.columns[column].type === 'DATE' ? '<i class="fa fa-calendar-o"></i>' : ''}
                            ${table.columns[column].type === 'TIMESTAMP' ? '<i class="fa fa-clock-o"></i>' : ''}
                            ${table.columns[column].type === 'INT' || table.columns[column].type === 'DOUBLE' ? '<i class="fa fa-hashtag"></i>' : ''}
                        </div>
                        ${ scope.metamodel.edit ? `<smss-btn class="smss-left smss-btn--icon" title="Edit Column: ${table.columns[column].alias}" ng-click="metamodel.editMetamodel('column', {'table':'${table.table}', 'column': '${table.columns[column].column}'})">
                            <i class="fa fa-edit"></i>
                        </smss-btn>` : ''}
                    </div>`;
                }
            }
            labelHolder += '</div>';


            return labelHolder;
        }


        /** Helpers */
        /**
         * @name addWatch
         * @desc add the watch
         */
        function addWatch(): void {
            if (watch) {
                watch();
            }

            watch = scope.$watch(function () {
                return JSON.stringify(scope.metamodel.metamodel);
            }, function (newValue: string, oldValue: string) {
                if (newValue !== oldValue) {
                    drawMetamodel();
                }
            });
        }

        /**
         * @name removeWatch
         * @desc remove the watch
         */
        function removeWatch(): void {
            if (watch) {
                watch();
            }
        }


        /**
         * @name initialize
         * @desc initialize the module
         */
        function initialize(): void {
            metamodelGraphEle = ele[0].querySelector('#metamodel__graph');

            // set up the plumbing
            plumbing = jsPlumb.jsPlumb.getInstance({
                Container: metamodelGraphEle
            });


            // add panzoom
            zoom = panzoom(metamodelGraphEle);

            scope.$on('$destroy', function () {
                destroyMetamodel();

                if (plumbing) {
                    plumbing.reset();
                }

                if (zoom) {
                    zoom.dispose();
                }
            });

            drawMetamodel();
        }

        initialize();
    }
}
