'use strict';

export default angular.module('app.landing.directive', [])
    .directive('landing', landingDirective);

import './landing.scss';

import Utility from '@/core/utility/utility.js';

landingDirective.$inject = ['ENDPOINT', '$state', '$timeout', 'semossCoreService'];

function landingDirective(ENDPOINT, $state, $timeout, semossCoreService) {
    landingCtrl.$inject = [];
    landingLink.$inject = [];

    return {
        restrict: 'E',
        template: require('./landing.directive.html'),
        controller: landingCtrl,
        link: landingLink,
        scope: {},
        bindToController: {},
        controllerAs: 'landing'
    };

    function landingCtrl() {}

    function landingLink(scope, ele) {
        let searchTimeout;


        scope.landing.search = {
            loading: true,
            input: '',
            raw: [],
            results: []
        };

        scope.landing.tags = {
            options: [],
            selected: []
        };

        scope.landing.sort = {
            options: ['Name', 'Last Modified'],
            selected: 'Name'
        };

        scope.landing.reverse = false;
        scope.landing.showDelete = false;
        scope.landing.appToDelete = '';

        scope.landing.searchApps = searchApps;
        scope.landing.filterApps = filterApps;
        scope.landing.sortApps = sortApps;
        scope.landing.reverseApps = reverseApps;
        scope.landing.changeMode = changeMode;

        scope.landing.edit = {
            open: false,
            app: undefined
        };

        scope.landing.openApp = openApp;
        scope.landing.openEdit = openEdit;
        scope.landing.openDelete = openDelete;
        scope.landing.closeDelete = closeDelete;
        scope.landing.onDelete = onDelete;

        /** Apps */

        /**
         * @name getApps
         * @desc gets the insights for the current selected app
         * @returns {void}
         */
        function getApps() {
            let message = semossCoreService.utility.random('query-pixel');

            scope.landing.search = {
                loading: true,
                input: '',
                raw: [],
                results: []
            };


            scope.landing.tags = {
                options: [],
                selected: []
            };

            // register message to come back to
            semossCoreService.once(message, function (response) {
                let output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType[0],
                    tags = {};

                scope.landing.search.loading = false;

                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                for (let appIdx = 0, appLen = output.length; appIdx < appLen; appIdx++) {
                    output[appIdx].image = getImage(output[appIdx].app_id);
                    output[appIdx].lastModified = new Date(output[appIdx].lastModified);
                    output[appIdx].lastModifiedDisplay = getDateFormat(output[appIdx].lastModified);

                    scope.landing.search.raw.push(output[appIdx]);

                    for (let tagIdx = 0, tagLen = output[appIdx].tags.length; tagIdx < tagLen; tagIdx++) {
                        tags[output[appIdx].tags[tagIdx]] = true;
                    }
                }


                scope.landing.tags.options = Object.keys(tags).sort();
                scope.landing.tags.selected = [];

                searchApps();
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'myApps',
                    components: [],
                    terminal: true
                }],
                listeners: [], // taken care of in the directive
                response: message
            });
        }

        /**
         * @name searchApps
         * @desc searchs the app insights
         * @returns {void}
         */
        function searchApps() {
            if (searchTimeout) {
                $timeout.cancel(searchTimeout);
            }

            // debounce
            searchTimeout = $timeout(function () {
                let searchTerm = String(scope.landing.search.input || '')
                    .toUpperCase()
                    .replace(/ /g, '_');


                scope.landing.search.results = scope.landing.search.raw.filter((app) => {
                    // check if has a valid tag
                    const tagLen = scope.landing.tags.selected.length;
                    if (tagLen > 0) {
                        let hasTag = false;

                        for (let tagIdx = 0; tagIdx < tagLen; tagIdx++) {
                            if (app.tags.indexOf(scope.landing.tags.selected[tagIdx]) > -1) {
                                hasTag = true;
                                break;
                            }
                        }

                        // not part of the tag
                        if (!hasTag) {
                            return false;
                        }
                    }

                    // needs to have an input
                    if (scope.landing.search.input) {
                        let name, description;

                        name = String(app.app_name || '')
                            .replace(/ /g, '_')
                            .toUpperCase();
                        description = String(app.description || '')
                            .replace(/ /g, '_')
                            .toUpperCase();

                        if (name.indexOf(searchTerm) === -1 && description.indexOf(searchTerm) === -1) {
                            return false;
                        }
                    }

                    return true;
                });

                sortApps();

                $timeout.cancel(searchTimeout);
            }, 300);
        }

        /**
         * @name filterApps
         * @desc reverse the apps
         * @param {string} tag - tag to search
         * @returns {void}
         */
        function filterApps(tag) {
            if (scope.landing.tags.selected.indexOf(tag) === -1) {
                scope.landing.tags.selected.push(tag);
                searchApps();
            }
        }

        /**
         * @name sortApps
         * @desc sort the apps
         * @returns {void}
         */
        function sortApps() {
            if (scope.landing.sort.selected === 'Name') {
                Utility.sort(scope.landing.search.results, 'app_name');
            } else if (scope.landing.sort.selected === 'Last Modified') {
                Utility.sort(scope.landing.search.results, 'lastModified');
            }

            if (scope.landing.reverse) {
                scope.landing.search.results.reverse();
            }
        }

        /**
         * @name reverseApps
         * @desc reverse the apps
         * @returns {void}
         */
        function reverseApps() {
            scope.landing.reverse = !scope.landing.reverse;
            sortApps();
        }


        /**
         * @name openApp
         * @desc open an app
         * @param {object} app - app to open
         * @return {void}
         */
        function openApp(app) {
            // close the app
            app.open = false;

            $state.go('home.app', {
                app: app.app_id
            });
        }

        /**
         * @name getImage
         * @desc gets the image for the app
         * @param {string} appId - appId of the image
         * @returns {void}
         */
        function getImage(appId) {
            let imageUpdates = semossCoreService.getOptions('imageUpdates');

            if (imageUpdates[appId]) {
                return imageUpdates[appId];
            }

            return semossCoreService.app.generateAppImageURL(appId);
        }

        /**
         * @name getDateFormat
         * @desc format a date into the wanted format
         * @param {date} date - date the date to format
         * @returns {string} formatted date
         */
        function getDateFormat(date) {
            return date.toLocaleDateString([], {
                year: 'numeric',
                month: 'short',
                day: 'numeric'
            }) + ' ' + date.toLocaleTimeString();
        }

        /**
         * @name openEdit
         * @desc open the overlay
         * @param {object} app - app to select
         * @returns {void}
         */
        function openEdit(app) {
            // close the app
            app.open = false;

            scope.landing.edit.open = true;
            scope.landing.edit.app = app;
        }

        /**
         * @name changeMode
         * @desc change the layout mode
         * @param {string} mode - the layout mode to change to
         * @returns {void}
         */
        function changeMode(mode) {
            scope.landing.mode = mode;
            window.localStorage.setItem('smss-landing-mode', mode);
        }

        /**
         * @name openDelete
         * @desc Opens the delete overlay and sets the app id to delete
         * @param {*} app - the app to delete
         * @returns {void}
         */
        function openDelete(app) {
            scope.landing.showDelete = true;
            scope.landing.appToDelete = app.app_id;
        }

        /**
         * @name closeDelete
         * @desc Closes the delete overlay and resets the app id
         * @returns {void}
         */
        function closeDelete() {
            scope.landing.showDelete = false;
            scope.landing.appToDelete = '';
        }

        /**
         * @name onDelete
         * @desc Deletes the app
         * @returns {void}
         */
        function onDelete() {
            const appId = scope.landing.appToDelete;
            if (appId && appId.length) {
                semossCoreService.once('delete-app-end', function (response) {
                    if (response.success) {
                        scope.landing.search.results = scope.landing.search.results.filter((app) => app.app_id !== appId);
                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'App successfully deleted'
                        });
                    }
                });
                semossCoreService.emit('delete-app', {
                    appId: appId,
                    closeApp: true
                });
            } else {
                semossCoreService.emit('alert', {
                    color: 'error',
                    text: 'Error deleting app.'
                });
            }
            scope.landing.appToDelete = '';
            scope.landing.showDelete = false;
        }

        /**
         * @name updateAppImage
         * @desc update the app image urls
         * @returns {void}
         */
        function updateAppImage() {
            // loop through scope.appHomeInsights.search.results and check imageUpdates to see if image needs to be updated
            let appIdx;
            for (appIdx = 0; appIdx < scope.landing.search.results.length; appIdx++) {
                scope.landing.search.results[appIdx].image = getImage(scope.landing.search.results[appIdx].app_id);
            }
        }

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize() {
            let appListener = semossCoreService.on('update-app', updateAppImage);

            // add listeners
            scope.$on('$destroy', function () {
                appListener();
            });

            if (window.localStorage.getItem('smss-landing-mode')) {
                scope.landing.mode = window.localStorage.getItem('smss-landing-mode');
            } else {
                scope.landing.mode = 'CARD';
                window.localStorage.setItem('smss-landing-mode', 'CARD');
            }
            getApps();
        }

        initialize();
    }
}
