'use strict';

import './home-search.scss';
import Utility from '@/style/src/utility.js';
export default angular.module('app.home.home-search', [])
    .directive('homeSearch', searchDirective);

searchDirective.$inject = ['$timeout', 'semossCoreService'];

function searchDirective($timeout, semossCoreService) {
    searchCtrl.$inject = ['$scope'];
    searchLink.$inject = ['scope', 'ele'];
    return {
        restrict: 'E',
        template: require('./home-search.directive.html'),
        controller: searchCtrl,
        link: searchLink,
        scope: {},
        bindToController: {
            target: '@',
            open: '='
        },
        controllerAs: 'homeSearch',
        transclude: true
    };

    function searchCtrl() {}

    function searchLink(scope, ele) {
        var targetEle,
            appendEle,
            searchContentEle,
            resultsEle,
            searchTimeout,
            suggestionTimeout;

        /** Initializing Vars ***/
        scope.homeSearch.options = {
            canCollect: false,
            loading: false,
            input: '',
            offset: 0,
            limit: 20,
            results: []
        };

        scope.homeSearch.apps = {
            show: false,
            raw: [],
            selected: []
        };

        scope.homeSearch.suggestions = {
            enabled: false, // Set to true to enable NLP Suggestions
            active: false,
            visible: false,
            options: [],
            filteredOptions: []
        };

        scope.homeSearch.recommendations = {
            enabled: false, // Set to true to enable NLP recommendations
            show: false,
            results: [],
            searchQuery: ''
        };


        // functions
        scope.homeSearch.changeSearch = changeSearch;
        scope.homeSearch.toggleApps = toggleApps;
        scope.homeSearch.toggleSuggestions = toggleSuggestions;
        scope.homeSearch.createViz = createViz;
        scope.homeSearch.getVizSvg = getVizSvg;
        scope.homeSearch.getNLPSuggestedInsights = getNLPSuggestedInsights;
        scope.homeSearch.createNLPSuggestedInsight = createNLPSuggestedInsight;
        scope.homeSearch.keydown = keydown;
        scope.homeSearch.keyup = keyup;
        scope.homeSearch.click = click;
        scope.homeSearch.searchList = searchList;

        /**
         * @name searchList
         * @desc called when the typeahead is being searched (will filter out values based on searchterm)
         * @param {string} searchTerm - the search string from the input
         * @returns {void}
         */
        function searchList(searchTerm) {
            var filteredOptions = Utility.filter(scope.homeSearch.suggestions.options, searchTerm, 'value');
            scope.homeSearch.suggestions.filteredOptions = filteredOptions;
        }

        /**
         * @name keydown
         * @desc - calls getSuggestions on space and backspace
         * @param {*} ev - event object
         * @param {*} type - if the keydown event came from the input or the list
         * @returns {void}
         */
        function keydown(ev, type) {
            if (type === 'input' && (ev.keyCode === 32)) { // 32: space, 8: backspace
                getSuggestions();
            }
        }
        /**
         * @name keyup
         * @desc - calls getSuggestions when using the enter key for the list
         * @param {*} ev  - event object
         * @returns {void}
         */
        function keyup(ev) {
            if (ev.keyCode === 13) {
                getSuggestions();
            }
        }

        /**
         * @name click
         * @desc - calls getSuggestions when an option is clicked from the list
         * @returns {void}
         */
        function click() {
            getSuggestions();
        }

        /**
         * @name setSearch
         * @desc set the search data
         * @returns {void}
         */
        function setSearch() {
            let message = semossCoreService.utility.random('meta');

            scope.homeSearch.options.loading = true;

            const installedPackages = semossCoreService.getWidgetState('installedPackages');
            if (installedPackages) {
                if (installedPackages.hasOwnProperty('R')) {
                    // NLP Search
                    if (installedPackages.R.indexOf('udpipe') > -1) {
                        scope.homeSearch.recommendations.enabled = true;
                    }
                    // NLP Suggestions
                    if (installedPackages.R.indexOf('openNLP') > -1 && installedPackages.R.indexOf('openNLPmodels.en') > -1) {
                        scope.homeSearch.suggestions.enabled = true;
                    }
                }
            }


            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType[0];

                scope.homeSearch.options.loading = false;

                changeSearch();

                if (type.indexOf('ERROR') > -1) {
                    return;
                }
                scope.homeSearch.apps.raw = JSON.parse(JSON.stringify(output));
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'myApps',
                    components: [],
                    terminal: true
                }],
                listeners: [],
                response: message
            });
        }

        /**
         * @name updateSearch
         * @desc update when the search is open or closed
         * @returns {void}
         */
        function updateSearch() {
            document.removeEventListener('click', onDocumentClick, true);

            appendEle.style.visibility = 'hidden';
            appendEle.style.opacity = '0';

            if (appendEle) {
                if (appendEle.parentNode !== null) {
                    appendEle.parentNode.removeChild(appendEle);
                }
            }


            if (scope.homeSearch.open) {
                let typeaheadInputEle = appendEle.querySelector('#home-search__bar__typeahead').querySelector('input');

                setSearch();


                // append the element
                document.body.appendChild(appendEle);

                // position the element
                const targetRect = targetEle.getBoundingClientRect();
                appendEle.style.top = (window.pageYOffset + targetRect.top) + 'px';
                appendEle.style.left = (window.pageXOffset + targetRect.left) + 'px';

                // make it visible
                appendEle.style.visibility = 'visible';
                appendEle.style.opacity = '1';

                // focus on it
                $timeout(function () {
                    typeaheadInputEle.focus();
                });

                document.addEventListener('click', onDocumentClick, true);
            }
        }

        /**
         * @name changeSearch
         * @desc called when something is searched
         * @returns {void}
         */
        function changeSearch() {
            getInsights(true);
        }

        /**
         * @name onDocumentClick
         * @desc hide active if the document is clicked
         * @param {event} event dom event
         * @returns {void}
         */
        function onDocumentClick(event) {
            if (event && event.target && searchContentEle.contains(event.target)) {
                return;
            }

            $timeout(function () {
                scope.homeSearch.open = false;
            });
        }

        /**
         * @name toggleApps
         * @desc called when the apps update
         * @returns {void}
         */
        function toggleApps() {
            getInsights(true);
            if (scope.homeSearch.suggestions.active) {
                getSuggestions();
            }
        }

        /**
         * @name getInsights
         * @desc gets the insights for the current selected app
         * @param {boolean} clear - reset the search
         * @returns {void}
         */
        function getInsights(clear) {
            // clear the recommendations
            scope.homeSearch.recommendations.show = false;
            scope.homeSearch.recommendations.results = [];
            scope.homeSearch.recommendations.query = '';

            if (searchTimeout) {
                $timeout.cancel(searchTimeout);
            }

            scope.homeSearch.options.loading = true;

            searchTimeout = $timeout(function (boundClear) {
                var message,
                    filteredApps = [],
                    appIdx,
                    appLen;

                if (boundClear) {
                    scope.homeSearch.options.canCollect = true;
                    scope.homeSearch.options.offset = 0;
                }

                filteredApps = [];
                for (appIdx = 0, appLen = scope.homeSearch.apps.selected.length; appIdx < appLen; appIdx++) {
                    filteredApps.push(scope.homeSearch.apps.selected[appIdx].app_id);
                }

                message = semossCoreService.utility.random('query-pixel');

                // register message to come back to
                semossCoreService.once(message, function (response) {
                    var output = response.pixelReturn[0].output,
                        type = response.pixelReturn[0].operationType[0],
                        insightIdx = 0,
                        insightLen;

                    if (type.indexOf('ERROR') > -1) {
                        return;
                    }

                    insightLen = output.length;

                    scope.homeSearch.options.loading = false;
                    if (boundClear) {
                        scope.homeSearch.options.results = [];
                    }

                    for (; insightIdx < insightLen; insightIdx++) {
                        scope.homeSearch.options.results.push(output[insightIdx]);
                    }
                    scope.homeSearch.options.canCollect = (insightLen === scope.homeSearch.options.limit);
                });

                semossCoreService.emit('query-pixel', {
                    commandList: [{
                        meta: true,
                        type: 'getInsights',
                        components: [
                            filteredApps,
                            scope.homeSearch.options.limit,
                            scope.homeSearch.options.offset,
                            scope.homeSearch.options.input,
                            []
                        ],
                        terminal: true
                    }],
                    listeners: [],
                    response: message
                });
            }.bind(null, clear), 300);
        }

        /**
         * @name getMoreInsights
         * @desc gets the insights for the current selected app
         * @returns {void}
         */
        function getMoreInsights() {
            if (!scope.homeSearch.options.canCollect) {
                return;
            }

            // check if it is at the bottom and going downwards
            if (((resultsEle.scrollTop + resultsEle.clientHeight) >= (resultsEle.scrollHeight * 0.75)) && !scope.homeSearch.options.loading) {
                // increment the offset to get more
                scope.homeSearch.options.offset += scope.homeSearch.options.limit;
                getInsights();
            }
        }

        /**
         * @name toggleSuggestions
         * @desc toggle suggestions on/ff
         * @returns {void}
         */
        function toggleSuggestions() {
            scope.homeSearch.suggestions.active = !scope.homeSearch.suggestions.active;

            scope.homeSearch.suggestions.options = [];
            scope.homeSearch.suggestions.filteredOptions = [];

            if (scope.homeSearch.suggestions.active) {
                getSuggestions();
            }
        }

        /**
         * @name getSuggestions
         * @desc get suggestions for next word on search
         * @returns {void}
         */
        function getSuggestions() {
            if (suggestionTimeout) {
                $timeout.cancel(suggestionTimeout);
            }

            suggestionTimeout = $timeout(function () {
                var message = semossCoreService.utility.random('query-pixel'),
                    filteredApps = [],
                    appIdx,
                    appLen;

                scope.homeSearch.suggestions.options = [];
                scope.homeSearch.suggestions.filteredOptions = [];

                filteredApps = [];
                for (appIdx = 0, appLen = scope.homeSearch.apps.selected.length; appIdx < appLen; appIdx++) {
                    filteredApps.push(scope.homeSearch.apps.selected[appIdx].app_id);
                }


                semossCoreService.once(message, function (response) {
                    var output = response.pixelReturn[0].output,
                        type = response.pixelReturn[0].operationType[0],
                        suggestionIdx,
                        suggestionLen;

                    if (type.indexOf('ERROR') > -1) {
                        return;
                    }

                    // TODO: they should sent back the whole sentence, then I can display 'similar' sentences
                    scope.homeSearch.suggestions.options = [];
                    for (suggestionIdx = 0, suggestionLen = output.length; suggestionIdx < suggestionLen; suggestionIdx++) {
                        scope.homeSearch.suggestions.options.push({display: output[suggestionIdx], value: scope.homeSearch.options.input + ' ' + output[suggestionIdx]});
                    }
                    scope.homeSearch.suggestions.filteredOptions = scope.homeSearch.suggestions.options;
                });
                semossCoreService.emit('query-pixel', {
                    commandList: [{
                        meta: true,
                        type: 'nlsQueryHelper',
                        components: [
                            scope.homeSearch.options.input,
                            filteredApps,
                            true
                        ],
                        terminal: true
                    }],
                    listeners: [],
                    response: message
                });
            }, 300);
        }

        /**
         * @name getNLPSuggestedInsights
         * @param {event} e - click event
         * @desc execute query using NaturalLanguageSearch to autocreate a visualization
         * @returns {void}
         */
        function getNLPSuggestedInsights() {
            var message = semossCoreService.utility.random('query-pixel'),
                filteredApps = [],
                appIdx,
                appLen;

            // clear the recommendations
            scope.homeSearch.recommendations.show = false;
            scope.homeSearch.recommendations.results = [];
            scope.homeSearch.recommendations.query = '';
            scope.homeSearch.options.loading = true;

            filteredApps = [];
            for (appIdx = 0, appLen = scope.homeSearch.apps.selected.length; appIdx < appLen; appIdx++) {
                filteredApps.push(scope.homeSearch.apps.selected[appIdx].app_id);
            }


            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType[0];

                scope.homeSearch.options.loading = false;

                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                scope.homeSearch.recommendations.show = true;
                scope.homeSearch.recommendations.query = scope.homeSearch.options.input;
                scope.homeSearch.recommendations.results = output;

                if (scope.homeSearch.recommendations.results.length === 1) {
                    createNLPSuggestedInsight(scope.homeSearch.recommendations.results[0]);
                }
            });


            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'naturalLanguageSearch',
                    components: [
                        scope.homeSearch.options.input,
                        filteredApps,
                        true
                    ],
                    terminal: true
                }],
                listeners: [],
                response: message
            });
        }

        /**
         * @name createNLPSuggestedInsight
         * @param {object} insight - selected recommended insight to create
         * @desc begins the process to create a recommended insight
         * @returns {void}
         */
        function createNLPSuggestedInsight(insight) {
            semossCoreService.emit('open', {
                type: 'pixel',
                options: insight,
                newSheet: true
            });

            scope.homeSearch.open = false;
        }

        /**
         * @name createViz
         * @param {event} $event - DOM event
         * @param {object} insight - the selected insight to create - selected by the user from the insight list
         * @desc begins the process to create a visualization
         * @returns {void}
         */
        function createViz($event, insight) {
            var newSheet = true;

            if ($event && ($event.metaKey || $event.ctrlKey || $event.keyCode === 17 || $event.keyCode === 224)) {
                newSheet = false;
            }

            semossCoreService.emit('open', {
                type: 'insight',
                options: insight,
                newSheet: newSheet
            });

            scope.homeSearch.open = false;
        }

        /** * Utility Functions ***/
        /**
         * @name getVizSvg
         * @desc gets the svg path from visualization service
         * @param {string} layout - layout of selected viz
         * @returns {string} svg path
         */
        function getVizSvg(layout) {
            var visualizations = {
                'TreeMap': {
                    title: 'TreeMap',
                    svgSrc: require('images/treemap.svg'),
                    layout: 'TreeMap'
                },
                'Radial': {
                    title: 'Radial',
                    svgSrc: require('images/radial-standard.svg'),
                    layout: 'Radial'
                },
                'Grid': {
                    title: 'Grid',
                    svgSrc: require('images/grid-standard.svg'),
                    layout: 'Grid'
                },
                'Graph': {
                    title: 'Graph',
                    svgSrc: require('images/graph-standard.svg'),
                    layout: 'Graph'
                },
                'VivaGraph': {
                    title: 'VivaGraph',
                    svgSrc: require('images/vivagraph-standard.svg'),
                    layout: 'VivaGraph'
                },
                'Column': {
                    title: 'Bar',
                    svgSrc: require('images/bar-standard.svg'),
                    layout: 'Column'
                },
                'Map': {
                    title: 'Map',
                    svgSrc: require('images/geo-svg-icon.svg'),
                    layout: 'Map'
                },
                'Choropleth': {
                    title: 'Choropleth',
                    svgSrc: require('images/choropleth-standard.svg'),
                    layout: 'Choropleth'
                },
                'HeatMap': {
                    title: 'Heat Map',
                    svgSrc: require('images/heatmap-standard.svg'),
                    layout: 'HeatMap'
                },
                'Line': {
                    title: 'Line',
                    svgSrc: require('images/line-standard.svg'),
                    layout: 'Line'
                },
                'Pie': {
                    title: 'Pie',
                    svgSrc: require('images/pie-standard.svg'),
                    layout: 'Pie'
                },
                'Scatter': {
                    title: 'Scatter',
                    svgSrc: require('images/scatter-standard.svg'),
                    layout: 'Scatter'
                },
                'ParallelCoordinates': {
                    title: 'Parallel Coordinates',
                    svgSrc: require('images/parallel-coordinates-standard.svg'),
                    layout: 'ParallelCoordinates'
                },
                'SingleAxisCluster': {
                    title: 'Single Axis Cluster',
                    svgSrc: require('images/single-axis-standard.svg'),
                    layout: 'SingleAxisCluster'
                },
                'Cluster': {
                    title: 'Cluster',
                    svgSrc: require('images/cluster-standard.svg'),
                    layout: 'Cluster'
                },
                'Dendrogram': {
                    title: 'Dendrogram',
                    svgSrc: require('images/dendrogram-standard.svg'),
                    layout: 'Dendrogram'
                },
                'ScatterplotMatrix': {
                    title: 'Scatter Matrix',
                    svgSrc: require('images/scatter-matrix-standard.svg'),
                    layout: 'ScatterplotMatrix'
                },
                'Sunburst': {
                    title: 'Sunburst',
                    svgSrc: require('images/sunburst-standard.svg'),
                    layout: 'Sunburst'
                },
                'Gantt': {
                    title: 'Gantt',
                    svgSrc: require('images/gantt-standard.svg'),
                    layout: 'Gantt'
                },
                'Sankey': {
                    title: 'Sankey',
                    svgSrc: require('images/sankey-standard.svg'),
                    layout: 'Sankey'
                },
                'dashboard': {
                    title: 'Dashboard',
                    svgSrc: require('images/dashboard-svg-icon.svg'),
                    layout: 'dashboard'
                },
                'panel': {
                    title: 'Panel',
                    svgSrc: require('images/dashboard-svg-icon.svg'),
                    layout: 'panel'
                },
                'text-widget': {
                    title: 'Text Widget',
                    svgSrc: require('images/text-widget-icon.svg'),
                    layout: 'TextWidget'
                },
                'html-widget': {
                    title: 'HTML',
                    svgSrc: require('images/html-widget.svg'),
                    layout: 'HtmlWidget'
                },
                'Pack': {
                    title: 'Circle Pack',
                    svgSrc: require('images/pack-standard.svg'),
                    layout: 'Pack'
                },
                'Cloud': {
                    title: 'Word Cloud',
                    svgSrc: require('images/cloud-standard.svg'),
                    layout: 'Cloud'
                },
                'Clustergram': {
                    title: 'Cluster',
                    svgSrc: require('images/clustergram-standard.svg'),
                    layout: 'Clustergram'
                },
                'Area': {
                    title: 'Area Chart',
                    svgSrc: require('images/area-standard.svg'),
                    layout: 'Area'
                },
                'kClusters': {
                    title: 'K Means Clustering',
                    svgSrc: require('images/k-clusters.svg'),
                    layout: 'kClusters'
                },
                'CircleView': {
                    title: 'Circle View',
                    svgSrc: require('images/circle-view.svg'),
                    layout: 'circleView'
                },
                'BoxWhisker': {
                    title: 'Box and Whisker',
                    svgSrc: require('images/boxwhisker-standard.svg'),
                    layout: 'boxwhisker'
                },
                'DualAxis': {
                    title: 'Dual Axis',
                    svgSrc: require('images/dual-axis.svg'),
                    layout: 'DualAxis'
                },
                'Bubble': {
                    title: 'Bubble',
                    svgSrc: require('images/bubble.svg'),
                    layout: 'Bubble'
                },
                'DashboardParam': {
                    title: 'DashboardParam',
                    svgSrc: require('images/param.svg'),
                    layout: 'DashboardParam'
                },
                'DashboardFilter': {
                    title: 'DashboardFilter',
                    svgSrc: require('images/filter.svg'),
                    layout: 'DashboardFilter'
                },
                /** * SPECIFIC **/
                'prerna.ui.components.playsheets.MashupPlaySheet': {
                    title: 'Mashup',
                    svgSrc: require('images/dashboard-svg-icon.svg'),
                    layout: 'Mashup'
                },
                'prerna.ui.components.playsheets.SankeyPlaySheet': {
                    title: 'Sankey',
                    svgSrc: require('images/sankey-svg-icon.svg'),
                    layout: 'prerna.ui.components.playsheets.SankeyPlaySheet'
                },
                'prerna.ui.components.specific.tap.MHSDashboardDrillPlaysheet': {
                    title: 'statusDashboard',
                    svgSrc: require('images/blue-logo.svg'),
                    layout: 'statusDashboard'
                },
                'prerna.ui.components.specific.anthem.AnthemPainpointsPlaysheet': {
                    title: 'statusDashboard',
                    svgSrc: require('images/blue-logo.svg'),
                    layout: 'statusDashboard'
                },
                'prerna.ui.components.specific.anthem.AnthemInitiativePlaysheet': {
                    title: 'statusDashboard',
                    svgSrc: require('images/blue-logo.svg'),
                    layout: 'statusDashboard'
                },
                /** TAP Specific Visualizations **/
                // TODO: Improve extensiblity
                'SystemSimilarity': {
                    title: 'System Similarity Heat Map',
                    svgSrc: require('images/-svg-icon.svg'),
                    layout: 'SystemSimilarity'
                },
                'prerna.ui.components.specific.tap.InterfaceGraphPlaySheet': {
                    title: 'Data Network Graph',
                    svgSrc: require('images/force-svg-icon.svg'),
                    layout: 'prerna.ui.components.specific.tap.InterfaceGraphPlaySheet'
                },
                'prerna.ui.components.specific.tap.SysSiteOptPlaySheet': {
                    title: 'Portfolio Rationalization Dashboard',
                    svgSrc: require('images/-svg-icon.svg'),
                    layout: 'prerna.ui.components.specific.tap.SysSiteOptPlaySheet'
                },
                'prerna.ui.components.specific.tap.SysCoverageSheetPortRat': {
                    title: 'System Coverage',
                    svgSrc: require('images/-svg-icon.svg'),
                    layout: 'prerna.ui.components.specific.tap.SysCoverageSheetPortRat'
                },
                'prerna.ui.components.specific.tap.HealthGridSheetPortRat': {
                    title: 'Health Grid',
                    svg: require('images/scatter-svg-icon.svg'),
                    layout: 'prerna.ui.components.specific.tap.HealthGridSheet'
                },
                'prerna.ui.components.specific.tap.genesisdeployment.MHSGenesisDeploymentStrategyPlaySheet': {
                    title: 'MHS Genesis Deployment Map',
                    svgSrc: require('images/-svg-icon.svg'),
                    layout: 'prerna.ui.components.specific.tap.genesisdeployment.MHSGenesisDeploymentStrategyPlaySheet'
                },
                'prerna.ui.components.specific.tap.HealthGridSheet': {
                    title: 'Health Grid',
                    svgSrc: require('images/scatter-svg-icon.svg'),
                    layout: 'prerna.ui.components.specific.tap.HealthGridSheet'
                },
                'prerna.ui.components.specific.tap.GraphTimePlaySheet': {
                    title: 'Network Timeline',
                    svgSrc: require('images/networktime-svg-icon.svg'),
                    layout: 'prerna.ui.components.specific.tap.GraphTimePlaySheet'
                },
                /** End TAP Specific Visualizations **/
                /** OUSD **/
                /** TODO BAD**/
                'prerna.ui.components.specific.ousd.SequencingDecommissioningPlaySheet': {
                    title: 'Grid',
                    svgSrc: require('images/grid-standard.svg'),
                    layout: 'prerna.ui.components.specific.ousd.SequencingDecommissioningPlaySheet'
                },
                'prerna.ui.components.specific.ousd.RoadmapCleanTablePlaySheet': {
                    title: 'Grid',
                    svgSrc: require('images/grid-standard.svg'),
                    layout: 'prerna.ui.components.specific.ousd.SequencingDecommissioningPlaySheet'
                },
                'prerna.ui.components.specific.ousd.RoadmapTimelineStatsPlaySheet': {
                    title: 'Gantt',
                    svgSrc: require('images/chart-gantt.svg'),
                    layout: 'prerna.ui.components.specific.ousd.RoadmapTimelineStatsPlaySheet'
                },
                'prerna.ui.components.specific.iatdd.AOAQueuingDashboard': {
                    title: 'AoA Queuing Dashboard',
                    svgSrc: require('images/dashboard-svg-icon.svg'),
                    layout: 'prerna.ui.components.specific.iatdd.AOAQueuingDashboard'
                },
                /** NAVY PEO **/
                'prerna.ui.components.specific.navypeo.NavyScoreboardPlaysheet': {
                    title: 'Scorecard',
                    svgSrc: require('images/-svg-icon.svg'),
                    layout: 'prerna.ui.components.specific.navypeo.NavyScoreboardPlaysheet'
                },
                /** NLP **/
                'NLP': {
                    title: 'NLP Visualization',
                    svgSrc: require('images/nlp.svg'),
                    layout: 'Grid'
                }
            };

            if (visualizations[layout]) {
                return visualizations[layout].svgSrc;
            }
            return require('images/-svg-icon.svg');
        }

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize() {
            targetEle = ele[0].querySelector(scope.homeSearch.target);
            appendEle = ele[0].querySelector('#home-search__append');


            searchContentEle = appendEle.querySelector('#home-search__content');
            resultsEle = appendEle.querySelector('#home-search__list__insights--results');
            resultsEle.addEventListener('scroll', getMoreInsights);

            scope.$watch('homeSearch.open', updateSearch);

            scope.$on('destroy', function () {});
        }

        initialize();
    }
}
