'use strict';

import angular from 'angular';

export default angular.module('app.edit-app.directive', [])
    .directive('editApp', editAppDirective);

import './edit-app.scss';

editAppDirective.$inject = ['$q', 'monolithService', 'semossCoreService', 'ENDPOINT', 'CONFIG'];

function editAppDirective($q: ng.IQService, monolithService: MonolithService, semossCoreService: SemossCoreService, ENDPOINT: EndPoint, CONFIG: any) {
    editAppCtrl.$inject = [];
    editAppLink.$inject = ['scope'];

    return {
        restrict: 'E',
        template: require('./edit-app.directive.html'),
        bindToController: {
            open: '=',
            app: '='
        },
        controller: editAppCtrl,
        controllerAs: 'editApp',
        link: editAppLink
    };

    function editAppCtrl() { }

    function editAppLink(scope) {
        scope.editApp.CONFIG = CONFIG;

        scope.editApp.name = {
            new: '',
            old: ''
        };

        scope.editApp.description = {
            new: '',
            old: ''
        };

        scope.editApp.tags = {
            new: '',
            old: ''
        };

        scope.editApp.image = {
            flow: undefined,
            src: ''
        };

        scope.editApp.users = {
            new: [],
            old: [],
            added: {
                selection: {},
                permission: 'READ_ONLY',
                id: '',
                name: '',
                options: []
            }
        };

        scope.editApp.save = save;
        scope.editApp.cancel = cancel;
        scope.editApp.checkImage = checkImage;
        scope.editApp.deleteUsers = deleteUsers;
        scope.editApp.searchUsers = searchUsers;
        scope.editApp.addUsers = addUsers;
        scope.editApp.deleteAppImage = deleteAppImage;
        /** General */
        /**
         * @name reset
         * @desc reset everything
         */
        function reset(): void {
            if (!scope.editApp.open) {
                return;
            }

            // name
            scope.editApp.name.new = scope.editApp.app.app_name;
            scope.editApp.name.old = scope.editApp.app.app_name;

            // description
            scope.editApp.description.new = scope.editApp.app.description;
            scope.editApp.description.old = scope.editApp.app.description;

            // tags
            scope.editApp.tags.new = semossCoreService.utility.freeze(scope.editApp.app.tags);
            scope.editApp.tags.old = semossCoreService.utility.freeze(scope.editApp.app.tags);

            // image
            const imageUpdates = semossCoreService.getOptions('imageUpdates');

            if (imageUpdates[scope.editApp.app.app_id]) {
                scope.editApp.image.src = imageUpdates[scope.editApp.app.app_id];
            } else {
                scope.editApp.image.src = semossCoreService.app.generateAppImageURL(scope.editApp.app.app_id);
            }

            scope.editApp.permission = 'READ_ONLY';

            // make call to get user's permission access for this insight so 'Public' can be disabled
            monolithService.getApps(false)
                .then(function (response) {
                    if (response && response.data) {
                        for (let appIdx = 0, appLen = response.data.length; appIdx < appLen; appIdx++) {
                            if (response.data[appIdx].app_id === scope.editApp.app.app_id) {
                                if (response.data[appIdx].app_permission === 'OWNER') {
                                    scope.editApp.permission = 'AUTHOR';
                                } else if (response.data[appIdx].app_permission === 'EDIT') {
                                    scope.editApp.permission = 'EDITOR';
                                } else {
                                    scope.editApp.permission = 'READ_ONLY';
                                }

                                break;
                            }
                        }
                    }
                });

            scope.editApp.users.new = [];
            scope.editApp.users.old = [];

            monolithService.getAppUsers(false, scope.editApp.app.app_id)
                .then(function (response) {
                    let users = response.data || [];
                    scope.editApp.users.new = JSON.parse(JSON.stringify(users));
                    scope.editApp.users.old = JSON.parse(JSON.stringify(users));
                });

            searchUsers('');
        }

        /**
         * @name save
         * @desc actually save everything
         * @returns {void}
         */
        function save(): void {
            let promises: ng.IPromise<any>[] = [],
                pixelComponents: PixelCommand[] = [];

            // name
            if (scope.editApp.name.new !== scope.editApp.name.old) {
                pixelComponents.push({
                    type: 'Pixel',
                    components: [
                        `ChangeAppName(app=["${scope.editApp.app.app_id}"], name=["${scope.editApp.name.new}"])`
                    ],
                    terminal: true
                });
            }

            // description
            if (scope.editApp.description.new !== scope.editApp.description.old) {
                pixelComponents.push({
                    type: 'Pixel',
                    components: [
                        `SetAppDescription(app=["${scope.editApp.app.app_id}"], description=["${scope.editApp.description.new}"])`
                    ],
                    terminal: true
                });
            }

            // tags
            if (JSON.stringify(scope.editApp.tags.new) !== JSON.stringify(scope.editApp.tags.old)) {
                let tags = scope.editApp.tags.new;

                pixelComponents.push({
                    type: 'Pixel',
                    components: [
                        `SetAppTags(app=["${scope.editApp.app.app_id}"], tags=${JSON.stringify(tags)})`
                    ],
                    terminal: true
                });
            }

            // file
            const file = scope.editApp.image.flow && scope.editApp.image.flow.files && scope.editApp.image.flow.files.length > 0 ? scope.editApp.image.flow.files[0].file : undefined;
            if (file) {
                promises.push(monolithService.uploadAppImage(scope.editApp.app.app_id, file));
            }

            // users
            let users = {},
                mapped = {},
                added: any[] = [],
                edited: any[] = [],
                deleted: any[] = [];

            // mapp all of the ids to the index in the array
            users = {};
            scope.editApp.users.old.forEach((value, idx) => {
                users[value.id] = idx;
            });

            for (let newIdx = 0, newLen = scope.editApp.users.new.length; newIdx < newLen; newIdx++) {
                // i keep track of what was already mapped, incase there are duplicates (there shouldn't be)
                if (mapped.hasOwnProperty(scope.editApp.users.new[newIdx].id)) {
                    continue;
                }
                // is it already added?
                if (users.hasOwnProperty(scope.editApp.users.new[newIdx].id)) {
                    // check if its been edited
                    if (scope.editApp.users.new[newIdx].permission !== scope.editApp.users.old[users[scope.editApp.users.new[newIdx].id]].permission) {
                        edited.push(scope.editApp.users.new[newIdx]);
                    }


                    // remove from mapping
                    delete users[scope.editApp.users.new[newIdx].id];
                } else {
                    added.push(scope.editApp.users.new[newIdx]);
                }


                mapped[scope.editApp.users.new[newIdx].id] = true;
            }

            // these are all the removed one (leftover)
            for (let user in users) {
                if (users.hasOwnProperty(user)) {
                    deleted.push(scope.editApp.users.old[users[user]]);
                }
            }

            if (added.length !== 0 || edited.length !== 0 || deleted.length !== 0) {
                added.forEach((user) => {
                    promises.push(monolithService.addAppUserPermission(false, scope.editApp.app.app_id, user.id, user.permission));
                });

                edited.forEach((user) => {
                    promises.push(monolithService.editAppUserPermission(false, scope.editApp.app.app_id, user.id, user.permission));
                });

                deleted.forEach((user) => {
                    promises.push(monolithService.removeAppUserPermission(false, scope.editApp.app.app_id, user.id));
                });
            }


            // message pixel
            if (pixelComponents.length > 0) {
                let message = semossCoreService.utility.random('execute-pixel'),
                    deffered = $q.defer();

                semossCoreService.once(message, function (response) {
                    let type = response.pixelReturn[0].operationType;

                    if (type.indexOf('ERROR') > -1) {
                        // because it is pixel, it will resolve
                        deffered.reject();
                        return;
                    }

                    deffered.resolve();
                });

                promises.push(deffered.promise);

                semossCoreService.emit('query-pixel', {
                    commandList: pixelComponents,
                    response: message
                });
            }

            if (promises.length === 0) {
                semossCoreService.emit('alert', {
                    color: 'warn',
                    text: 'Nothing to update.'
                });
            } else {
                $q.all(promises).then(function () {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully updated app.'
                    });

                    // // update with the new information
                    // scope.editApp.insight.app_name = scope.editApp.name.new;

                    // description
                    scope.editApp.app.description = scope.editApp.description.new;

                    // tags
                    scope.editApp.app.tags = scope.editApp.tags.new;

                    // image
                    if (file) {
                        // tag the app as updated image so we can refresh it next time we get it
                        semossCoreService.setOptions('imageUpdates', scope.editApp.app.app_id, semossCoreService.app.generateAppImageURL(scope.editApp.app.app_id) + '?time=' + new Date().getTime());
                        scope.editApp.app.image = file;

                        // TODO: Fix hack.....
                        semossCoreService.emit('update-app');
                    }


                    reset();
                }, function (error) {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: error || 'Error updating app.'
                    });
                });
            }

            // close the model
            scope.editApp.open = false;
        }

        /**
         * @name cancel
         * @desc do not save things
         */
        function cancel(): void {
            scope.editApp.open = false;
        }

        /**
         * @name deleteAppImage
         * @desc delete the app image
         * @returns {void}
         */
        function deleteAppImage(): void {
            monolithService.deleteAppImage(scope.editApp.app.app_id).then(response => {
                // scope.editApp.app.last_app_image_timestamp = response.last_app_image_timestamp;
                semossCoreService.emit('alert', {
                    color: 'success',
                    text: 'App image has been deleted.'
                });
                scope.editApp.image.flow.files = [];
                // tag the app as updated image so we can refresh it next time we get it
                semossCoreService.setOptions('imageUpdates', scope.editApp.app.app_id, semossCoreService.app.generateAppImageURL(scope.editApp.app.app_id) + '?time=' + new Date().getTime());
                scope.editApp.image.src = semossCoreService.getOptions('imageUpdates')[scope.editApp.app.app_id];
                semossCoreService.emit('update-app');
            });
        }

        /** Image */
        /**
         * @name checkImage
         * @desc update and save the image
         * @param {file} file - flow file
         */
        function checkImage(file: any): boolean {
            const fileExtension = file.getExtension();
            if (fileExtension !== 'png' && fileExtension !== 'svg' && fileExtension !== 'jpg' && fileExtension !== 'jpeg') {
                semossCoreService.emit('alert', {
                    color: 'error',
                    text: 'Image must be .png, .svg, .jpg, .jpeg'
                });

                return false;
            }


            return true;
        }

        /** Users */
        /**
         * @name deleteUsers
         * @desc delete a newly added user
         * @param {string}  index - user to delete
         * @returns {void}
         */
        function deleteUsers(deletedUser: string): void {
            var index = scope.editApp.users.new.indexOf(deletedUser);
            scope.editApp.users.new.splice(index, 1);
        }

        /**
         * @name searchUsers
         * @param search - search term
         * @desc search the users
         */
        function searchUsers(search: string): void {
            scope.editApp.users.added.options = [];

            // get list of users that meet the search input
            monolithService.getUserInformation(search)
                .then(function (data) {
                    scope.editApp.users.added.options = data.map((user) => {
                        user.display = user.name + ' | ' + user.email;
                        user.selection = {'name':user.name, 'id':user.id, 'email':user.email};
                        return user;
                    });
                });
        }

        /**
         * @name addUsers
         * @desc add the users to the list
         */
        function addUsers(): void {
            if (!scope.editApp.users.added.selection) {
                semossCoreService.emit('alert', {
                    color: 'warn',
                    text: 'Please select a user.'
                });
                return;
            }
            // set both the id and name for the display
            scope.editApp.users.added.id = scope.editApp.users.added.selection.id;
            if (!scope.editApp.users.added.id) {
                semossCoreService.emit('alert', {
                    color: 'warn',
                    text: 'Please select a user.'
                });
                return;
            }
            scope.editApp.users.added.name = scope.editApp.users.added.selection.name;

            let exists = false;
            for (let newIdx = 0, newLen = scope.editApp.users.new.length; newIdx < newLen; newIdx++) {
                if (scope.editApp.users.new[newIdx] === scope.editApp.users.added.id) {
                    // update the permission
                    scope.editApp.users.new[newIdx].permission = scope.editApp.users.added.permission;
                    exists = true;
                    break;
                }
            }

            if (!exists) {
                scope.editApp.users.new.push({
                    permission: scope.editApp.users.added.permission,
                    id: scope.editApp.users.added.id,
                    name : scope.editApp.users.added.name
                });
            }

            scope.editApp.users.added.permission = 'READ_ONLY';
            scope.editApp.users.added.id = '';
        }

        /** Utility */
        /**
         * @name initialize
         * @desc initialize the module
         */
        function initialize(): void {
            reset();

            scope.$watch(function () {
                return JSON.stringify(scope.editApp.app) + '_' + scope.editApp.open;
            }, function (newValue, oldValue) {
                if (newValue !== oldValue) {
                    reset();
                }
            });

            // listeners
            scope.$on('$destroy', function () {
                console.log('destroying editApp....');
            });
        }

        initialize();
    }
}
