'use strict';

import angular from 'angular';

import {
    GRAPH_TYPES
} from '@/core/constants.js';

import './database-meta.scss';

export default angular.module('app.database-meta.directive', [])
    .directive('databaseMeta', databaseMetaDirective);

databaseMetaDirective.$inject = ['$ocLazyLoad', 'semossCoreService'];

function databaseMetaDirective($ocLazyLoad, semossCoreService: SemossCoreService) {
    databaseMetaCtrl.$inject = [];
    databaseMetaLink.$inject = ['scope', 'ele', 'attrs', 'ctrl'];

    return {
        restrict: 'E',
        template: require('./database-meta.directive.html'),
        controllerAs: 'databaseMeta',
        scope: {},
        bindToController: {
            app: '='
        },
        controller: databaseMetaCtrl,
        link: databaseMetaLink
    };

    function databaseMetaCtrl() { }

    function databaseMetaLink(scope, ele, attrs, ctrl) {
        scope.databaseMeta.hasMetamodelChanged = hasMetamodelChanged;
        scope.databaseMeta.resetMetamodel = resetMetamodel;
        scope.databaseMeta.saveMetamodel = saveMetamodel;
        scope.databaseMeta.editItem = editItem;
        scope.databaseMeta.changeColumn = changeColumn;

        scope.databaseMeta.information = {
            type: '',
            original: {
                tables: {},
                relationships: []
            },
            metamodel: { // containers rendered information
                tables: {},
                relationships: []
            },
            allTables: {},
            showEditTables: false,
            showEditRelationships: false,
            showEditTable: false,
            showEditColumn: false,
            selectedTable: false,
            selectedColumn: false
        };

        /** Metamodel */
        /**
         * @name getMetamodel
         * @desc get a copy of the metamodel
         */
        function getMetamodel(): void {
            const message = semossCoreService.utility.random('query-pixel');

            semossCoreService.once(message, function (response) {
                const output = response.pixelReturn[0].output;

                scope.databaseMeta.information = {
                    type: GRAPH_TYPES.indexOf(scope.databaseMeta.app.app_type) > -1 ? 'GRAPH' : 'RDBMS',
                    original: {
                        tables: {},
                        relationships: []
                    },
                    metamodel: {
                        tables: {},
                        relationships: []
                    },
                    allTables: {},
                    showEditTables: false,
                    showEditRelationships: false,
                    showEditTable: false,
                    showEditColumn: false,
                    selectedTable: false,
                    selectedColumn: false
                };

                // add the rendered information
                // add the relationships
                if (output.edges) {
                    for (let edgeIdx = 0, edgeLen = output.edges.length; edgeIdx < edgeLen; edgeIdx++) {
                        scope.databaseMeta.information.metamodel.relationships.push({
                            fromTable: output.edges[edgeIdx].source,
                            fromColumn: output.edges[edgeIdx].sourceColumn || '',
                            toTable: output.edges[edgeIdx].target,
                            toColumn: output.edges[edgeIdx].targetColumn || '',
                            alias: output.edges[edgeIdx].relation
                        });
                    }
                }

                // add the props
                if (output.nodes) {
                    for (let nodeIdx = 0, nodeLen = output.nodes.length; nodeIdx < nodeLen; nodeIdx++) {
                        scope.databaseMeta.information.metamodel.tables[output.nodes[nodeIdx].conceptualName] = {
                            alias: output.nodes[nodeIdx].conceptualName,
                            table: output.nodes[nodeIdx].conceptualName,
                            position: output.positions && output.positions[output.nodes[nodeIdx].conceptualName] ? output.positions[output.nodes[nodeIdx].conceptualName] : {
                                top: 0,
                                left: 0
                            },
                            columns: {}
                        };

                        const typeInformation = getTypeInformation(
                            output.dataTypes[output.nodes[nodeIdx].conceptualName],
                            output.additionalDataTypes[output.nodes[nodeIdx].conceptualName]
                        );

                        // add in the concept
                        scope.databaseMeta.information.metamodel.tables[output.nodes[nodeIdx].conceptualName].columns[output.nodes[nodeIdx].conceptualName] = {
                            alias: output.nodes[nodeIdx].conceptualName,
                            column: output.nodes[nodeIdx].conceptualName,
                            table: output.nodes[nodeIdx].conceptualName,
                            isPrimKey: true,
                            type: typeInformation.type,
                            typeFormat: typeInformation.typeFormat,
                            description: output.descriptions[output.nodes[nodeIdx].conceptualName] ? output.descriptions[output.nodes[nodeIdx].conceptualName] : '',
                            logical: output.logicalNames[output.nodes[nodeIdx].conceptualName] ? output.logicalNames[output.nodes[nodeIdx].conceptualName] : []
                        };

                        // All Tables
                        scope.databaseMeta.information.allTables[output.nodes[nodeIdx].conceptualName] = {
                            alias: output.nodes[nodeIdx].conceptualName,
                            table: output.nodes[nodeIdx].conceptualName,
                            position: {
                                top: 0,
                                left: 0
                            },
                            columns: {}
                        };

                        scope.databaseMeta.information.allTables[output.nodes[nodeIdx].conceptualName].columns[output.nodes[nodeIdx].conceptualName] = {
                            alias: output.nodes[nodeIdx].conceptualName,
                            column: output.nodes[nodeIdx].conceptualName,
                            table: output.nodes[nodeIdx].conceptualName,
                            isPrimKey: true,
                            type: typeInformation.type,
                            typeFormat: typeInformation.typeFormat,
                            description: output.descriptions[output.nodes[nodeIdx].conceptualName] ? output.descriptions[output.nodes[nodeIdx].conceptualName] : '',
                            logical: output.logicalNames[output.nodes[nodeIdx].conceptualName] ? output.logicalNames[output.nodes[nodeIdx].conceptualName] : []
                        };

                        // NOTE: There may be a PROP that is the same as the TABLE. We overwrite this prop. It is OKAY.
                        for (let propIdx = 0, propLen = output.nodes[nodeIdx].propSet.length; propIdx < propLen; propIdx++) {
                            const column = output.nodes[nodeIdx].propSet[propIdx];
                            const concept = `${output.nodes[nodeIdx].conceptualName}__${column}`;

                            const typeInformation = getTypeInformation(
                                output.dataTypes[concept],
                                output.additionalDataTypes[concept]
                            );

                            scope.databaseMeta.information.metamodel.tables[output.nodes[nodeIdx].conceptualName].columns[column] = {
                                alias: column,
                                column: column,
                                table: output.nodes[nodeIdx].conceptualName,
                                isPrimKey: false,
                                type: typeInformation.type,
                                typeFormat: typeInformation.typeFormat,
                                description: output.descriptions[concept] ? output.descriptions[concept] : '',
                                logical: output.logicalNames[concept] ? output.logicalNames[concept] : []
                            };

                            // All Tables
                            scope.databaseMeta.information.allTables[column] = {
                                alias: column,
                                table: column,
                                position: {
                                    top: 0,
                                    left: 0
                                },
                                columns: {}
                            };

                            scope.databaseMeta.information.allTables[column].columns[column] = {
                                alias: column,
                                column: column,
                                table: column,
                                isPrimKey: true,
                                type: typeInformation.type,
                                typeFormat: typeInformation.typeFormat,
                                description: output.descriptions[concept] ? output.descriptions[concept] : '',
                                logical: output.logicalNames[concept] ? output.logicalNames[concept] : []
                            };
                        }
                    }
                }

                // save copy
                scope.databaseMeta.information.original = JSON.parse(JSON.stringify(scope.databaseMeta.information.metamodel));
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    type: 'getDatabaseMetamodel',
                    components: [
                        scope.databaseMeta.app.app_id,
                        ['dataTypes', 'additionalDataTypes', 'logicalNames', 'descriptions', 'positions']
                    ],
                    terminal: true,
                    meta: true
                }],
                response: message
            });
        }

        /**
         * @name hasMetamodelChanged
         * @desc check if the metamodel has changed
         */
        function hasMetamodelChanged(): boolean {
            return JSON.stringify(scope.databaseMeta.information.original) !== JSON.stringify(scope.databaseMeta.information.metamodel);
        }

        /**
         * @name resetMetamodel
         * @desc reset the changes
         */
        function resetMetamodel(): void {
            semossCoreService.emit('alert', {
                color: 'success',
                text: 'Reset Metamodel'
            });

            scope.databaseMeta.information.metamodel = JSON.parse(JSON.stringify(scope.databaseMeta.information.original));
        }

        /**
         * @name saveMetamodel
         * @desc save a copy of the metamodel
         */
        function saveMetamodel(): void {
            let addedNodes: any[] = [],
                removedNodes: any[] = [],
                addedProperties = {},
                removedProperties = {},
                changeAlias = {},
                changeDataTypes = {},
                changeDescription = {},
                changeLogical = {},
                currentRelationships = {},
                addedRelationships: any[] = [],
                removedRelationships: any[] = [],
                originalPositions = {},
                updatedPositions = {},
                components: any[] = [];

            // compare original and current
            if (!hasMetamodelChanged()) {
                // no changes
                return;
            }

            // check if a node has been added or removed
            // check if a property has been added or removed
            // check if the properties have been updated
            // check if the relationships have added or removed
            for (let table in scope.databaseMeta.information.original.tables) {
                if (scope.databaseMeta.information.original.tables.hasOwnProperty(table)) {
                    // doesn't exist in the metamodel it is a removed node
                    if (!scope.databaseMeta.information.metamodel.tables.hasOwnProperty(table)) {
                        removedNodes.push(scope.databaseMeta.information.original.tables[table]);
                    } else {
                        // chekck what properties are removed
                        for (let column in scope.databaseMeta.information.original.tables[table].columns) {
                            if (scope.databaseMeta.information.original.tables[table].columns.hasOwnProperty(column)) {
                                if (!scope.databaseMeta.information.metamodel.tables[table].columns.hasOwnProperty(column)) {
                                    if (!removedProperties.hasOwnProperty(table)) {
                                        removedProperties[table] = [];
                                    }

                                    removedProperties[table].push(scope.databaseMeta.information.original.tables[table].columns[column]);
                                }
                            }
                        }

                        // check what properties exists now
                        for (let column in scope.databaseMeta.information.metamodel.tables[table].columns) {
                            if (scope.databaseMeta.information.metamodel.tables[table].columns.hasOwnProperty(column)) {
                                if (!scope.databaseMeta.information.original.tables[table].columns.hasOwnProperty(column)) {
                                    if (!addedProperties.hasOwnProperty(table)) {
                                        addedProperties[table] = [];
                                    }

                                    addedProperties[table].push(scope.databaseMeta.information.metamodel.tables[table].columns[column]);
                                } else {
                                    if ((scope.databaseMeta.information.original.tables[table].columns[column].type !== scope.databaseMeta.information.metamodel.tables[table].columns[column].type) || (scope.databaseMeta.information.original.tables[table].columns[column].typeFormat !== scope.databaseMeta.information.metamodel.tables[table].columns[column].typeFormat)) {
                                        if (!changeDataTypes.hasOwnProperty(table)) {
                                            changeDataTypes[table] = [];
                                        }

                                        changeDataTypes[table].push(scope.databaseMeta.information.metamodel.tables[table].columns[column]);
                                    }

                                    if (scope.databaseMeta.information.original.tables[table].columns[column].alias !== scope.databaseMeta.information.metamodel.tables[table].columns[column].alias) {
                                        if (!changeAlias.hasOwnProperty(table)) {
                                            changeAlias[table] = [];
                                        }

                                        changeAlias[table].push(scope.databaseMeta.information.metamodel.tables[table].columns[column]);
                                    }

                                    if ((scope.databaseMeta.information.original.tables[table].columns[column].description !== scope.databaseMeta.information.metamodel.tables[table].columns[column].description)) {
                                        if (!changeDescription.hasOwnProperty(table)) {
                                            changeDescription[table] = [];
                                        }

                                        changeDescription[table].push(scope.databaseMeta.information.metamodel.tables[table].columns[column]);
                                    }

                                    if ((JSON.stringify(scope.databaseMeta.information.original.tables[table].columns[column].logical) !== JSON.stringify(scope.databaseMeta.information.metamodel.tables[table].columns[column].logical))) {
                                        if (!changeLogical.hasOwnProperty(table)) {
                                            changeLogical[table] = [];
                                        }

                                        changeLogical[table].push(scope.databaseMeta.information.metamodel.tables[table].columns[column]);
                                    }
                                }
                            }
                        }
                    }

                    // capture original
                    originalPositions[table] = scope.databaseMeta.information.original.tables[table].position;
                }
            }

            // save teh positions as a map
            for (let table in scope.databaseMeta.information.metamodel.tables) {
                if (scope.databaseMeta.information.metamodel.tables.hasOwnProperty(table)) {
                    // doesn't exist in the original it is a new node
                    if (!scope.databaseMeta.information.original.tables.hasOwnProperty(table)) {
                        addedNodes.push(scope.databaseMeta.information.metamodel.tables[table]);
                    }
                }

                // capture the updated
                updatedPositions[table] = scope.databaseMeta.information.metamodel.tables[table].position;
            }

            // map the current relationships
            for (let relationshipIdx = 0, relationshipLen = scope.databaseMeta.information.metamodel.relationships.length; relationshipIdx < relationshipLen; relationshipIdx++) {
                const
                    relation = scope.databaseMeta.information.metamodel.relationships[relationshipIdx],
                    name = `${relation.fromTable}__${relation.fromColumn}.${relation.toTable}__${relation.toColumn}`;

                currentRelationships[name] = relation;
            }

            // see what was removed
            for (let relationshipIdx = 0, relationshipLen = scope.databaseMeta.information.original.relationships.length; relationshipIdx < relationshipLen; relationshipIdx++) {
                const
                    relation = scope.databaseMeta.information.original.relationships[relationshipIdx],
                    name = `${relation.fromTable}__${relation.fromColumn}.${relation.toTable}__${relation.toColumn}`;


                if (!currentRelationships[name]) {
                    removedRelationships.push(relation);
                } else {
                    delete currentRelationships[name];
                }
            }

            // see what was added
            for (let name in currentRelationships) {
                if (currentRelationships.hasOwnProperty(name)) {
                    addedRelationships.push(currentRelationships[name]);
                }
            }

            // remove, add, modify (but do it out in - highest level -> lowest -> highest)
            if (removedRelationships.length > 0) {
                const
                    startTable: any[] = [],
                    startColumn: any[] = [],
                    endTable: any[] = [],
                    endColumn: any[] = [];

                for (let relationshipIdx = 0, relationshipLen = removedRelationships.length; relationshipIdx < relationshipLen; relationshipIdx++) {
                    const relation = removedRelationships[relationshipIdx];

                    startTable.push(relation.fromTable);
                    startColumn.push(relation.fromColumn || relation.fromTable);

                    endTable.push(relation.toTable);
                    endColumn.push(relation.toColumn || relation.toTable);
                }

                components.push({
                    'type': 'removeOwlRelationship',
                    'components': [
                        scope.databaseMeta.app.app_id,
                        startTable,
                        startColumn,
                        endTable,
                        endColumn
                    ],
                    'meta': true,
                    'terminal': true
                });
            }

            // remove based on the split
            for (let table in removedProperties) {
                if (removedProperties.hasOwnProperty(table)) {
                    for (let columnIdx = 0, columnLen = removedProperties[table].length; columnIdx < columnLen; columnIdx++) {
                        components.push({
                            'type': 'removeOwlProperty',
                            'components': [
                                scope.databaseMeta.app.app_id,
                                removedProperties[table][columnIdx].table,
                                removedProperties[table][columnIdx].column
                            ],
                            'meta': true,
                            'terminal': true
                        });
                    }
                }
            }

            // remove based on the conceptual name aka the table.
            for (let conceptIdx = 0, conceptLen = removedNodes.length; conceptIdx < conceptLen; conceptIdx++) {
                components.push({
                    'type': 'removeOwlConcept',
                    'components': [
                        scope.databaseMeta.app.app_id,
                        removedNodes[conceptIdx].table
                    ],
                    'meta': true,
                    'terminal': true
                });
            }

            for (let conceptIdx = 0, conceptLen = addedNodes.length; conceptIdx < conceptLen; conceptIdx++) {
                components.push({
                    'type': 'addOwlConcept',
                    'components': [
                        scope.databaseMeta.app.app_id,
                        addedNodes[conceptIdx].table,
                        addedNodes[conceptIdx].table,
                        addedNodes[conceptIdx].columns[addedNodes[conceptIdx].table].type,
                        addedNodes[conceptIdx].columns[addedNodes[conceptIdx].table].typeFormat,
                        addedNodes[conceptIdx].columns[addedNodes[conceptIdx].table].alias,
                        addedNodes[conceptIdx].columns[addedNodes[conceptIdx].table].description,
                        addedNodes[conceptIdx].columns[addedNodes[conceptIdx].table].logical
                    ],
                    'meta': true,
                    'terminal': true
                });

                for (let column in addedNodes[conceptIdx].columns) {
                    if (addedNodes[conceptIdx].columns.hasOwnProperty(column)) {
                        if (column !== addedNodes[conceptIdx].table) {
                            components.push({
                                'type': 'addOwlProperty',
                                'components': [
                                    scope.databaseMeta.app.app_id,
                                    addedNodes[conceptIdx].table,
                                    addedNodes[conceptIdx].columns[column].column,
                                    addedNodes[conceptIdx].columns[column].type,
                                    addedNodes[conceptIdx].columns[column].typeFormat,
                                    addedNodes[conceptIdx].columns[column].alias,
                                    addedNodes[conceptIdx].columns[column].description,
                                    addedNodes[conceptIdx].columns[column].logical
                                ],
                                'meta': true,
                                'terminal': true
                            });
                        }
                    }
                }
            }

            for (let table in addedProperties) {
                if (addedProperties.hasOwnProperty(table)) {
                    for (let columnIdx = 0, columnLen = addedProperties[table].length; columnIdx < columnLen; columnIdx++) {
                        components.push({
                            'type': 'addOwlProperty',
                            'components': [
                                scope.databaseMeta.app.app_id,
                                table,
                                addedProperties[table][columnIdx].column,
                                addedProperties[table][columnIdx].type,
                                addedProperties[table][columnIdx].typeFormat,
                                addedProperties[table][columnIdx].alias,
                                addedProperties[table][columnIdx].description,
                                addedProperties[table][columnIdx].logical
                            ],
                            'meta': true,
                            'terminal': true
                        });
                    }
                }
            }

            if (addedRelationships.length > 0) {
                const
                    startTable: any[] = [],
                    startColumn: any[] = [],
                    endTable: any[] = [],
                    endColumn: any[] = [];

                for (let relationshipIdx = 0, relationshipLen = addedRelationships.length; relationshipIdx < relationshipLen; relationshipIdx++) {
                    const relation = addedRelationships[relationshipIdx];

               
                    startTable.push(relation.fromTable);
                    startColumn.push(relation.fromColumn || relation.fromTable);

                    endTable.push(relation.toTable);
                    endColumn.push(relation.toColumn || relation.toTable);
                }

                components.push({
                    'type': 'addOwlRelationship',
                    'components': [
                        scope.databaseMeta.app.app_id,
                        startTable,
                        startColumn,
                        endTable,
                        endColumn
                    ],
                    'meta': true,
                    'terminal': true
                });
            }


            for (let table in changeDataTypes) {
                if (changeDataTypes.hasOwnProperty(table)) {
                    for (let columnIdx = 0, columnLen = changeDataTypes[table].length; columnIdx < columnLen; columnIdx++) {
                        if (changeDataTypes[table][columnIdx].isPrimKey) {
                            components.push({
                                'type': 'editOwlConceptDataType',
                                'components': [
                                    scope.databaseMeta.app.app_id,
                                    changeDataTypes[table][columnIdx].table,
                                    changeDataTypes[table][columnIdx].type,
                                    changeDataTypes[table][columnIdx].typeFormat
                                ],
                                'meta': true,
                                'terminal': true
                            });
                        } else {
                            components.push({
                                'type': 'editOwlPropertyDataType',
                                'components': [
                                    scope.databaseMeta.app.app_id,
                                    changeDataTypes[table][columnIdx].table,
                                    changeDataTypes[table][columnIdx].column,
                                    changeDataTypes[table][columnIdx].type,
                                    changeDataTypes[table][columnIdx].typeFormat
                                ],
                                'meta': true,
                                'terminal': true
                            });
                        }
                    }
                }
            }

            for (let table in changeAlias) {
                if (changeAlias.hasOwnProperty(table)) {
                    for (let columnIdx = 0, columnLen = changeAlias[table].length; columnIdx < columnLen; columnIdx++) {
                        if (changeAlias[table][columnIdx].isPrimKey) {
                            components.push({
                                'type': 'editOwlConceptConceptualName',
                                'components': [
                                    scope.databaseMeta.app.app_id,
                                    changeAlias[table][columnIdx].table,
                                    changeAlias[table][columnIdx].alias
                                ],
                                'meta': true,
                                'terminal': true
                            });
                        } else {
                            components.push({
                                'type': 'editOwlPropertyConceptualName',
                                'components': [
                                    scope.databaseMeta.app.app_id,
                                    changeAlias[table][columnIdx].table,
                                    changeAlias[table][columnIdx].column,
                                    changeAlias[table][columnIdx].alias
                                ],
                                'meta': true,
                                'terminal': true
                            });
                        }
                    }
                }
            }

            for (let table in changeDescription) {
                if (changeDescription.hasOwnProperty(table)) {
                    for (let columnIdx = 0, columnLen = changeDescription[table].length; columnIdx < columnLen; columnIdx++) {
                        if (changeDescription[table][columnIdx].isPrimKey) {
                            components.push({
                                'type': 'editOwlDescription',
                                'components': [
                                    scope.databaseMeta.app.app_id,
                                    changeDescription[table][columnIdx].table,
                                    false,
                                    changeDescription[table][columnIdx].description
                                ],
                                'meta': true,
                                'terminal': true
                            });
                        } else {
                            components.push({
                                'type': 'editOwlDescription',
                                'components': [
                                    scope.databaseMeta.app.app_id,
                                    changeDescription[table][columnIdx].table,
                                    changeDescription[table][columnIdx].column,
                                    changeDescription[table][columnIdx].description
                                ],
                                'meta': true,
                                'terminal': true
                            });
                        }
                    }
                }
            }

            for (let table in changeLogical) {
                if (changeLogical.hasOwnProperty(table)) {
                    for (let columnIdx = 0, columnLen = changeLogical[table].length; columnIdx < columnLen; columnIdx++) {
                        if (changeLogical[table][columnIdx].isPrimKey) {
                            components.push({
                                'type': 'editOwlLogicalNames',
                                'components': [
                                    scope.databaseMeta.app.app_id,
                                    changeLogical[table][columnIdx].table,
                                    false,
                                    changeLogical[table][columnIdx].logical
                                ],
                                'meta': true,
                                'terminal': true
                            });
                        } else {
                            components.push({
                                'type': 'editOwlLogicalNames',
                                'components': [
                                    scope.databaseMeta.app.app_id,
                                    changeLogical[table][columnIdx].table,
                                    changeLogical[table][columnIdx].column,
                                    changeLogical[table][columnIdx].logical
                                ],
                                'meta': true,
                                'terminal': true
                            });
                        }
                    }
                }
            }

            if (JSON.stringify(originalPositions) !== JSON.stringify(updatedPositions)) {
                components.push({
                    'type': 'saveOwlPositions',
                    'components': [
                        scope.databaseMeta.app.app_id,
                        updatedPositions
                    ],
                    'meta': true,
                    'terminal': true
                });
            }

            if (components.length === 0) {
                return;
            }

            const message = semossCoreService.utility.random('query-pixel');

            semossCoreService.once(message, function (response) {
                const type = response.pixelReturn[0].operationType;

                // if there is an error, don't reset
                if (type.indexOf('ERROR') > -1) {
                    return;
                }

                getMetamodel();
            });

            components.push({
                'type': 'syncAppWithLocalMaster',
                'components': [
                    scope.databaseMeta.app.app_id
                ],
                'meta': true,
                'terminal': true
            });

            semossCoreService.emit('query-pixel', {
                commandList: components,
                response: message
            });
        }

        /**
         * @name getTypeInformation
         * @param type - data type to set
         * @param typeFormat - typeFormat
         * @returns map - containing the type information
         */
        function getTypeInformation(type: string, typeFormat: string): { type: string, typeFormat: string } {
            let newType = type,
                newTypeFormat = typeFormat || '';

            // if (newType === 'INT' || newType === 'DOUBLE') {
            //     // ui considers int and double a type format for number,
            //     newTypeFormat = type;
            //     newType = 'NUMBER';
            // }

            if ((newType === 'DATE' || newType === 'TIMESTAMP') && !typeFormat) {
                // needs type format, must tell user
            }

            if (!newType) {
                newType = 'STRING';
                newTypeFormat = '';
            }

            return {
                type: newType,
                typeFormat: newTypeFormat
            };
        }

        /** Edit */
        /**
        /**
          * @name editItem
          * @desc selects a column to grab data about
          * @param type - type of edit (table, column)
          * @param options - options to save
         */
        function editItem(type: 'table' | 'column', options: any): void {
            if (type === 'table') {
                scope.databaseMeta.information.showEditTable = true;

                scope.databaseMeta.information.selectedTable = options.table;
            } else if (type === 'column') {
                scope.databaseMeta.information.showEditColumn = true;

                scope.databaseMeta.information.selectedTable = options.table;
                scope.databaseMeta.information.selectedColumn = options.column;
            }
        }

        /**
         * @name changeColumn
         * @param type - type that was formatted
         * @param typeFormat - new typeformat
         * @param alias - new alias for the column
         * @param description - new description for the column
         * @param logical - new logical for the column
         * @desc callback for header formatting
         */
        function changeColumn(type: string, typeFormat: string, alias: string, description: string, logical: string[]): void {
            // update the values
            scope.databaseMeta.information.metamodel.tables[scope.databaseMeta.information.selectedTable].columns[scope.databaseMeta.information.selectedColumn].type = type;
            scope.databaseMeta.information.metamodel.tables[scope.databaseMeta.information.selectedTable].columns[scope.databaseMeta.information.selectedColumn].typeFormat = typeFormat;
            scope.databaseMeta.information.metamodel.tables[scope.databaseMeta.information.selectedTable].columns[scope.databaseMeta.information.selectedColumn].alias = alias;
            scope.databaseMeta.information.metamodel.tables[scope.databaseMeta.information.selectedTable].columns[scope.databaseMeta.information.selectedColumn].description = description;
            scope.databaseMeta.information.metamodel.tables[scope.databaseMeta.information.selectedTable].columns[scope.databaseMeta.information.selectedColumn].logical = logical;

            // reset
            scope.databaseMeta.information.showEditColumn = false;
        }

        /** Utility */
        /**
         * @name initialize
         * @desc function that is called on directive load
         */
        function initialize(): void {
            getMetamodel();

            // load import chunk
            scope.databaseMeta.loadImport = false;
            import( /* webpackChunkName: "components/import" */ '../import/import.directive.js').then((module) => {
                $ocLazyLoad.load(module.default);
                scope.databaseMeta.loadImport = true;
            }).catch((err) => {
                console.error('Error: ', err);
            });
        }

        initialize();
    }
}
