'use strict';

import './app-home.scss';

export default angular.module('app.app-home.directive', [])
    .directive('appHome', appHomeDirective);


appHomeDirective.$inject = ['ENDPOINT', 'semossCoreService', '$state', '$timeout', 'PLAYGROUND', 'monolithService', 'CONFIG'];


function appHomeDirective(ENDPOINT, semossCoreService, $state, $timeout, PLAYGROUND, monolithService, CONFIG) {
    appHomeCtrl.$inject = [];
    appHomeLink.$inject = ['scope', 'ele'];

    return {
        restrict: 'E',
        template: require('./app-home.directive.html'),
        controller: appHomeCtrl,
        link: appHomeLink,
        scope: {},
        bindToController: {
            appId: '='
        },
        controllerAs: 'appHome'
    };

    function appHomeCtrl() {}

    function appHomeLink(scope, ele) {
        scope.appHome.appInfo = {};

        scope.appHome.CONFIG = CONFIG;

        scope.appHome.PLAYGROUND = PLAYGROUND;

        scope.appHome.informationTabs = [{
            name: 'Insights'
        },
        {
            name: 'Data'
        },
        {
            name: 'Collaboration'
        },
        {
            name: 'Scheduler'
        },
        {
            name: 'Settings'
        }
        ];

        scope.appHome.overlay = false;
        scope.appHome.authorsOverlay = false;
        scope.appHome.editorsOverlay = false;
        scope.appHome.viewersOverlay = false;
        scope.appHome.selectedTab = 'Insights';
        scope.appHome.deleteAppOverlay = false;
        scope.appHome.appUsers = {
            authors: [],
            editors: [],
            viewers: [],
            authorsClean: '',
            editorsClean: '',
            viewersClean: ''
        };

        scope.appHome.edit = {
            open: false,
            app: undefined
        };

        scope.appHome.toggleTab = toggleTab;
        scope.appHome.toggleOverlay = toggleOverlay;
        scope.appHome.closeApp = closeApp;
        scope.appHome.hideApp = hideApp;
        scope.appHome.deleteApp = deleteApp;
        scope.appHome.focus = focus;
        scope.appHome.openEdit = openEdit;
        scope.appHome.hideTab = hideTab;
        scope.appHome.exportApp = exportApp;
        scope.appHome.openDeleteAppOverlay = openDeleteAppOverlay;
        scope.appHome.closeDeleteAppOverlay = closeDeleteAppOverlay;


        /** Updates */
        /**
         * @name updateApp
         * @desc grab the app information
         * @returns {void}
         */
        function updateApp() {
            let currentAppId = semossCoreService.app.get('selectedApp'),
                message;

            if (scope.appHome.appId !== currentAppId) {
                return;
            }

            message = semossCoreService.utility.random('query-pixel');

            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    type = response.pixelReturn[0].operationType,
                    imageUpdates;

                if (type.indexOf('ERROR') > -1) {
                    return;
                }


                scope.appHome.appInfo = output;

                // image
                imageUpdates = semossCoreService.getOptions('imageUpdates');

                if (imageUpdates[scope.appHome.appInfo.app_id]) {
                    scope.appHome.appInfo.image = imageUpdates[scope.appHome.appInfo.app_id];
                } else {
                    scope.appHome.appInfo.image = semossCoreService.app.generateAppImageURL(scope.appHome.appInfo.app_id);
                }


                output = response.pixelReturn[1].output;
                type = response.pixelReturn[1].operationType;

                if (type.indexOf('ERROR') > -1) {
                    return;
                }


                formatUsers(output);
            });

            semossCoreService.emit('query-pixel', {
                commandList: [{
                    meta: true,
                    type: 'appInfo',
                    components: [
                        scope.appHome.appId
                    ],
                    terminal: true
                },
                {
                    meta: true,
                    type: 'appUsers',
                    components: [
                        scope.appHome.appId
                    ],
                    terminal: true
                }
                ],
                response: message
            });
        }


        /**
         * @name formatUsers
         * @desc format list of users
         * @param {array} users array of users
         * @returns {void}
         */
        function formatUsers(users) {
            var i;

            scope.appHome.appUsers = { // wipe app users to start anew
                authors: [],
                editors: [],
                viewers: [],
                authorsClean: '',
                editorsClean: '',
                viewersClean: ''
            };

            for (i = 0; i < users.length; i++) {
                if (users[i].permission === 'OWNER') {
                    scope.appHome.appUsers.authors.push(users[i]);
                    if (scope.appHome.appUsers.authors.length === 1) {
                        scope.appHome.appUsers.authorsClean += users[i].name;
                    } else {
                        scope.appHome.appUsers.authorsClean += ', ' + users[i].name;
                    }
                } else if (users[i].permission === 'EDIT') {
                    scope.appHome.appUsers.editors.push(users[i]);
                    if (scope.appHome.appUsers.editors.length === 1) {
                        scope.appHome.appUsers.editorsClean += users[i].name;
                    } else {
                        scope.appHome.appUsers.editorsClean += ', ' + users[i].name;
                    }
                } else if (users[i].permission === 'READ_ONLY') {
                    scope.appHome.appUsers.viewers.push(users[i]);
                    if (scope.appHome.appUsers.viewers.length === 1) {
                        scope.appHome.appUsers.viewersClean += users[i].name;
                    } else {
                        scope.appHome.appUsers.viewersClean += ', ' + users[i].name;
                    }
                }
            }
        }

        /**
         * @name toggleTab
         * @desc changes the active tab
         * @param {string} selectedTab - the new tab
         * @returns {void}
         */
        function toggleTab(selectedTab) {
            scope.appHome.selectedTab = selectedTab;
        }

        /** Overlay **/
        /**
         * @name toggleOverlay
         * @desc close the app
         * @returns {void}
         */
        function toggleOverlay() {
            scope.appHome.overlay = !scope.appHome.overlay;

            if (scope.appHome.overlay) {
                focus();
            }
        }

        /**
         * @name closeApp
         * @desc close the app
         * @returns {void}
         */
        function closeApp() {
            scope.appHome.overlay = false;
            $timeout(() => {
                semossCoreService.emit('close-app', {
                    appId: scope.appHome.appId
                });
            });
        }

        /**
         * @name focus
         * @desc focus on the first button in the overlay
         * @returns {void}
         */
        function focus() {
            var overlayHideEle = ele[0].querySelector('#overlay-hide'),
                overlayFocusTimeout;
            if (overlayHideEle) {
                overlayFocusTimeout = $timeout(function () {
                    overlayHideEle.focus();
                    $timeout.cancel(overlayFocusTimeout);
                });
            }
        }

        /**
         * @name hideApp
         * @desc close the app
         * @returns {void}
         */
        function hideApp() {
            scope.appHome.overlay = false;
            $timeout(() => {
                $state.go('home.landing');
            });
        }

        /**
         * @name hideTab
         * @param {*} tab name of the tab
         * @desc hides specified tab
         * @returns {boolean} true or false
         */
        function hideTab(tab) {
            var hiddenTabs = [
                'Scheduler',
                'Settings',
                'Collaboration'
            ];
            if (PLAYGROUND && hiddenTabs.indexOf(tab) > -1) {
                return true;
            }

            return false;
        }

        /** Edit */

        /**
         * @name openEdit
         * @desc open the overlay
         * @returns {void}
         */
        function openEdit() {
            scope.appHome.edit.open = true;
        }

        /** Export */
        /**
         * @name exportApp
         * @desc Generates a ZIP file containing the given database and prompts user to save
         * @return {void}
         */
        function exportApp() {
            var message = semossCoreService.utility.random('export');

            // register message to come back to
            semossCoreService.once(message, function (response) {
                var output = response.pixelReturn[0].output,
                    insightID = response.insightID;

                monolithService.downloadFile(insightID, output);

                semossCoreService.emit('alert', {
                    color: 'success',
                    text: 'Successfully exported app: ' + scope.appHome.appId
                });
            });

            semossCoreService.emit('meta-pixel', {
                insightID: 'new',
                commandList: [{
                    meta: true,
                    type: 'exportApp',
                    components: [
                        scope.appHome.appId
                    ],
                    terminal: true
                }],
                listeners: [],
                response: message
            });
        }

        /** Delete */
        /**
         * @name openDeleteAppOverlay
         * @desc open the delete overlay
         * @returns {void}
         */
        function openDeleteAppOverlay() {
            scope.appHome.deleteAppOverlay = true;
        }


        /**
         * @name closeDeleteAppOverlay
         * @desc closes the delete overlay
         * @return {void}
         */
        function closeDeleteAppOverlay() {
            scope.appHome.deleteAppOverlay = false;
        }

        /**
         * @name deleteApp
         * @desc deletes an app
         * @return {void}
         */
        function deleteApp() {
            // scope.appHome.loading = true;

            // register message to come back to
            semossCoreService.once('delete-app-end', function (response) {
                // scope.appHome.loading = false;

                if (response.success) {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'App successfully deleted'
                    });
                }
            });

            semossCoreService.emit('delete-app', {
                appId: scope.appHome.appId,
                closeApp: true
            });
        }

        /**
         * @name initialize
         * @desc initialize the module
         * @returns {void}
         */
        function initialize() {
            let appListener;

            appListener = semossCoreService.on('update-app', updateApp);

            scope.$on('$destroy', function () {
                console.log('destroying appHome....');
                appListener();
            });

            scope.appHome.securityEnabled = semossCoreService.getSecurity('securityEnabled');
            if (scope.appHome.securityEnabled) {
                // TODO bring this back once we create a new directive to handle db specific security
                // scope.appHome.informationTabs.push({
                //     name: 'Security'
                // });
            }

            updateApp();
        }

        initialize();
    }
}
