'use strict';

import './admin.scss';

import './admin-insights/admin-insights.directive.js';
import './admin-theme/admin-theme.directive.js';
import './admin-social/admin-social.directive.js';

export default angular.module('app.admin.directive', [
    'app.admin.admin-insights',
    'app.admin.admin-theme',
    'app.admin.admin-social'
]).directive('admin', adminDirective);


adminDirective.$inject = ['monolithService', '$state', 'semossCoreService', '$q', 'CONFIG'];

function adminDirective(monolithService, $state, semossCoreService, $q, CONFIG) {
    adminCtrl.$inject = [];
    adminLink.$inject = [];

    return {
        restrict: 'E',
        template: require('./admin.directive.html'),
        controller: adminCtrl,
        link: adminLink,
        scope: {},
        bindToController: {},
        controllerAs: 'admin'
    };

    function adminCtrl() {}

    function adminLink(scope, ele) {
        scope.admin.securityEnabled = true;
        scope.admin.CONFIG = CONFIG;
        scope.admin.selectedPage = 'database-permissions';
        scope.admin.groupData = [];
        scope.admin.allDBUserData = [];
        scope.admin.databaseOwnInfo = [];
        scope.admin.databaseAdminOwnInfo = [];
        scope.admin.databaseAccessInfo = [];
        scope.admin.permissionInfo = [];
        scope.admin.tempGroup = [];
        scope.admin.addGroup = [];
        scope.admin.removeGroup = [];
        scope.admin.removeUserPerm = [];
        scope.admin.addGroupPerm = [];
        scope.admin.removeGroupPerm = [];
        scope.admin.tempPerm = [];
        scope.admin.addUserPerm = [];
        scope.admin.editUserPerm = [];
        scope.admin.otherAddUser = [];
        scope.admin.otherAddGroup = [];
        scope.admin.otherRemoveUser = [];
        scope.admin.otherRemoveGroup = [];
        scope.admin.userInfo = [];
        scope.admin.togglePermission = [];
        scope.admin.userDatabase = {
            list: [],
            user: '',
            userName: ''
        };
        scope.admin.group = '';
        scope.admin.groupId = '';
        scope.admin.user = '';
        scope.admin.removeUser = false;
        scope.admin.dbPublicState = false;
        scope.admin.memberGroup = [];
        scope.admin.permissionUpdate = '';
        scope.admin.groupEditModal = false;
        scope.admin.groupAddNewModal = false;
        scope.admin.editPermissions = false;
        scope.admin.userEditModal = false;
        scope.admin.adminUser = false;
        scope.admin.loading = false;
        scope.admin.showDB = true;
        scope.admin.userGray = false;
        scope.admin.hideDB = false;
        scope.admin.userRegistrationState = false;
        scope.admin.alreadyChosen = false;
        scope.admin.editNameMode = false;
        scope.admin.editEmailMode = false;
        scope.admin.editGroupClose = false;
        scope.admin.editUserClose = false;
        scope.admin.pristine = false;
        scope.admin.showValidation = false;
        // scope.admin.totalItems = '';
        // scope.admin.currentGroup = 4;
        // scope.admin.currentDatabase = 4;
        // scope.admin.currentAdminUser = 4;
        // scope.admin.currentAdminDatabase = 4;
        // scope.admin.viewby = 8;
        // scope.admin.itemsPerPage = scope.admin.viewby;
        scope.admin.updateUser = {};
        scope.admin.addUser = {};
        scope.admin.name = '';
        scope.admin.overlay = {
            open: false
        };
        scope.admin.overlay.openAddNew = false;
        scope.admin.overlay.openEdit = false;
        scope.admin.overlay.openEditUser = false;
        scope.admin.overlay.openRemoveAccess = false;
        scope.admin.overlay.openEditDatabase = false;
        scope.admin.overlay.openAddUser = false;
        scope.admin.overlay.openUserDatabase = false;

        scope.admin.exitAdmin = exitAdmin;
        scope.admin.isAdmin = isAdmin;
        scope.admin.getAllDBUserData = getAllDBUserData;
        scope.admin.setTempGroupAddNew = setTempGroupAddNew;
        scope.admin.setTempGroupEdit = setTempGroupEdit;
        scope.admin.removeFromTempGroup = removeFromTempGroup;
        scope.admin.getUserInformation = getUserInformation;
        scope.admin.addToTempGroup = addToTempGroup;
        scope.admin.addToTempValidGroup = addToTempValidGroup;
        scope.admin.setAppVisibility = setAppVisibility;
        scope.admin.getApps = getApps;
        scope.admin.getAppUsers = getAppUsers;
        scope.admin.setTempPerm = setTempPerm;
        scope.admin.addNewPermission = addNewPermission;
        scope.admin.savePermissions = savePermissions;
        scope.admin.getDBManagerToggle = getDBManagerToggle;
        scope.admin.toggleDBManager = toggleDBManager;
        scope.admin.changeTempPerm = changeTempPerm;
        scope.admin.setDBManagerToggle = setDBManagerToggle;
        scope.admin.checkPermissionExists = checkPermissionExists;
        scope.admin.removeFromTempPerm = removeFromTempPerm;
        scope.admin.removeDBAccess = removeDBAccess;
        scope.admin.removeAdminDBAccess = removeAdminDBAccess;
        scope.admin.updateUserInfo = updateUserInfo;
        scope.admin.removeAdminUser = removeAdminUser;
        scope.admin.passwordValidate = passwordValidate;
        scope.admin.deleteDatabase = deleteDatabase;
        scope.admin.cloneUser = cloneUser;
        scope.admin.overlayTitle = overlayTitle;
        scope.admin.openOverlay = openOverlay;
        scope.admin.closeOverlay = closeOverlay;
        scope.admin.openAddNewOverlay = openAddNewOverlay;
        scope.admin.closeAddNewOverlay = closeAddNewOverlay;
        scope.admin.openAddUserOverlay = openAddUserOverlay;
        scope.admin.closeAddUserOverlay = closeAddUserOverlay;
        scope.admin.openEditOverlay = openEditOverlay;
        scope.admin.closeEditOverlay = closeEditOverlay;
        scope.admin.closeEditUserOverlay = closeEditUserOverlay;
        scope.admin.openEditUserOverlay = openEditUserOverlay;
        scope.admin.closeAccessOverlay = closeAccessOverlay;
        scope.admin.openEditDatabaseOverlay = openEditDatabaseOverlay;
        scope.admin.closeEditDatabaseOverlay = closeEditDatabaseOverlay;
        scope.admin.openUserDatabaseOverlay = openUserDatabaseOverlay;
        scope.admin.closeUserDatabaseOverlay = closeUserDatabaseOverlay;
        scope.admin.resetForm = resetForm;
        scope.admin.resetUserForm = resetUserForm;
        scope.admin.userDatabaseHide = userDatabaseHide;
        scope.admin.userGroupHide = userGroupHide;
        scope.admin.setAppGlobal = setAppGlobal;
        scope.admin.addCacUser = addCacUser;
        scope.admin.setUserDbPermission = setUserDbPermission;
        scope.admin.permissionOptions = [{
            display: 'Author',
            value: 'OWNER'
        },
        {
            display: 'Editor',
            value: 'EDIT'
        },
        {
            display: 'Read-Only',
            value: 'READ_ONLY'
        }
        ];

        /**
         * @name openEditDatabaseOverlay
         * @desc open the overlay
         * @returns {void}
         */
        function openEditDatabaseOverlay() {
            scope.admin.overlay.openEditDatabase = true;
            scope.admin.editGroupClose = false;
            scope.admin.editUserClose = false;
            scope.admin.otherAddGroup = [];
            scope.admin.otherAddUser = [];
            scope.admin.otherRemoveGroup = [];
            scope.admin.otherRemoveUser = [];
        }

        /**
         * @name closeEditDatabaseOverlay
         * @desc close the overlay
         * @returns {void}
         */
        function closeEditDatabaseOverlay() {
            scope.admin.overlay.openEditDatabase = false;
        }

        /**
         * @name openOverlay
         * @desc open the overlay
         * @returns {void}
         */
        function openOverlay() {
            scope.admin.overlay.open = true;
        }

        /**
         * @name closeOverlay
         * @desc close the overlay
         * @returns {void}
         */
        function closeOverlay() {
            scope.admin.overlay.open = false;
        }

        /**
         * @name closeAccessOverlay
         * @desc close the overlay
         * @returns {void}
         */
        function closeAccessOverlay() {
            scope.admin.overlay.openRemoveAccess = false;
        }

        /**
         * @name openAddNewOverlay
         * @desc open the overlay
         * @returns {void}
         */
        function openAddNewOverlay() {
            scope.admin.overlay.openAddNew = true;
            scope.admin.userGray = false;
        }

        /**
         * @name closeAddNewOverlay
         * @desc close the overlay
         * @returns {void}
         */
        function closeAddNewOverlay() {
            scope.admin.overlay.openAddNew = false;
        }

        /**
         * @name openAddUserOverlay
         * @desc open the overlay
         * @returns {void}
         */
        function openAddUserOverlay() {
            scope.admin.overlay.openAddUser = true;
            scope.admin.addUser = {};
        }

        /**
         * @name closeAddUserOverlay
         * @desc close the overlay
         * @returns {void}
         */
        function closeAddUserOverlay() {
            scope.admin.overlay.openAddUser = false;
        }

        /**
         * @name openUserDatabaseOverlay
         * @param {*} user the user to set in overlay
         * @desc open the overlay for user database permissions
         * @returns {void}
         */
        function openUserDatabaseOverlay(user) {
            scope.admin.overlay.openUserDatabase = true;
            scope.admin.userDatabase.user = user.id;
            scope.admin.userDatabase.userName = user.name;

            setUserDatabaseList();
        }

        /**
         * @name setUserDatabaseList
         * @desc sets the databases for this user and their permission
         * @returns {void}
         */
        function setUserDatabaseList() {
            monolithService.getAllUserDbs(scope.admin.userDatabase.user).then(function (response) {
                var appIdx, selectedIdx;

                for (appIdx = 0; appIdx < scope.admin.userDatabase.list.length; appIdx++) {
                    scope.admin.userDatabase.list[appIdx].author = false;
                    scope.admin.userDatabase.list[appIdx].editor = false;
                    scope.admin.userDatabase.list[appIdx].read_only = false;

                    scope.admin.userDatabase.list[appIdx].og_author = false;
                    scope.admin.userDatabase.list[appIdx].og_editor = false;
                    scope.admin.userDatabase.list[appIdx].og_read_only = false;
                    for (selectedIdx = 0; selectedIdx < response.length; selectedIdx++) {
                        if (scope.admin.userDatabase.list[appIdx].app_id === response[selectedIdx].app_id) {
                            // OWNER, EDIT, READ_ONLY
                            if (response[selectedIdx].app_permission === 'OWNER') {
                                scope.admin.userDatabase.list[appIdx].author = true;
                                scope.admin.userDatabase.list[appIdx].editor = false;
                                scope.admin.userDatabase.list[appIdx].read_only = false;
                            } else if (response[selectedIdx].app_permission === 'EDIT') {
                                scope.admin.userDatabase.list[appIdx].author = false;
                                scope.admin.userDatabase.list[appIdx].editor = true;
                                scope.admin.userDatabase.list[appIdx].read_only = false;
                            } else if (response[selectedIdx].app_permission === 'READ_ONLY') {
                                scope.admin.userDatabase.list[appIdx].author = false;
                                scope.admin.userDatabase.list[appIdx].editor = false;
                                scope.admin.userDatabase.list[appIdx].read_only = true;
                            }

                            // need to keep track of original values so we know whether to do add/edit/delete when values change
                            scope.admin.userDatabase.list[appIdx].og_author = scope.admin.userDatabase.list[appIdx].author;
                            scope.admin.userDatabase.list[appIdx].og_editor = scope.admin.userDatabase.list[appIdx].editor;
                            scope.admin.userDatabase.list[appIdx].og_read_only = scope.admin.userDatabase.list[appIdx].read_only;
                        }
                    }
                }
            });
        }

        /**
         * @name setUserDbPermission
         * @param {number} index the index to check
         * @param {string} permission the permission to work with
         * @desc sets the db permissions for this user
         * @returns {void}
         */
        function setUserDbPermission(index, permission) {
            var hasExistingPermissions = scope.admin.userDatabase.list[index].og_author ||
                                            scope.admin.userDatabase.list[index].og_editor ||
                                            scope.admin.userDatabase.list[index].og_read_only,
                ogValue = 'og_' + permission,
                databaseObj = scope.admin.userDatabase.list[index],
                dbPermission = '';

            if (permission === 'author') {
                dbPermission = 'OWNER';
            } else if (permission === 'editor') {
                dbPermission = 'EDIT';
            } else if (permission === 'read_only') {
                dbPermission = 'READ-ONLY';
            }

            if (hasExistingPermissions) {
                // here we will either edit the user access level or delete them from a db. if the og value is true then we will delete them
                if (databaseObj[ogValue]) {
                    // delete their user access
                    monolithService.removeAppUserPermission(true, databaseObj.app_id, scope.admin.userDatabase.user).then(function () {
                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Permission has been modified.'
                        });
                        setUserDatabaseList();
                    }, function () {
                        semossCoreService.emit('alert', {
                            color: 'error',
                            text: 'An error occurred while modifying permissions.'
                        });
                        setUserDatabaseList();
                    });
                } else {
                    // edit their user access
                    monolithService.editAppUserPermission(true, databaseObj.app_id, scope.admin.userDatabase.user, dbPermission).then(function () {
                        semossCoreService.emit('alert', {
                            color: 'success',
                            text: 'Permission has been modified.'
                        });
                        setUserDatabaseList();
                    }, function () {
                        semossCoreService.emit('alert', {
                            color: 'error',
                            text: 'An error occurred while modifying permissions.'
                        });
                        setUserDatabaseList();
                    });
                }
            } else {
                // user has no access to this db so we will add them as a user.
                monolithService.addAppUserPermission(true, databaseObj.app_id, scope.admin.userDatabase.user, dbPermission).then(function () {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Permission has been modified.'
                    });
                    setUserDatabaseList();
                }, function () {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'An error occurred while modifying permissions.'
                    });
                    setUserDatabaseList();
                });
            }
        }

        /**
         * @name openUserDatabaseOverlay
         * @param {*} user the user to set in overlay
         * @desc open the overlay for user database permissions
         * @returns {void}
         */
        function closeUserDatabaseOverlay() {
            scope.admin.overlay.openUserDatabase = false;
        }

        /**
         * @name openEditOverlay
         * @desc open the overlay
         * @returns {void}
         */
        function openEditOverlay() {
            scope.admin.overlay.openEdit = true;
            scope.admin.userGray = false;
        }

        /**
         * @name closeEditOverlay
         * @desc close the overlay
         * @returns {void}
         */
        function closeEditOverlay() {
            scope.admin.overlay.openEdit = false;
        }

        /**
         * @name openEditUserOverlay
         * @desc open the overlay
         * @returns {void}
         */
        function openEditUserOverlay() {
            scope.admin.overlay.openEditUser = true;
            scope.admin.editEmailMode = false;
            scope.admin.editNameMode = false;
            scope.admin.pristine = true;
        }

        /**
         * @name closeEditUserOverlay
         * @desc close the overlay
         * @returns {void}
         */
        function closeEditUserOverlay() {
            scope.admin.overlay.openEditUser = false;
        }

        /**
         * @name exitAdmin
         * @desc return to home page
         * @returns {void}
         */
        function exitAdmin() {
            $state.go('home.landing');
        }

        /**
         * @name isAdmin
         * @desc check if user is admin
         * @returns {void}
         */
        function isAdmin() {
            monolithService.isAdmin()
                .then(function (adminUser) {
                    scope.admin.adminUser = adminUser;
                });
        }

        /**
         * @name setDBState
         * @param {boolean} admin true or false
         * @param {string} appId - id of database
         * @param {boolean} isPublic - boolean
         * @desc set db to public to private
         * @returns {void}
         */
        function setAppGlobal(admin, appId, isPublic) {
            monolithService.setAppGlobal(admin, appId, isPublic)
                .then(function () {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'You have successfully made the app ' + (isPublic ? 'public.' : 'private.')
                    });
                }, function () {
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'An error occurred while toggling the public value.'
                    });
                });
        }

        /**
         * @name setTempPerm
         * @desc Function to set the temporary array for permissions
         * @returns {void}
         */
        function setTempPerm() {
            scope.admin.tempPerm = scope.admin.permissionInfo;
            scope.admin.addUserPerm = [];
            scope.admin.editUserPerm = [];
            scope.admin.removeUserPerm = [];
            scope.admin.addGroupPerm = [];
            scope.admin.removeGroupPerm = [];
        }

        /**
         * @name resetForm
         * @desc reset form to pristine
         * @param  {object} updateForm form
         * @returns {boolean} return bool
         */
        function resetForm(updateForm) {
            if (updateForm && scope.admin.overlay.openEditUser && scope.admin.pristine) {
                updateForm.$setPristine();
                updateForm.$setUntouched();
                scope.admin.pristine = false;
            }
            return scope.admin.overlay.openEditUser;
        }

        /**
         * @name resetUserForm
         * @desc reset form to pristine
         * @param  {object} addUserForm form
         * @returns {boolean} return bool
         */
        function resetUserForm(addUserForm) {
            if (addUserForm && scope.admin.overlay.openAddUser && scope.admin.pristine) {
                addUserForm.$setPristine();
                addUserForm.$setUntouched();
                scope.admin.pristine = false;
            }
            return scope.admin.overlay.openAddUser;
        }

        /**
         * @name getAllDBUserData
         * @desc Function to get group data from BE
         * @returns {void}
         */
        function getAllDBUserData() {
            monolithService.getAllDBUsers()
                .then(function (data) {
                    scope.admin.allDBUserData = data;
                }, function (error) {
                    console.log(error);
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Could not retrieve user information'
                    });
                });
        }

        /**
         * @name addCacUser
         * @desc function that is called to add new user
         * @param  {string} addUserForm - form name
         * @returns {void}
         */
        function addCacUser() {
            monolithService.addNewUser(scope.admin.addUser.cacId, scope.admin.addUser.admin)
                .then(function (data) {
                    console.log(data);
                    scope.admin.allDBUserData.push({
                        id: scope.admin.addUser.cacId,
                        admin: scope.admin.addUser.admin
                    });
                    scope.admin.addUser = {};
                    scope.admin.getAllDBUserData();

                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'You have successfully added new user'
                    });
                }, function () {
                    console.log('Error in update!');
                    scope.admin.getAllDBUserData();

                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Error in adding new user'
                    });
                });
        }

        /**
         * @name updateUserInfo
         * @desc function that is called to update user information
         * @param  {string} updateUserForm - form name
         * @returns {void}
         */
        function updateUserInfo(updateUserForm) {
            if (!updateUserForm.name.$dirty) {
                scope.admin.updateUser.name = '';
            }
            if (!updateUserForm.userEmail.$dirty) {
                scope.admin.updateUser.email = '';
            }
            if (!updateUserForm.userPassword.$dirty) {
                scope.admin.updateUser.password = '';
            }
            if (!updateUserForm.userAdmin.$dirty) {
                scope.admin.updateUser.admin = '';
            }
            monolithService.updateUser(scope.admin.updateUser.id, scope.admin.updateUser.name, scope.admin.updateUser.email, scope.admin.updateUser.password, scope.admin.updateUser.admin, scope.admin.updateUser.type, scope.admin.updateUser.username)
                .then(function (data) {
                    scope.admin.allDBUserData.push({
                        id: scope.admin.updateUser.id,
                        name: scope.admin.updateUser.name,
                        email: scope.admin.updateUser.email,
                        password: scope.admin.updateUser.password,
                        admin: scope.admin.updateUser.admin,
                        type: scope.admin.updateUser.type,
                        username: scope.admin.updateUser.username
                    });
                    scope.admin.updateUser = {};
                    updateUserForm.$setPristine();
                    updateUserForm.$setUntouched();
                    scope.admin.getAllDBUserData();

                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'You have successfully updated user information'
                    });
                }, function () {
                    console.log('Error in update!');
                    scope.admin.getAllDBUserData();

                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Error in update'
                    });
                });
        }

        /**
         * @name removeAdminUser
         * @desc Function to remove user
         * @param {string} userId the user id
         * @returns {void}
         */
        function removeAdminUser(userId) {
            monolithService.removeAdminUser(userId)
                .then(function () {
                    // get updated information back from BE
                    scope.admin.getAllDBUserData();
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully removed user'
                    });
                }, function (error) {
                    console.log(error);
                    // get updated information back from BE
                    scope.admin.getAllDBUserData();

                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Error in removing user'
                    });
                });
        }


        /**
         * @name passwordValidate
         * @desc function that validate password strength for user registration
         * @returns {void}
         */
        function passwordValidate() {
            scope.admin.showValidation = true;
            // When the user starts to type something inside the password field
            var inputpwd = ele[0].querySelector('#psw'),
                letter = ele[0].querySelector('#letter'),
                capital = ele[0].querySelector('#capital'),
                number = ele[0].querySelector('#number'),
                special = ele[0].querySelector('#special'),
                length = ele[0].querySelector('#length'),
                lowerCaseLetters = /[a-z]/g,
                upperCaseLetters = /[A-Z]/g,
                numbers = /[0-9]/g,
                specials = /[!@#\$%\^&\*]/g;


            // When the user clicks on the password field, show the message box
            ele[0].querySelector('#admin-message').style.display = 'block';

            if (inputpwd.value.match(lowerCaseLetters)) {
                letter.classList.remove('smss-color--error');
                letter.classList.add('smss-color--success');
            } else {
                letter.classList.remove('smss-color--success');
                letter.classList.add('smss-color--error');
            }

            if (inputpwd.value.match(upperCaseLetters)) {
                capital.classList.remove('smss-color--error');
                capital.classList.add('smss-color--success');
            } else {
                capital.classList.remove('smss-color--success');
                capital.classList.add('smss-color--error');
            }

            if (inputpwd.value.match(numbers)) {
                number.classList.remove('smss-color--error');
                number.classList.add('smss-color--success');
            } else {
                number.classList.remove('smss-color--success');
                number.classList.add('smss-color--error');
            }

            if (inputpwd.value.match(specials)) {
                special.classList.remove('smss-color--error');
                special.classList.add('smss-color--success');
            } else {
                special.classList.remove('smss-color--success');
                special.classList.add('smss-color--error');
            }
            // Validate length
            if (inputpwd.value.length >= 8) {
                length.classList.remove('smss-color--error');
                length.classList.add('smss-color--success');
            } else {
                length.classList.remove('smss-color--success');
                length.classList.add('smss-color--error');
            }
        }

        /**
         * @name removeDBAccess
         * @desc Function to remove user access for DB
         * @param {string} databaseId the database id
         * @returns {void}
         */
        function removeDBAccess(databaseId) {
            monolithService.removeDBAccess(databaseId)
                .then(function () {
                    // get updated information back from BE
                    getApps(false);

                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully removed access'
                    });
                }, function (error) {
                    console.log(error);
                    // get updated information back from BE
                    getApps(false);

                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Error removing access'
                    });
                });
        }

        /**
         * @name deleteDatabase
         * @desc deletes an app
         * @param {string} dabaseId the database id
         * @return {void}
         */
        function deleteDatabase(dabaseId) {
            scope.admin.loading = true;

            // register message to come back to
            semossCoreService.once('delete-app-end', function () {
                scope.admin.loading = false;

                semossCoreService.emit('alert', {
                    color: 'success',
                    text: 'App successfully deleted'
                });

                // TODO remove app from table list
                getApps(false);
                getApps(true);
            });

            semossCoreService.emit('delete-app', {
                appId: dabaseId,
                closeApp: false
            });
        }

        /**
         * @name removeAdminDBAccess
         * @desc Function to remove user access for DB
         * @param {string} databaseId database id
         * @returns {void}
         */
        function removeAdminDBAccess(databaseId) {
            monolithService.removeDBAccess(databaseId)
                .then(function () {
                    // get updated information back from BE
                    getApps(true);
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully removed access'
                    });
                }, function (error) {
                    console.log(error);
                    // get updated information back from BE
                    getApps(true);
                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Error removing access'
                    });
                });
        }


        /**
         * @name removeFromTempGroup
         * @desc Function to remove a user from the temporary group
         * Function adds the removed user to the object BE will receive
         * @param {object} member the member to remove
         * @returns {void}
         */
        function removeFromTempGroup(member) {
            var alreadyExists = false,
                item;
            for (item in scope.admin.memberGroup) {
                if (scope.admin.memberGroup[item].name === member.name) {
                    scope.admin.memberGroup.splice(item, 1);
                }
            }

            // add/remove to object that BE will receive

            for (item in scope.admin.addGroup) {
                if (scope.admin.addGroup[item] === member.id) {
                    alreadyExists = true;
                    scope.admin.addGroup.splice(item, 1);
                    break;
                }
            }
            if (!alreadyExists) {
                scope.admin.removeGroup.push(member.id);
            }
        }

        /**
         * @name getUserInformation
         * @desc Function to get user list from BE
         * @param {object} searchInput the search
         * @returns {void}
         */
        function getUserInformation(searchInput) {
            scope.admin.userInfo = [];

            if (searchInput) {
                scope.admin.permissionTemp.showOptions = true;
            } else {
                scope.admin.permissionTemp.showOptions = false;
            }

            // get list of users that meet the search input
            monolithService.getUserInformation(searchInput)
                .then(function (data) {
                    scope.admin.userInfo = data;
                }, function (error) {
                    console.log(error);
                    scope.admin.userInfo = [];

                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'No users found with that name'
                    });
                });
        }

        /**
         * @name addToTempGroup
         * @desc Function to add a user to the temporary group
         * Function adds the added user to the object BE will receive
         * @param {object} member the member to add
         * @returns {void}
         */
        function addToTempGroup(member) {
            var alreadyExists = false,
                item;
            scope.admin.alreadyChosen = false;
            for (item in scope.admin.memberGroup) {
                if (scope.admin.memberGroup[item].id === member.id) {
                    scope.admin.alreadyChosen = true;
                }
            }

            if (scope.admin.alreadyChosen) {
                semossCoreService.emit('alert', {
                    color: 'error',
                    text: 'User has already been added'
                });
            } else {
                scope.admin.memberGroup.push({
                    id: member.id,
                    name: member.name,
                    email: member.email
                });
                // add/remove to object that BE will receive
                for (item in scope.admin.removeGroup) {
                    if (scope.admin.removeGroup[item] === member.id) {
                        alreadyExists = true;
                        scope.admin.removeGroup.splice(item, 1);
                        break;
                    }
                }
                if (!alreadyExists) {
                    scope.admin.addGroup.push(member.id);
                }
            }
        }


        /**
         * @name userGroupHide
         * @desc Function to hide user if already added in group
         * @param {string} userId
         * @returns {boolean}
         */

        function userGroupHide(userId) {
            for (var member in scope.admin.memberGroup) {
                if (scope.admin.memberGroup[member].id === userId) {
                    return true;
                }
            }
            return false;
        }

        /**
         * @name userDatabaseHide
         * @desc Function to hide user if already added in group
         * @param {string} userId
         * @returns {boolean}
         */

        function userDatabaseHide(userId) {
            for (var user in scope.admin.tempPerm) {
                if (scope.admin.tempPerm[user].id === userId) {
                    return true;
                }
            }
            return false;
        }

        /**
         * @name addToTempValidGroup
         * @desc Function to add a user to the temporary group
         * Function adds the added user to the object BE will receive
         * @param {object} member the member to add
         * @param {string}groupId the group to add to
         * @returns {void}
         */
        function addToTempValidGroup(member, groupId) {
            scope.admin.alreadyChosen = false;

            for (var item in scope.admin.memberGroup) {
                if (scope.admin.memberGroup[item].id === member.id) {
                    scope.admin.alreadyChosen = true;
                }
            }
            if (scope.admin.alreadyChosen) {
                semossCoreService.emit('alert', {
                    color: 'error',
                    text: 'User has alreay been added'
                });
            } else {
                monolithService.checkGroupValid(member.id, groupId)
                    .then(function () {
                        scope.admin.memberGroup.push({
                            id: member.id,
                            name: member.name,
                            email: member.email
                        });
                        // add/remove to object that BE will receive
                        var alreadyExists = false;
                        for (item in scope.admin.removeGroup) {
                            if (scope.admin.removeGroup[item] === member.id) {
                                alreadyExists = true;
                                scope.admin.removeGroup.splice(item, 1);
                                break;
                            }
                        }
                        if (!alreadyExists) {
                            scope.admin.addGroup.push(member.id);
                        }
                    }, function (error) {
                        semossCoreService.emit('alert', {
                            color: 'error',
                            text: error.data
                        });
                    });
            }
        }

        /**
         * @name setTempGroupAddNew
         * @desc Function to set the temporary array for groups
         * @returns {void}
         */
        function setTempGroupAddNew() {
            scope.admin.tempGroup = [];
            scope.admin.overlay.openAddNew = true;
            if (scope.admin.overlay.openAddNew) {
                scope.admin.memberGroup = [];
            }
            scope.admin.addGroup = [];
            scope.admin.removeGroup = [];
        }

        /**
         * @name setTempGroupEdit
         * @desc Function to set the temporary array for groups
         * @returns {void}
         */
        function setTempGroupEdit() {
            scope.admin.tempGroup = [];
            scope.admin.overlay.openEdit = true;
            if (scope.admin.overlay.openEdit) {
                if (scope.admin.groupData.hasOwnProperty(scope.admin.groupId)) {
                    scope.admin.tempGroup = scope.admin.groupData[scope.admin.groupId];
                }
            }
            scope.admin.addGroup = [];
            scope.admin.removeGroup = [];
        }

        /**
         * @name overlayTitle
         * @param {string} type the edit type
         * @param {string} name name
         * @desc Function to set the overlay header
         * @returns {string} name of overlay
         */
        function overlayTitle(type, name) {
            if (type === 'EditGroup') {
                return 'Edit Mode for ' + name;
            }

            if (type === 'RemoveGroup') {
                return 'Are you sure you want to remove ' + name + ' ?';
            }

            if (type === 'RemoveDB') {
                return 'Confirm delete for ' + name + '.';
            }

            if (type === 'RemoveAccess') {
                return 'Are you sure you want to remove access for ' + name + ' ?';
            }

            if (type === 'RemoveUser') {
                return 'Are you sure you want to remove ' + name + ' ?';
            }

            if (type === 'RemoveAdminDB') {
                return 'Confirm delete for ' + name + '.';
            }
            if (type === 'EditUser') {
                return 'Edit Mode for  ' + name;
            }

            if (type === 'AddUser') {
                return 'Add User';
            }
            if (type === 'EditDatabases') {
                return 'Permissions for  ' + name;
            }

            return '';
        }

        /**
         * @name getApps
         * @param {boolean} admin admin call or regular call
         * @desc Function to get all databases I own from the BE
         * @returns {void}
         */
        function getApps(admin) {
            monolithService.getApps(admin)
                .then(function (data) {
                    var idx, tempData = data.data;

                    if (admin) {
                        tempData.forEach(function (item) {
                            item.app_global = (item.app_global);
                        });
                        scope.admin.databaseAdminOwnInfo = tempData;
                        scope.admin.userDatabase.list = semossCoreService.utility.freeze(tempData);
                    } else {
                        tempData.forEach(function (item) {
                            item.app_global = (item.app_global);
                            item.app_visibility = (item.app_visibility);
                        });

                        // changing the name to reflect what we want to show in the UI
                        for (idx = 0; idx < tempData.length; idx++) {
                            if (tempData[idx].app_permission === 'OWNER') {
                                tempData[idx].app_permission_display = 'AUTHOR';
                            } else if (tempData[idx].app_permission === 'EDIT') {
                                tempData[idx].app_permission_display = 'EDITOR';
                            } else if (tempData[idx].app_permission === 'READ_ONLY') {
                                tempData[idx].app_permission_display = tempData[idx].app_permission;
                            }
                        }

                        scope.admin.databaseOwnInfo = tempData;
                    }
                }, function (error) {
                    console.log(error);

                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Could not retrieve database information'
                    });
                });
        }

        /**
         * @name setAppVisibility
         * @param {string} databaseId the database id
         * @param {boolean} visibility visibility of cup
         * @desc Function to set visibility for databases
         * @returns {void}
         */
        function setAppVisibility(databaseId, visibility) {
            monolithService.setAppVisibility(databaseId, visibility)
                .then(function (data) {
                    console.log(data);
                    getApps(false);
                }, function (error) {
                    console.log(error);
                    getApps(false);

                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Could not update database visibility!'
                    });
                });
        }

        /**
         * @name cloneUser
         * @desc do not update values until click on update button
         * @param  {object} dbUser the db user
         * @returns {void}
         */
        function cloneUser(dbUser) {
            scope.admin.updateUser = angular.copy(dbUser);
            scope.admin.updateUser.password = '';
        }

        /**
         * @name toggleDBManager
         * @desc Function to toggle the DB manager dropdown
         * @returns {void}
         */
        function toggleDBManager() {
            var value = !(scope.admin.togglePermission);
            scope.admin.togglePermission = value;
        }

        /**
         * @name getDBManagerToggle
         * @desc Function to get toggle boolean for the DB manager dropdown
         * @returns {string} tottle persmission
         */
        function getDBManagerToggle() {
            return scope.admin.togglePermission;
        }

        /**
         * @name setDBManagerToggle
         * @desc Function to set the toggle for the DB manager dropdown
         * @param {string} value value
         * @returns {void}
         */
        function setDBManagerToggle(value) {
            scope.admin.togglePermission = value;
        }

        /**
         * @name getAppUsers
         * @param {boolean} admin admin making the call
         * @desc Function to get permission info for a specific DB from BE
         * @param {string} databaseId database id
         * @returns {void}
         */
        function getAppUsers(admin, databaseId) {
            scope.admin.permissionInfo = [];
            scope.admin.togglePermission = [];

            monolithService.getAppUsers(admin, databaseId)
                .then(function (data) {
                    scope.admin.permissionInfo = data.data;
                    scope.admin.togglePermission = {
                        users: true,
                        groups: true
                    };
                    scope.admin.setTempPerm();
                }, function (error) {
                    console.log(error);
                    scope.admin.permissionInfo = [];
                    scope.admin.togglePermission = [];
                    scope.admin.setTempPerm();

                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'Could not retrieve database permission information'
                    });
                });
        }

        /**
         * @name addNewPermission
         * @desc Function to add a new user or group permission to the temporary object
         * Function adds the new permission to the object BE will receive
         * @param {string} databaseId database id
         * @param {string} id user id
         * @param {string} name the name of user
         * @param {string} permission permission to give
         * @returns {void}
         */
        function addNewPermission(databaseId, id, name, permission) {
            scope.admin.tempPerm.push({
                id: id,
                name: name,
                permission: permission,
                new: true
            });

            scope.admin.addUserPerm.push({
                id: id,
                name: name,
                permission: permission
            });
        }

        /**
         * @name removeFromTempPerm
         * @desc Function to remove a user or group permission from the temporary object
         * Function adds the removed user to the object BE will receive
         * @param {string} permission permission
         * @returns {void}
         */
        function removeFromTempPerm(permission) {
            var addIdx, tempPermission = semossCoreService.utility.freeze(permission),
                permIdx, editIdx;
            if (tempPermission.new) {
                for (addIdx = 0; addIdx < scope.admin.addUserPerm.length; addIdx++) {
                    if (scope.admin.addUserPerm[addIdx].idx === tempPermission.idx) {
                        scope.admin.addUserPerm.splice(addIdx, 1);
                        break;
                    }
                }
            } else {
                tempPermission.new = true;
                scope.admin.removeUserPerm.push(tempPermission);
            }

            for (editIdx = 0; editIdx < scope.admin.editUserPerm.length; editIdx++) {
                if (scope.admin.editUserPerm[editIdx].id === tempPermission.id) {
                    scope.admin.editUserPerm.splice(editIdx, 1);
                    break;
                }
            }

            for (permIdx = 0; permIdx < scope.admin.tempPerm.length; permIdx++) {
                if (scope.admin.tempPerm[permIdx].id === tempPermission.id) {
                    scope.admin.tempPerm.splice(permIdx, 1);
                    break;
                }
            }
        }

        /**
         * @name changeTempPerm
         * @desc Function to change the permission of a user or group in the temporary object
         * Function adds the old permission to removal object and adds the new permission to adding object for BE
         * @param {string} permission permission
         * @returns {void}
         */
        function changeTempPerm(permission) {
            var addIdx;
            if (permission.new) {
                for (addIdx = 0; addIdx < scope.admin.addUserPerm.length; addIdx++) {
                    if (scope.admin.addUserPerm[addIdx].id === permission.id) {
                        scope.admin.addUserPerm[addIdx].permission = permission.permission;
                    }
                }
            } else {
                scope.admin.editUserPerm.push(permission);
            }
        }

        /**
         * @name checkPermissionExists
         * @desc Function to check whether object exists as a permission already
         * @param {string} option the option
         * @returns {boolean} the boolean to return
         */
        function checkPermissionExists(option) {
            scope.admin.alreadyChosen = false;
            for (var item in scope.admin.tempPerm) {
                if (scope.admin.tempPerm[item].id === option) {
                    scope.admin.alreadyChosen = true;

                    semossCoreService.emit('alert', {
                        color: 'error',
                        text: 'User has already been added'
                    });
                }
            }
            return scope.admin.alreadyChosen;
        }

        /**
         * @name savePermissions
         * @param {boolean} admin if admin made the call
         * @param {string} databaseId the database id
         * @desc Function to save any changes in DB permissions
         * Function will send object to BE for updates
         * @returns {void}
         */
        function savePermissions(admin, databaseId) {
            var addIdx, editIdx, removeIdx,
                queuedAdd = [],
                queuedRemove = [],
                queuedEdit = [];

            for (addIdx = 0; addIdx < scope.admin.addUserPerm.length; addIdx++) {
                queuedAdd.push(monolithService.addAppUserPermission(admin, databaseId, scope.admin.addUserPerm[addIdx].id, scope.admin.addUserPerm[addIdx].permission));
            }

            for (editIdx = 0; editIdx < scope.admin.editUserPerm.length; editIdx++) {
                queuedEdit.push(monolithService.editAppUserPermission(admin, databaseId, scope.admin.editUserPerm[editIdx].id, scope.admin.editUserPerm[editIdx].permission));
            }

            for (removeIdx = 0; removeIdx < scope.admin.removeUserPerm.length; removeIdx++) {
                queuedRemove.push(monolithService.removeAppUserPermission(admin, databaseId, scope.admin.removeUserPerm[removeIdx].id));
            }

            if (queuedAdd.length > 0) {
                $q.all(queuedAdd).then(function () {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully added permission information for the app.'
                    });
                });
            }

            if (queuedEdit.length > 0) {
                $q.all(queuedEdit).then(function () {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully edited permission information for the app.'
                    });
                });
            }

            if (queuedRemove.length > 0) {
                $q.all(queuedRemove).then(function () {
                    semossCoreService.emit('alert', {
                        color: 'success',
                        text: 'Successfully removed permission information for the app.'
                    });
                });
            }

            scope.admin.addUserPerm = [];
            scope.admin.editUserPerm = [];
            scope.admin.removeUserPerm = [];
            scope.admin.tempPerm = [];
            scope.admin.permissionInfo = [];
            scope.admin.togglePermission = [];
        }

        /** Initialize */
        /**
         * @name initialize
         * @desc function that is called on directive load
         * @returns {void}
         */
        function initialize() {
            semossCoreService.isSecurityEnabled().then(function (response) {
                scope.admin.securityEnabled = response;
                if (scope.admin.securityEnabled) {
                    scope.admin.selectedPage = 'database-permissions';
                    getApps(false);
                    isAdmin();
                } else {
                    scope.admin.selectedPage = 'global-themes';
                }
            });
        }

        initialize();

        // cleanup
        scope.$on('$destroy', function () {
            console.log('destroying admin...');
        });
    }
}
