// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.openqa.selenium.htmlunit;

import java.io.IOException;

import org.openqa.selenium.Keys;
import org.openqa.selenium.WebDriverException;
import org.openqa.selenium.WebElement;

import com.gargoylesoftware.htmlunit.html.HtmlElement;
import com.gargoylesoftware.htmlunit.html.HtmlFileInput;
import com.gargoylesoftware.htmlunit.html.Keyboard;

/**
 * Implements keyboard operations using the HtmlUnit WebDriver.
 *
 */
public class HtmlUnitKeyboard implements org.openqa.selenium.interactions.Keyboard {
  private KeyboardModifiersState modifiersState = new KeyboardModifiersState();
  private final HtmlUnitDriver parent;
  private HtmlElement lastElement;

  HtmlUnitKeyboard(HtmlUnitDriver parent) {
    this.parent = parent;
  }

  private HtmlUnitWebElement getElementToSend(WebElement toElement) {
    WebElement sendToElement = toElement;
    if (sendToElement == null) {
      sendToElement = parent.switchTo().activeElement();
    }

    return (HtmlUnitWebElement) sendToElement;
  }

  @Override
  public void sendKeys(CharSequence... keysToSend) {
    WebElement toElement = parent.switchTo().activeElement();

    HtmlUnitWebElement htmlElem = getElementToSend(toElement);
    htmlElem.sendKeys(false, keysToSend);
  }

  public void sendKeys(HtmlElement element, String currentValue, InputKeysContainer keysToSend,
      boolean releaseAllAtEnd) {
    keysToSend.setCapitalization(modifiersState.isShiftPressed());

    if (parent.isJavascriptEnabled() && !(element instanceof HtmlFileInput)) {
      try {
        String keysSequence = keysToSend.toString();
        Keyboard keyboard = asHtmlUnitKeyboard(lastElement != element, keysSequence, true);
        if (releaseAllAtEnd) {
          if (isShiftPressed()) {
            addToKeyboard(keyboard, Keys.SHIFT.charAt(0), false);
          }
          if (isAltPressed()) {
            addToKeyboard(keyboard, Keys.ALT.charAt(0), false);
          }
          if (isCtrlPressed()) {
            addToKeyboard(keyboard, Keys.CONTROL.charAt(0), false);
          }
        }
        element.type(keyboard);
      } catch (IOException e) {
        throw new WebDriverException(e);
      }
    }
    lastElement = element;
  }

  private Keyboard asHtmlUnitKeyboard(final boolean startAtEnd, final CharSequence keysSequence,
      final boolean isPress) {
    Keyboard keyboard = new Keyboard(startAtEnd);
    for (int i = 0; i < keysSequence.length(); i++) {
      char ch = keysSequence.charAt(i);
      addToKeyboard(keyboard, ch, isPress);
    }
    return keyboard;
  }

  private void addToKeyboard(final Keyboard keyboard, char ch, final boolean isPress) {
    if (HtmlUnitKeyboardMapping.isSpecialKey(ch)) {
      int keyCode = HtmlUnitKeyboardMapping.getKeysMapping(ch);
      if (isPress) {
        keyboard.press(keyCode);
        modifiersState.storeKeyDown(ch);
      }
      else {
        keyboard.release(keyCode);
        modifiersState.storeKeyUp(ch);
      }
    }
    else {
      keyboard.type(ch);
    }
  }

  @Override
  public void pressKey(CharSequence keyToPress) {
    WebElement toElement = parent.switchTo().activeElement();

    HtmlUnitWebElement htmlElement = getElementToSend(toElement);
    HtmlElement element = (HtmlElement) htmlElement.element;
    try {
      element.type(asHtmlUnitKeyboard(lastElement != element, keyToPress, true));
    } catch (IOException e) {
      throw new WebDriverException(e);
    }
  }

  @Override
  public void releaseKey(CharSequence keyToRelease) {
    WebElement toElement = parent.switchTo().activeElement();

    HtmlUnitWebElement htmlElement = getElementToSend(toElement);
    HtmlElement element = (HtmlElement) htmlElement.element;
    try {
      element.type(asHtmlUnitKeyboard(lastElement != element, keyToRelease, false));
    } catch (IOException e) {
      throw new WebDriverException(e);
    }
  }

  public boolean isShiftPressed() {
    return modifiersState.isShiftPressed();
  }

  public boolean isCtrlPressed() {
    return modifiersState.isCtrlPressed();
  }

  public boolean isAltPressed() {
    return modifiersState.isAltPressed();
  }

}
